/*
  MaemoRate - Rate maemo packages
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Upload.h"

#include <cstring>

#include <curl/curl.h>

#include <iostream>

#include <Lum/Base/String.h>

#define ERROR_START    "<div id=\"error\">"
#define REALNAME_START "<a href=\"/profile/edit/\">"

static wchar_t characters[16]= {L'0',
                                L'1',
                                L'2',
                                L'3',
                                L'4',
                                L'5',
                                L'6',
                                L'7',
                                L'8',
                                L'9',
                                L'A',
                                L'B',
                                L'C',
                                L'D',
                                L'E',
                                L'F'
                              };

static std::wstring URLEncode(const std::wstring& text)
{
  std::wstring result(text);
  size_t       i;

  i=0;
  while (i<result.length()) {
    if (result[i]>255) {
      result.erase(i,1);
    }
    else if (result[i]==L' ') {
      result[i]=L'+';
      i++;
    }
    else if (!((result[i]>=L'a'&& result[i]<=L'z') ||
               (result[i]>=L'A'&& result[i]<=L'Z') ||
               (result[i]>=L'0'&& result[i]<=L'9') ||
               result[i]==L'-' ||
               result[i]==L'_' ||
               result[i]==L'.' ||
               result[i]==L'!' ||
               result[i]==L'~' ||
               result[i]==L'*' ||
               result[i]==L'\'' ||
               result[i]==L'(' ||
               result[i]==L')')) {
      result.insert(i+1,1,characters[result[i]/16]);
      result.insert(i+2,1,characters[result[i]%16]);
      result[i]=L'%';
      i+=3;
    }
    else {
      i++;
    }
  }

  return result;
}

size_t WriteFunction(void* ptr, size_t size, size_t nmemb, void* stream)
{
  UploadTask* task=static_cast<UploadTask*>(stream);

  task->AppendToBuffer((const char*)ptr,size*nmemb);

  return size*nmemb;
}

UploadTask::UploadTask()
{
  // no code
}

void UploadTask::SetLogin(const std::wstring& username, const std::wstring& password)
{
  this->username=username;
  this->password=password;
}

void UploadTask::SetJobs(const std::list<Rating>& jobs)
{
  this->jobs=jobs;
}

void UploadTask::Run()
{
  CURL        *curl;
  CURLcode    result;
  size_t      steps=jobs.size()+1;
  size_t      currentStep=0;
  size_t      a,b;
  std::string url;
  std::string realname;

  std::cout << "Steps: " << steps << std::endl;

  error.clear();

  curl=curl_easy_init();

  if (curl==NULL) {
    error="Cannot initiale curl handle!";
    return;
  }

  result=curl_easy_setopt(curl,CURLOPT_VERBOSE,1);
  if (result!=CURLE_OK) {
    error=curl_easy_strerror(result);
    curl_easy_cleanup(curl);
    return;
  }

  result=curl_easy_setopt(curl,CURLOPT_NOSIGNAL,1);
  if (result!=CURLE_OK) {
    error=curl_easy_strerror(result);
    curl_easy_cleanup(curl);
    return;
  }

  result=curl_easy_setopt(curl,CURLOPT_SSL_VERIFYPEER,0);
  if (result!=CURLE_OK) {
    error=curl_easy_strerror(result);
    curl_easy_cleanup(curl);
    return;
  }

  result=curl_easy_setopt(curl,CURLOPT_COOKIEFILE,"");
  if (result!=CURLE_OK) {
    error=curl_easy_strerror(result);
    curl_easy_cleanup(curl);
    return;
  }

  result=curl_easy_setopt(curl,CURLOPT_WRITEFUNCTION,WriteFunction);
  if (result!=CURLE_OK) {
    error=curl_easy_strerror(result);
    curl_easy_cleanup(curl);
    return;
  }

  result=curl_easy_setopt(curl,CURLOPT_WRITEDATA,this);
  if (result!=CURLE_OK) {
    error=curl_easy_strerror(result);
    curl_easy_cleanup(curl);
    return;
  }

  url=std::string("https://downloads.maemo.org/")+Lum::Base::WStringToString(ossoVersion);
  result=curl_easy_setopt(curl,CURLOPT_URL,url.c_str());
  if (result!=CURLE_OK) {
    error=curl_easy_strerror(result);
    curl_easy_cleanup(curl);
    return;
  }

  std::wstring params=L"username="+URLEncode(username)+L"&password="+URLEncode(password)+L"&midcom_services_auth_frontend_form_submit=Login";
  std::string tmp=Lum::Base::WStringToString(params);

  std::cout << "POST URL is: '" << tmp << "'" << std::endl;

  curl_easy_setopt(curl,CURLOPT_POSTFIELDS,tmp.c_str());
  if (result!=CURLE_OK) {
    error=curl_easy_strerror(result);
    curl_easy_cleanup(curl);
    return;
  }

  buffer.clear();

  SetAction(L"Logging in...");

  if (curl_easy_perform(curl)==CURLE_OK) {
    std::cout << buffer << std::endl;

    if (buffer.find("midcom_services_auth_frontend_form_submit")==std::string::npos) {
      SetAction(L"Logged in.");
      SetProgress(20);
    }
    else {
      a=buffer.find(ERROR_START);
      if (a==std::string::npos) {
        std::cout << buffer << std::endl;
        error="Unable to log in: Unknown error!";
        curl_easy_cleanup(curl);
        return;
      }

      a+=strlen(ERROR_START);

      b=buffer.find("<",a);
      if (b==std::string::npos) {
        std::cout << buffer << std::endl;
        error="Unable to log in: Unknown error!";
        curl_easy_cleanup(curl);
        return;
      }

      error="Unable to log in: '"+buffer.substr(a,b-a)+"'";
      curl_easy_cleanup(curl);
      return;
    }
  }
  else {
    std::cout << buffer << std::endl;
    error=curl_easy_strerror(result);
    curl_easy_cleanup(curl);
    return;
  }

  a=buffer.find(REALNAME_START);
  if (a==std::string::npos) {
    std::cout << buffer << std::endl;
    error="Unable to log in: Unknown error!";
    curl_easy_cleanup(curl);
    return;
  }

  a+=strlen(REALNAME_START);

  b=buffer.find("</a",a);
  if (b==std::string::npos) {
    std::cout << buffer << std::endl;
    error="Unable to log in: Unknown error!";
    curl_easy_cleanup(curl);
    return;
  }

  realname=buffer.substr(a,b-a);

  currentStep++;
  SetProgress((100*currentStep)/steps);

  SetAction(L"Rating application'rivers'...");

  for (std::list<Rating>::iterator iter=jobs.begin(); iter!=jobs.end(); ++iter) {
    SetAction(L"Uploading rating for application '"+iter->package+L"'...");

    url=std::string("https://downloads.maemo.org/product/")+
        Lum::Base::WStringToString(ossoVersion)+"/"+
        Lum::Base::WStringToString(iter->package);
    result=curl_easy_setopt(curl,CURLOPT_URL,url.c_str());
    if (result!=CURLE_OK) {
      error=curl_easy_strerror(result);
      curl_easy_cleanup(curl);
      return;
    }

    params=std::wstring(L"_qf__net_nehmer_comments=")+
           L"&author="+URLEncode(Lum::Base::StringToWString(realname))+
           std::wstring(L"&rating=")+URLEncode(Lum::Base::NumberToWString(iter->score))+
           L"&content="+URLEncode(iter->comment)+
           L"&midcom_helper_datamanager2_save=Save";
    tmp=Lum::Base::WStringToString(params);

    std::cout << "POST URL is: '" << tmp << "'" << std::endl;

    curl_easy_setopt(curl,CURLOPT_POSTFIELDS,tmp.c_str());
    if (result!=CURLE_OK) {
      error=curl_easy_strerror(result);
      curl_easy_cleanup(curl);
      return;
    }

    buffer.clear();

    if (curl_easy_perform(curl)!=CURLE_OK) {
      std::cout << buffer << std::endl;
      error=curl_easy_strerror(result);
      curl_easy_cleanup(curl);
      return;
    }

    if (buffer.find("<h2>Comments</h2>")) {
      std::cout << buffer << std::endl;
      error="Unable to log in: Looks like rating did not succeed!";
      curl_easy_cleanup(curl);
      return;
    }

    iter->pushed=true;
    currentStep++;
    SetProgress((100*currentStep)/steps);
  }

  SetAction(L"Done.");
  SetProgress(100);

  curl_easy_cleanup(curl);
}

void UploadTask::AppendToBuffer(const char* text, size_t size)
{
  buffer.append(text,size);
}

bool UploadTask::HasError() const
{
  return !error.empty();
}

std::wstring UploadTask::GetError() const
{
  return Lum::Base::StringToWString(error);
}

void UploadTask::GetFinishedJobs(std::set<Rating>& finishedJobs) const
{
  finishedJobs.clear();

  for (std::list<Rating>::const_iterator iter=jobs.begin(); iter!=jobs.end(); ++iter) {
    if (iter->pushed) {
      finishedJobs.insert(*iter);
    }
  }
}

