/*
  MaemoRate - Rate maemo packages
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <curl/curl.h>

#include <fstream>
#include <iostream>

#include <Lum/OS/Main.h>

#include <Lum/Base/L10N.h>
#include <Lum/Base/Object.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/Msg.h>
#include <Lum/Dlg/Password.h>

#include <Lum/Images/Loader.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/Selection.h>
#include <Lum/Model/Table.h>

#include <Lum/OS/Main.h>

#include <Lum/ButtonRow.h>
#include <Lum/Dialog.h>
#include <Lum/Space.h>
#include <Lum/Table.h>
#include <Lum/Text.h>
#include <Lum/WindowGroup.h>

#include "Configuration.h"
#include "Package.h"
#include "PackageList.h"
#include "Upload.h"
#include "RateDialog.h"

#include "config.h"

static Lum::AppInfo   info;

Lum::Images::ImageRef fullStar;
Lum::Images::ImageRef emptyStar;


class MainDialog : public Lum::Dialog
{
private:
  Lum::Model::ActionRef              rateAction;
  Lum::Model::ActionRef              uploadAction;
  Lum::Model::ActionRef              aboutAction;
  Lum::Model::ListTableRef           packages;
  Lum::Model::SingleLineSelectionRef selection;
  PackageListTask                    packageListTask;
  UploadTask                         uploadTask;

public:
  MainDialog()
  : rateAction(new Lum::Model::Action()),
    uploadAction(new Lum::Model::Action()),
    aboutAction(new Lum::Model::Action()),
    packages(new Lum::Model::ListTable(2)),
    selection(new Lum::Model::SingleLineSelection()),
    packageListTask(packages)
  {
    rateAction->Disable();

    AttachModel(packageListTask.GetFinishedAction());
    AttachModel(uploadTask.GetFinishedAction());

    AttachModel(GetOpenedAction());
    AttachModel(GetClosedAction());

    AttachModel(selection);

    AttachModel(rateAction);
    AttachModel(uploadAction);
    AttachModel(aboutAction);

    DetectOssoVersion();
  }

  ~MainDialog()
  {
    UnattachModel(aboutAction);
    UnattachModel(uploadAction);
    UnattachModel(rateAction);

    UnattachModel(selection);

    UnattachModel(GetClosedAction());
    UnattachModel(GetOpenedAction());

    UnattachModel(uploadTask.GetFinishedAction());
    UnattachModel(packageListTask.GetFinishedAction());
  }

  virtual void PreInit()
  {
    Lum::Button           *button;
    Lum::Panel            *hpanel,*vpanel;
    Lum::Table            *table;
    Lum::Text             *text;
    Lum::WindowGroup      *wGroup;
    Lum::Model::HeaderRef headerModel;

    wGroup=new Lum::WindowGroup();
    wGroup->SetFlex(true,true);

    hpanel=new Lum::HPanel();
    hpanel->SetFlex(true,true);

    headerModel=new Lum::Model::HeaderImpl();
    headerModel->AddColumn(L"Package",Lum::Base::Size::stdCharWidth,10,true);
    headerModel->AddColumn(L"Rating",Lum::Base::Size::modePixel,100);

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetMinWidth(Lum::Base::Size::stdCharWidth,30);
    table->SetMinHeight(Lum::Base::Size::stdCharHeight,10);
    table->RequestFocus();
    table->SetModel(packages);
    table->SetHeaderModel(headerModel);
    table->SetSelection(selection);
    table->SetShowHeader(true);
    table->GetTableView()->SetAutoFitColumns(true);
    //table->GetTableView()->SetAutoHSize(true);
    hpanel->Add(table);

    hpanel->Add(new Lum::HSpace());

    vpanel=new Lum::VPanel();
    vpanel->SetFlex(false,true);

    button=new Lum::Button(L"_Rate");
    button->RequestFocus();
    button->SetFlex(true,false);
    button->SetModel(rateAction);
    vpanel->Add(button);

    vpanel->Add(new Lum::VSpace());

    button=new Lum::Button(L"_Upload rates");
    button->RequestFocus();
    button->SetFlex(true,false);
    button->SetModel(uploadAction);
    vpanel->Add(button);

    vpanel->Add(new Lum::VSpace(true));

    text=new Lum::Text();
    text->SetFlex(true,false);
    text->SetText(ossoVersion,Lum::OS::Font::bold,Lum::Text::centered);
    vpanel->Add(text);

    vpanel->Add(new Lum::VSpace(true));

    button=new Lum::Button(L"_About");
    button->RequestFocus();
    button->SetFlex(true,false);
    button->SetModel(aboutAction);
    vpanel->Add(button);

    hpanel->Add(vpanel);

    wGroup->SetMain(hpanel);

    SetTop(wGroup);

    packageListTask.SetParent(this);
    packageListTask.SetCaption(L"Reading package list...");

    uploadTask.SetParent(this);
    uploadTask.SetCaption(L"Uploading ratings...");

    Dialog::PreInit();
  }

  void DetectOssoVersion()
  {
    std::ifstream file;

    file.open("/etc/osso_software_version",std::ios::in);

    if (file) {
      std::string version;

      file >> version;
      file.close();

      if (version.find("2006")!=std::string::npos) {
        ossoVersion=L"OS2006";
      }
      else if (version.find("2007")!=std::string::npos) {
        ossoVersion=L"OS2007";
      }
      else if (version.find("2008")!=std::string::npos) {
        ossoVersion=L"OS2008";
      }
    }

    if (ossoVersion.empty()) {
      ossoVersion=L"OS2007";
    }
  }

  void RatePackage()
  {
    Package                          *package;
    Rating                           rating;
    std::set<Rating>::const_iterator iter;

    package=dynamic_cast<Package*>(packages->GetEntry(selection->GetLine()));

    rating.package=package->GetName();
    rating.version=package->GetVersion();
    rating.pushed=false;

    iter=ratings.find(rating);
    if (iter!=ratings.end()) {
      rating=*iter;
    }

    if (GetRate(this,rating)) {
      configurationChanged=true;
      if (iter!=ratings.end()) {
        ratings.erase(iter);
      }
      ratings.insert(rating);
      package->UpdateScore(rating.score);
    }
  }

  void InitiateUpload()
  {
    std::wstring                     username(::username);
    std::wstring                     password(::password);
    bool                             savePassword(!username.empty());
    std::set<Rating>::const_iterator iter;
    std::list<Rating>                jobs;

    for (std::set<Rating>::const_iterator iter=ratings.begin(); iter!=ratings.end(); ++iter) {
      if (!iter->pushed && !iter->comment.empty()) {
        jobs.push_back(*iter);
      }
    }

    if (jobs.size()==0) {
      Lum::Dlg::Msg::ShowOk(this,
                            L"Nothing to upload!",
                            L"There are currently no new, completed ratings to upload!");
      return;
    }

    if (Lum::Dlg::Password::GetPassword(this->GetWindow(),
                                        L"Please enter password...",
                                        L"Please enter your login information for the\n"
                                        L"Maemo downloads portal:",
                                        L"Passwords will be stored uncrypted!",
                                        username,password,savePassword)) {
      rateAction->Disable();
      uploadAction->Disable();
      packages->Disable();
      uploadTask.SetLogin(username,password);
      uploadTask.SetJobs(jobs);
      uploadTask.Start();

      if (!savePassword) {
        ::username.clear();
        ::password.clear();
        configurationChanged=true;
      }
      else if (username!=::username || password!=::password) {
        ::username=username;
        ::password=password;
        configurationChanged=true;
      }
    }
  }

  void EvaluateUploadResult()
  {
    std::set<Rating> jobs;
    size_t           i;

    if (uploadTask.HasError()) {
      Lum::Dlg::Msg::ShowOk(this,
                            L"Error while upload!",
                            L"There was an error during upload:\n"+
                            uploadTask.GetError());
    }

    uploadTask.GetFinishedJobs(jobs);

    selection->Clear();

    packages->Off();

    // Remove dall uploaded packages from list!

    i=1;
    while (i<=packages->GetRows()) {
      std::set<Rating>::const_iterator iter;
      Package                          *package;
      Rating                           rating;

      package=dynamic_cast<Package*>(packages->GetEntry(i));

      rating.package=package->GetName();
      rating.version=package->GetVersion();

      iter=jobs.find(rating);
      if (iter!=jobs.end()) {
        packages->Delete(i);
      }
      else {
        i++;
      }
    }

    packages->On();

    uploadAction->Enable();
    packages->Enable();
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      if (!LoadConfig()) {
        // TODO: Some dialog?
        configurationChanged=true;
      }

      packages->Disable();
      packages->Off();
      packageListTask.Start();
    }
    else if (model==rateAction && rateAction->IsFinished()) {
      RatePackage();
    }
    else if (model==uploadAction && uploadAction->IsFinished()) {
      InitiateUpload();
    }
    else if (model==aboutAction && aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }
    else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      /*if (configurationChanged && !SaveConfig()) {
        std::cerr << "Error while storing configuration!" << std::endl;
      }*/
    }
    else if (model==selection) {
      if (selection->HasSelection()) {
        rateAction->Enable();
      }
      else {
        rateAction->Disable();
      }
    }
    else if (model==packageListTask.GetFinishedAction() && packageListTask.GetFinishedAction()->IsFinished()) {
      packages->Sort(1);
      packages->On();
      packages->Enable();
    }
    else if (model==uploadTask.GetFinishedAction() && uploadTask.GetFinishedAction()->IsFinished()) {
      EvaluateUploadResult();
    }

    Dialog::Resync(model,msg);
  }
};

class Main : public Lum::OS::MainDialog<MainDialog>
{
public:
  bool Prepare()
  {
#if defined(APP_DATADIR)
    Lum::Base::Path::SetApplicationDataDir(Lum::Base::StringToWString(APP_DATADIR));
#endif

    if (curl_global_init(CURL_GLOBAL_ALL)!=0) {
      std::cerr << "Cannot initialize curl" << std::endl;
      return false;
    }

    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(_(L"ABOUT_DESC",L"Rate the installed applications..."));
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2007, Tim Teulings");
    info.SetLicense(L"GNU Public License");

    if (Lum::OS::MainDialog<MainDialog>::Prepare()) {
      fullStar=Lum::Images::Factory::factory->CreateImage();
      if (!Lum::Images::loader->Load(L"images/orange_star.png",fullStar)) {
        Lum::Base::Path path;

	path.SetNativeDir(Lum::Base::StringToWString(APP_DATADIR));
        path.AppendDir(L"images");
	path.SetBaseName(L"orange_star.png");
	      
        if (!Lum::Images::loader->Load(path.GetPath(),fullStar)) {
           std::cout << "Cannot load image 'orange_star.png'!" << std::endl;
          return false;
	}
      }

      emptyStar=Lum::Images::Factory::factory->CreateImage();
      if (!Lum::Images::loader->Load(L"images/grey_star.png",emptyStar)) {
         Lum::Base::Path path;

	path.SetNativeDir(Lum::Base::StringToWString(APP_DATADIR));
        path.AppendDir(L"images");
	path.SetBaseName(L"grey_star.png");
	      
        if (!Lum::Images::loader->Load(path.GetPath(),emptyStar)) {
           std::cout << "Cannot load image 'grey_star.png'!" << std::endl;
          return false;
	}
      }

      return true;
    }
    else {
      return false;
    }
  }

  void Cleanup()
  {
    Lum::OS::MainDialog<MainDialog>::Cleanup();
    curl_global_cleanup();
  }
};


LUM_MAIN(Main,L"MaemoRate")
