/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of OSM2Go.
 *
 * OSM2Go is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * OSM2Go is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSM2Go.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <sys/stat.h>
#include "appdata.h"

static const char *icon_paths[] = {
  "/usr/share/" PACKAGE "/icons" ,   // final installation path 
  "./data/icons",                    // local path for testing
  ""
};

static const char *icon_exts[] = {
  "gif", "png", "jpg", ""
};

static gchar*
icon_file_exists(const gchar *dir, const gchar *file) {
  gchar *fullname;
  struct stat s;
  gint status, idx;

  idx = 0;
  while(icon_exts[idx][0]) {
    fullname = g_strdup_printf("%s/%s.%s", dir, file, icon_exts[idx]);
    status = stat(fullname, &s);
    if(status == 0 && S_ISREG(s.st_mode))
      return fullname;

    idx++;
    g_free(fullname);
  }
  return NULL;
}

GdkPixbuf *icon_load(appdata_t *appdata, const char *name) {
  if(!name) return NULL;

  /* check if icon list already contains an icon of that name */
  icon_t **icon = &(appdata->icon);
  while(*icon) {
    if(strcmp(name, (*icon)->name) == 0) {
      //      printf("reuse existing icon\n");
      (*icon)->use++;
      return (*icon)->buf;
    }

    icon = &((*icon)->next);
  }

  gint i=0;
  do {
    gchar *fullname = icon_file_exists(icon_paths[i], name);
    if(fullname) {
      GdkPixbuf *pix = gdk_pixbuf_new_from_file(fullname, NULL);
      g_free(fullname);

      //      printf("Successfully loaded icon %s to %p\n", name, pix);
      *icon = g_new0(icon_t, 1);
      (*icon)->name = strdup(name);
      (*icon)->buf = pix;
      (*icon)->use = 1;

      return pix;
    }

    i++;
  } while(icon_paths[i][0]);

  printf("Icon %s not found\n", name);
  return NULL;
}

GtkWidget *icon_widget_load(appdata_t *appdata, const char *name) {
  GdkPixbuf *pix = icon_load(appdata, name);
  if(!pix) return NULL;

  return gtk_image_new_from_pixbuf(pix);
}

void icon_free(appdata_t *appdata, GdkPixbuf *buf) {
  icon_t **icon = &(appdata->icon);

  //  printf("request to free icon %p\n", buf);

  while(*icon) {
    //    printf("   -> %s %p\n", (*icon)->name, (*icon)->buf);

    if(buf == (*icon)->buf) {
      (*icon)->use--;
      if(!(*icon)->use) {
	printf("freeing unused icon\n");

	g_free((*icon)->name);
	gdk_pixbuf_unref((*icon)->buf);
	icon_t *next = (*icon)->next;
	g_free(*icon);
	*icon = next;

      } else 
	printf("keeping icon still in use by %d\n", (*icon)->use);

      return;
    }
    icon = &((*icon)->next);
  }

  printf("ERROR: icon to be freed not found\n");
}

void icon_free_all(appdata_t *appdata) {
  icon_t *icon = appdata->icon;

  while(icon) {
    g_free(icon->name);
    gdk_pixbuf_unref(icon->buf);
    icon_t *next = icon->next;
    g_free(icon);
    icon = next;
  }

  appdata->icon = NULL;
}
