/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of OSM2Go.
 *
 * OSM2Go is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * OSM2Go is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSM2Go.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "appdata.h"

/* -------------- picking one relation from the global list ---------- */

typedef struct {
  GtkWidget *dialog;
} relall_context_t;

enum {
  RELALL_COL_ID = 0,
  RELALL_COL_TYPE,
  RELALL_COL_DATA,
  RELALL_NUM_COLS
};

relation_t *relation_select(appdata_t *appdata, GtkWidget *parent) {
  relall_context_t *context = g_new0(relall_context_t, 1);
  
  context->dialog = gtk_dialog_new_with_buttons(_("Select existing relation"),
	GTK_WINDOW(parent), GTK_DIALOG_MODAL,
	GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, 
	GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, 
	NULL);
  
  gtk_dialog_set_default_response(GTK_DIALOG(context->dialog), 
				  GTK_RESPONSE_ACCEPT);

  /* making the dialog a little wider makes it less "crowded" */
#ifdef USE_HILDON
  gtk_window_set_default_size(GTK_WINDOW(context->dialog), 500, 300);
#else
  gtk_window_set_default_size(GTK_WINDOW(context->dialog), 400, 200);
#endif

  //  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(context->dialog)->vbox),
  //  		     relation_list(context, chain), TRUE, TRUE, 0);

  /* ----------------------------------- */

  gtk_widget_show_all(context->dialog);
  gtk_dialog_run(GTK_DIALOG(context->dialog));
  gtk_widget_destroy(context->dialog);

  return NULL;
}

/* --------------- relation dialog for an item (node or way) ----------- */

typedef struct {
  relation_item_t *item;
  appdata_t *appdata;
  GtkWidget *dialog, *view;
  GtkListStore *store;
  GtkWidget *but_add, *but_remove;
} relitem_context_t;

enum {
  RELITEM_COL_ID = 0,
  RELITEM_COL_TYPE,
  RELITEM_COL_DATA,
  RELITEM_NUM_COLS
};

static void relation_list_selected(relitem_context_t *context, 
				   gboolean selected) {
  gtk_widget_set_sensitive(context->but_remove, selected);
}

static gboolean
relation_list_selection_func(GtkTreeSelection *selection, GtkTreeModel *model,
		     GtkTreePath *path, gboolean path_currently_selected,
		     gpointer userdata) {
  relitem_context_t *context = (relitem_context_t*)userdata;
  GtkTreeIter iter;
    
  if(gtk_tree_model_get_iter(model, &iter, path)) {
    g_assert(gtk_tree_path_get_depth(path) == 1);
    relation_list_selected(context, TRUE);
  }
  
  return TRUE; /* allow selection state to change */
}

static void relation_remove_item(relation_t *relation, relation_item_t *item) {

  printf("remove item of type %d from relation #%ld\n", 
	 item->type, relation->id);

  member_t **member = &relation->member;
  while(*member) {
    if(((*member)->type == item->type) &&
       (((item->type == NODE) && (item->node == (*member)->node)) ||
	((item->type == WAY) && (item->way == (*member)->way)) ||
      ((item->type == RELATION) && (item->relation == (*member)->relation)))) {
      
      member_t *next = (*member)->next;
      osm_member_free(*member);
      *member = next;

      relation->flags |= OSM_FLAG_DIRTY;

      return;
    } else
      member = &(*member)->next;
  }
  g_assert(0);
}

static void on_relation_add(GtkWidget *but, relitem_context_t *context) {
  /* open a dialog where the user can pick from a list of all */
  /* relations */
  relation_t *rel = relation_select(context->appdata, context->dialog);
  if(rel) {
    printf("add it to item ...\n");

  }
}

static void on_relation_remove(GtkWidget *but, relitem_context_t *context) {
  GtkTreeSelection *selection;
  GtkTreeModel     *model;
  GtkTreeIter       iter;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(context->view));
  if(gtk_tree_selection_get_selected(selection, &model, &iter)) {
    relation_t *relation;
    gtk_tree_model_get(model, &iter, RELITEM_COL_DATA, &relation, -1);

    g_assert(relation);

    relation_remove_item(relation, context->item);

    /* and remove from store */
    gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
  }
  
  relation_list_selected(context, FALSE);
}

static GtkWidget *relation_list(relitem_context_t *context, 
				relation_chain_t *chain) {
  GtkWidget *vbox = gtk_vbox_new(FALSE,3);
  context->view = gtk_tree_view_new();

  gtk_tree_selection_set_select_function(
	 gtk_tree_view_get_selection(GTK_TREE_VIEW(context->view)), 
	 relation_list_selection_func, 
	 context, NULL);


  /* --- "ID" column --- */
  GtkCellRenderer *renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(context->view),
	-1, _("ID"), renderer, "text", RELITEM_COL_ID, NULL);

  /* --- "Type" column --- */
  renderer = gtk_cell_renderer_text_new();
  g_object_set(renderer, "ellipsize", PANGO_ELLIPSIZE_END, NULL);
  GtkTreeViewColumn *column = 
    gtk_tree_view_column_new_with_attributes(_("Type"), renderer, 
		 "text", RELITEM_COL_TYPE, NULL);
  gtk_tree_view_column_set_expand(column, TRUE);
  gtk_tree_view_insert_column(GTK_TREE_VIEW(context->view), column, -1);

  /* build and fill the store */
  context->store = gtk_list_store_new(RELITEM_NUM_COLS, 
		G_TYPE_STRING, G_TYPE_STRING, G_TYPE_POINTER);

  gtk_tree_view_set_model(GTK_TREE_VIEW(context->view), 
			  GTK_TREE_MODEL(context->store));

  GtkTreeIter iter;
  char *no_type = "<unknown type>";
  while(chain) {
    char *id = g_strdup_printf("#%ld", chain->relation->id);
    char *type = osm_tag_get_by_key(chain->relation->tag, "type");
    if(!type) type = _(no_type);

    /* Append a row and fill in some data */
    gtk_list_store_append(context->store, &iter);
    gtk_list_store_set(context->store, &iter,
	       RELITEM_COL_ID, id,
	       RELITEM_COL_TYPE, type,
	       RELITEM_COL_DATA, chain->relation,
	       -1);

    g_free(id);

    /* free chain while parsing it */
    relation_chain_t *next = chain->next;
    g_free(chain);
    chain = next;
  }
  
  g_object_unref(context->store);

  /* put it into a scrolled window */
  GtkWidget *scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), 
				 GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled_window), 
				      GTK_SHADOW_ETCHED_IN);
  gtk_container_add(GTK_CONTAINER(scrolled_window), context->view);

  gtk_box_pack_start_defaults(GTK_BOX(vbox), scrolled_window);

  /* ------- button box ------------ */

  GtkWidget *hbox = gtk_hbox_new(TRUE,3);

  context->but_add = gtk_button_new_with_label(_("Add..."));
  gtk_box_pack_start_defaults(GTK_BOX(hbox), context->but_add);
  gtk_signal_connect(GTK_OBJECT(context->but_add), "clicked",
    		     GTK_SIGNAL_FUNC(on_relation_add), context);

  context->but_remove = gtk_button_new_with_label(_("Remove"));
  gtk_box_pack_start_defaults(GTK_BOX(hbox), context->but_remove);
  gtk_signal_connect(GTK_OBJECT(context->but_remove), "clicked",
  		     GTK_SIGNAL_FUNC(on_relation_remove), context);

  relation_list_selected(context, FALSE);

  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  return vbox;
}

void relation_add_dialog(appdata_t *appdata, relation_item_t *relitem) {
  relitem_context_t *context = g_new0(relitem_context_t, 1);
  map_t *map = appdata->map;
  g_assert(map);

  context->appdata = appdata;
  context->item = relitem;

  relation_chain_t *chain = NULL;
  char *str = NULL;
  switch(relitem->type) {
  case NODE:
    chain = osm_node_to_relation(appdata->osm, relitem->node);
    str = g_strdup_printf(_("Relations for node #%ld"), relitem->node->id);
    break;
  case WAY:
    chain = osm_way_to_relation(appdata->osm, relitem->way);
    str = g_strdup_printf(_("Relations for way #%ld"), relitem->way->id);
    break;
  default:
    g_assert((relitem->type == NODE) || (relitem->type == WAY));
    break;
  }
  
  context->dialog = gtk_dialog_new_with_buttons(str,
	GTK_WINDOW(appdata->window), GTK_DIALOG_MODAL,
	GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL);
  g_free(str);
  
  gtk_dialog_set_default_response(GTK_DIALOG(context->dialog), 
				  GTK_RESPONSE_CLOSE);

  /* making the dialog a little wider makes it less "crowded" */
#ifdef USE_HILDON
  gtk_window_set_default_size(GTK_WINDOW(context->dialog), 500, 300);
#else
  gtk_window_set_default_size(GTK_WINDOW(context->dialog), 400, 200);
#endif

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(context->dialog)->vbox),
  		     relation_list(context, chain), TRUE, TRUE, 0);

  /* ----------------------------------- */

  gtk_widget_show_all(context->dialog);
  gtk_dialog_run(GTK_DIALOG(context->dialog));
  gtk_widget_destroy(context->dialog);

  g_free(context);
}
