/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of OSM2Go.
 *
 * OSM2Go is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * OSM2Go is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSM2Go.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "appdata.h"

void map_edit_way_add_segment(map_t *map, gint x, gint y) {

  /* convert mouse position to canvas (world) position */
  canvas_window2world(map->canvas, x, y, &x, &y);
  
  /* check if this was a double click. This is the case if */
  /* the last node placed is less than 5 pixels from the current */
  /* position */
  node_t *lnode = osm_way_get_last_node(map->action.way);
  if(lnode && (map->state->zoom * sqrt((lnode->lpos.x-x)*(lnode->lpos.x-x)+ 
		       (lnode->lpos.y-y)*(lnode->lpos.y-y))) < 5) {
    printf("detected double click -> simulate ok click\n");
    map_hl_touchnode_clear(map);
    map_action_ok(map->appdata);
  } else {
    
    /* use the existing node if one was touched */
    node_t *node = map_hl_touchnode_get_node(map);
    if(node) {
      printf("  re-using node #%ld\n", node->id);
      map_hl_touchnode_clear(map);
    } else {
      if(!osm_position_within_bounds(map->appdata->osm, x, y)) 
	map_outside_error(map->appdata);
      else 
	node = osm_node_new(map->appdata->osm, x, y);
    }
    
    if(node) {
      g_assert(map->action.way);
      osm_way_append_node(map->action.way, node);
      
      switch(osm_node_chain_length(map->action.way->node_chain)) {
      case 1:
	/* replace "place first node..." message */
	statusbar_set(map->appdata, _("Place next node of way"), FALSE);
	break;
	
      case 2:
	/* two nodes are enough for a valid way */
	icon_bar_map_cancel_ok(map->appdata, TRUE, TRUE);
	break;
	
      default:
	break;
      }
      
      /* remove prior version of this way */
      map_item_chain_destroy(&map->action.way->map_item_chain);
      
      /* draw current way */
      map_way_color(map->appdata->potlatch, map->action.way);
      map_way_draw(map, map->action.way);
    }
  }
}

void map_edit_way_add_cancel(map_t *map) {

  printf("  removing temporary way\n");
  g_assert(map->action.way);

  /* remove all nodes that have been created for this way */
  /* (their way count will be 0 after removing the way) */
  node_chain_t *chain = map->action.way->node_chain;
  while(chain) {
    node_chain_t *next = chain->next;
    
    printf("    node #%ld (used by %d)\n", 
	   chain->node->id, chain->node->ways);
    
    chain->node->ways--;
    if(!chain->node->ways && (chain->node->id == ID_ILLEGAL)) {
      printf("      -> freeing temp node\n");
      osm_node_free(chain->node);
    }
    g_free(chain);
    chain = next;
  }
  map->action.way->node_chain = NULL;
  
  /* remove ways visual representation */
  map_item_chain_destroy(&map->action.way->map_item_chain);
  
  osm_way_free(map->action.way);
  map->action.way = NULL;
}

void map_edit_way_add_ok(map_t *map) {
  g_assert(map->action.way);
    
  /* transfer all nodes that have been created for this way */
  /* into the node chain */
  
  /* (their way count will be 0 after removing the way) */
  node_chain_t *chain = map->action.way->node_chain;
  while(chain) {
    printf("    node #%ld (used by %d)\n", 
	   chain->node->id, chain->node->ways);
    
    /* a node may have been a stand-alone node before, so remove its */
    /* visual representation as its now drawn as part of the way */
    /* (if at all) */
    if(chain->node->id != ID_ILLEGAL) 
      map_item_chain_destroy(&chain->node->map_item_chain);
    
    map_node_draw(map, chain->node);
    
    /* we can be sure that no node gets inserted twice (even if twice in */
    /* the ways chain) because it gets assigned a non-ID_ILLEGAL id when */
    /* being moved to the osm node chain */
    if(chain->node->id == ID_ILLEGAL) 
      osm_node_attach(map->appdata->osm, chain->node);
    
    chain = chain->next;
  }
  
  /* now move the way itself into the main data structure */
  osm_way_attach(map->appdata->osm, map->action.way);
  
  map_way_select(map->appdata, map->action.way);
  
  map->action.way = NULL;
  
  /* let the user specify some tags for the new way */
  info_dialog(map->appdata);
}

void map_edit_way_node_add_highlight(map_t *map, map_item_t *item, 
				     gint x, gint y) {
  if(map_item_is_selected_way(map, item)) {
    gint nx, ny;
    canvas_window2world(map->canvas, x, y, &nx, &ny);
    if(canvas_item_get_segment(item->item, nx, ny) >= 0)
      map_hl_cursor_draw(map, x, y, FALSE, RADIUS);
  }
}

void map_edit_way_node_add(map_t *map, gint x, gint y) {
  /* check if we are still hovering above the selected way */
  map_item_t *item = map_item_at(map, x, y);
  if(item && map_item_is_selected_way(map, item)) {
    /* convert mouse position to canvas (world) position */
    canvas_window2world(map->canvas, x, y, &x, &y);
    gint insert_after = canvas_item_get_segment(item->item, x, y)+1;
    if(insert_after > 0) {
      /* create new node */
      node_t* node = osm_node_new(map->appdata->osm, x, y);
      osm_node_attach(map->appdata->osm, node);
      
      /* insert it into ways chain of nodes */
      way_t *way = item->way;
      
      /* search correct position */
      node_chain_t **chain = &way->node_chain;
      while(insert_after--) {
	g_assert(*chain);
	chain = &(*chain)->next;
      }
      
      /* and actually do the insertion */
      node_chain_t *new_chain_item = g_new0(node_chain_t, 1);
      new_chain_item->node = node;
      new_chain_item->next = *chain;
      *chain = new_chain_item;
      
      /* clear selection */
      map_item_deselect(map->appdata);
      
      /* remove prior version of this way */
      map_item_chain_destroy(&way->map_item_chain);
      
      /* draw the updated way */
      map_way_draw(map, way);
      
      /* remember that this node is contained in one way */
      node->ways=1;
      
      /* and that the way needs to be uploaded */
      way->flags |= OSM_FLAG_DIRTY;
      
      /* put gui into idle state */
      map_action_set(map->appdata, MAP_ACTION_IDLE);
      
      /* and redo it */
      map_way_select(map->appdata, way);
    }
  }
}  

void map_edit_way_cut_highlight(map_t *map, map_item_t *item, gint x, gint y) {

  if(map_item_is_selected_way(map, item)) {
    gint nx, ny, seg;
    canvas_window2world(map->canvas, x, y, &nx, &ny);
    seg = canvas_item_get_segment(item->item, nx, ny);
    if(seg >= 0) {
      gint x0, y0, x1, y1;
      canvas_item_get_segment_pos(item->item, seg, &x0, &y0, &x1, &y1);
      
      gint width = (item->way->draw_flags & 
		    OSM_DRAW_FLAG_STROKE)?2*STROKE_WIDTH:3*WAY_WIDTH;
      map_hl_segment_draw(map, width, x0, y0, x1, y1);
    }
  } else if(map_item_is_selected_node(map, item)) {
    node_t *nfirst = osm_way_get_first_node(map->selected.way);
    node_t *nlast = osm_way_get_last_node(map->selected.way);

    /* cutting a way at its first or last node doesn't make much sense ... */
    if((nfirst != item->node) && (nlast != item->node)) 
      map_hl_cursor_draw(map, item->node->lpos.x, item->node->lpos.y, 
			 TRUE, 2*RADIUS);
  }
}

/* cut the currently selected way at the current cursor position */
void map_edit_way_cut(map_t *map, gint x, gint y) {

  /* check if we are still hovering above the selected way */
  map_item_t *item = map_item_at(map, x, y);
  if(item && (map_item_is_selected_way(map, item) || 
	      map_item_is_selected_node(map, item))) {
    gboolean cut_at_node = map_item_is_selected_node(map, item);
    
    /* convert mouse position to canvas (world) position */
    canvas_window2world(map->canvas, x, y, &x, &y);
    
    gint cut_at = -1;
    way_t *way = NULL;
    if(cut_at_node) {
      printf("  cut at node\n");
      
      /* node must not be first or last node of way */
      g_assert(map->selected.type == MAP_TYPE_WAY);
      
      if((osm_way_get_first_node(map->selected.way) != item->node) &&
	 (osm_way_get_last_node(map->selected.way) != item->node)) {
	way = map->selected.way;

	cut_at = 0;
	node_chain_t *chain = way->node_chain;
	while(chain && chain->node != item->node) {
	  chain = chain->next;
	  cut_at++;
	}

      } else
	printf("  won't cut as it's last or first node\n");
      
    } else {
      printf("  cut at segment\n");
      cut_at = canvas_item_get_segment(item->item, x, y);
      if(cut_at >= 0) way = item->way;
    }
    
    if(way) {
      /* create a duplicate of the currently selected way */
      way_t *new = osm_way_new();
      
      /* if this is a closed way, reorder (rotate) it, so the */
      /* place to cut is adjecent to the begin/end of the way. */
      /* this prevents a cut polygon to be split into two ways */
      g_assert(way->node_chain);
      if(way->node_chain->node == osm_way_get_last_node(way)) {
	printf("CLOSED WAY -> rotate by %d\n", cut_at);
	osm_way_rotate(way, cut_at);
	cut_at = 0;
      }
      
      /* ------------  copy all tags ------------- */
      new->tag = osm_tags_copy(way->tag, TRUE);
      
      /* ---- transfer relation membership from way to new ----- */
      relation_chain_t *rchain = osm_way_to_relation(map->appdata->osm, way);
      
      while(rchain) {
	relation_chain_t *next = rchain->next;
	printf("way is part of relation #%ld\n", rchain->relation->id);
	
	/* make new member of the same relation */
	
	/* walk member chain. save role of way if its being found. */
	member_t **member = &rchain->relation->member;
	char *role = NULL;
	while(*member) { 
	  /* save role of way */
	  if(((*member)->type == WAY) && ((*member)->way == way))
	    role = (*member)->role;
	  member = &(*member)->next;
	}
	
	printf("  adding new to relation\n");
	*member = g_new0(member_t, 1);
	(*member)->type = WAY;
	(*member)->way = new;
	if(role) (*member)->role = g_strdup(role);
	member = &(*member)->next;
	
	rchain->relation->flags |= OSM_FLAG_DIRTY;
	
	g_free(rchain);
	rchain = next;
      }
      
      /* move parts of node_chain to the new way */
      printf("  moving everthing after segment %d to new way\n", cut_at);

      node_chain_t *chain = way->node_chain;
      while(cut_at--) {
	g_assert(chain);
	chain = chain->next;
      }
      
      /* attach remaining nodes to new way */
      new->node_chain = chain->next;
      
      /* terminate remainig chain on old way */
      chain->next = NULL;
      
      /* if we cut at a node, this node is now part of both ways. so */
      /* create a copy of the last node of the old way and prepend it to */
      /* the new way */
      if(cut_at_node) {
	node_chain_t *first = g_new0(node_chain_t, 1);
	first->next = new->node_chain;
	first->node = osm_way_get_last_node(way);
	first->node->ways++;
	new->node_chain = first;
      }

      /* now move the way itself into the main data structure */
      osm_way_attach(map->appdata->osm, new);
      
      /* clear selection */
      map_item_deselect(map->appdata);
      
      /* remove prior version of this way */
      printf("remove visible version of way #%ld\n", way->id);
      map_item_chain_destroy(&way->map_item_chain);
      
      /* swap chains of the old way is to be destroyed due to a lack */
      /* of nodes */
      if(osm_way_number_of_nodes(way) < 2) {
	printf("swapping ways to avoid destruction of original way\n");
	node_chain_t *tmp = way->node_chain;
	way->node_chain = new->node_chain;
	new->node_chain = tmp;
      }

      /* the way may still only consist of a single node. */
      /* remove it then */
      if(osm_way_number_of_nodes(way) < 2) {
	printf("original way has less than 2 nodes left, deleting it\n");
	map_way_delete(map->appdata, way);
	item = NULL;
      } else {
	printf("original way still has %d nodes\n",
	       osm_way_number_of_nodes(way));
	
	/* draw the updated old way */
	map_way_color(map->appdata->potlatch, way);
	map_way_draw(map, way);
	
	/* remember that the way needs to be uploaded */
	way->flags |= OSM_FLAG_DIRTY;
	}
      
      if(osm_way_number_of_nodes(new) < 2) {
	printf("new way has less than 2 nodes, deleting it\n");
	map_way_delete(map->appdata, new);
	new = NULL;
      } else {
	
	/* colorize the new way before drawing */
	map_way_color(map->appdata->potlatch, new);
	  map_way_draw(map, new);
      }
      
      /* put gui into idle state */
      map_action_set(map->appdata, MAP_ACTION_IDLE);      
      
      /* and redo selection if way still exists */
      if(item) 
	map_way_select(map->appdata, way);
      else if(new)
	map_way_select(map->appdata, new);
    }
  }
}
