/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of OSM2Go.
 *
 * OSM2Go is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * OSM2Go is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSM2Go.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "appdata.h"

static void map_way_color(potlatch_conf_t *potlatch, way_t *way) {
  way->color = way->fill_color = MAP_COLOR_NONE;
  way->draw_flags = 0;

  potlatch_color_t *plcol = potlatch->colors;

  /* a highway/waterway/railway is never a polygon */
  if(!osm_way_is_way(way)) {
  
    /* check if this way is a polygon */
    node_chain_t *node_chain = way->node_chain;
    while(node_chain && node_chain->next) node_chain = node_chain->next;
    if(way->node_chain && node_chain && 
       way->node_chain->node == node_chain->node) {
      way->draw_flags |= OSM_DRAW_FLAGS_POLYGON;
      
      /* default fill color is light grey */
      way->fill_color = (0xc0c0c0 << 8) | FILL_TRANSP;
    }
  }
  
  while(plcol) {
    if(osm_way_has_key_or_value(way, plcol->tag)) {
      way->color = (plcol->color << 8) | 0xff;
      if(plcol->stroke) way->draw_flags |= OSM_DRAW_FLAGS_STROKE;
      
      if(plcol->fill_color != COLOR_NONE)
	way->fill_color = (plcol->fill_color << 8) | FILL_TRANSP;
    }
    plcol = plcol->next;
  }
  
  /* use dark grey/no stroke/not filled for everything unknown */
  if(way->color == MAP_COLOR_NONE) way->color = DARK_GREY;
}

static void map_init_colors(potlatch_conf_t *potlatch, osm_t *osm) {
  way_t *way = osm->way;

  printf("preparing colors\n");

  /* colorize ways */
  while(way) {
    map_way_color(potlatch, way);
    way = way->next;
  }
}

static void map_statusbar(map_t *map, map_item_t *map_item) {
  char *item_str = NULL;
  item_id_t id = ID_ILLEGAL;
  tag_t *tag = NULL;
  char *str = NULL;
  
  switch(map_item->type) {
  case MAP_TYPE_NODE:
    item_str = "Node";
    id = map_item->node->id;
    tag = map_item->node->tag;
    break;

  case MAP_TYPE_WAY:
    item_str = "Way";
    id = map_item->way->id;
    tag = map_item->way->tag;
    break;

  default:
    break;
  }

  if(id == ID_ILLEGAL) 
    str = g_strdup_printf(_("Unknown item"));
  else {
    str = g_strdup_printf("%s #%ld", item_str, id);

    /* add some tags ... */
    while(tag) {
      /* we don't have much space, so ignore created_by tag */
      if(strcasecmp(tag->key, "created_by") != 0) {
	char *old = str;
	str = g_strdup_printf("%s, %s/%s", old, tag->key, tag->value);
	g_free(old);
      }
      tag = tag->next;
    }
  }
  
  statusbar_set(map->appdata, str);
  g_free(str);
}

static void map_node_select(appdata_t *appdata, node_t *node) {
  map_t *map = appdata->map;
  map_item_t *map_item = &map->selected;
  
  g_assert(!map->highlight);

  map_item->type      = MAP_TYPE_NODE;
  map_item->node      = node;
  map_item->highlight = FALSE;

  /* node may not have any visible representation at all */
  if(node->map_item_chain) 
    map_item->item = node->map_item_chain->map_item->item;
  else
    map_item->item = NULL;

  map_statusbar(map, map_item);
  icon_bar_map_item_selected(appdata, map_item, TRUE);

  /* highlight node */
  gint x = map_item->node->lpos.x, y = map_item->node->lpos.y;

  /* create a copy of this map item and mark it as being a highlight */
  map_item_t *new_map_item = g_new0(map_item_t, 1);
  memcpy(new_map_item, map_item, sizeof(map_item_t));
  new_map_item->highlight = TRUE;
  map_hl_circle_new(map, CANVAS_GROUP_NODES_HL, new_map_item, 
		    x, y, 2*RADIUS, HIGHLIGHT_COLOR);

  if(!map_item->item) {
    /* and draw a fake node */
    new_map_item = g_new0(map_item_t, 1);
    memcpy(new_map_item, map_item, sizeof(map_item_t));
    new_map_item->highlight = TRUE;
    map_hl_circle_new(map, CANVAS_GROUP_NODES_HL, new_map_item, 
		      x, y, RADIUS, HIGHLIGHT2_COLOR);
  }
}

static void map_way_select(appdata_t *appdata, way_t *way) {
  map_t *map = appdata->map;
  map_item_t *map_item = &map->selected;
  
  g_assert(!map->highlight);

  map_item->type      = MAP_TYPE_WAY;
  map_item->way       = way;
  map_item->highlight = FALSE;
  map_item->item      = way->map_item_chain->map_item->item;

  map_statusbar(map, map_item);
  icon_bar_map_item_selected(appdata, map_item, TRUE);

  node_chain_t *node_chain = map_item->way->node_chain;
  while(node_chain) {
    map_item_t item;
    item.type = MAP_TYPE_NODE;
    item.node = node_chain->node;

    if(!map_hl_item_is_highlighted(map, &item)) {

      /* create a new map item for every node */
      map_item_t *new_map_item = g_new0(map_item_t, 1);
      new_map_item->type = MAP_TYPE_NODE;
      new_map_item->node = node_chain->node;
      new_map_item->highlight = TRUE;
    
      gint x = node_chain->node->lpos.x;
      gint y = node_chain->node->lpos.y;

      map_hl_circle_new(map, CANVAS_GROUP_NODES_HL, new_map_item, 
			x, y, RADIUS, HIGHLIGHT2_COLOR);
    }

    node_chain = node_chain->next;
  }

  /* a way needs at least 2 points to be drawn */
  guint nodes = osm_way_number_of_nodes(way);
  if(nodes > 1) {
    
    /* allocate space for nodes */
    canvas_points_t *points = canvas_points_new(nodes);

    int node = 0;
    node_chain = map_item->way->node_chain;
    while(node_chain) {
      canvas_point_set_pos(points, node++, &node_chain->node->lpos);
      node_chain = node_chain->next;
    }
    
    /* create a copy of this map item and mark it as being a highlight */
    map_item_t *new_map_item = g_new0(map_item_t, 1);
    memcpy(new_map_item, map_item, sizeof(map_item_t));
    new_map_item->highlight = TRUE;
    
    map_hl_polyline_new(map, CANVAS_GROUP_WAYS_HL, new_map_item, points, 
			(map_item->way->draw_flags & OSM_DRAW_FLAGS_STROKE)?
	                 2*STROKE_WIDTH:3*WAY_WIDTH, HIGHLIGHT_COLOR);

    canvas_points_free(points);
  }
}

static void map_item_select(appdata_t *appdata, map_item_t *map_item) {
  switch(map_item->type) {
  case MAP_TYPE_NODE:
    map_node_select(appdata, map_item->node);
    break;
  case MAP_TYPE_WAY:
    map_way_select(appdata, map_item->way);
    break;
  default:
    g_assert((map_item->type == MAP_TYPE_NODE)||
	     (map_item->type == MAP_TYPE_WAY));
    break;
  }
}

static void map_item_deselect(appdata_t *appdata) {

  /* remove statusbar message */
  statusbar_set(appdata, NULL);

  /* disable/enable icons in icon bar */
  icon_bar_map_item_selected(appdata, NULL, FALSE);

  /* remove highlight */
  map_hl_remove(appdata);

  /* forget about selection */
  appdata->map->selected.type = MAP_TYPE_ILLEGAL;
}

/* called whenever a map item is to be destroyed */
static gint map_item_destroy_event(GtkWidget *widget, gpointer data) {
  map_item_t *map_item = (map_item_t*)data;

  //  printf("destroying map_item @ %p\n", map_item);

  /* remove item from nodes/ways map_item_chain */
  map_item_chain_t **chain = NULL;
  if(map_item->type == MAP_TYPE_NODE)
    chain = &map_item->node->map_item_chain;
  else if(map_item->type == MAP_TYPE_WAY)
    chain = &map_item->way->map_item_chain;

  /* there must be a chain with content, otherwise things are broken */
  g_assert(chain);
  g_assert(*chain);

  /* search current map_item, ... */
  while(*chain && (*chain)->map_item != map_item)
    chain = &(*chain)->next;

  g_assert(*chain);

  /* ... remove it from chain and free it */
  map_item_chain_t *tmp = *chain;
  *chain = (*chain)->next;

  g_free(tmp);
  g_free(map_item);

  return FALSE;
}

static canvas_item_t *map_node_new(map_t *map, node_t *node, gint radius, 
				   gint width, canvas_color_t fill, canvas_color_t border) {

  map_item_t *map_item = g_new0(map_item_t, 1);
  map_item->type = MAP_TYPE_NODE;
  map_item->node = node;
  map_item->item = canvas_circle_new(map, CANVAS_GROUP_NODES, 
		     node->lpos.x, node->lpos.y, radius, width, fill, border);
  
  /* attach map_item to nodes map_item_chain */
  map_item_chain_t **chain = &node->map_item_chain;
  while(*chain) chain = &(*chain)->next;
  *chain = g_new0(map_item_chain_t, 1);
  (*chain)->map_item = map_item;

  canvas_item_set_user_data(map_item->item, map_item);

  canvas_item_destroy_connect(map_item->item, 
          G_CALLBACK(map_item_destroy_event), map_item);

  return map_item->item;
}

static canvas_item_t *map_way_new(map_t *map, canvas_group_t group, way_t *way, 
				  canvas_points_t *points, gint width, canvas_color_t color,
				  canvas_color_t fill_color) {
  map_item_t *map_item = g_new0(map_item_t, 1);
  map_item->type = MAP_TYPE_WAY;
  map_item->way = way;

  if(way->draw_flags & OSM_DRAW_FLAGS_POLYGON) {
    map_item->item = canvas_polygon_new(map, group, points, width, color, fill_color);
  } else {
    map_item->item = canvas_polyline_new(map, group, points, width, color);
  }

  /* attach map_item to ways map_item_chain */
  map_item_chain_t **chain = &way->map_item_chain;
  while(*chain) chain = &(*chain)->next;
  *chain = g_new0(map_item_chain_t, 1);
  (*chain)->map_item = map_item;

  canvas_item_set_user_data(map_item->item, map_item);

  canvas_item_destroy_connect(map_item->item, 
	      G_CALLBACK(map_item_destroy_event), map_item);

  return map_item->item;
}

void map_show_node(map_t *map, node_t *node) {
  map_node_new(map, node, S_RADIUS, 0, NODE_WAY_COLOR, 0);
}

static void map_way_draw(map_t *map, way_t *way) {

  /* don't draw a way that's not there anymore */
  if(way->flags & OSM_FLAG_DELETED)
    return;

  /* allocate space for nodes */
  /* a way needs at least 2 points to be drawn */
  guint nodes = osm_way_number_of_nodes(way);
  if(nodes > 1) {
    canvas_points_t *points = canvas_points_new(nodes);
    
    int node = 0;
    node_chain_t *node_chain = way->node_chain;
    while(node_chain) {
      canvas_point_set_pos(points, node++, &node_chain->node->lpos);
      node_chain = node_chain->next;
    }
    
    /* draw way */
    if(way->draw_flags & OSM_DRAW_FLAGS_POLYGON) {
      map_way_new(map, CANVAS_GROUP_POLYGONS, way, points, 2, DARK_GREY, way->fill_color);
    } else {
      map_way_new(map, CANVAS_GROUP_WAYS, way, points, WAY_WIDTH, way->color, NO_COLOR);
      
      if(way->draw_flags & OSM_DRAW_FLAGS_STROKE)
	map_way_new(map, CANVAS_GROUP_WAYS_OL, way, points, STROKE_WIDTH, DARK_GREY, NO_COLOR);
    }
    canvas_points_free(points);
  }
}

static void map_node_draw(map_t *map, node_t *node) {
  /* don't draw a node that's not there anymore */
  if(node->flags & OSM_FLAG_DELETED)
    return;
  
  if(!node->ways) 
    map_node_new(map, node, RADIUS, RADIUS/2.0, NODE_COLOR, NODE_WAY_COLOR);
  else if(osm_node_has_tag(node)) 
    map_node_new(map, node, S_RADIUS, 0, NODE_WAY_COLOR, 0);
}

static void map_item_draw(map_t *map, map_item_t *map_item) {
  switch(map_item->type) {
  case MAP_TYPE_NODE:
    map_node_draw(map, map_item->node);
    break;
  case MAP_TYPE_WAY:
    map_way_draw(map, map_item->way);
    break;
  default:
    g_assert((map_item->type == MAP_TYPE_NODE) ||
	     (map_item->type == MAP_TYPE_WAY));
  }
}

static void map_item_remove(map_t *map, map_item_t *map_item) {
  map_item_chain_t *chain = NULL;

  switch(map_item->type) {
  case MAP_TYPE_NODE:
    chain = map_item->node->map_item_chain;
    break;
  case MAP_TYPE_WAY:
    chain = map_item->way->map_item_chain;
    break;
  default:
    g_assert((map_item->type == MAP_TYPE_NODE) ||
	     (map_item->type == MAP_TYPE_WAY));
  }

  while(chain) {
    map_item_chain_t *next = chain->next;
    canvas_item_destroy(chain->map_item->item);
    chain = next;
  }
}

static void map_item_init(potlatch_conf_t *potlatch, map_item_t *map_item) {
  if(map_item->type == MAP_TYPE_WAY)
    map_way_color(potlatch, map_item->way);
}

void map_item_redraw(appdata_t *appdata, map_item_t *map_item) {
  map_item_t item = *map_item;

  /* check if the item to be redrawn is the selected one */
  gboolean is_selected = FALSE;
  if(map_item->ptr == appdata->map->selected.ptr) {
    map_item_deselect(appdata);
    is_selected = TRUE;
  }

  map_item_remove(appdata->map, &item);
  map_item_init(appdata->potlatch, &item);
  map_item_draw(appdata->map, &item);

  /* restore selection if there was one */
  if(is_selected) 
    map_item_select(appdata, &item);
}

static void map_draw(map_t *map, osm_t *osm) {
  g_assert(map->canvas);

  printf("drawing ways ...\n");
  way_t *way = osm->way;
  while(way) {
    map_way_draw(map, way);
    way = way->next;
  }

  printf("drawing single nodes ...\n");
  node_t *node = osm->node;
  while(node) {
    map_node_draw(map, node);
    node = node->next;
  }
}

static gint map_destroy_event(GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  map_t *map = appdata->map;

  printf("destroying map\n");

  g_free(map);
  appdata->map = NULL;

  return FALSE;
}

/* get the item at position x, y */
static map_item_t *map_item_at(map_t *map, gint x, gint y) {	  
  printf("map check at %d/%d\n", x, y);

  canvas_window2world(map->canvas, x, y, &x, &y);

  printf("world check at %d/%d\n", x, y);
  
  canvas_item_t *item = canvas_get_item_at(map->canvas, x, y);
	  
  if(!item) {
    printf("  there's no item\n");
    return NULL;
  }

  printf("  there's an item\n");

  map_item_t *map_item = canvas_item_get_user_data(item);

  if(!map_item) {
    printf("  item has no user data!\n");
    return NULL;
  }
    
  if(map_item->highlight)
    printf("  item is highlight\n");    

  switch(map_item->type) {
  case MAP_TYPE_NODE:
    printf("  item is node #%ld\n", map_item->node->id);
    break;
  case MAP_TYPE_WAY:
    printf("  item is way #%ld\n", map_item->way->id);
    break;
  default:
    printf("  unknown item\n");
    break;
  }

  return map_item;
}    

/* get the real item (no highlight) at x, y */
static map_item_t *map_real_item_at(map_t *map, gint x, gint y) { 
  map_item_t *map_item = map_item_at(map, x, y);

  /* no item or already a real one */
  if(!map_item || !map_item->highlight) return map_item;

  /* get the item (parent) this item is the highlight of */
  map_item_t *parent = NULL;
  switch(map_item->type) {
  case MAP_TYPE_NODE:
    if(map_item->node->map_item_chain)
      parent = map_item->node->map_item_chain->map_item;

    if(parent)
      printf("  using parent item node #%ld\n", parent->node->id);      
    break;
  case MAP_TYPE_WAY:
    if(map_item->way->map_item_chain)
      parent = map_item->way->map_item_chain->map_item;

    if(parent)
      printf("  using parent item way #%ld\n", parent->way->id);      
    break;
  default:
    g_assert((map_item->type == MAP_TYPE_NODE) ||
	     (map_item->type == MAP_TYPE_WAY)); 
    break;
  }
  
  if(parent) 
    map_item = parent;
  else 
    printf("  no parent, working on highlight itself\n");

  return map_item;
}

static void map_set_zoom(map_t *map, double zoom) {
  map->zoom = zoom;
  canvas_set_zoom(map->canvas, map->zoom);
}

static gboolean map_scroll_event(GtkWidget *widget, GdkEventScroll *event,
				 gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  if(event->type == GDK_SCROLL) {
    if(event->direction) 
      map_set_zoom(appdata->map, appdata->map->zoom / 1.2);
    else
      map_set_zoom(appdata->map, appdata->map->zoom * 1.2);
  }

  return TRUE;
}

static gboolean distance_above(map_t *map, gint x, gint y, gint limit) {
  gint sx, sy;

#ifdef USE_GNOMECANVAS
  gnome_canvas_get_scroll_offsets(GNOME_CANVAS(map->canvas), &sx, &sy);

  /* add offsets generated by mouse within map and map scrolling */
  sx = (x-map->pen_down.at.x) + (map->pen_down.so.x-sx);
  sy = (y-map->pen_down.at.y) + (map->pen_down.so.y-sy);
#else
  /* add offsets generated by mouse within map and map scrolling */
  sx = (x-map->pen_down.at.x);
  sy = (y-map->pen_down.at.y);
#endif

  return(sx*sx + sy*sy > limit*limit);
}

static void map_do_scroll(map_t *map, gint x, gint y) {
  gint sx, sy;

  canvas_get_scroll_offsets(map->canvas, &sx, &sy);
  sx -= x-map->pen_down.at.x;
  sy -= y-map->pen_down.at.y;
  canvas_scroll_to(map->canvas, sx, sy);
}

static gboolean item_is_selected_node(map_t *map, map_item_t *map_item) {
  printf("check if item is a selected node\n");

  if(map->selected.type == MAP_TYPE_ILLEGAL) {
    printf("  nothing is selected\n");
    return FALSE;
  }

  /* clicked the highlight directly */
  if(map_item->type != MAP_TYPE_NODE) {
    printf("  didn't click node\n");
    return FALSE;
  }

  if(map->selected.type == MAP_TYPE_NODE) {
    printf("  selected item is a node\n");

    if(map_item->node == map->selected.node) {
      printf("  requested item is a selected node\n");
      return TRUE;
    }
    printf("  but it's not the requested one\n");
    return FALSE;

  } else if(map->selected.type == MAP_TYPE_WAY) {
    printf("  selected item is a way\n");

    node_chain_t *node_chain = map->selected.way->node_chain;
    while(node_chain) {
      if(node_chain->node == map_item->node) {
	printf("  requested item is part of selected way\n");
	return TRUE;
      } 
      node_chain = node_chain->next;
    }
    printf("  but it doesn't include the requested node\n");
    return FALSE;
    
  } else {
    printf("  selected item is unknown\n");
    return FALSE;
  }

  g_assert(0);
  return TRUE;
}

void map_highlight_refresh(appdata_t *appdata) {
  map_t *map = appdata->map;
  map_item_t old = map->selected;

  printf("type to refresh is %d\n", old.type);

  map_item_deselect(appdata);
  map_item_select(appdata, &old);
}

static void map_node_move(appdata_t *appdata, map_item_t *map_item, 
			  gint ex, gint ey) {

  map_t *map = appdata->map;
  osm_t *osm = appdata->osm;

  g_assert(map_item->type == MAP_TYPE_NODE);
  node_t *node = map_item->node;

  printf("released dragged node #%ld\n", node->id);
  printf("  was at %d %d (%f %f)\n", 
	 node->lpos.x, node->lpos.y,
	 node->pos.lat, node->pos.lon);
	    
  /* convert mouse position to canvas (world) position */
  canvas_window2world(map->canvas, ex, ey, &node->lpos.x, &node->lpos.y);
	    
  /* convert screen position back to ll */
  lpos2pos(osm->bounds, &node->lpos, &node->pos);

  printf("  now at %d %d (%f %f)\n", 
	 node->lpos.x, node->lpos.y, node->pos.lat, node->pos.lon);
  
  /* now update the visual representation of the node */
  canvas_item_t *item = map_item->item;
  
  if(!node->ways) 
    canvas_item_set_pos(item, &node->lpos, RADIUS);
  else {
    /* only move item of it's not the highlight. highlight will be */
    /* redrawed entirely lateron */
    if(!map_item->highlight)
      canvas_item_set_pos(item, &node->lpos, S_RADIUS);
  }

  /* update ways, node is part of */
  way_t *way = osm->way;
  while(way) {
    if(osm_node_in_way(way, node)) {
      printf("node is part of way #%ld\n", way->id);
      
      /* rebuild ways positions */
      
      /* allocate space for nodes */
      guint nodes = osm_way_number_of_nodes(way);
      if(nodes > 1) {
	canvas_points_t *points = canvas_points_new(nodes);
      
	int node = 0;
	node_chain_t *node_chain = way->node_chain;
	while(node_chain) {
	  canvas_point_set_pos(points, node++, &node_chain->node->lpos);
	  node_chain = node_chain->next;
	}
	
	/* do something with points */
	map_item_chain_t *chain = way->map_item_chain;
	while(chain) {
	  canvas_item_set_points(chain->map_item->item, points);
	  chain = chain->next;
	}
	
	canvas_points_free(points);
      }
    }
    
    way = way->next;
  }
  
  /* and mark the node as dirty */
  node->flags |= OSM_FLAG_DIRTY;

  /* update highlight */
  map_highlight_refresh(appdata);
}
			  
static void map_handle_click(appdata_t *appdata, map_t *map) {

  /* problem: on_item may be the highlight itself! So store it! */
  map_item_t map_item;
  if(map->pen_down.on_item) map_item = *map->pen_down.on_item;
  else                      map_item.type = MAP_TYPE_ILLEGAL;

  /* if we aready have something selected, then de-select it */
  map_item_deselect(appdata);
  
  /* select the clicked item (if there was one) */
  if(map_item.type != MAP_TYPE_ILLEGAL) {
    switch(map_item.type) {
    case MAP_TYPE_NODE:
      map_node_select(appdata, map_item.node);
      break;

    case MAP_TYPE_WAY:
      map_way_select(appdata, map_item.way);
      break;

    default:
      g_assert((map_item.type == MAP_TYPE_NODE) ||
	       (map_item.type == MAP_TYPE_WAY));
      break;
    }
  }
}

static void map_button_press(map_t *map, gint x, gint y) {

  printf("left button pressed\n");
  map->pen_down.is = TRUE;
      
  /* save press position */
  map->pen_down.at.x = x;
  map->pen_down.at.y = y;
  map->pen_down.drag = FALSE;     // don't assume drag yet

#ifdef USE_GNOMECANVAS  
  /* save initial scroll offset */
  gnome_canvas_get_scroll_offsets(GNOME_CANVAS(map->canvas), 
		  &map->pen_down.so.x, &map->pen_down.so.y);
#endif  

  /* determine wether this press was on an item */
  map->pen_down.on_item = map_real_item_at(map, x, y);
  
  /* check if the clicked item is a highlighted node as the user */
  /* might want to drag that */
  map->pen_down.on_selected_node = FALSE;
  if(map->pen_down.on_item)
    map->pen_down.on_selected_node = 
      item_is_selected_node(map, map->pen_down.on_item);

  switch(map->action) {
  case MAP_ACTION_NODE_ADD:
    map_hl_cursor_draw(map, x, y);
    break;
  default:
    break;
  }
}

/* move the background image (wms data) during wms adjustment */
static void map_bg_adjust(map_t *map, gint x, gint y) {    
  g_assert(map->appdata);
  g_assert(map->appdata->osm);
  g_assert(map->appdata->osm->bounds);

  x += map->appdata->osm->bounds->min.x + map->bg_offset.x -
    map->pen_down.at.x;
  y += map->appdata->osm->bounds->min.y + map->bg_offset.y -
    map->pen_down.at.y;

  canvas_image_move(map->bg_item, x, y, map->bg_scale.x, map->bg_scale.y);
}

static void map_button_release(map_t *map, gint x, gint y) {
  map->pen_down.is = FALSE;

  switch(map->action) {
  case MAP_ACTION_BG_ADJUST:
    map_bg_adjust(map, x, y);
    map->bg_offset.x += x - map->pen_down.at.x;
    map->bg_offset.y += y - map->pen_down.at.y;
    break;

  case MAP_ACTION_IDLE:
    /* check if distance to press is above drag limit */
    if(!map->pen_down.drag)
      map->pen_down.drag = distance_above(map, x, y, MAP_DRAG_LIMIT);
    
    if(!map->pen_down.drag) {
      printf("left button released after click\n");
      
      map_item_t old_sel = map->selected;
      map_handle_click(map->appdata, map);
      
      if((old_sel.type != MAP_TYPE_ILLEGAL) && 
	 (old_sel.ptr == map->selected.ptr)) {
	printf("re-selected same item of type %d/%d, "
	       "pushing it to the bottom\n", old_sel.type, 
	       map->selected.type);
	
	if(!map->selected.item) {
	  printf("  item has no visible representation to push\n");
	} else {
	  canvas_item_to_bottom(map->selected.item);
	  
	  /* update clicked item, to correctly handle the click */
	  map->pen_down.on_item = 
	    map_real_item_at(map, map->pen_down.at.x, map->pen_down.at.y);
	  
	  map_handle_click(map->appdata, map);
	}
      }
    } else {
      printf("left button released after drag\n");
      
      /* just scroll if we didn't drag an selected item */
      if(!map->pen_down.on_selected_node)
	map_do_scroll(map, x, y);
      else {
	printf("released after dragging node\n");
	map_hl_cursor_clear(map);
	map_hl_touchnode_clear(map);
	
	/* now actually move the node */
	map_node_move(map->appdata, map->pen_down.on_item, x, y);
      }
    }
    break;

  case MAP_ACTION_NODE_ADD:
    printf("released after NODE ADD\n");
    map_hl_cursor_clear(map);

    /* convert mouse position to canvas (world) position */
    canvas_window2world(map->canvas, x, y, &x, &y);
    node_t *node = osm_node_new(map->appdata->osm, x, y);
    map_node_draw(map, node);

    map_action_set(map->appdata, MAP_ACTION_IDLE);

    map_item_deselect(map->appdata);
    map_node_select(map->appdata, node);
    break;

  default:
    break;
  }
}

static gboolean map_button_event(GtkWidget *widget, GdkEventButton *event,
				       gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  map_t *map = appdata->map;

  if(event->button == 1) {
    gint x = event->x, y = event->y;

    if(event->type == GDK_BUTTON_PRESS) 
      map_button_press(map, x, y);

    if(event->type == GDK_BUTTON_RELEASE) 
      map_button_release(map, x, y);
  }

  return FALSE;  /* forward to further processing */
}

static void map_touchnode_update(appdata_t *appdata, gint x, gint y) {
  map_t *map = appdata->map;

  map_hl_touchnode_clear(map);

  /* check if we are close to another node */
  g_assert(map->pen_down.on_item);
  g_assert(map->pen_down.on_item->type == MAP_TYPE_NODE);

  /* check if we are close to one of the other nodes */
  canvas_window2world(appdata->map->canvas, x, y, &x, &y);
  node_t *node = appdata->osm->node;
  while(node) {
    /* don't highlight the dragged node itself and don't highlight */
    /* deleted ones */
    if((node != map->pen_down.on_item->node) && 
       (!(node->flags & OSM_FLAG_DELETED))) {
      gint nx = x - node->lpos.x;
      gint ny = y - node->lpos.y;
	  
      if(nx*nx + ny*ny < RADIUS*RADIUS)
	map_hl_touchnode_draw(map, node->lpos.x, node->lpos.y);

    }
    node = node->next;
  }
}


static gboolean map_motion_notify_event(GtkWidget *widget, 
                             GdkEventMotion *event, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  map_t *map = appdata->map;
  gint x, y;
  GdkModifierType state;

#ifdef USE_HILDON
  /* reduce update frequency on hildon to keep screen update fluid */
  static guint32 last_time = 0;

  if(event->time - last_time < 100) return FALSE;
  last_time = event->time;
#endif

  if(!map->pen_down.is) 
    return FALSE;

#ifdef USE_GNOMECANVAS
  /* handle hints, hints are handled by goocanvas directly */
  if(event->is_hint)
    gdk_window_get_pointer(event->window, &x, &y, &state);
  else 
#endif
  {
    x = event->x;
    y = event->y;
    state = event->state;
  }

  /* check if distance to press is above drag limit */
  if(!map->pen_down.drag)
    map->pen_down.drag = distance_above(map, x, y, MAP_DRAG_LIMIT);
  
  switch(map->action) {
  case MAP_ACTION_BG_ADJUST:
    map_bg_adjust(map, x, y);
    break;

  case MAP_ACTION_IDLE:
    if(map->pen_down.drag) {
      /* just scroll if we didn't drag an selected item */
      if(!map->pen_down.on_selected_node)
	map_do_scroll(map, x, y);
      else {
	map_hl_cursor_draw(map, x, y);
	map_touchnode_update(appdata, x, y);
      }
    }
    break;
    
  case MAP_ACTION_NODE_ADD:
    map_hl_cursor_draw(map, x, y);
    break;
    
  default:
    break;
  }


  return FALSE;  /* forward to further processing */
}

gboolean map_key_press_event(appdata_t *appdata, GdkEventKey *event) {
  /* map needs to be there to handle buttons */
  if(!appdata->map->canvas) 
    return FALSE;

  if(event->type == GDK_KEY_PRESS) {
    switch(event->keyval) {

#ifdef USE_HILDON
    case HILDON_HARDKEY_INCREASE:
#else
    case '+':
#endif
      map_set_zoom(appdata->map, appdata->map->zoom*2);
      printf("zoom is now %f\n", appdata->map->zoom);
      return TRUE;
      break;

#ifdef USE_HILDON
    case HILDON_HARDKEY_DECREASE:
#else
    case '-':
#endif
      map_set_zoom(appdata->map, appdata->map->zoom/2);
      printf("zoom is now %f\n", appdata->map->zoom);
      return TRUE;
      break;
    }

    printf("key event %d\n", event->keyval);
  }

  return FALSE;
}

GtkWidget *map_new(appdata_t *appdata) {
  map_t *map = appdata->map = g_new0(map_t, 1);

  map->zoom = 0.25;
  map->pen_down.at.x = -1;
  map->pen_down.at.y = -1;
  map->appdata = appdata;
  map->action = MAP_ACTION_IDLE;

#ifdef USE_GNOMECANVAS
  map->canvas = gnome_canvas_new_aa();  // _aa

  /* create the groups */
  canvas_group_t group;
  for(group = 0; group < CANVAS_GROUPS; group++) 
    map->group[group] = gnome_canvas_item_new(
	       	gnome_canvas_root(GNOME_CANVAS(map->canvas)),
	       	GNOME_TYPE_CANVAS_GROUP,
	       	NULL);       

  gtk_widget_modify_bg(map->canvas, GTK_STATE_NORMAL, 
		       &map->canvas->style->white);

#else
  map->canvas = goo_canvas_new();

  g_object_set(G_OBJECT(map->canvas), "anchor", GTK_ANCHOR_CENTER, NULL);

  GooCanvasItem *root = goo_canvas_get_root_item(GOO_CANVAS(map->canvas));

  /* create the groups */
  canvas_group_t group;
  for(group = 0; group < CANVAS_GROUPS; group++) 
    map->group[group] = goo_canvas_group_new(root, NULL);       

#endif

  gtk_widget_set_events(map->canvas,
                          GDK_BUTTON_PRESS_MASK
			| GDK_BUTTON_RELEASE_MASK
			| GDK_SCROLL_MASK
			| GDK_POINTER_MOTION_MASK
    			| GDK_POINTER_MOTION_HINT_MASK);
  
  gtk_signal_connect(GTK_OBJECT(map->canvas), 
     "button_press_event", G_CALLBACK(map_button_event), appdata);
  gtk_signal_connect(GTK_OBJECT(map->canvas), 
     "button_release_event", G_CALLBACK(map_button_event), appdata);
  gtk_signal_connect(GTK_OBJECT(map->canvas), 
     "motion_notify_event", G_CALLBACK(map_motion_notify_event), appdata);
  gtk_signal_connect(GTK_OBJECT(map->canvas), 
     "scroll_event", G_CALLBACK(map_scroll_event), appdata);

  gtk_signal_connect(GTK_OBJECT(map->canvas), 
     "destroy", G_CALLBACK(map_destroy_event), appdata);

  return map->canvas;
}

void map_init(appdata_t *appdata) {
  map_t *map = appdata->map;

  /* set initial zoom */
  map_set_zoom(map, map->zoom);
  map_init_colors(appdata->potlatch, appdata->osm);
  map_draw(map, appdata->osm);

  canvas_set_bounds(map->canvas,
	appdata->osm->bounds->min.x, appdata->osm->bounds->min.y,
	appdata->osm->bounds->max.x, appdata->osm->bounds->max.y);

  canvas_scroll_to(map->canvas, 0,0);
}


void map_item_set_flags(map_item_t *map_item, int set, int clr) {

  switch(map_item->type) {
  case MAP_TYPE_NODE:
    map_item->node->flags |=  set;
    map_item->node->flags &= ~clr;
    break;

  case MAP_TYPE_WAY:
    map_item->way->flags |=  set;
    map_item->way->flags &= ~clr;
    break;

  default:
    g_assert(0);
    break;
  }
}

void map_clear(appdata_t *appdata) {
  map_t *map = appdata->map;

  printf("freeing map contents\n");

  /* remove a possibly existing highlight */
  map_item_deselect(appdata);
  
  canvas_group_t group;
  for(group=0;group<CANVAS_GROUPS;group++) {

    /* destroy the entire group */
    canvas_item_destroy(map->group[group]);

#ifdef USE_GNOMECANVAS
    /* and create an empty new one */
    map->group[group] = gnome_canvas_item_new(
		      gnome_canvas_root(GNOME_CANVAS(map->canvas)),
		      GNOME_TYPE_CANVAS_GROUP,
		      NULL);       
#else
    GooCanvasItem *root = goo_canvas_get_root_item(GOO_CANVAS(map->canvas));
    map->group[group] = goo_canvas_group_new(root, NULL);       
#endif
  }
}

void map_paint(appdata_t *appdata) {
  map_t *map = appdata->map;

  map_init_colors(appdata->potlatch, appdata->osm);
  map_draw(map, appdata->osm);
}

/* called from several icons like e.g. "node_add" */
void map_action_set(appdata_t *appdata, map_action_t action) {
  printf("map action set to %d\n", action);

  /* returning from this action */
  switch(appdata->map->action) {
  case MAP_ACTION_BG_ADJUST:
    gtk_widget_set_sensitive(appdata->menu_item_wms_adjust, TRUE);
    break;
  default:
    break;
  }

  appdata->map->action = action;

  /* enable/disable ok/cancel buttons */
  // MAP_ACTION_IDLE=0, _NODE_ADD, _BG_ADJUST
  const gboolean ok_state[] = { FALSE, FALSE, TRUE };
  const gboolean cancel_state[] = { FALSE, TRUE, TRUE };
  g_assert(action < sizeof(ok_state)/sizeof(gboolean));

  icon_bar_map_cancel_ok(appdata, cancel_state[action], ok_state[action]);

  switch(action) {
  case MAP_ACTION_BG_ADJUST:
    /* an existing selection only causes confusion ... */
    gtk_widget_set_sensitive(appdata->menu_item_wms_adjust, FALSE);
    map_item_deselect(appdata);
    break;

  default:
    break;
  }

  icon_bar_map_action_idle(appdata, action == MAP_ACTION_IDLE);

  const char *str_state[] = { 
    NULL, 
    _("Please place a node"),
    _("Adjust background image position"),
  };

  statusbar_set(appdata, str_state[action]);
}

void map_action_cancel(appdata_t *appdata) {
  map_t *map = appdata->map;

  switch(map->action) {
  case MAP_ACTION_BG_ADJUST:
    /* undo all changes to bg_offset */
    map->bg_offset.x = appdata->project->wms_offset.x;
    map->bg_offset.y = appdata->project->wms_offset.y;

    gint x = map->appdata->osm->bounds->min.x + map->bg_offset.x;
    gint y = map->appdata->osm->bounds->min.y + map->bg_offset.y;
    canvas_image_move(map->bg_item, x, y, map->bg_scale.x, map->bg_scale.y);
    break;

  default:
    break;
  }

  map_action_set(appdata, MAP_ACTION_IDLE);
}

void map_action_ok(appdata_t *appdata) {
  map_t *map = appdata->map;

  switch(map->action) {
  case MAP_ACTION_BG_ADJUST:
    /* save changes to bg_offset in project */
    appdata->project->wms_offset.x = map->bg_offset.x;
    appdata->project->wms_offset.y = map->bg_offset.y;
    appdata->project->dirty = TRUE;
    break;

  default:
    break;
  }

  map_action_set(appdata, MAP_ACTION_IDLE);
}

static void map_item_chain_destroy(map_item_chain_t *chain) {
  while(chain) {
    map_item_chain_t *next = chain->next;
    canvas_item_destroy(chain->map_item->item);
    chain = next;
  }
}

/* called from icon "trash" */
void map_delete_selected(appdata_t *appdata) {
  map_t *map = appdata->map;

  /* work on local copy since de-selecting destroys the selection */
  map_item_t item = map->selected;

  /* deleting the selected item de-selects it ... */
  map_item_deselect(appdata);

  switch(item.type) {
  case MAP_TYPE_NODE:
    printf("request to delete node #%ld\n", item.node->id);

    /* check if this node is part of a way with two nodes only. */
    /* we cannot delete this as this would also delete the way */
    way_chain_t *way_chain = osm_node_to_way(appdata->osm, item.node);
    if(way_chain) {
      gboolean short_way = FALSE;
      
      /* free the chain of relations */
      while(way_chain && !short_way) {
	way_chain_t *next = way_chain->next;

	if(osm_way_number_of_nodes(way_chain->way) <= 2) 
	  short_way = TRUE;

	g_free(way_chain);
	way_chain = next;
      }

      if(short_way) {
	messagef(GTK_WIDGET(appdata->window), _("Node in short way"), 
		 _("This node cannot be deleted (yet) as it's part of a way "
		   "consisting of only two nodes."));
	return;
      }
    }

    /* check if this node is part of a relation */
    relation_chain_t *rel_chain = osm_node_to_relation(appdata->osm, item.node);
    if(rel_chain) {
      
      /* free the chain of relations */
      while(rel_chain) {
	relation_chain_t *next = rel_chain->next;
	g_free(rel_chain);
	rel_chain = next;
      }

      messagef(GTK_WIDGET(appdata->window), _("Node in relation"), 
       _("This node cannot be deleted (yet) as it's part of a relation"));

      return;
    }

    /* remove it visually from the screen */
    map_item_chain_destroy(item.node->map_item_chain);

    /* and mark it "deleted" in the database */
    way_chain_t *chain = osm_node_delete(appdata->osm, item.node, FALSE);

    /* redraw all affected ways */
    while(chain) {
      way_chain_t *next = chain->next;

      if(osm_way_number_of_nodes(chain->way) == 1) {
	printf("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX\n");
	printf("TODO: Only 1 node left on way -> delete way also\n");
	printf("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX\n");
      }

      map_item_t item;
      item.type = MAP_TYPE_WAY;
      item.way = chain->way;
      map_item_redraw(appdata, &item);

      g_free(chain);

      chain = next;
    }

    break;

  case MAP_TYPE_WAY:
    printf("NYI: request to delete way #%ld\n", item.way->id);
    g_assert(0);
    break;

  default:
    g_assert((item.type == MAP_TYPE_NODE) ||
	     (item.type == MAP_TYPE_WAY));
    break;
  }
}

/* ----------------------- track related stuff ----------------------- */

void map_track_draw_seg(map_t *map, track_seg_t *seg) {
  /* a track_seg needs at least 2 points to be drawn */
  guint pnum = track_seg_points(seg);
  printf("seg of length %d\n", pnum);

  if(pnum == 1) {
    g_assert(!seg->item);

    seg->item = canvas_circle_new(map, CANVAS_GROUP_TRACK,
		  seg->track_point->lpos.x, seg->track_point->lpos.y, 
		  1.5*WAY_WIDTH, 0, TRACK_COLOR, NO_COLOR);
  }

  if(pnum > 1) {
    
    /* allocate space for nodes */
    canvas_points_t *points = canvas_points_new(pnum);
    
    int point = 0;
    track_point_t *track_point = seg->track_point;
    while(track_point) {
      points->coords[point++] = track_point->lpos.x;
      points->coords[point++] = track_point->lpos.y;
      track_point = track_point->next;
    }
    
    /* there may be a circle (one point line) */
    if(seg->item)
      canvas_item_destroy(seg->item);

    seg->item = canvas_polyline_new(map, CANVAS_GROUP_TRACK,
		  points, 3*WAY_WIDTH, TRACK_COLOR);

    canvas_points_free(points);
  }
}

void map_track_update_seg(map_t *map, track_seg_t *seg) {
  /* a track_seg needs at least 2 points to be drawn */
  guint pnum = track_seg_points(seg);
  printf("seg of length %d\n", pnum);

  if(pnum > 1) {
    
    /* allocate space for nodes */
    canvas_points_t *points = canvas_points_new(pnum);
    
    int point = 0;
    track_point_t *track_point = seg->track_point;
    while(track_point) {
      canvas_point_set_pos(points, point++, &track_point->lpos);
      track_point = track_point->next;
    }
    
    g_assert(seg->item);
    canvas_item_set_points(seg->item, points);
    canvas_points_free(points);
  }
}

void map_track_draw(map_t *map, track_t *track) {
  track_seg_t *seg = track->track_seg;

  /* draw all segments */
  while(seg) {
    map_track_draw_seg(map, seg);
    seg = seg->next;
  }
}

void map_track_remove(appdata_t *appdata) {
  track_t *track = appdata->track.track;

  printf("removing track\n");

  g_assert(track);

  /* remove all segments */
  track_seg_t *seg = track->track_seg;
  while(seg) {
    if(seg->item) {
      canvas_item_destroy(seg->item);
      seg->item = NULL;
    }

    seg = seg->next;
  }
}

void map_track_pos(appdata_t *appdata, lpos_t *lpos) {
  if(appdata->track.gps_item) {
    canvas_item_destroy(appdata->track.gps_item);
    appdata->track.gps_item = NULL;
  }

  if(lpos)
    appdata->track.gps_item = canvas_circle_new(appdata->map, CANVAS_GROUP_GPS, 
	lpos->x, lpos->y, 1.5 * WAY_WIDTH, 0, TRACK_GPS_COLOR, NO_COLOR);
}

/* ------------------- map background ------------------ */

void map_remove_bg_image(map_t *map) {
  if(!map) return;

  if(map->bg_item) {
    canvas_item_destroy(map->bg_item);
    map->bg_item = NULL;
  }
}

static gint map_bg_item_destroy_event(GtkWidget *widget, gpointer data) {
  map_t *map = (map_t*)data;

  /* destroying background item */

  map->bg_item = NULL;
  if(map->bg_pix) {
    printf("destroying background item\n");
    gdk_pixbuf_unref(map->bg_pix);
    map->bg_pix = NULL;
  }
  return FALSE;
}

void map_set_bg_image(map_t *map, char *filename) {
  bounds_t *bounds = map->appdata->osm->bounds;

  map_remove_bg_image(map);

  map->bg_pix = gdk_pixbuf_new_from_file(filename, NULL);

  /* calculate required scale factor */
  map->bg_scale.x = (float)(bounds->max.x - bounds->min.x)/
    (float)gdk_pixbuf_get_width(map->bg_pix);
  map->bg_scale.y = (float)(bounds->max.y - bounds->min.y)/
    (float)gdk_pixbuf_get_height(map->bg_pix);

  map->bg_item = canvas_image_new(map, CANVAS_GROUP_BG, map->bg_pix, 
	  bounds->min.x, bounds->min.y, map->bg_scale.x, map->bg_scale.y);

  canvas_item_destroy_connect(map->bg_item, 
          G_CALLBACK(map_bg_item_destroy_event), map);
}  
