/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of OSM2Go.
 *
 * OSM2Go is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * OSM2Go is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSM2Go.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "appdata.h"

#include <sys/stat.h>

#include <libxml/parser.h>
#include <libxml/tree.h>

#if !defined(LIBXML_TREE_ENABLED) || !defined(LIBXML_OUTPUT_ENABLED)
#error "libxml doesn't support required tree or output"
#endif

typedef struct {
  appdata_t *appdata;
  project_t *project;
  GtkWidget *dialog, *fsize, *diff_stat, *diff_remove;
  GtkWidget *desc, *server;
  GtkWidget *minlat, *minlon, *maxlat, *maxlon;
  area_edit_t area_edit;
} project_context_t;

void project_free(project_t *project) {
  if(project->name)       g_free(project->name);
  if(project->desc)       g_free(project->desc);
  if(project->server)     g_free(project->server);

  if(project->username)   g_free(project->username);
  if(project->password)   g_free(project->password);
 
  if(project->wms_server) g_free(project->wms_server);
  if(project->wms_path)   g_free(project->wms_path);
 
  if(project->path)       g_free(project->path);
  if(project->osm)        g_free(project->osm);
  g_free(project);
}

/* return file length or -1 on error */
static gsize file_length(char *name) {
  GMappedFile *gmap = g_mapped_file_new(name, FALSE, NULL);
  if(!gmap) return -1;
  gsize size = g_mapped_file_get_length(gmap); 
  g_mapped_file_free(gmap);
  return size;
}

void project_filesize(project_context_t *context) {
  char *str = NULL;

  printf("Checking size of %s\n", context->project->osm);

  if(!g_file_test(context->project->osm, G_FILE_TEST_EXISTS)) {
    str = g_strdup(_("Not downloaded!"));
    gtk_dialog_set_response_sensitive(GTK_DIALOG(context->dialog), 
				      GTK_RESPONSE_ACCEPT, 0);
  } else {
    if(!context->project->data_dirty)
      str = g_strdup_printf(_("%d bytes present"), 
			    file_length(context->project->osm));
    else
      str = g_strdup_printf(_("Outdated, please download!"));

    /* project also must not be dirty to proceed */
    gtk_dialog_set_response_sensitive(GTK_DIALOG(context->dialog), 
			GTK_RESPONSE_ACCEPT, !context->project->data_dirty);
  }

  if(str) {
    gtk_label_set_text(GTK_LABEL(context->fsize), str); 
    g_free(str);
  }
}

void project_diffstat(project_context_t *context) {
  char *str = NULL;

  if(diff_present(context->project))
    str = g_strdup(_("present"));
  else
    str = g_strdup(_("not present"));

  gtk_label_set_text(GTK_LABEL(context->diff_stat), str); 
  g_free(str);
}

static void project_update(project_context_t *context) {

  /* fetch values from dialog */
  if(context->project->desc) g_free(context->project->desc);
  context->project->desc = g_strdup(gtk_entry_get_text(
                                       GTK_ENTRY(context->desc)));
  
  if(context->project->server) g_free(context->project->server);
  context->project->server = g_strdup(gtk_entry_get_text(
				       GTK_ENTRY(context->server)));
}

static void on_edit_clicked(GtkButton *button, gpointer data) {
  project_context_t *context = (project_context_t*)data;

  if(area_edit(&context->area_edit)) {
    printf("coordinates changed!!\n");

    pos_lon_label_set(context->minlat, context->project->min.lat);
    pos_lon_label_set(context->minlon, context->project->min.lon);
    pos_lon_label_set(context->maxlat, context->project->max.lat);
    pos_lon_label_set(context->maxlon, context->project->max.lon);
  }
}

static void on_download_clicked(GtkButton *button, gpointer data) {
  project_context_t *context = (project_context_t*)data;

  project_update(context);

  printf("download %s\n", context->project->osm);

  gboolean do_it = TRUE;
  /* check if diff exists and issue a warning */
  if(diff_present(context->project)) {
    GtkWidget *dialog = gtk_message_dialog_new(
	       GTK_WINDOW(context->dialog),
	       GTK_DIALOG_DESTROY_WITH_PARENT,
	       GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
	       _("You have saved changes for the current data set. "
		 "Downloading new data may result in loss of these changes. "
		 "Do you really want to download new data?"));
      
    gtk_window_set_title(GTK_WINDOW(dialog), _("Really download?"));
      
    /* set the active flag again if the user answered "no" */
    if(GTK_RESPONSE_NO == gtk_dialog_run(GTK_DIALOG(dialog)))
      do_it = FALSE;
    
    gtk_widget_destroy(dialog);
  }

  if(!do_it) {
    printf("Download cancelled\n");
    return;
  }

  if(osm_download(context->dialog, context->appdata, context->project)) {
    context->project->data_dirty = FALSE;
    project_filesize(context);
  } else
    printf("download failed\n"); 

}

static void on_diff_remove_clicked(GtkButton *button, gpointer data) {
  project_context_t *context = (project_context_t*)data;

  printf("clicked diff remove\n");

  GtkWidget *dialog = gtk_message_dialog_new(
	     GTK_WINDOW(context->dialog),
	     GTK_DIALOG_DESTROY_WITH_PARENT,
	     GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
	       _("Do you really want to remove the diff file? This "
		 "will delete all changes you've made so far and which "
		 "you didn't upload yet."));
      
    gtk_window_set_title(GTK_WINDOW(dialog), _("Really remove?"));
      
    /* set the active flag again if the user answered "no" */
    if(GTK_RESPONSE_YES == gtk_dialog_run(GTK_DIALOG(dialog))) {
      diff_remove(context->project);
      project_diffstat(context);
      gtk_widget_set_sensitive(context->diff_remove,  FALSE);
    }

    gtk_widget_destroy(dialog);

}

void project_save(appdata_t *appdata, project_t *project) {
  char *project_file = g_strdup_printf("%s%s.proj", 
		       project->path, project->name);

  printf("saving project to %s\n", project_file);

  LIBXML_TEST_VERSION;

  xmlDocPtr doc = xmlNewDoc(BAD_CAST "1.0");
  xmlNodePtr node, root_node = xmlNewNode(NULL, BAD_CAST "proj");
  xmlNewProp(root_node, BAD_CAST "name", BAD_CAST project->name);
  if(project->data_dirty)
    xmlNewProp(root_node, BAD_CAST "dirty", BAD_CAST "true");

  xmlDocSetRootElement(doc, root_node);

  node = xmlNewChild(root_node, NULL, BAD_CAST "server", 
		     BAD_CAST project->server);
  xmlNewProp(node, BAD_CAST "username", BAD_CAST project->username);
  xmlNewProp(node, BAD_CAST "password", BAD_CAST project->password);

  xmlNewChild(root_node, NULL, BAD_CAST "desc", BAD_CAST project->desc);
  xmlNewChild(root_node, NULL, BAD_CAST "osm", BAD_CAST project->osm);

  node = xmlNewChild(root_node, NULL, BAD_CAST "min", NULL);
  char str[32];
  g_ascii_dtostr(str, sizeof(str), project->min.lat);
  xmlNewProp(node, BAD_CAST "lat", BAD_CAST str);
  g_ascii_dtostr(str, sizeof(str), project->min.lon);
  xmlNewProp(node, BAD_CAST "lon", BAD_CAST str);

  node = xmlNewChild(root_node, NULL, BAD_CAST "max", NULL);
  g_ascii_dtostr(str, sizeof(str), project->max.lat);
  xmlNewProp(node, BAD_CAST "lat", BAD_CAST str);
  g_ascii_dtostr(str, sizeof(str), project->max.lon);
  xmlNewProp(node, BAD_CAST "lon", BAD_CAST str);

  node = xmlNewChild(root_node, NULL, BAD_CAST "wms", NULL);
  xmlNewProp(node, BAD_CAST "server", BAD_CAST project->wms_server);
  xmlNewProp(node, BAD_CAST "path", BAD_CAST project->wms_path);
  snprintf(str, sizeof(str), "%d", project->wms_offset.x);
  xmlNewProp(node, BAD_CAST "x-offset", BAD_CAST str);
  snprintf(str, sizeof(str), "%d", project->wms_offset.y);
  xmlNewProp(node, BAD_CAST "y-offset", BAD_CAST str);

  xmlSaveFormatFileEnc(project_file, doc, "UTF-8", 1);
  xmlFreeDoc(doc);
  xmlCleanupParser();

  g_free(project_file);
}

project_t *project_open(appdata_t *appdata, char *name) {
  project_t *project = g_new0(project_t, 1);

  LIBXML_TEST_VERSION;

  /* build project path */
  project->path = g_strdup_printf("%s%s/", 
		  appdata->settings->base_path, name);
  project->name = g_strdup(name);

  /* make sure project base path exists */
  if(g_mkdir_with_parents(project->path, S_IRWXU) != 0) {
    errorf(GTK_WIDGET(appdata->window), 
	   _("Unable to access project path %s"), project->path);
    return NULL;
  }

  char *project_file = g_strdup_printf("%s%s.proj", project->path, name);

  printf("project file = %s\n", project_file);
  if(!g_file_test(project_file, G_FILE_TEST_EXISTS)) {
    printf("project doesn't exist, setting default values\n");

    project->desc = g_strdup(_("OSM2Go test project"));

    /* no data downloaded yet */
    project->data_dirty = TRUE;

    /* use global server/access settings */
    project->server   = g_strdup(appdata->settings->server);
    project->username = g_strdup(appdata->settings->username);
    project->password = g_strdup(appdata->settings->password);

    /* dito for wms settings */
    project->wms_server = g_strdup(appdata->settings->wms_server);
    project->wms_path   = g_strdup(appdata->settings->wms_path);

    /* build project osm file name */
    project->osm = g_strdup_printf("%s%s.osm", project->path, name);

    /* around my home ... */
    project->min.lat = 49.02;  project->min.lon = 8.38;
    project->max.lat = 49.04;  project->max.lon = 8.40;

  } else {
    xmlDoc *doc = NULL;
    xmlNode *root_element = NULL;

    /* parse the file and get the DOM */
    if((doc = xmlReadFile(project_file, NULL, 0)) == NULL) {
      printf("error: could not parse file %s\n", project_file);
      project_free(project);
      g_free(project_file);
      return NULL;
    }
  
    /* Get the root element node */
    root_element = xmlDocGetRootElement(doc);

    xmlNode *cur_node = NULL;
    for (cur_node = root_element; cur_node; cur_node = cur_node->next) {
      if (cur_node->type == XML_ELEMENT_NODE) {
	if(strcasecmp((char*)cur_node->name, "proj") == 0) {
	  char *str;

	  if((str = (char*)xmlGetProp(cur_node, BAD_CAST "dirty"))) {
	    project->data_dirty = (strcasecmp(str, "true") == 0);
	    xmlFree(str);
	  } else
	    project->data_dirty = FALSE;

	  xmlNode *node = cur_node->children;

	  while(node != NULL) {
	    if(node->type == XML_ELEMENT_NODE) {

	      if(strcasecmp((char*)node->name, "desc") == 0) {
		str = (char*)xmlNodeListGetString(doc, node->children, 1);
		project->desc = g_strdup(str);
		printf("desc = %s\n", project->desc);
		xmlFree(str);
	      } else if(strcasecmp((char*)node->name, "server") == 0) {
		str = (char*)xmlNodeListGetString(doc, node->children, 1);
		project->server = g_strdup(str);
		printf("server = %s\n", project->server);
		xmlFree(str);

		if((str = (char*)xmlGetProp(node, BAD_CAST "username"))) {
		  project->username = g_strdup(str);
		  xmlFree(str);
		} 
		if((str = (char*)xmlGetProp(node, BAD_CAST "password"))) {
		  project->password = g_strdup(str);
		  xmlFree(str);
		} 

	      } else if(strcasecmp((char*)node->name, "wms") == 0) {
		if((str = (char*)xmlGetProp(node, BAD_CAST "server"))) {
		  project->wms_server = g_strdup(str);
		  xmlFree(str);
		} 
		if((str = (char*)xmlGetProp(node, BAD_CAST "path"))) {
		  project->wms_path = g_strdup(str);
		  xmlFree(str);
		} 
		if((str = (char*)xmlGetProp(node, BAD_CAST "x-offset"))) {
		  project->wms_offset.x = strtoul(str, NULL, 10);
		  xmlFree(str);
		} 
		if((str = (char*)xmlGetProp(node, BAD_CAST "y-offset"))) {
		  project->wms_offset.y = strtoul(str, NULL, 10);
		  xmlFree(str);
		} 

	      } else if(strcasecmp((char*)node->name, "osm") == 0) {
		str = (char*)xmlNodeListGetString(doc, node->children, 1);
		project->osm = g_strdup(str);
		printf("osm = %s\n", project->osm);
		xmlFree(str);
	      } else if(strcasecmp((char*)node->name, "min") == 0) {
		if((str = (char*)xmlGetProp(node, BAD_CAST "lat"))) {
		  project->min.lat = g_ascii_strtod(str, NULL);
		  xmlFree(str);
		} 
		if((str = (char*)xmlGetProp(node, BAD_CAST "lon"))) {
		  project->min.lon = g_ascii_strtod(str, NULL);
		  xmlFree(str);
		}
	      } else if(strcasecmp((char*)node->name, "max") == 0) {
		if((str = (char*)xmlGetProp(node, BAD_CAST "lat"))) {
		  project->max.lat = g_ascii_strtod(str, NULL);
		  xmlFree(str);
		}
		if((str = (char*)xmlGetProp(node, BAD_CAST "lon"))) {
		  project->max.lon = g_ascii_strtod(str, NULL);
		  xmlFree(str);
		}
	      }
	    }
	    node = node->next;
	  }
	}
      }
    }
  }

  /* ------------ set some default that may be missing ----------- */
  /* ------- e.g. from project files saved by old versions ------- */
  if(!project->wms_server)
    project->wms_server = g_strdup(appdata->settings->wms_server);

  if(!project->wms_path)
    project->wms_path = g_strdup(appdata->settings->wms_path);


  /* ------------ project dialog ------------- */

  project_context_t *context = g_new0(project_context_t, 1);
  context->appdata = appdata;
  context->project = project;

  context->area_edit.min = &project->min;
  context->area_edit.max = &project->max;

  char *str = g_strdup_printf(_("Project - %s"), name);
  context->area_edit.parent = 
    context->dialog = gtk_dialog_new_with_buttons(str,
	  GTK_WINDOW(appdata->window), GTK_DIALOG_MODAL,
          GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
	  GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, NULL);
  g_free(str);

#ifdef USE_HILDON
  /* making the dialog a little wider makes it less "crowded" */
  gtk_window_set_default_size(GTK_WINDOW(context->dialog), 640, 100);
#else
  gtk_window_set_default_size(GTK_WINDOW(context->dialog), 400, 100);
#endif

  GtkWidget *download, *label;
  GtkWidget *table = gtk_table_new(4, 6, FALSE);  // x, y

  label = gtk_label_new(_("Description:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
  gtk_table_attach_defaults(GTK_TABLE(table),  label, 0, 1, 0, 1);
  context->desc = gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(context->desc), project->desc);
  gtk_table_attach_defaults(GTK_TABLE(table),  context->desc, 1, 4, 0, 1);

  gtk_table_set_row_spacing(GTK_TABLE(table), 0, 4);

  label = gtk_label_new(_("Latitude"));
  gtk_table_attach_defaults(GTK_TABLE(table),  label, 1, 2, 1, 2);
  label = gtk_label_new(_("Longitude"));
  gtk_table_attach_defaults(GTK_TABLE(table),  label, 2, 3, 1, 2);

  label = gtk_label_new(_("Min:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
  gtk_table_attach_defaults(GTK_TABLE(table),  label, 0, 1, 2, 3);
  context->minlat = pos_lat_label_new(project->min.lat);
  gtk_table_attach_defaults(GTK_TABLE(table), context->minlat, 1, 2, 2, 3);
  context->minlon = pos_lon_label_new(project->min.lon);
  gtk_table_attach_defaults(GTK_TABLE(table), context->minlon, 2, 3, 2, 3);

  label = gtk_label_new(_("Max:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
  gtk_table_attach_defaults(GTK_TABLE(table),  label, 0, 1, 3, 4);
  context->maxlat = pos_lat_label_new(project->max.lat);
  gtk_table_attach_defaults(GTK_TABLE(table), context->maxlat, 1, 2, 3, 4);
  context->maxlon = pos_lon_label_new(project->max.lon);
  gtk_table_attach_defaults(GTK_TABLE(table), context->maxlon, 2, 3, 3, 4);

  GtkWidget *edit = gtk_button_new_with_label(_("Edit..."));
  gtk_signal_connect(GTK_OBJECT(edit), "clicked",
  		     (GtkSignalFunc)on_edit_clicked, context);
  gtk_table_attach(GTK_TABLE(table), edit, 3, 4, 2, 4, GTK_EXPAND | GTK_FILL,0,0,0);

  gtk_table_set_row_spacing(GTK_TABLE(table), 3, 4);

  label = gtk_label_new(_("Server:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
  gtk_table_attach_defaults(GTK_TABLE(table),  label, 0, 1, 4, 5);
  context->server = gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(context->server), project->server);
  gtk_table_attach_defaults(GTK_TABLE(table),  context->server, 1, 4, 4, 5);

  label = gtk_label_new(_("OSM file:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
  gtk_table_attach_defaults(GTK_TABLE(table),  label, 0, 1, 5, 6);
  GtkWidget *hbox = gtk_hbox_new(FALSE, 0);
  context->fsize = gtk_label_new(_(""));
  gtk_misc_set_alignment(GTK_MISC(context->fsize), 0.f, 0.5f);
  project_filesize(context);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), context->fsize);
  download = gtk_button_new_with_label(_("Download..."));
  gtk_signal_connect(GTK_OBJECT(download), "clicked",
		     (GtkSignalFunc)on_download_clicked, context);
  gtk_box_pack_start(GTK_BOX(hbox), download, FALSE, FALSE, 0);
  gtk_table_attach_defaults(GTK_TABLE(table), hbox, 1, 4, 5, 6);

  label = gtk_label_new(_("Diff file:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
  gtk_table_attach_defaults(GTK_TABLE(table),  label, 0, 1, 6, 7);
  hbox = gtk_hbox_new(FALSE, 0);
  context->diff_stat = gtk_label_new(_(""));
  gtk_misc_set_alignment(GTK_MISC(context->diff_stat), 0.f, 0.5f);
  project_diffstat(context);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), context->diff_stat);
  context->diff_remove = gtk_button_new_with_label(_("Remove..."));
  if(!diff_present(project)) 
    gtk_widget_set_sensitive(context->diff_remove,  FALSE);
  gtk_signal_connect(GTK_OBJECT(context->diff_remove), "clicked",
		     (GtkSignalFunc)on_diff_remove_clicked, context);
  gtk_box_pack_start(GTK_BOX(hbox), context->diff_remove, FALSE, FALSE, 0);
  gtk_table_attach_defaults(GTK_TABLE(table), hbox, 1, 4, 6, 7);
  

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(context->dialog)->vbox), 
			      table);
  gtk_widget_show_all(context->dialog);

  if(GTK_RESPONSE_ACCEPT == gtk_dialog_run(GTK_DIALOG(context->dialog))) {
    /* todo: save project */
    project_update(context);

    project_save(appdata, project);
  } else {
    project_free(project);
    project = NULL;
  }


  gtk_widget_destroy(context->dialog);

  g_free(context);
  g_free(project_file);
  return project;
}

