/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of OSM2Go.
 *
 * OSM2Go is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * OSM2Go is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSM2Go.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ctype.h>

#include "appdata.h"

#define TAG_STATE  GTK_STATE_PRELIGHT

void pos_lat_str(char *str, int len, float latitude) {
  if(len > 10) len = 10;
  g_ascii_dtostr(str, len, latitude);
}

void pos_lon_str(char *str, int len, float longitude) {
  if(len > 10) len = 10;
  g_ascii_dtostr(str, len, longitude);
}

float pos_parse_lat(char *str) {
  return g_ascii_strtod(str, NULL);
}

float pos_parse_lon(char *str) {
  return g_ascii_strtod(str, NULL);
}

static gboolean mark(GtkWidget *widget, gboolean valid) {
  gtk_widget_set_state(widget, valid?GTK_STATE_NORMAL:TAG_STATE);
  return valid;
}

static void callback_modified_lat(GtkWidget *widget, gpointer data ) {
  float i = pos_parse_lat((char*)gtk_entry_get_text(GTK_ENTRY(widget)));
  mark(widget, !isnan(i));
}

/* a entry that is colored red when being "active" */
GtkWidget *pos_lat_entry_new(float lat) {
  GdkColor color;
  GtkWidget *widget = gtk_entry_new();
  gdk_color_parse("#ff0000", &color);
  gtk_widget_modify_text(widget, TAG_STATE, &color);

  char str[32];
  pos_lat_str(str, sizeof(str), lat);
  gtk_entry_set_text(GTK_ENTRY(widget), str);

  g_signal_connect(G_OBJECT(widget), "changed",
                   G_CALLBACK(callback_modified_lat), NULL);

  return widget;
}

static void callback_modified_lon(GtkWidget *widget, gpointer data ) {
  float i = pos_parse_lon((char*)gtk_entry_get_text(GTK_ENTRY(widget)));
  mark(widget, !isnan(i));
}

/* a entry that is colored red when filled with invalid coordinate */
GtkWidget *pos_lon_entry_new(float lon) {
  GdkColor color;
  GtkWidget *widget = gtk_entry_new();
  gdk_color_parse("#ff0000", &color);
  gtk_widget_modify_text(widget, TAG_STATE, &color);

  char str[32];
  pos_lon_str(str, sizeof(str), lon);
  gtk_entry_set_text(GTK_ENTRY(widget), str);

  g_signal_connect(G_OBJECT(widget), "changed",
                   G_CALLBACK(callback_modified_lon), NULL);

  return widget;
}

float pos_lat_get(GtkWidget *widget) {
  char *p = (char*)gtk_entry_get_text(GTK_ENTRY(widget));
  return pos_parse_lat(p);
}

float pos_lon_get(GtkWidget *widget) {
  char *p = (char*)gtk_entry_get_text(GTK_ENTRY(widget));
  return pos_parse_lon(p);
}

void pos2lpos(bounds_t *bounds, pos_t *pos, lpos_t *lpos) {
  lpos->x = POS_EQ_RADIUS * DEG2RAD(pos->lon);
  lpos->y = POS_EQ_RADIUS * log(tan(M_PI/4 + DEG2RAD(pos->lat)/2)); 

  lpos->x = ( lpos->x - bounds->center.x) * bounds->scale;
  lpos->y = (-lpos->y + bounds->center.y) * bounds->scale;
}

/* the maps center is special as it isn't offset (by itself) */
void pos2lpos_center(pos_t *pos, lpos_t *lpos) {
  lpos->x = POS_EQ_RADIUS * DEG2RAD(pos->lon);
  lpos->y = POS_EQ_RADIUS * log(tan(M_PI/4 + DEG2RAD(pos->lat)/2)); 
}

void lpos2pos(bounds_t *bounds, lpos_t *lpos, pos_t *pos) {
  lpos_t tmp = *lpos;

  tmp.x = ( tmp.x/bounds->scale) + bounds->center.x;
  tmp.y = (-tmp.y/bounds->scale) + bounds->center.y;

  pos->lon = RAD2DEG(tmp.x / POS_EQ_RADIUS);
  pos->lat = RAD2DEG(2 * atan(exp(tmp.y/POS_EQ_RADIUS)) - M_PI/2);
}
