/*
 * This file is part of OSM2Go.
 *
 * OSM2Go is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * OSM2Go is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSM2Go.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <string.h>

#include <curl/curl.h>

#include "appdata.h"

void messagef(GtkWidget *parent, char *title, const char *fmt, ...) {
  va_list args;
  va_start( args, fmt );
  char *buf = g_strdup_vprintf(fmt, args);
  va_end( args );

  printf("%s: \"%s\"\n", title, buf); 
  
  GtkWidget *dialog = gtk_message_dialog_new(
		     GTK_WINDOW(parent),
                     GTK_DIALOG_DESTROY_WITH_PARENT,
                     GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
                     buf);

  gtk_window_set_title(GTK_WINDOW(dialog), title);

  gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);

  g_free(buf);
}

void errorf(GtkWidget *parent, const char *fmt, ...) {
  va_list args;
  va_start( args, fmt );
  char *buf = g_strdup_vprintf(fmt, args);
  va_end( args );

  printf("errorf(\"%s\")\n", buf); 
  
  GtkWidget *dialog = gtk_message_dialog_new(
		     GTK_WINDOW(parent),
                     GTK_DIALOG_DESTROY_WITH_PARENT,
                     GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
                     buf);

  gtk_window_set_title(GTK_WINDOW(dialog), _("ERROR"));

  gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);

  g_free(buf);
}

/******************** begin of menu *********************/

static void 
cb_menu_about(GtkWidget *window, gpointer data) {
  GtkAboutDialog *about = GTK_ABOUT_DIALOG(gtk_about_dialog_new());

  gtk_about_dialog_set_name(about, "OSM2Go");
  gtk_about_dialog_set_version(about, VERSION);
  gtk_about_dialog_set_copyright(about, 
	 _("OSM2Go (c) 2008 by\n" 
	 "Till Harbaum <till@harbaum.org>"));

  gtk_about_dialog_set_website(about,
       _("http://www.harbaum.org/till/maemo"));
  
  gtk_about_dialog_set_comments(about, 
       _("Maemo OSM Editor"));

  gtk_widget_show_all(GTK_WIDGET(about));
  gtk_dialog_run(GTK_DIALOG(about));
  gtk_widget_destroy(GTK_WIDGET(about));
}

void on_window_destroy (GtkWidget *widget, gpointer data);

static void 
cb_menu_quit(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  gtk_widget_destroy(GTK_WIDGET(appdata->window));
}

static void 
cb_menu_upload(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  if(!appdata->osm || !appdata->project) return;

  osm_upload(appdata, appdata->osm, appdata->project);
}

static void 
cb_menu_wms_import(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  wms_import(appdata);
}

static void 
cb_menu_wms_clear(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  wms_remove(appdata);
}

#if 1
static void 
cb_menu_redraw(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  /* redraw the entire map by destroying all map items and redrawing them */
  track_save(appdata->project, appdata->track.track);
  diff_save(appdata->project, appdata->osm);
  map_clear(appdata);
  osm_free(appdata->osm);

  appdata->osm = osm_parse(appdata->project->osm);
  diff_restore(appdata, appdata->project, appdata->osm);
  map_paint(appdata);

  appdata->track.track = track_restore(appdata, appdata->project);
  if(appdata->track.track)
    map_track_draw(appdata->map, appdata->track.track);
}
#endif

#ifdef USE_HILDON
static void 
cb_menu_fullscreen(GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t *)data;

  if(gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(widget)))
    gtk_window_fullscreen(GTK_WINDOW(appdata->window));
  else
    gtk_window_unfullscreen(GTK_WINDOW(appdata->window));
}
#endif

static void 
cb_menu_track_import(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  /* open a file selector */
  GtkWidget *dialog;
  
#ifdef USE_HILDON
  dialog = hildon_file_chooser_dialog_new(GTK_WINDOW(appdata->window), 
					  GTK_FILE_CHOOSER_ACTION_OPEN);
#else
  dialog = gtk_file_chooser_dialog_new (_("Import track file"),
			GTK_WINDOW(appdata->window),
			GTK_FILE_CHOOSER_ACTION_OPEN,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
			NULL);
#endif
  
  /* use path if one is present */
  if(appdata->track.import_path) 
    gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog), 
					appdata->track.import_path);

  gtk_widget_show_all (GTK_WIDGET(dialog));
  if (gtk_dialog_run (GTK_DIALOG(dialog)) == GTK_FM_OK) {
    char *filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));

    /* load a track */
    track_do(appdata, TRACK_IMPORT, filename);
    if(appdata->track.track) {

      /* save path if gpx was successfully loaded */
      char *r = strrchr(filename, '/');

      /* there is a delimiter, use everything left of it as path */
      if(r) {
	*r = 0;
	if(appdata->track.import_path) g_free(appdata->track.import_path);
	appdata->track.import_path = g_strdup(filename);
	/* restore path ... just in case ... */
	*r = '/';
      }
    }
    g_free (filename);
  }
  
  gtk_widget_destroy (dialog);
}

static void 
cb_menu_track_gps(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  if(gtk_check_menu_item_get_active(
    GTK_CHECK_MENU_ITEM(appdata->track.menu_item_gps))) {
    track_do(appdata, TRACK_GPS, NULL);
  } else {
    track_do(appdata, TRACK_NONE, NULL);
  }
}

static void 
cb_menu_track_export(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  messagef(GTK_WIDGET(appdata->window), _("NIY"),
	   _("Track export is not yet supported."));
}

static void 
cb_menu_track_clear(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  track_do(appdata, TRACK_NONE, NULL);
}

void menu_create(appdata_t *appdata) { 
  GtkWidget *menu, *item, *submenu;  
  menu = gtk_menu_new();

#ifdef USE_HILDON
  item = gtk_menu_item_new_with_label( _("View") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  submenu = gtk_menu_new();
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);
  
  appdata->fullscreen_menu_item = 
    item = gtk_check_menu_item_new_with_label( _("Fullscreen") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_fullscreen), 
		   appdata);
#endif

  item = gtk_menu_item_new_with_label( _("Upload...") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_upload), appdata);

#if 1 // testing
  item = gtk_menu_item_new_with_label( _("Redraw...") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_redraw), appdata);
#endif

  /* -------------------- wms submenu -------------------- */

  item = gtk_menu_item_new_with_label( _("WMS") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  submenu = gtk_menu_new();
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);
  
  item = gtk_menu_item_new_with_label( _("Import...") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_wms_import), 
		   appdata);

  item = gtk_menu_item_new_with_label( _("Clear...") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_wms_clear), 
		   appdata);


  /* -------------------- track submenu -------------------- */

  item = gtk_menu_item_new_with_label( _("Track") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  submenu = gtk_menu_new();
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);
  
  appdata->track.menu_item_import =
    item = gtk_menu_item_new_with_label( _("Import...") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_track_import), 
		   appdata);

  appdata->track.menu_item_export =
    item = gtk_menu_item_new_with_label( _("Export...") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_track_export), 
		   appdata);

  appdata->track.menu_item_clear =
    item = gtk_menu_item_new_with_label( _("Clear") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_track_clear), 
		   appdata);

  appdata->track.menu_item_gps =
    item = gtk_check_menu_item_new_with_label( _("GPS") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_track_gps), 
		   appdata);
  
  /* ------------------------------------------------------- */

  gtk_menu_append(GTK_MENU_SHELL(menu), gtk_separator_menu_item_new());


  item = gtk_menu_item_new_with_label( _("About...") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_about), appdata);


  item = gtk_menu_item_new_with_label( _("Quit") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_quit), appdata);

#ifdef USE_HILDON
  hildon_window_set_menu(appdata->window, GTK_MENU(menu));
#else
  /* attach ordinary gtk menu */
  GtkWidget *menu_bar = gtk_menu_bar_new();

  GtkWidget *root_menu = gtk_menu_item_new_with_label (_("Menu"));
  gtk_widget_show(root_menu);

  gtk_menu_bar_append(menu_bar, root_menu); 
  gtk_menu_item_set_submenu(GTK_MENU_ITEM (root_menu), menu);

  gtk_widget_show(menu_bar);
  gtk_box_pack_start(GTK_BOX(appdata->vbox), menu_bar, 0, 0, 0);
#endif
}

/********************* end of menu **********************/


void cleanup(appdata_t *appdata) {
#ifdef USE_HILDON
  if(appdata->osso_context)
    osso_deinitialize(appdata->osso_context);

  appdata->program = NULL;
#endif

  /* let gtk clean up first */
  while(gtk_events_pending())
    gtk_main_iteration();

  map_remove_bg_image(appdata->map);

  osm_free(appdata->osm);

  curl_global_cleanup();

  icon_free_all(appdata);

  gps_release(appdata);

  puts("everything is gone");
}

void on_window_destroy (GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  gtk_main_quit();
  appdata->window = NULL;
}

gboolean on_window_key_press(GtkWidget *widget, 
			 GdkEventKey *event, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  int handled = FALSE;

  //  printf("key event %d\n", event->keyval);

  // the map handles some keys on its own ...
  switch(event->keyval) {
#ifdef USE_HILDON

#if 0
  case HILDON_HARDKEY_SELECT:
    handled = TRUE;
    break;
#endif
    
  case HILDON_HARDKEY_FULLSCREEN:
    {
      gboolean fullscreen = !gtk_check_menu_item_get_active(
	       GTK_CHECK_MENU_ITEM(appdata->fullscreen_menu_item));
      gtk_check_menu_item_set_active(
	       GTK_CHECK_MENU_ITEM(appdata->fullscreen_menu_item), fullscreen);

      if(fullscreen)
	gtk_window_fullscreen(GTK_WINDOW(appdata->window));
      else
	gtk_window_unfullscreen(GTK_WINDOW(appdata->window));

      handled = TRUE;
    }
    break;
#endif
  }

  if(!handled) 
    handled = map_key_press_event(appdata, event);
  
  return handled;
}

int main(int argc, char *argv[]) {
  appdata_t appdata;

  /* init appdata */
  memset(&appdata, 0, sizeof(appdata));

  printf("Using locale for %s in %s\n", PACKAGE, LOCALEDIR);

  setlocale(LC_ALL, "");
  bindtextdomain(PACKAGE, LOCALEDIR);
  bind_textdomain_codeset(PACKAGE, "UTF-8");
  textdomain(PACKAGE);

  /* Must initialize libcurl before any threads are started */
  curl_global_init(CURL_GLOBAL_ALL);

  g_thread_init(NULL);
  
  gps_init(&appdata);

  gtk_init (&argc, &argv);

#ifdef USE_HILDON
  printf("Installing osso context for \"org.harbaum." PACKAGE "\"\n");
  appdata.osso_context = osso_initialize("org.harbaum."PACKAGE, 
					 VERSION, TRUE, NULL);
  if(appdata.osso_context == NULL) 
    fprintf(stderr, "error initiating osso context\n");
#endif

#ifdef USE_HILDON
  /* Create the hildon program and setup the title */
  appdata.program = HILDON_PROGRAM(hildon_program_get_instance());
  g_set_application_name("OSM2Go");
  
  /* Create HildonWindow and set it to HildonProgram */
  appdata.window = HILDON_WINDOW(hildon_window_new());
  hildon_program_add_window(appdata.program, appdata.window);
#else
  /* Create a Window. */
  appdata.window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(appdata.window), "OSM2Go");
  /* Set a decent default size for the window. */
  gtk_window_set_default_size(GTK_WINDOW(appdata.window), 640, 480);
  gtk_window_set_icon(GTK_WINDOW(appdata.window), icon_load(&appdata, "osm2go"));
#endif

  g_signal_connect(G_OBJECT(appdata.window), "destroy", 
		   G_CALLBACK(on_window_destroy), &appdata);

  g_signal_connect(G_OBJECT(appdata.window), "key_press_event",
 		   G_CALLBACK(on_window_key_press), &appdata);

  appdata.vbox = gtk_vbox_new(FALSE,0);
  menu_create(&appdata);

  /* user specific init */
  appdata.settings = settings_load();  

  /* load potlatch config files to resemble potlatch view */
  appdata.potlatch = potlatch_conf_load();

  /* ----------------------- setup main window ---------------- */

  GtkWidget *hbox = gtk_hbox_new(FALSE,0);
  GtkWidget *vbox = gtk_vbox_new(FALSE,0);

  gtk_box_pack_start(GTK_BOX(vbox), map_new(&appdata), TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), statusbar_new(&appdata), FALSE, FALSE, 0);

  gtk_box_pack_start(GTK_BOX(hbox), iconbar_new(&appdata), FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), gtk_vseparator_new(), FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);

  gtk_box_pack_start(GTK_BOX(appdata.vbox), hbox, TRUE, TRUE, 0);

  gtk_container_add(GTK_CONTAINER(appdata.window), appdata.vbox);

  gtk_widget_show_all(GTK_WIDGET(appdata.window));

  /* -------------- load the project ------------- */

  appdata.project = project_open(&appdata, "test");
  if(!appdata.project) {
    cleanup(&appdata);
    return 0;
  }    

  printf("loading osm\n");
  appdata.osm = osm_parse(appdata.project->osm);
  if(!appdata.osm) {
    printf("parsing failed\n");
    cleanup(&appdata);
    return 0;
  }

  printf("parsing ok\n");

  printf("loading diff\n");
  diff_restore(&appdata, appdata.project, appdata.osm);

  map_init(&appdata);

  appdata.track.track = track_restore(&appdata, appdata.project);
  if(appdata.track.track)
    map_track_draw(appdata.map, appdata.track.track);

  wms_load(&appdata);

  /* ------------ jump into main loop ---------------- */

  gtk_main();

  track_save(appdata.project, appdata.track.track);

  /* save a diff if there are dirty entries */
  diff_save(appdata.project, appdata.osm);

  cleanup(&appdata);

  return 0;
}
