/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of OSM2Go.
 *
 * OSM2Go is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * OSM2Go is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSM2Go.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ctype.h>

#include "appdata.h"

#define TAG_STATE  GTK_STATE_PRELIGHT

void pos_lat_str(char *str, int len, float latitude) {
  char *c = _("N");
  float integral, fractional;

  if(latitude < 0) { latitude = fabs(latitude); c = _("S"); }
  fractional = modff(latitude, &integral);

  snprintf(str, len, "%s %02d° %06.3f'", c, (int)integral, fractional*60.0);
}

void pos_lon_str(char *str, int len, float longitude) {
  char *c = _("E");
  float integral, fractional;

  if(longitude < 0) { longitude = fabs(longitude); c = _("W"); }
  fractional = modff(longitude, &integral);

  snprintf(str, len, "%s %03d° %06.3f'", c, (int)integral, fractional*60.0);
}

float pos_parse_lat(char *str) {
  int integral_int;
  float fractional;
  char c;

  if(sscanf(str, "%c %d° %f'", &c, &integral_int, &fractional) == 3) {
    c = toupper(c);

    if(c != 'S' && c != 'N')
      return NAN;

    /* prevent -0.0 */
    if(!integral_int && (fractional == 0.0))
      return 0.0;

    return ((c == 'S')?-1:+1) * (integral_int + fractional/60.0);
  }

  return NAN;
}

float pos_parse_lon(char *str) {
  int integral_int;
  float fractional;
  char c;

  if(sscanf(str, "%c %d° %f'", &c, &integral_int, &fractional) == 3) {
    c = toupper(c);

    /* O is german "Ost" for "East" */
    if(c != 'E' && c != 'W' && c != 'O')
      return NAN;

    /* prevent -0.0 */
    if(!integral_int && (fractional == 0.0))
      return 0.0;

    return ((c == 'W')?-1:+1) * (integral_int + fractional/60.0);
  }

  return NAN;
}

static gboolean mark(GtkWidget *widget, gboolean valid) {
  gtk_widget_set_state(widget, valid?GTK_STATE_NORMAL:TAG_STATE);
  return valid;
}

static void callback_modified_lat(GtkWidget *widget, gpointer data ) {
  float i = pos_parse_lat((char*)gtk_entry_get_text(GTK_ENTRY(widget)));
  mark(widget, !isnan(i));
}

/* a entry that is colored red when being "active" */
GtkWidget *pos_lat_entry_new(float lat) {
  GdkColor color;
  GtkWidget *widget = gtk_entry_new();
  gdk_color_parse("#ff0000", &color);
  gtk_widget_modify_text(widget, TAG_STATE, &color);

  char str[32];
  pos_lat_str(str, sizeof(str), lat);
  gtk_entry_set_text(GTK_ENTRY(widget), str);

  g_signal_connect(G_OBJECT(widget), "changed",
                   G_CALLBACK(callback_modified_lat), NULL);

  return widget;
}

static void callback_modified_lon(GtkWidget *widget, gpointer data ) {
  float i = pos_parse_lon((char*)gtk_entry_get_text(GTK_ENTRY(widget)));
  mark(widget, !isnan(i));
}

/* a entry that is colored red when filled with invalid coordinate */
GtkWidget *pos_lon_entry_new(float lon) {
  GdkColor color;
  GtkWidget *widget = gtk_entry_new();
  gdk_color_parse("#ff0000", &color);
  gtk_widget_modify_text(widget, TAG_STATE, &color);

  char str[32];
  pos_lon_str(str, sizeof(str), lon);
  gtk_entry_set_text(GTK_ENTRY(widget), str);

  g_signal_connect(G_OBJECT(widget), "changed",
                   G_CALLBACK(callback_modified_lon), NULL);

  return widget;
}

float pos_lat_get(GtkWidget *widget) {
  char *p = (char*)gtk_entry_get_text(GTK_ENTRY(widget));
  return pos_parse_lat(p);
}

float pos_lon_get(GtkWidget *widget) {
  char *p = (char*)gtk_entry_get_text(GTK_ENTRY(widget));
  return pos_parse_lon(p);
}
