/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of OSM2Go.
 *
 * OSM2Go is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * OSM2Go is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSM2Go.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#define __USE_XOPEN
#include <time.h>

#include <libxml/parser.h>
#include <libxml/tree.h>

#include "appdata.h"

#ifndef LIBXML_TREE_ENABLED
#error "Tree not enabled in libxml"
#endif

/* determine where a node/way/relation read from the osm file */
/* is inserted into the internal database */
// #define OSM_SORT_ID
#define OSM_SORT_LAST
// #define OSM_SORT_FIRST

/* ------------------------- user handling --------------------- */

static void osm_bounds_free(bounds_t *bounds) {
  free(bounds);
}

static void osm_bounds_dump(bounds_t *bounds) {
  printf("\nBounds: %f->%f %f->%f\n", 
	 bounds->ll_min.lat, bounds->ll_max.lat, 
	 bounds->ll_min.lon, bounds->ll_max.lon);

  char str[32];
  utm2str(str, sizeof(str), &bounds->center);
  printf("   -> map center is at %s\n", str);
}

static bounds_t *osm_parse_osm_bounds(osm_t *osm, 
		     xmlDocPtr doc, xmlNode *a_node) {
  char *prop;

  if(osm->bounds) {
    errorf(NULL, "Doubly defined bounds");
    return NULL;
  }

  bounds_t *bounds = g_new0(bounds_t, 1);

  bounds->ll_min.lat = bounds->ll_min.lon = NAN;
  bounds->ll_max.lat = bounds->ll_max.lon = NAN;

  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"minlat"))) {
    bounds->ll_min.lat = g_ascii_strtod(prop, NULL);
    xmlFree(prop);
  }

  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"maxlat"))) {
    bounds->ll_max.lat = g_ascii_strtod(prop, NULL);
    xmlFree(prop);
  }

  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"minlon"))) {
    bounds->ll_min.lon = g_ascii_strtod(prop, NULL);
    xmlFree(prop);
  }

  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"maxlon"))) { 
    bounds->ll_max.lon = g_ascii_strtod(prop, NULL); 
    xmlFree(prop); 
  }

  if(isnan(bounds->ll_min.lat) || isnan(bounds->ll_min.lon) || 
     isnan(bounds->ll_max.lat) || isnan(bounds->ll_max.lon)) {
    errorf(NULL, "Invalid coordinate in bounds (%f/%f/%f/%f)",
	   bounds->ll_min.lat, bounds->ll_min.lon, 
	   bounds->ll_max.lat, bounds->ll_max.lon);

    g_free(bounds);
    return NULL;
  }


  /* calculate map zone which will be used as a reference for all */
  /* drawing/projection later on */
  pos_t center = { (bounds->ll_max.lat + bounds->ll_min.lat)/2, 
		   (bounds->ll_max.lon + bounds->ll_min.lon)/2 };

  utm_pos2utm(&center, &bounds->center, NULL);

  utm_t utm;
  utm_pos2utm(&bounds->ll_min, &utm, &bounds->center);
  bounds->min.x = utm.easting - bounds->center.easting;
  bounds->min.y = utm.northing - bounds->center.northing;
  utm_pos2utm(&bounds->ll_max, &utm, &bounds->center);
  bounds->max.x = utm.easting - bounds->center.easting;
  bounds->max.y = utm.northing - bounds->center.northing;

  return bounds;
}

/* ------------------------- user handling --------------------- */

void osm_users_free(user_t *user) {
  while(user) {
    user_t *next = user->next;

    if(user->name) g_free(user->name);
    g_free(user);

    user = next;
  }
}

void osm_users_dump(user_t *user) {
  printf("\nUser list:\n");
  while(user) {
    printf("Name: %s\n", user->name);
    user = user->next;
  }
}

static user_t *osm_user(osm_t *osm, char *name) {

  /* search through user list */
  user_t **user = &osm->user;
  while(*user && strcasecmp((*user)->name, name) < 0) 
    user = &(*user)->next;

  /* end of list or inexact match? create new user entry! */
  if(!*user || strcasecmp((*user)->name, name)) {
    user_t *new = g_new0(user_t, 1);
    new->name = g_strdup(name);
    new->next = *user;
    *user = new;

    return new;
  }

  return *user;
}

static
time_t convert_iso8601(const char *str) {
  tzset();

  struct tm ctime;
  memset(&ctime, 0, sizeof(struct tm));
  strptime(str, "%FT%T%z", &ctime);
 
  return mktime(&ctime) - timezone;
}

/* -------------------- tag handling ----------------------- */

void osm_tag_free(tag_t *tag) {
  if(tag->key)   g_free(tag->key);
  if(tag->value) g_free(tag->value);
  g_free(tag);
}

static void osm_tags_free(tag_t *tag) {
  while(tag) {
    tag_t *next = tag->next;
    osm_tag_free(tag);
    tag = next;
  }
}

static void osm_tags_dump(tag_t *tag) {
  while(tag) {
    printf("Key/Val: %s/%s\n", tag->key, tag->value);
    tag = tag->next;
  }
}

tag_t *osm_parse_osm_tag(osm_t *osm, xmlDocPtr doc, xmlNode *a_node) {
  xmlNode *cur_node = NULL;

  /* allocate a new tag structure */
  tag_t *tag = g_new0(tag_t, 1);

  char *prop;
  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"k"))) {
    if(strlen(prop) > 0) tag->key = g_strdup(prop);
    xmlFree(prop);
  }

  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"v"))) {
    if(strlen(prop) > 0) tag->value = g_strdup(prop);
    xmlFree(prop);
  }

  if(!tag->key || !tag->value) {
    printf("incomplete tag key/value %s/%s\n", tag->key, tag->value);
    osm_tags_free(tag);
    return NULL;
  }

  for (cur_node = a_node->children; cur_node; cur_node = cur_node->next) 
    if (cur_node->type == XML_ELEMENT_NODE) 
      printf("found unhandled osm/node/tag/%s\n", cur_node->name);

  return tag;
}

/* ------------------- node handling ------------------- */

static void osm_node_free(node_t *node) {
  /* there must not be anything left in this chain */
  g_assert(!node->map_item_chain);

  osm_tags_free(node->tag);
  g_free(node);
}

static void osm_nodes_free(node_t *node) {
  while(node) {
    node_t *next = node->next;
    osm_node_free(node);
    node = next;
  }
}

void osm_node_dump(node_t *node) {
  char buf[64];
  struct tm tm;
    
  printf("Id:      %lu\n", node->id);
  printf("User:    %s\n", node->user?node->user->name:"<unspecified>");
  printf("Visible: %s\n", node->visible?"yes":"no");
  
  localtime_r(&node->time, &tm);
  strftime(buf, sizeof(buf), "%a, %d %b %Y %H:%M:%S %Z", &tm);
  printf("Time:    %s\n", buf);
  osm_tags_dump(node->tag);
}

void osm_nodes_dump(node_t *node) {
  printf("\nNode list:\n");
  while(node) {
    osm_node_dump(node);
    printf("\n");
    node = node->next;
  }
}

static node_t *osm_parse_osm_node(osm_t *osm, 
			  xmlDocPtr doc, xmlNode *a_node) {
  xmlNode *cur_node = NULL;

  /* allocate a new node structure */
  node_t *node = g_new0(node_t, 1);
  node->pos.lat = node->pos.lon = NAN;

  char *prop;
  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"id"))) {
    node->id = strtoul(prop, NULL, 10);
    xmlFree(prop);
  }

  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"lat"))) {
    node->pos.lat = g_ascii_strtod(prop, NULL);
    xmlFree(prop);
  }

  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"lon"))) {
    node->pos.lon = g_ascii_strtod(prop, NULL);
    xmlFree(prop);
  }

  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"user"))) {
    node->user = osm_user(osm, prop);
    xmlFree(prop);
  }

  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"visible"))) {
    node->visible = (strcasecmp(prop, "true") == 0);
    xmlFree(prop);
  }

  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"timestamp"))) {
    node->time = convert_iso8601(prop);
    xmlFree(prop);
  }

  /* scan for tags and attach a list of tags */
  tag_t **tag = &node->tag;
  for (cur_node = a_node->children; cur_node; cur_node = cur_node->next) {
    if (cur_node->type == XML_ELEMENT_NODE) {
      if(strcasecmp((char*)cur_node->name, "tag") == 0) {
	/* attach tag to node */
      	*tag = osm_parse_osm_tag(osm, doc, cur_node);
	if(*tag) tag = &((*tag)->next);
      } else
	printf("found unhandled osm/node/%s\n", cur_node->name);
    }
  }

  /* create utm coordinates */
  utm_t utm;
  utm_pos2utm(&node->pos, &utm, &osm->bounds->center);
  node->lpos.x =   utm.easting - osm->bounds->center.easting;
  node->lpos.y = -utm.northing + osm->bounds->center.northing;

  return node;
}

/* ------------------- way handling ------------------- */

void osm_node_chain_free(node_chain_t *node_chain) {
  while(node_chain) {
    g_assert(node_chain->node->ways);

    node_chain_t *next = node_chain->next;
    node_chain->node->ways++;
    g_free(node_chain);
    node_chain = next;
  }
}

static void osm_ways_free(way_t *way) {
  while(way) {
    way_t *next = way->next;

    osm_node_chain_free(way->node_chain);
    osm_tags_free(way->tag);

    /* there must not be anything left in this chain */
    g_assert(!way->map_item_chain);

    g_free(way);

    way = next;
  }
}

void osm_way_dump(way_t *way) {
  char buf[64];
  struct tm tm;

  printf("Id:      %lu\n", way->id);
  printf("User:    %s\n", way->user?way->user->name:"<unspecified>");
  printf("Visible: %s\n", way->visible?"yes":"no");
  node_chain_t *node_chain = way->node_chain;
  while(node_chain) {
    printf("  Node:  %lu\n", node_chain->node->id);
    node_chain = node_chain->next;
  }
  
  localtime_r(&way->time, &tm);
  strftime(buf, sizeof(buf), "%a, %d %b %Y %H:%M:%S %Z", &tm);
  printf("Time:    %s\n", buf);
  osm_tags_dump(way->tag);
}

void osm_ways_dump(way_t *way) {
  printf("\nWay list:\n");
  while(way) {
    osm_way_dump(way);
    printf("\n");
    way = way->next;
  }
}

node_chain_t *osm_parse_osm_way_nd(osm_t *osm, 
			  xmlDocPtr doc, xmlNode *a_node) {
  char *prop;

  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"ref"))) {
    item_id_t id = strtoul(prop, NULL, 10);
    node_chain_t *node_chain = g_new0(node_chain_t, 1);

    /* search matching node */
    node_chain->node = osm->node;
    while(node_chain->node && node_chain->node->id != id)
      node_chain->node = node_chain->node->next;

    if(!node_chain->node) printf("Node id %lu not found\n", id);

    if(node_chain->node) 
      node_chain->node->ways++;

    xmlFree(prop);

    return node_chain;
  }

  return NULL;
}

static way_t *osm_parse_osm_way(osm_t *osm, 
			  xmlDocPtr doc, xmlNode *a_node) {
  xmlNode *cur_node = NULL;

  /* allocate a new way structure */
  way_t *way = g_new0(way_t, 1);

  char *prop;
  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"id"))) {
    way->id = strtoul(prop, NULL, 10);
    xmlFree(prop);
  }

  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"user"))) {
    way->user = osm_user(osm, prop);
    xmlFree(prop);
  }

  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"visible"))) {
    way->visible = (strcasecmp(prop, "true") == 0);
    xmlFree(prop);
  }

  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"timestamp"))) {
    way->time = convert_iso8601(prop);
    xmlFree(prop);
  }

  /* scan for tags/nodes and attach their lists */
  tag_t **tag = &way->tag;
  node_chain_t **node_chain = &way->node_chain;

  for (cur_node = a_node->children; cur_node; cur_node = cur_node->next) {
    if (cur_node->type == XML_ELEMENT_NODE) {
      if(strcasecmp((char*)cur_node->name, "tag") == 0) {
	/* attach tag to node */
      	*tag = osm_parse_osm_tag(osm, doc, cur_node);
	if(*tag) tag = &((*tag)->next);
      } else if(strcasecmp((char*)cur_node->name, "nd") == 0) {
	*node_chain = osm_parse_osm_way_nd(osm, doc, cur_node);
	if(*node_chain) 
	  node_chain = &((*node_chain)->next);
      } else
	printf("found unhandled osm/node/%s\n", cur_node->name);
    }
  }

  return way;
}

/* ------------------- relation handling ------------------- */

static void osm_relations_free(relation_t *relation) {
  while(relation) {
    relation_t *next = relation->next;

    osm_tags_free(relation->tag);

    member_t *member = relation->member;
    while(member) {
      member_t *next = member->next;
      if(member->role) g_free(member->role);
      g_free(member);
      member = next;
    }

    g_free(relation);

    relation = next;
  }
}

void osm_relations_dump(relation_t *relation) {
  printf("\nRelation list:\n");
  while(relation) {
    char buf[64];
    struct tm tm;

    printf("Id:      %lu\n", relation->id);
    printf("User:    %s\n", 
	   relation->user?relation->user->name:"<unspecified>");
    printf("Visible: %s\n", relation->visible?"yes":"no");

    member_t *member = relation->member;
    while(member) {
      switch(member->type) {
      case ILLEGAL:
      case NODE_ID:
      case WAY_ID:
	break;

      case NODE:
	if(member->node)
	  printf(" Member: Node, id = %lu, role = %s\n", 
		 member->node->id, member->role);
	break;

      case WAY:
	if(member->way)
	printf(" Member: Way, id = %lu, role = %s\n", 
	       member->way->id, member->role);
	break;
      }

      member = member->next;
    }
    
    localtime_r(&relation->time, &tm);
    strftime(buf, sizeof(buf), "%a, %d %b %Y %H:%M:%S %Z", &tm);
    printf("Time:    %s\n", buf);
    osm_tags_dump(relation->tag);
    
    printf("\n");
    relation = relation->next;
  }
}

static member_t *osm_parse_osm_relation_member(osm_t *osm, 
			  xmlDocPtr doc, xmlNode *a_node) {
  char *prop;
  member_t *member = g_new0(member_t, 1);
  member->type = ILLEGAL;

  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"type"))) {
    if(strcasecmp(prop, "way") == 0)       member->type = WAY;
    else if(strcasecmp(prop, "node") == 0) member->type = NODE;
    xmlFree(prop);
  }

  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"ref"))) {
    item_id_t id = strtoul(prop, NULL, 10);

    switch(member->type) {
    case ILLEGAL:
      printf("Unable to store illegal type\n");
      break;

    case WAY:
      /* search matching way */
      member->way = osm->way;
      while(member->way && member->way->id != id)
	member->way = member->way->next;

      if(!member->way) {
	member->type = WAY_ID;
	member->id = id;
      }
      break;

    case NODE:
      /* search matching node */
      member->node = osm->node;
      while(member->node && member->node->id != id)
	member->node = member->node->next;

      if(!member->node) {
	member->type = NODE_ID;
	member->id = id;
      }
      break;

    case WAY_ID:
    case NODE_ID:
      break;
    }

    xmlFree(prop);
  }

  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"role"))) {
    if(strlen(prop) > 0) member->role = g_strdup(prop);
    xmlFree(prop);
  }

  return member;
}

static relation_t *osm_parse_osm_relation(osm_t *osm, 
			  xmlDocPtr doc, xmlNode *a_node) {
  xmlNode *cur_node = NULL;

  /* allocate a new relation structure */
  relation_t *relation = g_new0(relation_t, 1);

  char *prop;
  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"id"))) {
    relation->id = strtoul(prop, NULL, 10);
    xmlFree(prop);
  }

  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"user"))) {
    relation->user = osm_user(osm, prop);
    xmlFree(prop);
  }

  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"visible"))) {
    relation->visible = (strcasecmp(prop, "true") == 0);
    xmlFree(prop);
  }

  if((prop = (char*)xmlGetProp(a_node, (unsigned char*)"timestamp"))) {
    relation->time = convert_iso8601(prop);
    xmlFree(prop);
  }

  /* scan for tags and attach a list of tags */
  tag_t **tag = &relation->tag;
  member_t **member = &relation->member;

  for (cur_node = a_node->children; cur_node; cur_node = cur_node->next) {
    if (cur_node->type == XML_ELEMENT_NODE) {
      if(strcasecmp((char*)cur_node->name, "tag") == 0) {
	/* attach tag to node */
      	*tag = osm_parse_osm_tag(osm, doc, cur_node);
	if(*tag) tag = &((*tag)->next);
      } else if(strcasecmp((char*)cur_node->name, "member") == 0) {
	*member = osm_parse_osm_relation_member(osm, doc, cur_node);
	if(*member) member = &((*member)->next);
      } else
	printf("found unhandled osm/node/%s\n", cur_node->name);
    }
  }

  return relation;
}

/* ----------------------- generic xml handling -------------------------- */

/* parse loc entry */
static void osm_parse_osm(osm_t *osm, xmlDocPtr doc, xmlNode * a_node) {
  xmlNode *cur_node = NULL;

  for (cur_node = a_node->children; cur_node; cur_node = cur_node->next) {
    if (cur_node->type == XML_ELEMENT_NODE) {
      if(strcasecmp((char*)cur_node->name, "bounds") == 0) 
      	osm->bounds = osm_parse_osm_bounds(osm, doc, cur_node);
      else if(strcasecmp((char*)cur_node->name, "node") == 0) {
	/* parse node and attach it to chain */
      	node_t *new = osm_parse_osm_node(osm, doc, cur_node);
	if(new) {
	  node_t **node = &osm->node;

#ifdef OSM_SORT_ID
	  /* search chain of nodes */
	  while(*node && ((*node)->id < new->id)) 
	    node = &(*node)->next;
#endif
	  
#ifdef OSM_SORT_LAST
	  while(*node) node = &(*node)->next;
#endif

	  /* insert into chain */
	  new->next = *node;
	  *node = new;
	}
      } else if(strcasecmp((char*)cur_node->name, "way") == 0) {
	/* parse way and attach it to chain */
      	way_t *new = osm_parse_osm_way(osm, doc, cur_node);
	if(new) {
	  way_t **way = &osm->way;

#ifdef OSM_SORT_ID
	  /* insert into chain */
	  while(*way && ((*way)->id < new->id))
	    way = &(*way)->next;
#endif
	  
#ifdef OSM_SORT_LAST
	  while(*way) way = &(*way)->next;
#endif

	  /* insert into chain */
	  new->next = *way;
	  *way = new;
	}
      } else if(strcasecmp((char*)cur_node->name, "relation") == 0) {
	/* parse relation and attach it to chain */
      	relation_t *new = osm_parse_osm_relation(osm, doc, cur_node);
	if(new) {
	  relation_t **relation = &osm->relation;

#ifdef OSM_SORT_ID
	  /* search chain of ways */
	  while(*relation && ((*relation)->id < new->id)) 
	    relation = &(*relation)->next;
#endif
	  
#ifdef OSM_SORT_LAST
	  while(*relation) relation = &(*relation)->next;
#endif

	  /* insert into chain */
	  new->next = *relation;
	  *relation = new;
	}
      } else
	printf("found unhandled osm/%s\n", cur_node->name);
	
    }
  }
}

/* parse root element and search for "osm" */
static osm_t *osm_parse_root(xmlDocPtr doc, xmlNode * a_node) {
  osm_t *osm;
  xmlNode *cur_node = NULL;

  /* allocate memory to hold osm file description */
  osm = g_new0(osm_t, 1);

  for (cur_node = a_node; cur_node; cur_node = cur_node->next) {
    if (cur_node->type == XML_ELEMENT_NODE) {
      /* parse osm osm file ... */
      if(strcasecmp((char*)cur_node->name, "osm") == 0) 
      	osm_parse_osm(osm, doc, cur_node);
      else 
	printf("found unhandled %s\n", cur_node->name);
    }
  }

  return osm;
}

static osm_t *osm_parse_doc(xmlDocPtr doc) {
  osm_t *osm;

  /* Get the root element node */
  xmlNode *root_element = xmlDocGetRootElement(doc);

  osm = osm_parse_root(doc, root_element);  

  /*free the document */
  xmlFreeDoc(doc);

  /*
   * Free the global variables that may
   * have been allocated by the parser.
   */
  xmlCleanupParser();

  return osm;
}

/* ------------------ osm handling ----------------- */

void osm_free(osm_t *osm) {
  if(!osm) return;

  osm_bounds_free(osm->bounds);
  osm_users_free(osm->user);
  osm_nodes_free(osm->node);
  osm_ways_free(osm->way);
  osm_relations_free(osm->relation);
  g_free(osm);
}

void osm_dump(osm_t *osm) {
  osm_bounds_dump(osm->bounds);
  osm_users_dump(osm->user);
  osm_nodes_dump(osm->node);
  osm_ways_dump(osm->way);
  osm_relations_dump(osm->relation);
}

osm_t *osm_parse(char *filename) {
  xmlDoc *doc = NULL;

  LIBXML_TEST_VERSION;

  /* parse the file and get the DOM */
  if ((doc = xmlReadFile(filename, NULL, 0)) == NULL) {
    xmlErrorPtr	errP = xmlGetLastError();
    errorf(NULL, "While parsing \"%s\":\n\n%s", filename, errP->message);
    return NULL;
  }
  
  return osm_parse_doc(doc); 
}

/* ------------------------- misc access functions -------------- */

char *osm_way_get_value(way_t *way, char *key) {
  tag_t *tag = way->tag;

  while(tag) {
    if(strcasecmp(tag->key, key) == 0)
      return tag->value;

    tag = tag->next;
  }

  return NULL;
}

char *osm_node_get_value(node_t *node, char *key) {
  tag_t *tag = node->tag;

  while(tag) {
    if(strcasecmp(tag->key, key) == 0)
      return tag->value;

    tag = tag->next;
  }

  return NULL;
}

gboolean osm_way_has_key_or_value(way_t *way, char *str) {
  tag_t *tag = way->tag;

  while(tag) {
    if(tag->key && strcasecmp(tag->key, str) == 0)
      return TRUE;

    if(tag->value && strcasecmp(tag->value, str) == 0)
      return TRUE;

    tag = tag->next;
  }

  return FALSE;
}

gboolean osm_way_is_way(way_t *way) {
  tag_t *tag = way->tag;

  while(tag) {
    if(tag->key) {
      if(strcasecmp(tag->key, "highway") == 0)
	return TRUE;

      if(strcasecmp(tag->key, "waterway") == 0)
	return TRUE;

      if(strcasecmp(tag->key, "railway") == 0)
	return TRUE;
    }
    tag = tag->next;
  }

  return FALSE;
}

gboolean osm_node_has_tag(node_t *node) {
  tag_t *tag = node->tag;

  if(tag && strcasecmp(tag->key, "created_by") == 0)
    tag = tag->next;

  return tag != NULL;
}

/* return true if node is part of way */
gboolean osm_node_in_way(way_t *way, node_t *node) {
  node_chain_t *node_chain = way->node_chain;
  while(node_chain) {
    if(node_chain->node == node)
      return TRUE;

    node_chain = node_chain->next;
  }
  return FALSE;
}

static void osm_generate_tags(tag_t *tag, xmlNodePtr node) {
  while(tag) {
    /* make sure "created_by" tag contains our id */
    if(strcasecmp(tag->key, "created_by") == 0) {
      g_free(tag->value);
      tag->value = g_strdup(PACKAGE " v" VERSION);
    }

    xmlNodePtr tag_node = xmlNewChild(node, NULL, BAD_CAST "tag", NULL);
    xmlNewProp(tag_node, BAD_CAST "k", BAD_CAST tag->key);
    xmlNewProp(tag_node, BAD_CAST "v", BAD_CAST tag->value);
    tag = tag->next;
  }
}

/* build xml representation for a node or way */
char *osm_generate_xml(osm_t *osm, type_t type, void *item) {
  char str[32];
  xmlChar *result = NULL;
  int len = 0;

  LIBXML_TEST_VERSION;

  xmlDocPtr doc = xmlNewDoc(BAD_CAST "1.0");
  xmlNodePtr root_node = xmlNewNode(NULL, BAD_CAST "osm");
  xmlNewProp(root_node, BAD_CAST "version", BAD_CAST "0.5");
  xmlNewProp(root_node, BAD_CAST "generator", BAD_CAST PACKAGE " V" VERSION);
  xmlDocSetRootElement(doc, root_node);

  switch(type) {
  case NODE: 
    {
      node_t *node = (node_t*)item;
      xmlNodePtr node_node = xmlNewChild(root_node, NULL, 
					 BAD_CAST "node", NULL);
      /* new nodes don't have an id, but get one after the upload */
      if(!(node->flags & OSM_FLAG_NEW)) {
	snprintf(str, sizeof(str), "%u", (unsigned)node->id);
	xmlNewProp(node_node, BAD_CAST "id", BAD_CAST str);
      }
      g_ascii_dtostr(str, sizeof(str), node->pos.lat);
      xmlNewProp(node_node, BAD_CAST "lat", BAD_CAST str);
      g_ascii_dtostr(str, sizeof(str), node->pos.lon);
      xmlNewProp(node_node, BAD_CAST "lon", BAD_CAST str);    
      osm_generate_tags(node->tag, node_node);
    }
    break;

  case WAY: 
    {
      way_t *way = (way_t*)item;
      xmlNodePtr way_node = xmlNewChild(root_node, NULL, BAD_CAST "way", NULL);
      snprintf(str, sizeof(str), "%u", (unsigned)way->id);
      xmlNewProp(way_node, BAD_CAST "id", BAD_CAST str);
      
      node_chain_t *node_chain = way->node_chain;
      while(node_chain) {
	xmlNodePtr nd_node = xmlNewChild(way_node, NULL, BAD_CAST "nd", NULL);
	char *str = g_strdup_printf("%ld", node_chain->node->id);
	xmlNewProp(nd_node, BAD_CAST "ref", BAD_CAST str);
	g_free(str);
	node_chain = node_chain->next;
      }
      
      osm_generate_tags(way->tag, way_node);
    } 
    break;

  default:
    printf("neither NODE nor WAY\n");
    g_assert(0);
    break;
  }

  xmlDocDumpFormatMemoryEnc(doc, &result, &len, "UTF-8", 1);
  xmlFreeDoc(doc);
  xmlCleanupParser();

  //  puts("xml encoding result:");
  //  puts((char*)result);

  return (char*)result;
}

/* build xml representation for a node */
char *osm_generate_xml_node(osm_t *osm, node_t *node) {
  return osm_generate_xml(osm, NODE, node);
}

/* build xml representation for a way */
char *osm_generate_xml_way(osm_t *osm, way_t *way) {
  return osm_generate_xml(osm, WAY, way);
}

node_t *osm_get_node_by_id(osm_t *osm, item_id_t id) {
  node_t *node = osm->node;
  while(node) {
    if(node->id == id)
      return node;

    node = node->next;
  }
  
  return NULL;
}

way_t *osm_get_way_by_id(osm_t *osm, item_id_t id) {
  way_t *way = osm->way;
  while(way) {
    if(way->id == id)
      return way;

    way = way->next;
  }
  
  return NULL;
}

/* ---------- edit functions ------------- */

item_id_t osm_new_node_id(osm_t *osm) {
  item_id_t id = -1;

  while(TRUE) {
    gboolean found = FALSE;
    node_t *node = osm->node;
    while(node) {
      if(node->id == id)
	found = TRUE;

      node = node->next;
    }

    /* no such id so far -> use it */
    if(!found) return id;

    id--;
  }
  g_assert(0);
  return 0;
}

node_t *osm_node_new(osm_t *osm, gint x, gint y) {
  printf("Attaching new node\n");

  node_t *node = g_new0(node_t, 1);
  node->id = osm_new_node_id(osm);
  node->lpos.x = x; 
  node->lpos.y = y;
  node->visible = TRUE;
  node->flags = OSM_FLAG_NEW;
  node->time = time(NULL);

  /* todo: id, user ... */

  /* add created_by tag */
  node->tag = g_new0(tag_t, 1);
  node->tag->key = g_strdup("created_by");
  node->tag->value = g_strdup(PACKAGE " v" VERSION);

  /* convert screen position back to utm/ll */
  utm_t utm = osm->bounds->center;  /* set zone/band */
  utm.easting  =  node->lpos.x + osm->bounds->center.easting;
  utm.northing = -node->lpos.y + osm->bounds->center.northing;
  
  utm_utm2pos(&utm, &node->pos);
  printf("  new at %d %d (%f %f)\n", 
	 node->lpos.x, node->lpos.y, node->pos.lat, node->pos.lon);

  /* attach to end of node list */
  node_t **lnode = &osm->node;
  while(*lnode) lnode = &(*lnode)->next;  
  *lnode = node;

  return node;
}

/* returns pointer to chain of ways affected by this deletion */
way_chain_t *osm_node_delete(osm_t *osm, node_t *node, gboolean permanently) {
  way_chain_t *way_chain = NULL, **cur_way_chain = &way_chain;

  /* new nodes aren't stored on the server and are just deleted permanently */
  if(node->flags & OSM_FLAG_NEW) {
    printf("About to delete NEW node -> force permanent delete\n");
    permanently = TRUE;
  }

  /* first remove node from all ways using it */
  way_t *way = osm->way;
  while(way) {
    node_chain_t **chain = &(way->node_chain);
    gboolean modified = FALSE;
    while(*chain) {
      /* remove node from chain */
      if(node == (*chain)->node) {
	modified = TRUE;
	node_chain_t *next = (*chain)->next;
	g_free(*chain);
	*chain = next;
      } else
	chain = &((*chain)->next);
    }

    if(modified) {
      way->flags |= OSM_FLAG_DIRTY;
      
      /* and add the way to the list of affected ways */
      *cur_way_chain = g_new0(way_chain_t, 1);
      (*cur_way_chain)->way = way;
      cur_way_chain = &((*cur_way_chain)->next);
    }

    way = way->next;
  }

  if(!permanently) {
    printf("mark as deleted\n");
    node->flags |= OSM_FLAG_DELETED;
  } else {
    printf("permanent delete\n");

    /* remove it from the chain */
    node_t **cnode = &osm->node;
    int found = 0;

    while(*cnode) {
      if(*cnode == node) {
	found++;
	*cnode = (*cnode)->next;

	osm_node_free(node);
      } else
	cnode = &((*cnode)->next);
    }
    g_assert(found == 1);
  }

  return way_chain;
}

guint osm_way_number_of_nodes(way_t *way) {
  guint nodes = 0;
  node_chain_t *chain = way->node_chain;
  while(chain) {
    nodes++;
    chain = chain->next;
  }
  return nodes;
}

/* return all relations a node is in */
relation_chain_t *osm_node_to_relation(osm_t *osm, node_t *node) {
  relation_chain_t *rel_chain = NULL, **cur_rel_chain = &rel_chain;

  relation_t *relation = osm->relation;
  while(relation) {
    gboolean is_member = FALSE;

    member_t *member = relation->member;
    while(member) {
      if((member->type == NODE) && (member->node == node))
	is_member = TRUE;

      member = member->next;
    }

    /* node is a member of this relation, so move it to the member chain */
    if(is_member) {
      *cur_rel_chain = g_new0(relation_chain_t, 1);
      (*cur_rel_chain)->relation = relation;
      cur_rel_chain = &((*cur_rel_chain)->next);
    }

    relation = relation->next;
  }

  return rel_chain;
}

/* return all ways a node is in */
way_chain_t *osm_node_to_way(osm_t *osm, node_t *node) {
  way_chain_t *chain = NULL, **cur_chain = &chain;

  way_t *way = osm->way;
  while(way) {
    gboolean is_member = FALSE;

    node_chain_t *node_chain = way->node_chain;
    while(node_chain) {
      if(node_chain->node == node)
	is_member = TRUE;

      node_chain = node_chain->next;
    }

    /* node is a member of this relation, so move it to the member chain */
    if(is_member) {
      *cur_chain = g_new0(way_chain_t, 1);
      (*cur_chain)->way = way;
      cur_chain = &((*cur_chain)->next);
    }

    way = way->next;
  }

  return chain;
}
