/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of OSM2Go.
 *
 * OSM2Go is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * OSM2Go is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSM2Go.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "appdata.h"

#define WGS84_EQUATORIAL_RADIUS     (6378137.0)
#define WGS84_ECCENTRICITY_SQUARED  (0.00669438)
#define ECC_SQUARED  (WGS84_ECCENTRICITY_SQUARED)
#define ECC_SQUARED_CUBED  (ECC_SQUARED*ECC_SQUARED*ECC_SQUARED)
#define ECC_PRIME_SQUARED  ((ECC_SQUARED)/(1-ECC_SQUARED))

void utm_utm2pos(utm_t *utm, pos_t *pos) {
  // converts UTM coords to lat/long.  Equations from USGS Bulletin 1532
  // East Longitudes are positive, West longitudes are negative.
  // North latitudes are positive, South latitudes are negative
  // Lat and Long are in decimal degrees.
  // Written by Chuck Gantz- chuck.gantz@globalstar.com

  UTM_FLOAT k0 = 0.9996;
  UTM_FLOAT a = WGS84_EQUATORIAL_RADIUS;
  UTM_FLOAT ecc_squared = ECC_SQUARED;
  UTM_FLOAT ecc_prime_squared;
  UTM_FLOAT e1 = (1-sqrt(1-ecc_squared))/(1+sqrt(1-ecc_squared));
  UTM_FLOAT N1, T1, C1, R1, D, M;
  UTM_FLOAT LongOrigin;
  UTM_FLOAT mu, phi1, phi1Rad;
  UTM_FLOAT x, y;
  int northern_hemisphere;          // 1 for northern hemispher, 0 for southern

  // remove 500,000 meter offset for longitude
  x = utm->easting + 0.5 - 500000.0;
  y = utm->northing + 0.5;

  if((utm->band - 'N') >= 0)
    northern_hemisphere = 1;        // point is in northern hemisphere
  else {
    northern_hemisphere = 0;        //point is in southern hemisphere
    // remove 10,000,000 meter offset used for southern hemisphere
    y -= 10000000.0;
  }

  // +3 puts origin in middle of zone
  LongOrigin = (utm->zone - 1)*6 - 180 + 3;

  ecc_prime_squared = (ecc_squared)/(1-ecc_squared);

  M = y / k0;
  mu = M/(a*(1-ecc_squared/4-3*ecc_squared*ecc_squared/64-
	     5*ecc_squared*ecc_squared*ecc_squared/256));

  phi1Rad = mu  + (3*e1/2-27*e1*e1*e1/32)*sin(2*mu)
    + (21*e1*e1/16-55*e1*e1*e1*e1/32)*sin(4*mu)
    +(151*e1*e1*e1/96)*sin(6*mu);
  phi1 = RAD2DEG(phi1Rad);

  N1 = a/sqrt(1-ecc_squared*sin(phi1Rad)*sin(phi1Rad));
  T1 = tan(phi1Rad)*tan(phi1Rad);
  C1 = ecc_prime_squared*cos(phi1Rad)*cos(phi1Rad);
  R1 = a*(1-ecc_squared)/pow(1-ecc_squared*sin(phi1Rad)*sin(phi1Rad), 1.5);
  D = x/(N1*k0);

  pos->lat = RAD2DEG(phi1Rad -
                (N1*tan(phi1Rad)/R1)*(D*D/2-(5+3*T1+10*
		    C1-4*C1*C1-9*ecc_prime_squared)*D*D*D*D/24
                 +(61+90*T1+298*C1+45*T1*T1-252*ecc_prime_squared-
                    3*C1*C1)*D*D*D*D*D*D/720));

  pos->lon = LongOrigin + RAD2DEG((D-(1+2*T1+C1)*D*D*D/6+
		   (5-2*C1+28*T1-3*C1*C1+8*ecc_prime_squared+24*T1*T1)
		   *D*D*D*D*D/120)/cos(phi1Rad));
}

static char UTMLetterDesignator(UTM_FLOAT Lat) {
  // This routine determines the correct UTM letter designator for 
  // the given latitude returns 'Z' if latitude is outside the UTM 
  // limits of 84N to 80S
  // Written by Chuck Gantz- chuck.gantz@globalstar.com
  char LetterDesignator;

  if((84 >= Lat) && (Lat >= 72)) LetterDesignator = 'X';
  else if((72 > Lat) && (Lat >= 64)) LetterDesignator = 'W';
  else if((64 > Lat) && (Lat >= 56)) LetterDesignator = 'V';
  else if((56 > Lat) && (Lat >= 48)) LetterDesignator = 'U';
  else if((48 > Lat) && (Lat >= 40)) LetterDesignator = 'T';
  else if((40 > Lat) && (Lat >= 32)) LetterDesignator = 'S';
  else if((32 > Lat) && (Lat >= 24)) LetterDesignator = 'R';
  else if((24 > Lat) && (Lat >= 16)) LetterDesignator = 'Q';
  else if((16 > Lat) && (Lat >= 8)) LetterDesignator = 'P';
  else if(( 8 > Lat) && (Lat >= 0)) LetterDesignator = 'N';
  else if(( 0 > Lat) && (Lat >= -8)) LetterDesignator = 'M';
  else if((-8> Lat) && (Lat >= -16)) LetterDesignator = 'L';
  else if((-16 > Lat) && (Lat >= -24)) LetterDesignator = 'K';
  else if((-24 > Lat) && (Lat >= -32)) LetterDesignator = 'J';
  else if((-32 > Lat) && (Lat >= -40)) LetterDesignator = 'H';
  else if((-40 > Lat) && (Lat >= -48)) LetterDesignator = 'G';
  else if((-48 > Lat) && (Lat >= -56)) LetterDesignator = 'F';
  else if((-56 > Lat) && (Lat >= -64)) LetterDesignator = 'E';
  else if((-64 > Lat) && (Lat >= -72)) LetterDesignator = 'D';
  else if((-72 > Lat) && (Lat >= -80)) LetterDesignator = 'C';

  // This is here as an error flag to show that the Latitude is 
  // outside the UTM limits
  else LetterDesignator = 'Z';
  
  return LetterDesignator;
}


void utm_pos2utm(pos_t *pos, utm_t *utm, utm_t *ref) {
  // converts lat/long to UTM coords.  Equations from USGS Bulletin 1532
  // East Longitudes are positive, West longitudes are negative.
  // North latitudes are positive, South latitudes are negative
  // Lat and Long are in decimal degrees
  // Written by Chuck Gantz- chuck.gantz@globalstar.com
  
  UTM_FLOAT a = WGS84_EQUATORIAL_RADIUS;
  UTM_FLOAT k0 = 0.9996;

  UTM_FLOAT LongOrigin;
  UTM_FLOAT N, T, C, A, M;

  // Make sure the longitude is between -180.00 .. 179.9
  UTM_FLOAT LongTemp = (pos->lon+180)- (int)((pos->lon+180)/360)*360-180; 

  UTM_FLOAT LatRad = DEG2RAD(pos->lat);
  UTM_FLOAT LongRad = DEG2RAD(LongTemp);
  UTM_FLOAT LongOriginRad;

  if(!ref) {
    // no reference zone given
    utm->zone = (int)((LongTemp + 180)/6) + 1;
    
    if(pos->lat >= 56.0 && pos->lat < 64.0 && 
       LongTemp >= 3.0 && LongTemp < 12.0 )
      utm->zone = 32;
    
    // Special zones for Svalbard
    if( pos->lat >= 72.0 && pos->lat < 84.0 ) {
      if(      LongTemp >= 0.0  && LongTemp <  9.0 ) utm->zone = 31;
      else if( LongTemp >= 9.0  && LongTemp < 21.0 ) utm->zone = 33;
      else if( LongTemp >= 21.0 && LongTemp < 33.0 ) utm->zone = 35;
      else if( LongTemp >= 33.0 && LongTemp < 42.0 ) utm->zone = 37;
    }
  } else
    utm->zone = ref->zone;

  LongOrigin = (utm->zone - 1)*6 - 180 + 3;  //+3 puts origin in middle of zone
  LongOriginRad = DEG2RAD(LongOrigin);

  // compute the UTM Zone from the latitude and longitude
  utm->band = UTMLetterDesignator(pos->lat);

  N = a/sqrt(1-ECC_SQUARED*sin(LatRad)*sin(LatRad));
  T = tan(LatRad)*tan(LatRad);
  C = ECC_PRIME_SQUARED*cos(LatRad)*cos(LatRad);
  A = cos(LatRad)*(LongRad-LongOriginRad);

  M = a*((1 - ECC_SQUARED/4 - 3*ECC_SQUARED*ECC_SQUARED/64 -
          5*ECC_SQUARED_CUBED/256)*LatRad
         - (3*ECC_SQUARED/8 + 3*ECC_SQUARED*ECC_SQUARED/32 +
            45*ECC_SQUARED_CUBED/1024)*sin(2*LatRad)
         + (15*ECC_SQUARED*ECC_SQUARED/256 +
            45*ECC_SQUARED_CUBED/1024)*sin(4*LatRad)
         - (35*ECC_SQUARED_CUBED/3072)*sin(6*LatRad));

  utm->easting = 0.5 + k0*N*(A+(1-T+C)*A*A*A/6
               + (5-18*T+T*T+72*C-58*ECC_PRIME_SQUARED)*A*A*A*A*A/120)
               + 500000.0;

  utm->northing = 0.5 + k0*(M+N*tan(LatRad)*(A*A/2+(5-T+9*C+4*C*C)*A*A*A*A/24
                 + (61-58*T+T*T+600*C-330*ECC_PRIME_SQUARED)*A*A*A*A*A*A/720));

  // 10000000 meter offset for southern hemisphere
  if(pos->lat < 0)
    utm->northing += 10000000.0;
}

// convert utm coordiante into human readable form
void utm2str(char *str, int size, utm_t *utm) {
  
  snprintf(str, size, "%d%c E %ld N %ld", utm->zone, utm->band,
	   utm->easting, utm->northing);
}
