/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of OSM2Go.
 *
 * OSM2Go is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * OSM2Go is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSM2Go.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MAP_H
#define MAP_H

#include "gpx.h"
#include <libgnomecanvas/libgnomecanvas.h>

/* -------- all sizes are in meters ---------- */
#define DARK_GREY        0x606060ff
#define WAY_WIDTH        3.0
#define STROKE_WIDTH     6.0
#define RADIUS           4.0          // stand-alone nodes
#define S_RADIUS         2.5          // nodes with tags (!= created_by)
#define MAP_COLOR_NONE   0x0
#define FILL_TRANSP      0x40
#define HIGHLIGHT_COLOR  0xffff0080   // stand alone nodes and ways
#define HIGHLIGHT2_COLOR 0xff000080   // nodes of a selected way
#define TOUCHNODE_COLOR  0x0000ff80   // node touched during drag

/* the "drag limit" is the number of pixels the mouse/pen has to */
/* be moved so the action is actually considered dragging. This is */
/* to prevent accidential dragging when the user only intended to click */
/* something. This is especially useful when using a touchscreen. */
#ifndef USE_HILDON
#define MAP_DRAG_LIMIT   8
#else
#define MAP_DRAG_LIMIT   16
#endif

typedef enum { MAP_TYPE_ILLEGAL=0, MAP_TYPE_NODE, MAP_TYPE_WAY } map_type_t;
typedef enum { MAP_ACTION_IDLE=0, MAP_ACTION_NODE_ADD } map_action_t;

typedef struct map_highlight_s {
  GnomeCanvasItem *item;
  struct map_highlight_s *next;
} map_highlight_t;

typedef struct map_item_s {
  map_type_t type;
  gboolean highlight;
  union {
    node_t *node;
    way_t *way;
    void *ptr;
  };
  GnomeCanvasItem *item;
} map_item_t;

/* this is a chain of map_items which is attached to all entries */
/* in the gpx tree (node_t, way_t, ...) to be able to get a link */
/* to the screen representation of a give node/way/etc */
typedef struct map_item_chain_s {
  map_item_t *map_item;
  struct map_item_chain_s *next;
} map_item_chain_t;

typedef enum { MAP_GROUP_POLYGONS=0, MAP_GROUP_WAYS_HL, MAP_GROUP_WAYS_OL, 
	       MAP_GROUP_WAYS, MAP_GROUP_NODES_HL, MAP_GROUP_NODES, 
	       MAP_GROUP_DRAW, MAP_GROUP_TRACK, MAP_GROUPS } map_group_t;

typedef struct map_s {
  appdata_t *appdata;

  GtkWidget *canvas;
  float zoom;                      // zoom level (1.0 = 1m/pixel

  map_highlight_t *highlight;      // list of elements used for highlighting

  GnomeCanvasItem *group[MAP_GROUPS];

  map_item_t selected;             // the currently selected item (node or way)
  
  GnomeCanvasItem *cursor;         // the cursor visible on the draw layer
  GnomeCanvasItem *touchnode;      // the blue "touch node" on the draw layer

  map_action_t action;             // current action in progress

  /* variables required for pen/mouse handling */
  struct {
    gboolean is;
    gboolean drag;
    map_item_t *on_item;
    struct { gint x,y; } at;    // point mouse button was last pressed
    struct { gint x,y; } so;    // initial scroll offset 
    gboolean on_selected_node;  // the currently clicked node 
                                // (may be part of a selected way)
  } pen_down;

} map_t;

GtkWidget *map_new(appdata_t *appdata);
void map_init(appdata_t *appdata);
gboolean map_key_press_event(appdata_t *appdata, GdkEventKey *event);
void map_item_set_flags(map_item_t *map_item, int set, int clr);
void map_show_node(map_t *map, node_t *node);
void map_cmenu_show(map_t *map);
void map_highlight_refresh(appdata_t *appdata);

void map_item_redraw(appdata_t *appdata, map_item_t *map_item);

void map_clear(appdata_t *appdata);
void map_paint(appdata_t *appdata);

void map_action_set(appdata_t *appdata, map_action_t action);
void map_delete_selected(appdata_t *appdata);

#endif // MAP_H
