/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of OSM2Go.
 *
 * OSM2Go is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * OSM2Go is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSM2Go.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "appdata.h"

static void map_way_color(potlatch_conf_t *potlatch, way_t *way) {
  way->color = way->fill_color = MAP_COLOR_NONE;
  way->draw_flags = 0;

  potlatch_color_t *plcol = potlatch->colors;

  /* a highway/waterway/railway is never a polygon */
  if(!gpx_way_is_way(way)) {
  
    /* check if this way is a polygon */
    node_chain_t *node_chain = way->node_chain;
    while(node_chain && node_chain->next) node_chain = node_chain->next;
    if(way->node_chain && node_chain && 
       way->node_chain->node == node_chain->node) {
      way->draw_flags |= GPX_DRAW_FLAGS_POLYGON;
      
      /* default fill color is light grey */
      way->fill_color = (0xc0c0c0 << 8) | FILL_TRANSP;
    }
  }
  
  while(plcol) {  // (way->color == MAP_COLOR_NONE && plcol) 
    if(gpx_way_has_key_or_value(way, plcol->tag)) {
      way->color = (plcol->color << 8) | 0xff;
      if(plcol->stroke) way->draw_flags |= GPX_DRAW_FLAGS_STROKE;
      
      if(plcol->fill_color != COLOR_NONE)
	way->fill_color = (plcol->fill_color << 8) | FILL_TRANSP;
    }
    plcol = plcol->next;
  }
  
  /* use dark grey/no stroke/not filled for everything unknown */
  if(way->color == MAP_COLOR_NONE) way->color = DARK_GREY;
}

static void map_init_colors(potlatch_conf_t *potlatch, gpx_t *gpx) {
  way_t *way = gpx->way;

  printf("preparing colors\n");

  /* colorize ways */
  while(way) {
    map_way_color(potlatch, way);
    way = way->next;
  }
}

static void map_statusbar(map_t *map, map_item_t *map_item) {
  char *item_str = NULL;
  item_id_t id = ID_ILLEGAL;
  tag_t *tag = NULL;
  char *str = NULL;
  
  switch(map_item->type) {
  case MAP_TYPE_NODE:
    item_str = "Node";
    id = map_item->node->id;
    tag = map_item->node->tag;
    break;

  case MAP_TYPE_WAY:
    item_str = "Way";
    id = map_item->way->id;
    tag = map_item->way->tag;
    break;

  default:
    break;
  }

  if(id == ID_ILLEGAL) 
    str = g_strdup_printf("Unknown item");
  else {
    str = g_strdup_printf("%s #%ld", item_str, id);

    /* add some tags ... */
    while(tag) {
      /* we don't have much space, so ignore created_by tag */
      if(strcasecmp(tag->key, "created_by") != 0) {
	char *old = str;
	str = g_strdup_printf("%s, %s/%s", old, tag->key, tag->value);
	g_free(old);
      }
      tag = tag->next;
    }
  }
  
  statusbar_set(map->appdata, str);
  g_free(str);
}

static void map_node_select(appdata_t *appdata, node_t *node) {
  map_t *map = appdata->map;
  map_item_t *map_item = &map->selected;
  
  g_assert(!map->highlight);

  map_item->type      = MAP_TYPE_NODE;
  map_item->node      = node;
  map_item->highlight = FALSE;

  /* node may not have any visible representation at all */
  if(node->map_item_chain) 
    map_item->item = node->map_item_chain->map_item->item;
  else
    map_item->item = NULL;

  map_statusbar(map, map_item);
  icon_bar_map_item_selected(appdata, map_item, TRUE);

  /* highlight node */
  gint x = map_item->node->lpos.x, y = map_item->node->lpos.y;

  /* create a copy of this map item and mark it as being a highlight */
  map_item_t *new_map_item = g_new0(map_item_t, 1);
  memcpy(new_map_item, map_item, sizeof(map_item_t));
  new_map_item->highlight = TRUE;

  map_hl_new(map, GNOME_CANVAS_GROUP(map->group[MAP_GROUP_NODES_HL]),
	     GNOME_TYPE_CANVAS_ELLIPSE, new_map_item,
	     "x1", (double)x-2*RADIUS, "x2", (double)x+2*RADIUS,
	     "y1", (double)y-2*RADIUS, "y2", (double)y+2*RADIUS,
	     "fill_color_rgba", HIGHLIGHT_COLOR, 
	     "width-units", 0.0,
	     NULL);       

  if(!map_item->item) {
    /* and draw a fake node */
    new_map_item = g_new0(map_item_t, 1);
    memcpy(new_map_item, map_item, sizeof(map_item_t));
    new_map_item->highlight = TRUE;

    map_hl_new(map, 
	       GNOME_CANVAS_GROUP(map->group[MAP_GROUP_NODES_HL]),
	       GNOME_TYPE_CANVAS_ELLIPSE, new_map_item,
	       "x1", (double)x-RADIUS, "x2", (double)x+RADIUS,
	       "y1", (double)y-RADIUS, "y2", (double)y+RADIUS,
	       "fill_color_rgba", HIGHLIGHT2_COLOR, 
	       "width-units", 0.0,
	       NULL);       
  }
}

static void map_way_select(appdata_t *appdata, way_t *way) {
  map_t *map = appdata->map;
  map_item_t *map_item = &map->selected;
  
  g_assert(!map->highlight);

  map_item->type      = MAP_TYPE_WAY;
  map_item->way       = way;
  map_item->highlight = FALSE;
  map_item->item      = way->map_item_chain->map_item->item;

  map_statusbar(map, map_item);
  icon_bar_map_item_selected(appdata, map_item, TRUE);

  node_chain_t *node_chain = map_item->way->node_chain;
  while(node_chain) {
    map_item_t item;
    item.type = MAP_TYPE_NODE;
    item.node = node_chain->node;

    if(!map_hl_item_is_highlighted(map, &item)) {

      /* create a new map item for every node */
      map_item_t *new_map_item = g_new0(map_item_t, 1);
      new_map_item->type = MAP_TYPE_NODE;
      new_map_item->node = node_chain->node;
      new_map_item->highlight = TRUE;
    
      gint x = node_chain->node->lpos.x;
      gint y = node_chain->node->lpos.y;
      map_hl_new(map, 
		 GNOME_CANVAS_GROUP(map->group[MAP_GROUP_NODES_HL]),
		 GNOME_TYPE_CANVAS_ELLIPSE, new_map_item,
		 "x1", (double)x-RADIUS, "x2", (double)x+RADIUS,
		 "y1", (double)y-RADIUS, "y2", (double)y+RADIUS,
		 "fill_color_rgba", HIGHLIGHT2_COLOR, 
		 "width-units", 0.0,
		 NULL);       
    }

    node_chain = node_chain->next;
  }

  /* a way needs at least 2 points to be drawn */
  guint nodes = gpx_way_number_of_nodes(way);
  if(nodes > 1) {
    
    /* allocate space for nodes */
    GnomeCanvasPoints *points = gnome_canvas_points_new(nodes);

    int node = 0;
    node_chain = map_item->way->node_chain;
    while(node_chain) {
      points->coords[node++] = node_chain->node->lpos.x;
      points->coords[node++] = node_chain->node->lpos.y;
      
      node_chain = node_chain->next;
    }
    
    /* create a copy of this map item and mark it as being a highlight */
    map_item_t *new_map_item = g_new0(map_item_t, 1);
    memcpy(new_map_item, map_item, sizeof(map_item_t));
    new_map_item->highlight = TRUE;
    
    map_hl_new(map, GNOME_CANVAS_GROUP(map->group[MAP_GROUP_WAYS_HL]),
	       GNOME_TYPE_CANVAS_LINE, new_map_item,
	       "points", points,
	       "fill_color_rgba", HIGHLIGHT_COLOR,
	       "width-units", (map_item->way->draw_flags & GPX_DRAW_FLAGS_STROKE)?
	       2*STROKE_WIDTH:3*WAY_WIDTH,
	       "join-style", GDK_JOIN_ROUND,
	       "cap-style", GDK_CAP_ROUND,
	       NULL);   
  
    gnome_canvas_points_free(points);
  }
}

static void map_item_select(appdata_t *appdata, map_item_t *map_item) {
  switch(map_item->type) {
  case MAP_TYPE_NODE:
    map_node_select(appdata, map_item->node);
    break;
  case MAP_TYPE_WAY:
    map_way_select(appdata, map_item->way);
    break;
  default:
    g_assert((map_item->type == MAP_TYPE_NODE)||
	     (map_item->type == MAP_TYPE_WAY));
    break;
  }
}

static void map_item_deselect(appdata_t *appdata) {

  /* remove statusbar message */
  statusbar_set(appdata, NULL);

  /* disable/enable icons in icon bar */
  icon_bar_map_item_selected(appdata, NULL, FALSE);

  /* remove highlight */
  map_hl_remove(appdata);

  /* forget about selection */
  appdata->map->selected.type = MAP_TYPE_ILLEGAL;
}

/* called whenever a map item is to be destroyed */
static gint map_item_destroy_event(GtkWidget *widget, gpointer data) {
  map_item_t *map_item = (map_item_t*)data;

  //   printf("destroying map_item @ %p\n", map_item);

  /* remove item from nodes/ways map_item_chain */
  map_item_chain_t **chain = NULL;
  if(map_item->type == MAP_TYPE_NODE)
    chain = &map_item->node->map_item_chain;
  else if(map_item->type == MAP_TYPE_WAY)
    chain = &map_item->way->map_item_chain;

  /* there must be a chain with content, otherwise things are broken */
  g_assert(chain);
  g_assert(*chain);

  /* search current map_item, ... */
  while(*chain && (*chain)->map_item != map_item)
    chain = &(*chain)->next;

  g_assert(*chain);

  /* ... remove it from chain and free it */
  map_item_chain_t *tmp = *chain;
  *chain = (*chain)->next;

  g_free(tmp);
  g_free(map_item);

  return FALSE;
}

static GnomeCanvasItem *map_item_new(GnomeCanvasGroup *parent,
				     GType type, map_item_t *map_item,
				     const gchar *first_arg_name,
				     va_list args) {

  map_item->item = g_object_new(type, NULL);
  gnome_canvas_item_construct(map_item->item, parent, first_arg_name, args);
  gtk_object_set_user_data(GTK_OBJECT(map_item->item), map_item);

  gtk_signal_connect(GTK_OBJECT(map_item->item), 
     "destroy", G_CALLBACK(map_item_destroy_event), map_item);
  
  return map_item->item;
};

static GnomeCanvasItem *map_node_new(GnomeCanvasGroup *parent,
				     GType type, node_t *node,
				     const gchar *first_arg_name,
				     ...) {
  va_list args;
  va_start( args, first_arg_name);

  map_item_t *map_item = g_new0(map_item_t, 1);
  map_item->type = MAP_TYPE_NODE;
  map_item->node = node;

  /* attach map_item to nodes map_item_chain */
  map_item_chain_t **chain = &node->map_item_chain;
  while(*chain) chain = &(*chain)->next;
  *chain = g_new0(map_item_chain_t, 1);
  (*chain)->map_item = map_item;

  return map_item_new(parent, type, map_item, first_arg_name, args);
}

static GnomeCanvasItem *map_way_new(GnomeCanvasGroup *parent,
				    GType type, way_t *way,
				    const gchar *first_arg_name,
				    ...) {
  va_list args;
  va_start( args, first_arg_name);

  map_item_t *map_item = g_new0(map_item_t, 1);
  map_item->type = MAP_TYPE_WAY;
  map_item->way = way;

  /* attach map_item to ways map_item_chain */
  map_item_chain_t **chain = &way->map_item_chain;
  while(*chain) chain = &(*chain)->next;
  *chain = g_new0(map_item_chain_t, 1);
  (*chain)->map_item = map_item;

  return map_item_new(parent, type, map_item, first_arg_name, args);
}

void map_show_node(map_t *map, node_t *node) {
  glong x = node->lpos.x, y = node->lpos.y;

  map_node_new(GNOME_CANVAS_GROUP(map->group[MAP_GROUP_NODES]),
	       GNOME_TYPE_CANVAS_ELLIPSE, node,
	       "x1", (double)x-S_RADIUS, "x2", (double)x+S_RADIUS,
	       "y1", (double)y-S_RADIUS, "y2", (double)y+S_RADIUS,
	       "fill_color_rgba", 0x000000ff,
	       "width-units", 0.0,
	       NULL);   
}

static void map_way_draw(map_t *map, way_t *way) {

  /* don't draw a way that's not there anymore */
  if(way->flags & GPX_FLAG_DELETED)
    return;

  /* allocate space for nodes */
  /* a way needs at least 2 points to be drawn */
  guint nodes = gpx_way_number_of_nodes(way);
  if(nodes > 1) {
    GnomeCanvasPoints *points = gnome_canvas_points_new(nodes);
    
    int node = 0;
    node_chain_t *node_chain = way->node_chain;
    while(node_chain) {
      /* mirror map vertically within bounds */
      points->coords[node++] = node_chain->node->lpos.x;
      points->coords[node++] = node_chain->node->lpos.y;
      node_chain = node_chain->next;
    }
    
    /* draw way */
    if(way->draw_flags & GPX_DRAW_FLAGS_POLYGON) {
      map_way_new(GNOME_CANVAS_GROUP(map->group[MAP_GROUP_POLYGONS]),
		  GNOME_TYPE_CANVAS_POLYGON, way,
		  "points", points,
		  "fill_color_rgba", way->fill_color,
		  "width-units", 2.0,
		  "outline_color_rgba", DARK_GREY,
		  "join-style", GDK_JOIN_ROUND,
		  "cap-style", GDK_CAP_ROUND,
		  NULL);   
    } else {
      map_way_new(GNOME_CANVAS_GROUP(map->group[MAP_GROUP_WAYS]),
		  GNOME_TYPE_CANVAS_LINE, way,
		  "points", points,
		  "fill_color_rgba", way->color,
		  "width-units", WAY_WIDTH,
		  "join-style", GDK_JOIN_ROUND,
		  "cap-style", GDK_CAP_ROUND,
		  NULL);   
      
      if(way->draw_flags & GPX_DRAW_FLAGS_STROKE)
	map_way_new(GNOME_CANVAS_GROUP(map->group[MAP_GROUP_WAYS_OL]),
		    GNOME_TYPE_CANVAS_LINE, way,
		    "points", points,
		    "fill_color_rgba", DARK_GREY,
		    "width-units", STROKE_WIDTH,
		    "join-style", GDK_JOIN_ROUND,
		    "cap-style", GDK_CAP_ROUND,
		    NULL);   
    }
    gnome_canvas_points_free(points);
  }
}

static void map_node_draw(map_t *map, node_t *node) {
  glong x = node->lpos.x, y = node->lpos.y;

  /* don't draw a node that's not there anymore */
  if(node->flags & GPX_FLAG_DELETED)
    return;
  
  if(!node->ways) {
    map_node_new(GNOME_CANVAS_GROUP(map->group[MAP_GROUP_NODES]),
		 GNOME_TYPE_CANVAS_ELLIPSE, node,
		 "x1", (double)x-RADIUS, "x2", (double)x+RADIUS,
		 "y1", (double)y-RADIUS, "y2", (double)y+RADIUS,
		 "fill_color_rgba", 0x008800ff,
		 "width-units", RADIUS/2.0,
		 "outline-color-rgba", 0x000000ff,
		 NULL);   
  } else if(gpx_node_has_tag(node)) {
    map_node_new(GNOME_CANVAS_GROUP(map->group[MAP_GROUP_NODES]),
		 GNOME_TYPE_CANVAS_ELLIPSE, node,
		 "x1", (double)x-S_RADIUS, "x2", (double)x+S_RADIUS,
		 "y1", (double)y-S_RADIUS, "y2", (double)y+S_RADIUS,
		 "fill_color_rgba", 0x000000ff,
		 "width-units", 0.0,
		 NULL);   
  }
}

static void map_item_draw(map_t *map, map_item_t *map_item) {
  switch(map_item->type) {
  case MAP_TYPE_NODE:
    map_node_draw(map, map_item->node);
    break;
  case MAP_TYPE_WAY:
    map_way_draw(map, map_item->way);
    break;
  default:
    g_assert((map_item->type == MAP_TYPE_NODE) ||
	     (map_item->type == MAP_TYPE_WAY));
  }
}

static void map_item_remove(map_t *map, map_item_t *map_item) {
  map_item_chain_t *chain = NULL;

  switch(map_item->type) {
  case MAP_TYPE_NODE:
    chain = map_item->node->map_item_chain;
    break;
  case MAP_TYPE_WAY:
    chain = map_item->way->map_item_chain;
    break;
  default:
    g_assert((map_item->type == MAP_TYPE_NODE) ||
	     (map_item->type == MAP_TYPE_WAY));
  }

  while(chain) {
    map_item_chain_t *next = chain->next;
    gtk_object_destroy(GTK_OBJECT(chain->map_item->item));
    chain = next;
  }
}

static void map_item_init(potlatch_conf_t *potlatch, map_item_t *map_item) {
  if(map_item->type == MAP_TYPE_WAY)
    map_way_color(potlatch, map_item->way);
}

void map_item_redraw(appdata_t *appdata, map_item_t *map_item) {
  map_item_t item = *map_item;

  /* check if the item to be redrawn is the selected one */
  gboolean is_selected = FALSE;
  if(map_item->ptr == appdata->map->selected.ptr) {
    map_item_deselect(appdata);
    is_selected = TRUE;
  }

  map_item_remove(appdata->map, &item);
  map_item_init(appdata->potlatch, &item);
  map_item_draw(appdata->map, &item);

  /* restore selection if there was one */
  if(is_selected) 
    map_item_select(appdata, &item);
}

static void map_draw(map_t *map, gpx_t *gpx) {
  g_assert(map->canvas);

  printf("drawing ways ...\n");
  way_t *way = gpx->way;
  while(way) {
    map_way_draw(map, way);
    way = way->next;
  }

  printf("drawing single nodes ...\n");
  node_t *node = gpx->node;
  while(node) {
    map_node_draw(map, node);
    node = node->next;
  }
}

static gint map_destroy_event(GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  map_t *map = appdata->map;

  printf("destroying map\n");

  g_free(map);
  appdata->map = NULL;

  return FALSE;
}

/* get the item at position x, y */
static map_item_t *map_item_at(map_t *map, gint x, gint y) {	  
  double wx, wy;
  gnome_canvas_window_to_world(GNOME_CANVAS(map->canvas), x, y, &wx, &wy);

  printf("world check at %f/%f\n", wx, wy);
  
  GnomeCanvasItem *item = 
    gnome_canvas_get_item_at(GNOME_CANVAS(map->canvas), wx, wy);
	  
  if(!item) {
    printf("  there's no item\n");
    return NULL;
  }

  printf("  there's an item\n");

  map_item_t *map_item = 
    (map_item_t*)gtk_object_get_user_data(GTK_OBJECT(item));

  if(!map_item) {
    printf("  item has no user data!\n");
    return NULL;
  }
    
  if(map_item->highlight)
    printf("  item is highlight\n");    

  switch(map_item->type) {
  case MAP_TYPE_NODE:
    printf("  item is node #%ld\n", map_item->node->id);
    break;
  case MAP_TYPE_WAY:
    printf("  item is way #%ld\n", map_item->way->id);
    break;
  default:
    printf("  unknown item\n");
    break;
  }

  return map_item;
}    

/* get the real item (no highlight) at x, y */
static map_item_t *map_real_item_at(map_t *map, gint x, gint y) { 
  map_item_t *map_item = map_item_at(map, x, y);

  /* no item or already a real one */
  if(!map_item || !map_item->highlight) return map_item;

  /* get the item (parent) this item is the highlight of */
  map_item_t *parent = NULL;
  switch(map_item->type) {
  case MAP_TYPE_NODE:
    if(map_item->node->map_item_chain)
      parent = map_item->node->map_item_chain->map_item;

    if(parent)
      printf("  using parent item node #%ld\n", parent->node->id);      
    break;
  case MAP_TYPE_WAY:
    if(map_item->way->map_item_chain)
      parent = map_item->way->map_item_chain->map_item;

    if(parent)
      printf("  using parent item way #%ld\n", parent->way->id);      
    break;
  default:
    g_assert((map_item->type == MAP_TYPE_NODE) ||
	     (map_item->type == MAP_TYPE_WAY)); 
    break;
  }
  
  if(parent) 
    map_item = parent;
  else 
    printf("  no parent, working on highlight itself\n");

  return map_item;
}


static gboolean map_scroll_event(GtkWidget *widget, GdkEventScroll *event,
				 gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  if(event->type == GDK_SCROLL) {
    if(event->direction) 
      appdata->map->zoom /= 1.2;
    else
      appdata->map->zoom *= 1.2;

    gnome_canvas_set_pixels_per_unit(GNOME_CANVAS(appdata->map->canvas), 
				     appdata->map->zoom);
  }

  return TRUE;
}

static gboolean distance_above(map_t *map, gint x, gint y, gint limit) {
  gint sx, sy;

  gnome_canvas_get_scroll_offsets(GNOME_CANVAS(map->canvas), &sx, &sy);

  /* add offsets generated by mouse within map and map scrolling */
  sx = (x-map->pen_down.at.x) + (map->pen_down.so.x-sx);
  sy = (y-map->pen_down.at.y) + (map->pen_down.so.y-sy);

  return(sx*sx + sy*sy > limit*limit);
}

static void map_do_scroll(map_t *map, gint x, gint y) {
  gint sx, sy;
  gnome_canvas_get_scroll_offsets(GNOME_CANVAS(map->canvas), &sx, &sy);
  sx -= (x-map->pen_down.at.x);
  sy -= (y-map->pen_down.at.y);
  gnome_canvas_scroll_to(GNOME_CANVAS(map->canvas), sx, sy);
}

static gboolean item_is_selected_node(map_t *map, map_item_t *map_item) {
  printf("check if item is a selected node\n");

  if(map->selected.type == MAP_TYPE_ILLEGAL) {
    printf("  nothing is selected\n");
    return FALSE;
  }

  /* clicked the highlight directly */
  if(map_item->type != MAP_TYPE_NODE) {
    printf("  didn't click node\n");
    return FALSE;
  }

  if(map->selected.type == MAP_TYPE_NODE) {
    printf("  selected item is a node\n");

    if(map_item->node == map->selected.node) {
      printf("  requested item is a selected node\n");
      return TRUE;
    }
    printf("  but it's not the requested one\n");
    return FALSE;

  } else if(map->selected.type == MAP_TYPE_WAY) {
    printf("  selected item is a way\n");

    node_chain_t *node_chain = map->selected.way->node_chain;
    while(node_chain) {
      if(node_chain->node == map_item->node) {
	printf("  requested item is part of selected way\n");
	return TRUE;
      } 
      node_chain = node_chain->next;
    }
    printf("  but it doesn't include the requested node\n");
    return FALSE;
    
  } else {
    printf("  selected item is unknown\n");
    return FALSE;
  }

  g_assert(0);
  return TRUE;
}

void map_highlight_refresh(appdata_t *appdata) {
  map_t *map = appdata->map;
  map_item_t old = map->selected;

  printf("type to refresh is %d\n", old.type);

  map_item_deselect(appdata);
  map_item_select(appdata, &old);
}

static void map_node_move(appdata_t *appdata, map_item_t *map_item, 
			  gint ex, gint ey) {

  map_t *map = appdata->map;
  gpx_t *gpx = appdata->gpx;

  g_assert(map_item->type == MAP_TYPE_NODE);
  node_t *node = map_item->node;

  printf("released dragged node #%ld\n", node->id);
  printf("  was at %d %d (%f %f)\n", 
	 node->lpos.x, node->lpos.y,
	 node->pos.lat, node->pos.lon);
	    
  struct { double x, y;} pos;	  
	    
  /* convert mouse position to canvas (world) position */
  gnome_canvas_window_to_world(GNOME_CANVAS(map->canvas), 
			       ex, ey, &pos.x, &pos.y);
	    
  /* convert to integer */
  node->lpos.x = pos.x; node->lpos.y = pos.y;
	    
  /* convert screen position back to utm/ll */
  utm_t utm = gpx->bounds->center;  /* set zone/band */
  utm.easting  =  node->lpos.x + gpx->bounds->center.easting;
  utm.northing = -node->lpos.y + gpx->bounds->center.northing;
  
  utm_utm2pos(&utm, &node->pos);
  printf("  now at %d %d (%f %f)\n", 
	 node->lpos.x, node->lpos.y, node->pos.lat, node->pos.lon);
  
  /* now update the visual representation of the node */
  GnomeCanvasItem *item = map_item->item;
  glong x = node->lpos.x, y = node->lpos.y;
  
  if(!node->ways) 
    gnome_canvas_item_set(item,
			  "x1", (double)x-RADIUS, "x2", (double)x+RADIUS,
			  "y1", (double)y-RADIUS, "y2", (double)y+RADIUS,
			  NULL);
  else {
    /* only move item of it's not the highlight. highlight will be */
    /* redrawed entirely lateron */
    if(!map_item->highlight)
      gnome_canvas_item_set(item,
			    "x1", (double)x-S_RADIUS, "x2", (double)x+S_RADIUS,
			    "y1", (double)y-S_RADIUS, "y2", (double)y+S_RADIUS,
			    NULL);
  }

  /* update ways, node is part of */
  way_t *way = gpx->way;
  while(way) {
    if(gpx_node_in_way(way, node)) {
      printf("node is part of way #%ld\n", way->id);
      
      /* rebuild ways positions */
      
      /* allocate space for nodes */
      guint nodes = gpx_way_number_of_nodes(way);
      if(nodes > 1) {
	GnomeCanvasPoints *points = gnome_canvas_points_new(nodes);
      
	int node = 0;
	node_chain_t *node_chain = way->node_chain;
	while(node_chain) {
	  /* mirror map vertically within bounds */
	  points->coords[node++] = node_chain->node->lpos.x;
	  points->coords[node++] = node_chain->node->lpos.y;
	  node_chain = node_chain->next;
	}
	
	/* do something with points */
	map_item_chain_t *chain = way->map_item_chain;
	while(chain) {
	  gnome_canvas_item_set(chain->map_item->item, 
				"points", points, NULL);
	  chain = chain->next;
	}
	
	gnome_canvas_points_free(points);
      }
    }
    
    way = way->next;
  }
  
  /* and mark the node as dirty */
  node->flags |= GPX_FLAG_DIRTY;

  /* update highlight */
  map_highlight_refresh(appdata);
}
			  
static void map_handle_click(appdata_t *appdata, map_t *map) {

  /* problem: on_item may be the highlight itself! So store it! */
  map_item_t map_item;
  if(map->pen_down.on_item) map_item = *map->pen_down.on_item;
  else                      map_item.type = MAP_TYPE_ILLEGAL;

  /* if we aready have something selected, then de-select it */
  map_item_deselect(appdata);
  
  /* select the clicked item (if there was one) */
  if(map_item.type != MAP_TYPE_ILLEGAL) {
    switch(map_item.type) {
    case MAP_TYPE_NODE:
      map_node_select(appdata, map_item.node);
      break;

    case MAP_TYPE_WAY:
      map_way_select(appdata, map_item.way);
      break;

    default:
      g_assert((map_item.type == MAP_TYPE_NODE) ||
	       (map_item.type == MAP_TYPE_WAY));
      break;
    }
  }
}

static void map_button_press(map_t *map, gint x, gint y) {

  printf("left button pressed\n");
  map->pen_down.is = TRUE;
      
  /* save press position */
  map->pen_down.at.x = x;
  map->pen_down.at.y = y;
  map->pen_down.drag = FALSE;     // don't assume drag yet
  
  /* save initial scroll offset */
  gnome_canvas_get_scroll_offsets(GNOME_CANVAS(map->canvas), 
				  &map->pen_down.so.x, &map->pen_down.so.y);
  
  /* determine wether this press was on an item */
  map->pen_down.on_item = map_real_item_at(map, x, y);
  
  /* check if the clicked item is a highlighted node as the user */
  /* might want to drag that */
  map->pen_down.on_selected_node = FALSE;
  if(map->pen_down.on_item)
    map->pen_down.on_selected_node = 
      item_is_selected_node(map, map->pen_down.on_item);

  switch(map->action) {
  case MAP_ACTION_IDLE:
    break;
  case MAP_ACTION_NODE_ADD:
    map_hl_cursor_draw(map, x, y);
    break;
  default:
    break;
  }
}

static void map_button_release(map_t *map, gint x, gint y) {
  map->pen_down.is = FALSE;

  switch(map->action) {
  case MAP_ACTION_IDLE:
    /* check if distance to press is above drag limit */
    if(!map->pen_down.drag)
      map->pen_down.drag = distance_above(map, x, y, MAP_DRAG_LIMIT);
    
    if(!map->pen_down.drag) {
      printf("left button released after click\n");
      
      map_item_t old_sel = map->selected;
      map_handle_click(map->appdata, map);
      
      if((old_sel.type != MAP_TYPE_ILLEGAL) && 
	 (old_sel.ptr == map->selected.ptr)) {
	printf("re-selected same item of type %d/%d, "
	       "pushing it to the bottom\n", old_sel.type, 
	       map->selected.type);
	
	if(!map->selected.item) {
	  printf("  item has no visible representation to push\n");
	} else {
	  gnome_canvas_item_lower_to_bottom(map->selected.item);
	  
	  /* update clicked item, to correctly handle the click */
	  map->pen_down.on_item = 
	    map_real_item_at(map, map->pen_down.at.x, map->pen_down.at.y);
	  
	map_handle_click(map->appdata, map);
	}
      }
    } else {
      printf("left button released after drag\n");
      
      /* just scroll if we didn't drag an selected item */
      if(!map->pen_down.on_selected_node)
	map_do_scroll(map, x, y);
      else {
	printf("released after dragging node\n");
	map_hl_cursor_clear(map);
	map_hl_touchnode_clear(map);
	
	/* now actually move the node */
	map_node_move(map->appdata, map->pen_down.on_item, x, y);
      }
    }
    break;

  case MAP_ACTION_NODE_ADD:
    printf("released after NODE ADD\n");
    map_hl_cursor_clear(map);

    double px, py;
    /* convert mouse position to canvas (world) position */
    gnome_canvas_window_to_world(GNOME_CANVAS(map->canvas), x, y, &px, &py);
    node_t *node = gpx_node_new(map->appdata->gpx, px, py);
    map_node_draw(map, node);

    map->action = MAP_ACTION_IDLE;

    map_item_deselect(map->appdata);
    map_node_select(map->appdata, node);
    break;

  default:
    break;
  }
}

static gboolean map_button_event(GtkWidget *widget, GdkEventButton *event,
				       gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  map_t *map = appdata->map;

  if(event->button == 1) {
    gint x = event->x, y = event->y;

    if(event->type == GDK_BUTTON_PRESS) 
      map_button_press(map, x, y);

    if(event->type == GDK_BUTTON_RELEASE) 
      map_button_release(map, x, y);
  }

  return FALSE;  /* forward to further processing */
}

static void map_touchnode_update(appdata_t *appdata, gint x, gint y) {
  map_t *map = appdata->map;

  map_hl_touchnode_clear(map);

  /* check if we are close to another node */
  g_assert(map->pen_down.on_item);
  g_assert(map->pen_down.on_item->type == MAP_TYPE_NODE);

  /* check if we are close to one of the other nodes */
  double wx, wy;
  gnome_canvas_window_to_world(GNOME_CANVAS(appdata->map->canvas), 
			       x, y, &wx, &wy);
  gint iwx = wx, iwy = wy;
  node_t *node = appdata->gpx->node;
  while(node) {
    /* don't highlight the dragged node itself and don't highlight */
    /* deleted ones */
    if((node != map->pen_down.on_item->node) && 
       (!(node->flags & GPX_FLAG_DELETED))) {
      gint nx = iwx - node->lpos.x;
      gint ny = iwy - node->lpos.y;
	  
      if(nx*nx + ny*ny < RADIUS*RADIUS)
	map_hl_touchnode_draw(map, node->lpos.x, node->lpos.y);

    }
    node = node->next;
  }
}

static gboolean map_motion_notify_event(GtkWidget *widget, 
                             GdkEventMotion *event, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  map_t *map = appdata->map;
  gint x, y;
  GdkModifierType state;

  if(!map->pen_down.is) 
    return FALSE;

  /* handle hints */
  if(event->is_hint)
    gdk_window_get_pointer(event->window, &x, &y, &state);
  else {
    x = event->x;
    y = event->y;
    state = event->state;
  }

  /* check if distance to press is above drag limit */
  if(!map->pen_down.drag)
    map->pen_down.drag = distance_above(map, x, y, MAP_DRAG_LIMIT);
  
  switch(map->action) {
  case MAP_ACTION_IDLE:
    if(map->pen_down.drag) {
      /* just scroll if we didn't drag an selected item */
      if(!map->pen_down.on_selected_node)
	map_do_scroll(map, x, y);
      else {
	map_hl_cursor_draw(map, x, y);
	map_touchnode_update(appdata, x, y);
      }
    }
    break;
    
  case MAP_ACTION_NODE_ADD:
    map_hl_cursor_draw(map, x, y);
    break;
    
  default:
    break;
  }

  return FALSE;  /* forward to further processing */
}

gboolean map_key_press_event(appdata_t *appdata, GdkEventKey *event) {
  /* map needs to be there to handle buttons */
  if(!appdata->map->canvas) 
    return FALSE;

  if(event->type == GDK_KEY_PRESS) {
    switch(event->keyval) {

#ifdef USE_HILDON
    case HILDON_HARDKEY_INCREASE:
#else
    case '+':
#endif
      appdata->map->zoom *= 2;
      printf("zoom is now %f\n", appdata->map->zoom);
      gnome_canvas_set_pixels_per_unit(GNOME_CANVAS(appdata->map->canvas), 
				       appdata->map->zoom);
      return TRUE;
      break;

#ifdef USE_HILDON
    case HILDON_HARDKEY_DECREASE:
#else
    case '-':
#endif
      appdata->map->zoom /= 2;
      printf("zoom is now %f\n", appdata->map->zoom);
      gnome_canvas_set_pixels_per_unit(GNOME_CANVAS(appdata->map->canvas), 
				       appdata->map->zoom);
      return TRUE;
      break;
    }

    printf("key event %d\n", event->keyval);
  }

  return FALSE;
}

GtkWidget *map_new(appdata_t *appdata) {
  map_t *map = appdata->map = g_new0(map_t, 1);

  map->zoom = 1.0;
  map->pen_down.at.x = -1;
  map->pen_down.at.y = -1;
  map->appdata = appdata;
  map->action = MAP_ACTION_IDLE;

  map->canvas = gnome_canvas_new_aa();  // _aa

  /* create the groups */
  map_group_t group;
  for(group = 0; group < MAP_GROUPS; group++) {
    map->group[group] = gnome_canvas_item_new(
	       	gnome_canvas_root(GNOME_CANVAS(map->canvas)),
	       	GNOME_TYPE_CANVAS_GROUP,
	       	NULL);       
  }

  gtk_widget_modify_bg(map->canvas, GTK_STATE_NORMAL, 
		       &map->canvas->style->white);

  gtk_widget_set_events(map->canvas,
                          GDK_BUTTON_PRESS_MASK
			| GDK_BUTTON_RELEASE_MASK
			| GDK_SCROLL_MASK
			| GDK_POINTER_MOTION_MASK
    			| GDK_POINTER_MOTION_HINT_MASK);
  
  gtk_signal_connect(GTK_OBJECT(map->canvas), 
     "button_press_event", G_CALLBACK(map_button_event), appdata);
  gtk_signal_connect(GTK_OBJECT(map->canvas), 
     "button_release_event", G_CALLBACK(map_button_event), appdata);
  gtk_signal_connect(GTK_OBJECT(map->canvas), 
     "motion_notify_event", G_CALLBACK(map_motion_notify_event), appdata);
  gtk_signal_connect(GTK_OBJECT(map->canvas), 
     "scroll_event", G_CALLBACK(map_scroll_event), appdata);

  gtk_signal_connect(GTK_OBJECT(map->canvas), 
     "destroy", G_CALLBACK(map_destroy_event), appdata);

  int cx, cy;
  gnome_canvas_w2c(GNOME_CANVAS(map->canvas), 0,0, &cx, &cy);
  gnome_canvas_scroll_to(GNOME_CANVAS(map->canvas), cx, cy);

  return map->canvas;
}

void map_init(appdata_t *appdata) {
  map_t *map = appdata->map;

  /* set initial zoom */
  gnome_canvas_set_pixels_per_unit(GNOME_CANVAS(map->canvas), 
				   map->zoom);

  map_init_colors(appdata->potlatch, appdata->gpx);
  map_draw(map, appdata->gpx);

  /* adjust scroll region to map bounds */
  gnome_canvas_set_scroll_region(GNOME_CANVAS(map->canvas), 
	 appdata->gpx->bounds->min.x, appdata->gpx->bounds->min.y,
	 appdata->gpx->bounds->max.x, appdata->gpx->bounds->max.y);

}


void map_item_set_flags(map_item_t *map_item, int set, int clr) {

  switch(map_item->type) {
  case MAP_TYPE_NODE:
    map_item->node->flags |=  set;
    map_item->node->flags &= ~clr;
    break;

  case MAP_TYPE_WAY:
    map_item->way->flags |=  set;
    map_item->way->flags &= ~clr;
    break;

  default:
    g_assert(0);
    break;
  }
}

void map_clear(appdata_t *appdata) {
  map_t *map = appdata->map;

  printf("freeing map contents\n");

  /* remove a possibly existing highlight */
  map_item_deselect(appdata);

  map_group_t group;
  for(group=0;group<MAP_GROUPS;group++) {
    /* destroy the entire group */
    gtk_object_destroy(GTK_OBJECT(map->group[group]));

    /* and create an empty new one */
    map->group[group] = gnome_canvas_item_new(
		      gnome_canvas_root(GNOME_CANVAS(map->canvas)),
		      GNOME_TYPE_CANVAS_GROUP,
		      NULL);       
  }
}

void map_paint(appdata_t *appdata) {
  map_t *map = appdata->map;

  map_init_colors(appdata->potlatch, appdata->gpx);
  map_draw(map, appdata->gpx);
}

/* called from several icons like e.g. "node_add" */
void map_action_set(appdata_t *appdata, map_action_t action) {
  printf("map action set to %d\n", action);
  appdata->map->action = action;
}

static void map_item_chain_destroy(map_item_chain_t *chain) {
  while(chain) {
    map_item_chain_t *next = chain->next;
    gtk_object_destroy(GTK_OBJECT(chain->map_item->item));
    chain = next;
  }
}

/* called from icon "trash" */
void map_delete_selected(appdata_t *appdata) {
  map_t *map = appdata->map;

  /* work on local copy since de-selecting destroys the selection */
  map_item_t item = map->selected;

  /* deleting the selected item de-selects it ... */
  map_item_deselect(appdata);

  switch(item.type) {
  case MAP_TYPE_NODE:
    printf("request to delete node #%ld\n", item.node->id);

    /* check if this node is part of a way with two nodes only. */
    /* we cannot delete this as this would also delete the way */
    way_chain_t *way_chain = gpx_node_to_way(appdata->gpx, item.node);
    if(way_chain) {
      gboolean short_way = FALSE;
      
      /* free the chain of relations */
      while(way_chain && !short_way) {
	way_chain_t *next = way_chain->next;

	if(gpx_way_number_of_nodes(way_chain->way) <= 2) 
	  short_way = TRUE;

	g_free(way_chain);
	way_chain = next;
      }

      if(short_way) {
	messagef(GTK_WIDGET(appdata->window), _("Node in short way"), 
		 _("This node cannot be deleted (yet) as it's part of a way "
		   "consisting of only two nodes."));
	return;
      }
    }

    /* check if this node is part of a relation */
    relation_chain_t *rel_chain = gpx_node_to_relation(appdata->gpx, item.node);
    if(rel_chain) {
      
      /* free the chain of relations */
      while(rel_chain) {
	relation_chain_t *next = rel_chain->next;
	g_free(rel_chain);
	rel_chain = next;
      }

      messagef(GTK_WIDGET(appdata->window), _("Node in relation"), 
	       _("This node cannot be deleted (yet) as it's part of a relation"));

      return;
    }

    /* remove it visually from the screen */
    map_item_chain_destroy(item.node->map_item_chain);

    /* and mark it "deleted" in the database */
    way_chain_t *chain = gpx_node_delete(appdata->gpx, item.node, FALSE);

    /* redraw all affected ways */
    while(chain) {
      way_chain_t *next = chain->next;

      if(gpx_way_number_of_nodes(chain->way) == 1) {
	printf("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX\n");
	printf("TODO: Only 1 node left on way -> delete way also\n");
	printf("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX\n");
      }

      map_item_t item;
      item.type = MAP_TYPE_WAY;
      item.way = chain->way;
      map_item_redraw(appdata, &item);

      g_free(chain);

      chain = next;
    }

    break;

  case MAP_TYPE_WAY:
    printf("NYI: request to delete way #%ld\n", item.way->id);
    g_assert(0);
    break;

  default:
    g_assert((item.type == MAP_TYPE_NODE) ||
	     (item.type == MAP_TYPE_WAY));
    break;
  }
}
