/*
 * This file is part of OSM2Go.
 *
 * OSM2Go is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * OSM2Go is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSM2Go.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <string.h>

#include <curl/curl.h>

#include "appdata.h"

void messagef(GtkWidget *parent, char *title, const char *fmt, ...) {
  va_list args;
  va_start( args, fmt );
  char *buf = g_strdup_vprintf(fmt, args);
  va_end( args );

  printf("%s: \"%s\"\n", title, buf); 
  
  GtkWidget *dialog = gtk_message_dialog_new(
		     GTK_WINDOW(parent),
                     GTK_DIALOG_DESTROY_WITH_PARENT,
                     GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
                     buf);

  gtk_window_set_title(GTK_WINDOW(dialog), title);

  gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);

  g_free(buf);
}

void errorf(GtkWidget *parent, const char *fmt, ...) {
  va_list args;
  va_start( args, fmt );
  char *buf = g_strdup_vprintf(fmt, args);
  va_end( args );

  printf("errorf(\"%s\")\n", buf); 
  
  GtkWidget *dialog = gtk_message_dialog_new(
		     GTK_WINDOW(parent),
                     GTK_DIALOG_DESTROY_WITH_PARENT,
                     GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
                     buf);

  gtk_window_set_title(GTK_WINDOW(dialog), _("ERROR"));

  gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);

  g_free(buf);
}

/******************** begin of menu *********************/

static void 
cb_menu_about(GtkWidget *window, gpointer data) {
  GtkAboutDialog *about = GTK_ABOUT_DIALOG(gtk_about_dialog_new());

  gtk_about_dialog_set_name(about, "OSM2Go");
  gtk_about_dialog_set_version(about, VERSION);
  gtk_about_dialog_set_copyright(about, 
	 _("OSM2Go (c) 2008 by\n" 
	 "Till Harbaum <till@harbaum.org>"));

  gtk_about_dialog_set_website(about,
       _("http://www.harbaum.org/till/maemo"));
  
  gtk_about_dialog_set_comments(about, 
       _("Maemo OSM Editor"));

  gtk_widget_show_all(GTK_WIDGET(about));
  gtk_dialog_run(GTK_DIALOG(about));
  gtk_widget_destroy(GTK_WIDGET(about));
}

void on_window_destroy (GtkWidget *widget, gpointer data);

static void 
cb_menu_quit(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  gtk_widget_destroy(appdata->window);
}

static void 
cb_menu_upload(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  if(!appdata->gpx || !appdata->project) return;

  osm_upload(appdata, appdata->gpx, appdata->project);
}

static void 
cb_menu_redraw(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  /* redraw the entire map by destroying all map items and redrawing them */
  diff_save(appdata->project, appdata->gpx);
  map_clear(appdata);
  gpx_free(appdata->gpx);

  appdata->gpx = gpx_parse(appdata->project->osm);
  diff_restore(appdata, appdata->project, appdata->gpx);
  map_paint(appdata);
}

#ifdef USE_HILDON
static void 
cb_menu_fullscreen(GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t *)data;

  if(gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(widget)))
    gtk_window_fullscreen(GTK_WINDOW(appdata->window));
  else
    gtk_window_unfullscreen(GTK_WINDOW(appdata->window));
}
#endif

void menu_create(appdata_t *appdata) { 
  GtkWidget *menu, *item;  
  menu = gtk_menu_new();

#ifdef USE_HILDON
  appdata->fullscreen_menu_item = 
    item = gtk_check_menu_item_new_with_label( _("Fullscreen") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_fullscreen), 
		   appdata);
#endif

  item = gtk_menu_item_new_with_label( _("Upload...") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_upload), appdata);

  item = gtk_menu_item_new_with_label( _("Redraw...") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_redraw), appdata);

  gtk_menu_append(GTK_MENU_SHELL(menu), gtk_separator_menu_item_new());

  item = gtk_menu_item_new_with_label( _("About...") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_about), appdata);


  item = gtk_menu_item_new_with_label( _("Quit") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_quit), appdata);

#ifdef USE_HILDON
  hildon_window_set_menu(appdata->window, GTK_MENU(menu));
#else
  /* attach ordinary gtk menu */
  GtkWidget *menu_bar = gtk_menu_bar_new();

  GtkWidget *root_menu = gtk_menu_item_new_with_label (_("Menu"));
  gtk_widget_show(root_menu);

  gtk_menu_bar_append(menu_bar, root_menu); 
  gtk_menu_item_set_submenu(GTK_MENU_ITEM (root_menu), menu);

  gtk_widget_show(menu_bar);
  gtk_box_pack_start(GTK_BOX(appdata->vbox), menu_bar, 0, 0, 0);
#endif
}

/********************* end of menu **********************/


void cleanup(appdata_t *appdata) {
#ifdef USE_HILDON
  if(appdata->osso_context)
    osso_deinitialize(appdata->osso_context);

  appdata->program = NULL;
#endif

  /* let gtk clean up first */
  while(gtk_events_pending())
    gtk_main_iteration();

  gpx_free(appdata->gpx);

  curl_global_cleanup();

  icon_free_all(appdata);

  puts("everything is gone");
}

void on_window_destroy (GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  gtk_main_quit();
  appdata->window = NULL;
}

gboolean on_window_key_press(GtkWidget *widget, 
			 GdkEventKey *event, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  int handled = FALSE;

  //  printf("key event %d\n", event->keyval);

  // the map handles some keys on its own ...
  switch(event->keyval) {
#ifdef USE_HILDON

#if 0
  case HILDON_HARDKEY_SELECT:
    handled = TRUE;
    break;
#endif
    
  case HILDON_HARDKEY_FULLSCREEN:
    {
      gboolean fullscreen = !gtk_check_menu_item_get_active(
	       GTK_CHECK_MENU_ITEM(appdata->fullscreen_menu_item));
      gtk_check_menu_item_set_active(
	       GTK_CHECK_MENU_ITEM(appdata->fullscreen_menu_item), fullscreen);

      if(fullscreen)
	gtk_window_fullscreen(GTK_WINDOW(appdata->window));
      else
	gtk_window_unfullscreen(GTK_WINDOW(appdata->window));

      handled = TRUE;
    }
    break;
#endif
  }

  if(!handled) 
    handled = map_key_press_event(appdata, event);
  
  return handled;
}

int main(int argc, char *argv[]) {
  appdata_t appdata;

  /* init appdata */
  memset(&appdata, 0, sizeof(appdata));

  printf("Using locale for %s in %s\n", PACKAGE, LOCALEDIR);

  setlocale(LC_ALL, "");
  bindtextdomain(PACKAGE, LOCALEDIR);
  bind_textdomain_codeset(PACKAGE, "UTF-8");
  textdomain(PACKAGE);

  /* Must initialize libcurl before any threads are started */
  curl_global_init(CURL_GLOBAL_ALL);

  g_thread_init(NULL);
  
  gtk_init (&argc, &argv);

#ifdef USE_HILDON
  printf("Installing osso context for \"org.harbaum." PACKAGE "\"\n");
  appdata.osso_context = osso_initialize("org.harbaum."PACKAGE, 
					 VERSION, TRUE, NULL);
  if(appdata.osso_context == NULL) 
    fprintf(stderr, "error initiating osso context\n");
#endif

#ifdef USE_HILDON
  /* Create the hildon program and setup the title */
  appdata.program = HILDON_PROGRAM(hildon_program_get_instance());
  g_set_application_name("OSM2Go");
  
  /* Create HildonWindow and set it to HildonProgram */
  appdata.window = HILDON_WINDOW(hildon_window_new());
  hildon_program_add_window(appdata.program, appdata.window);
#else
  /* Create a Window. */
  appdata.window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(appdata.window), "OSM2Go");
  /* Set a decent default size for the window. */
  gtk_window_set_default_size(GTK_WINDOW(appdata.window), 640, 480);
  gtk_window_set_icon(GTK_WINDOW(appdata.window), icon_load(&appdata, "osm2go"));
#endif

  g_signal_connect(G_OBJECT(appdata.window), "destroy", 
		   G_CALLBACK(on_window_destroy), &appdata);

  g_signal_connect(G_OBJECT(appdata.window), "key_press_event",
 		   G_CALLBACK(on_window_key_press), &appdata);

  appdata.vbox = gtk_vbox_new(FALSE,0);
  menu_create(&appdata);

  /* user specific init */
  appdata.settings = settings_load();  

  /* load potlatch config files to resemble potlatch view */
  appdata.potlatch = potlatch_conf_load();

  /* ----------------------- setup main window ---------------- */

  GtkWidget *hbox = gtk_hbox_new(FALSE,0);
  GtkWidget *vbox = gtk_vbox_new(FALSE,0);

  gtk_box_pack_start(GTK_BOX(vbox), map_new(&appdata), TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), statusbar_new(&appdata), FALSE, FALSE, 0);

  gtk_box_pack_start(GTK_BOX(hbox), iconbar_new(&appdata), FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), gtk_vseparator_new(), FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);

  gtk_box_pack_start(GTK_BOX(appdata.vbox), hbox, TRUE, TRUE, 0);

  gtk_container_add(GTK_CONTAINER(appdata.window), appdata.vbox);

  gtk_widget_show_all(GTK_WIDGET(appdata.window));

  /* -------------- load the project ------------- */

  appdata.project = project_open(&appdata, "test");
  if(!appdata.project) {
    cleanup(&appdata);
    return 0;
  }    

  printf("loading gpx\n");
  appdata.gpx = gpx_parse(appdata.project->osm);
  if(!appdata.gpx) {
    printf("parsing failed\n");
    cleanup(&appdata);
    return 0;
  }

  printf("parsing ok\n");

  printf("loading diff\n");
  diff_restore(&appdata, appdata.project, appdata.gpx);

  map_init(&appdata);

  /* ------------ jump into main loop ---------------- */

  gtk_main();

  /* save a diff if there are dirty entries */
  diff_save(appdata.project, appdata.gpx);

  //  osm_upload(&appdata, appdata.gpx, project);

  cleanup(&appdata);

  return 0;
}
