/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of OSM2Go.
 *
 * OSM2Go is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * OSM2Go is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSM2Go.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef GPX_H
#define GPX_H

#include <math.h>

#define GPX_FLAG_DIRTY    (1<<0)
#define GPX_FLAG_DELETED  (1<<1)
#define GPX_FLAG_NEW      (1<<2)

typedef gulong item_id_t;

#define ID_ILLEGAL  ((item_id_t)0)

typedef struct bounds {
  pos_t ll_min, ll_max;
  lpos_t min, max;
  utm_t center;
} bounds_t;

typedef struct user {
  char *name;
  struct user *next;
} user_t;

typedef struct tag {
  char *key, *value;
  struct tag *next;
} tag_t;

typedef struct node {
  item_id_t id;
  pos_t pos;
  lpos_t lpos;
  user_t *user;
  gboolean visible;
  time_t time;
  tag_t *tag;
  int ways;
  int flags;

  /* a link to the visual representation on screen */
  struct map_item_chain_s *map_item_chain;

  struct node *next;
} node_t;

typedef struct node_chain {
  node_t *node;
  struct node_chain *next;
} node_chain_t;

#define GPX_DRAW_FLAGS_POLYGON  (1<<0)
#define GPX_DRAW_FLAGS_STROKE   (1<<1)

typedef struct way {
  item_id_t id;
  user_t *user;
  gboolean visible;
  time_t time;
  tag_t *tag;
  node_chain_t *node_chain;
  int flags;

  /* visual representation from potlatch conf */
  guint color, fill_color;
  guint draw_flags;

  /* a link to the visual representation on screen */
  struct map_item_chain_s *map_item_chain;

  struct way *next;
} way_t;

typedef struct way_chain {
  way_t *way;
  struct way_chain *next;
} way_chain_t;

typedef enum {
  ILLEGAL=0, NODE, WAY, NODE_ID, WAY_ID
} type_t;

typedef struct member {
  type_t type; 
  char   *role;

  union {
    node_t *node;
    way_t *way;
    item_id_t id;
  };

  struct member *next;
} member_t;

typedef struct relation {
  item_id_t id;
  user_t *user;
  gboolean visible;
  time_t time;
  tag_t *tag;
  member_t *member;
  int flags;

  struct relation *next;
} relation_t;

typedef struct relation_chain {
  relation_t *relation;
  struct relation_chain *next;
} relation_chain_t;

typedef struct gpx {
  bounds_t *bounds;   // original bounds as they appear in the file
  user_t *user;
  node_t *node;
  way_t  *way;
  relation_t  *relation;
} gpx_t;

#include <libxml/parser.h>
#include <libxml/tree.h>

gpx_t *gpx_parse(char *filename);
tag_t *gpx_parse_osm_tag(gpx_t *gpx, xmlDocPtr doc, xmlNode *a_node);
node_chain_t *gpx_parse_osm_way_nd(gpx_t *gpx, xmlDocPtr doc, xmlNode *a_node);
void gpx_dump(gpx_t *gpx);
void gpx_free(gpx_t *gpx);

gboolean gpx_way_has_key_or_value(way_t *way, char *str);
gboolean gpx_way_is_way(way_t *way);
char *gpx_way_get_value(way_t *way, char *key);
char *gpx_node_get_value(node_t *node, char *key);
gboolean gpx_node_has_tag(node_t *node);

void gpx_node_dump(node_t *node);
void gpx_way_dump(way_t *way);

void gpx_tag_free(tag_t *tag);
void gpx_node_chain_free(node_chain_t *node_chain);

gboolean gpx_node_in_way(way_t *way, node_t *node);

char *gpx_generate_xml_node(gpx_t *gpx, node_t *node);
char *gpx_generate_xml_way(gpx_t *gpx, way_t *way);

node_t *gpx_get_node_by_id(gpx_t *gpx, item_id_t id);
way_t *gpx_get_way_by_id(gpx_t *gpx, item_id_t id);

guint gpx_way_number_of_nodes(way_t *way);
relation_chain_t *gpx_node_to_relation(gpx_t *gpx, node_t *node);
way_chain_t *gpx_node_to_way(gpx_t *gpx, node_t *node);

/* ----------- edit functions ----------- */
node_t *gpx_node_new(gpx_t *gpx, gint x, gint y);
way_chain_t *gpx_node_delete(gpx_t *gpx, node_t *node, gboolean permanently);

#endif /* GPX_H */
