/*******************************************************************************
This file is part of mdictionary.

mdictionary is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

mdictionary is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License 
along with mdictionary; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Copyright 2006-2008 ComArch S.A.
*******************************************************************************/
/** \defgroup XDXFEngine Dictionary Engine - XDXF format
 * \brief XDXF-based dictionary engine.
 *
 * This is library with dictionary engine supporting XDXF dictionaries. XDXF is
 * based on XML language. For more information, please go to:
 * \li http://xdxf.sourceforge.net/
 *
 * TODO: currently XDXF engine does not support all function from engine API!
 * It need implementation of API version 0.2 function, like getting icon path
 * etc.
 */
/*@{*/
/** \file engine_xdxf.h
 * \brief Header for XDXF-based dictionary engine.
 */

#ifndef _DICTIONARY_ENGINE_XDXF
#define _DICTIONARY_ENGINE_XDXF

#ifdef __cplusplus
        extern "C" {
#endif

/* headers with unix types/functions - onl for timers */
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <unistd.h>

#include <glib.h> /* header with GLIB definitions/functions/types */
#include <libgnomevfs/gnome-vfs.h> /* header with gnome-vfs - recommended I/O
                                    * API for maemo */
#include <expat.h> /* header with expat - XML Parser API */
#include <string.h> /* manipulating strings */
#include <dictionary_engine.h> /* header wit engine API */


/** \name Timers */
/*@{*/
/** \brief Flags telling that we want to start timer. */
#define TIMER_START     TRUE
/** \brief Flags telling that we want to stop timer. */
#define TIMER_STOP      FALSE

/** \brief Start/stop timers.
 *
 * \param start do we want start new timer or end the last one
 * \param message string which shoul be printed by function while debbuging
 * \return -1.0 if we start or seconds passed from start if we want to 
 * stop timer
 */
/* in final releases timer capapilities should be removed for increase
 * performance */
/* static double timer(gboolean start, gchar* message); */

/*@}*/


/** \brief Version of XDXF engine. */
#define DIC_ENG_VERSION "0.1b"

/** \brief Short description of format supported by the current engine. */
#define DIC_ENG_FORMAT  "XDXF"

/** \brief Buffer's length used while generating cache for dictionary. */
#define DICT_CACHEING_BUFF_SIZE                 16*1024

/** \brief Buffer's length used while searching for words list. */
#define DICT_SEARCHING_WORD_LIST_BUFF_SIZE      16*1024

/** \brief Buffer's length used while searching for translation. */
#define DICT_SEARCHING_WORD_TRAN_BUFF_SIZE      16*1024

/** \brief Maximum length of word in dictionary.
 *
 * Engine use this value while searching in cache file. If this would be too
 * low value engine would work incorrectly, but too big value will decrease
 * performance of searching. 512 is optimal in most cases.
 */
#define DICT_MAX_WORD_LENGTH   512

/** \brief Translate boolean value into string. */
#define PRINT_STATE(state) ( (state) ? "TRUE" : "FALSE" )


/** \brief Structure used while dict_eng_module_check() is working.
 */
struct _XDXFCheckingData {
	gboolean further;
	/**< \brief do we need to continue checking */
	gboolean good;
	/**< \brief returned value telling if given file was proper XDXF
	 * dictionary */
	guint    deep;
	/**< \brief keep depth of XDXF structure while parsing file */
};
typedef struct _XDXFCheckingData XDXFCheckingData;

/** \brief Structure to help parse xdxf file for searching words list.
 */
struct _XDXFWordsListData {
	gchar*   last_word;
	/**< \brief last found word */
	gchar*   pattern;
	/**< \brief pattern for words to search */
	guint    pattern_len;
	/**< \brief length of pattern */
	guint    last_word_length;
	/**< \brief length of last found word */
	GArray*  result;
	/**< \brief result of searching - GArray with words matching pattern */
	guint    one_word;
	/**< \brief 1 while engine is parsing whole particular word (k tag), 0 
	 * otherwise */
	gboolean cont;
	/**< \brief do we need to continue searching */
};
typedef struct _XDXFWordsListData       XDXFWordsListData;

/** \brief Structure to help parse xdxf file for searching word's translation.
 */
struct _XDXFWordsTransData {
	gchar*          last_word;
	/**< \brief last found word in dictionary */
	gchar*          word;
	/**< \brief word to search translation for */
	guint           word_len;
	/**< \brief length of word */
	guint           last_word_length;
	/**< \brief length of last found word */
	gchar*          translation;
	/**< \brief found translation or NULL if such was not found */
	guint           one_word;
	/**< \brief 1 while engine is parsing whole particular word (k tag), 0
	 * otherwise */
	gboolean        cont;
	/**< \brief do we need to continue searching */
	gulong          last_start;
	/**< \brief keeps offset in file of last found article 
	 * (word plus translation) */
	XML_Parser*     parser;
	/**< \brief pointer to expat XML parser */
	gboolean        found;
	/**< \brief telling if translation was found */
	GnomeVFSHandle* xdxf;
	/**< \brief pointer to dictionary file */
};
typedef struct _XDXFWordsTransData      XDXFWordsTransData;

/** \brief Structure to help make optimization possible
 */
struct _XDXFCacheData {
	gchar*          buffer;
	/**< \brief buffer with output part of cache file */
	long            last_start;
	/**< \brief keeps offset in file of the beggining of last found article
	 */
	long            last_stop;
	/**< \brief keeps offset in file of the end of last found article */
	long            last_length;
	/**< \brief keeps length of last found article */
	GnomeVFSHandle* cache;
	/**< \brief pointer to cache file */
	XML_Parser      parser;
	/**< \brief pointer to expat XML parser */
	int             state;
	/**< \brief 1 if parser is parsing article key (word), 2 if parser is
	 * parsing article value (translation), 0 otherwise. */
	long            buffer_length;
	/**< \brief buffer's length */
};
typedef struct _XDXFCacheData           XDXFCacheData;

/** \brief Internal data structure for representing part of file.
 */
struct _FilePart {
	guint offset;
	/**< \brief offset in file of file part */
	guint length;
	/**< \brief length of file part */
};
typedef struct _FilePart        FilePart;

/** \brief Internal data structure of XDXF Engine.
 */
struct _XDXFData {
	GnomeVFSHandle*         xdxf;
	/**< \brief pointer to *.xdxf file */
	GnomeVFSHandle*         cache;
	/**< \brief pointer to cache file */
	gchar*                  dict_path;
	/**< \brief path to dictionary */
	EngineStatus            last_error;
	/**< \brief status of last taken action */
	gboolean                auto_free;
	/**< \brief auto free mechanism status: FALSE - off, TRUE - on */

	cb_progress             cb_progress_caching;
	/**< \brief pointer to callback function called while informing about 
	 * caching progress */
	gpointer                cb_progress_caching_data;
	/**< \brief pointer to data passed to callback function called while
	 * informing about caching progress */
	gdouble                 cb_progress_caching_seed;
	/**< \brief how often progress callback should be called. 0.01 mean 
	 * that after each 1% of work callback shoul be called */

	cb_progress             cb_progress_word_list;
	/**< \brief pointer to callback function called while informing about
	 * words list searching progress */
	gpointer                cb_progress_word_list_data;
	/**< \brief pointer to data passed to callback function called while
	 * informing about words list searching progress */
	gdouble                 cb_progress_word_list_seed;
	/**< \brief how often progress callback should be called. 0.01 mean
	 * that after each 1% of work callback shoul be called */

	cb_progress             cb_progress_word_trans;
	/**< \brief pointer to callback function called while informing about
	 * word's translation searching progress */
	gpointer                cb_progress_word_trans_data;
	/**< \brief pointer to data passed to callback function called while
	 * informing about word's translation searching progress */
	gdouble                 cb_progress_word_trans_seed;
	/**< \brief how often progress callback should be called. 0.01 mean
	 * that after each 1% of work callback shoul be called */

	cb_word_list            cb_search_word_list;
	/**< \brief pointer to callback function called after words list is
	 * found */
	gpointer                cb_search_word_list_data;
	/**< \brief pointer to data passed to callback function called after
	 * words list is found */

	cb_word_translation     cb_search_word_trans;
	/**< \brief pointer to callback function called after word's translation
	 * is found */
	gpointer                cb_search_word_trans_data;
	/**< \brief pointer to data passed to callback function called after
	 * word's translation is found */
};
typedef struct _XDXFData        XDXFData;


/** \name Parsing Expat's callbacks */
/*@{*/

/** \brief Checking XML file is proper XDXF file - tag starts.
 *
 * See Expat documentation for more information about parser callbacks.
 */
static void     is_xdxf_file_start(void *data,
                                   const char *el,
                                   const char **attr);

/** \brief Checking XML file is proper XDXF file - tag ends.
 *
 * See Expat documentation for more information about parser callbacks.
 */
static void     is_xdxf_file_end(void *data,  const char *el);

/** \brief Searching for words list - tag start.
 *
 * See Expat documentation for more information about parser callbacks.
 */
static void     search_word_list_start(void *data,
                                       const char *el,
                                       const char **attr);
/** \brief Searching for words list - tag ends.
 *
 * See Expat documentation for more information about parser callbacks.
 */
static void     search_word_list_end(void *data, const char *el);

/** \brief Searching for words list - text node.
 *
 * See Expat documentation for more information about parser callbacks.
 */
static void     search_word_list_text(void *data, const XML_Char *txt, int len);

/** \brief Searching for word's translation - tag start.
 *
 * See Expat documentation for more information about parser callbacks.
 */
static void     search_word_trans_start(void *data,
                                        const char *el,
                                        const char **attr);

/** \brief Searching for word's translation - tag ends.
 *
 * See Expat documentation for more information about parser callbacks.
 */
static void     search_word_trans_end(void *data, const char *el);

/** \brief Searching for word's translation - text node.
 *
 * See Expat documentation for more information about parser callbacks.
 */
static void     search_word_trans_text(void *data,
                                       const XML_Char *txt,
                                       int len);
/*@}*/


/** \brief Return particular part of file. */
static gchar* read_file_part(FilePart* part, GnomeVFSHandle* file);

/** \brief Convert string to proper path name. */
static gchar*           string_to_path(gchar** string);

/** \brief Tells if file is in XDXF format (file should exist). */
static gboolean         is_xdxf_file(gchar* file);

/** \brief Get file's lenght. */
static guint64          get_file_size(GnomeVFSHandle* file);

/** \brief Return how many records (from cache file) are in the current buffer.
 */
static guint            get_max_length(gchar* a, guint length);

/** \brief Searching for word's translation in cache file. */
static gchar* word_translation_cache(XDXFData* data, gchar* word);

/** \brief Searching for word's translation in XDXF file. */
static gchar* word_translation_xdxf(XDXFData* data, gchar* word);

/** \brief Searching for words list in cache file. */
static void word_list_cache(XDXFData* data,
                            gchar* pattern,
                            GArray* result,
                            gpointer cb_data);

/** \brief Searching for words list in XDXF file. */
static void word_list_xdxf(XDXFData* data,
                           gchar* pattern,
                           GArray* result,
                           gpointer cb_data);

/** \name Module functions */
/*@{*/

/** \brief dict_eng_module_check() function implementation. */
gboolean        xdxf_engine_check(gchar* location);

/** \brief dict_eng_module_get_description() function implementation. */
gchar*          xdxf_engine_description();

/** \brief dict_eng_module_get_format() function implementation. */
gchar*          xdxf_engine_format();

/** \brief dict_eng_module_get_version() function implementation. */
gchar*          xdxf_engine_version();

/** \brief dict_eng_module_create() function implementation. */
Engine*         xdxf_engine_create(gchar* location, 
                              EngineOptimizationFlag flags,
                              cb_progress progress_handler,
                              gpointer progress_data,
                              gdouble seed);
/*@}*/


/** \name Particular dictionary function */
/*@{*/

/** \brief dict_eng_destroy() function implementation. */
void            xdxf_engine_close(Engine* engine);

/** \brief dict_eng_get_location() function implementation. */
gchar*          xdxf_engine_location(Engine* engine);

/** \brief dict_eng_optimize() function implementation. */
void            xdxf_engine_optimize(Engine* engine);

/** \brief dict_eng_is_optimized() function implementation. */
gboolean        xdxf_engine_is_optimized(Engine* engine);

/** \brief dict_eng_set_auto_free() function implementation. */
void            xdxf_engine_set_auto_free(Engine* engine, gboolean state);

/** \brief dict_eng_set_callback() function implementation. */
gpointer        xdxf_engine_set_callbacks(Engine* engine,
                                     gchar* event,
                                     gpointer c_handler,
                                     gpointer user_data);

/** \brief dict_eng_set_progress_seed() function implementation. */
void            xdxf_engine_set_progress_seed(Engine* engine,
                                         gchar* signal,
                                         gdouble seed);

/** \brief dict_eng_search_word_list() function implementation. */
void            xdxf_engine_search_word_list(Engine* engine,
                                             gchar* pattern,
                                             gpointer data);

/** \brief dict_eng_search_word_translation() function implementation. */
void            xdxf_engine_search_word_translation(Engine* engine,
                                                    gchar* word,
                                                    gpointer data);

/*** \brief dict_eng_search_word_translation_extended() function implementation.
 */
/* this function was removed from engine API */
/*void            xdxf_engine_search_word_translation_extended(Engine* engine,
                                                        gchar* word);
*/

/** \brief dict_eng_get_last_status() function implementation. */
EngineStatus    xdxf_engine_error(Engine* engine);

/** \brief dict_eng_status_message() function implementation. */
gchar*          xdxf_engine_error_message(EngineStatus error);

/** \brief dict_eng_add_word() function implementation. */
gboolean        xdxf_engine_add_word(Engine* engine,
                                     gchar*  word,
                                     gchar*  translation);

/** \brief dict_eng_remove_word() function implementation. */
gboolean        xdxf_engine_remove_word(Engine* engine, gchar*  word);
/*@}*/

/** \brief implementation of engine_global_functions(void) function. */
EngineModule    engine_global_functions();

#ifdef __cplusplus
}
#endif
#endif

/*@}*/
