/******************************************************************************
This file is part of mdictionary.

mdictionary is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

mdictionary is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License 
along with mdictionary; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Copyright 2006 ComArch S.A.
******************************************************************************/

/** \addtogroup StarDictEngine
 */
/*@{*/
/** \file engine_stardict.c
 * \brief Code of StarDict-based dictionary engine.
 */

#include <engine_stardict.h>
//------------------------------------------------------------------------------

#define eg_debug g_debug
#define eg_warning g_warning


/** \brief Read and return part of compressed file (dictzip)
 *
 * @param part <b>FilePart*</b> defines file part which we want to read
 * @param data <b>SDData*</b> structure from which we get filename of file to 
 * read (data->dic_file_name)
 * @return <b>gchar*</b> buffer with content of file part, or NULL if there 
 * were some problems
 */
static gchar* sd_read_file_part_dz(FilePart* part, gchar* file) 
{
	eg_debug("-> %s() called.\n",__FUNCTION__);
	timer(TIMER_START,(gchar*)__FUNCTION__);
	gchar* result = NULL;
	
	// read into the buffer 'result'
	dictData* header = NULL;
	header = dict_data_open(file , 0 );
	result = dict_data_read_(header,part->offset,part->length,NULL,NULL);
	dict_data_close(header);

	timer(TIMER_STOP,(gchar*)__FUNCTION__);
	eg_debug("<- %s() returned string=\n%s.\n",__FUNCTION__,result);
	return result;
}
//------------------------------------------------------------------------------
/** \brief Read and return part of uncompressed file.
 *
 * @param part <b>FilePart*</b> defines file part which we want to read
 * @param data <b>SDData*</b> structure from which we get filename of file to 
 * read (data->dic_file_name)
 * @return <b>gchar*</b> buffer with content of file part, or NULL if there 
 * were some problems
 */
static gchar* sd_read_file_part(FilePart* part, gchar* file)
{
	eg_debug("-> %s() called.\n",__FUNCTION__);
	timer(TIMER_START,(gchar*)__FUNCTION__);
	guint length = part->length;

	// open file
	gint fd = open(file, O_RDONLY);
	if(fd == -1)
	{
		eg_debug("---> Error: could not open *.dict file!\n");
		timer(TIMER_STOP,(gchar*)__FUNCTION__);
		eg_debug("<- %s() finished with error.\n",
			__FUNCTION__);
		return NULL;
	}

	// read from file to buffer
	gchar* result = (gchar*)g_try_malloc0(length+1);
	gint readed = (gint)read(fd, result, length);
	if(readed != part->length)
	{
		eg_debug("---> Error: could not read *.dict file!\n");
		g_free(result);
		result = NULL;
	}
	result[length] = '\0';

	timer(TIMER_STOP,(gchar*)__FUNCTION__);
	eg_debug("<- %s() returned string=\n%s.\n",__FUNCTION__,result);
	return result;
}
//------------------------------------------------------------------------------
/** \brief Search offset and length of article in dictionary for word.
 *
 * @param data <b>SDData*</b> where to search for descriptor of article
 * @param word <b>gchar*</b> for what word we are looking format
 * @return <b>FilePart*</b> describing part of file containing article about 
 * word or NULL if there were some problems
 */
FilePart* sd_find_file_part(SDData* data, gchar* word)
{
	eg_debug("-> %s() called.\n",__FUNCTION__);
	timer(TIMER_START,(gchar*)__FUNCTION__);

	FilePart* result = NULL;
	data->sd_seek_idx(data->idx_file, 0, _FILES_WRAPPER_BEG);
	gchar buffer[WORD_LIST_BUFFER_LENGTH + 1];
	gint readed =0;
	gint offset = 0;
	gchar* tmp = NULL;
	guint len = 0;
	guint pl = strlen(word);
	gboolean further = FALSE;
	do
	{
		readed =  data->sd_read_idx(data->idx_file,
					buffer + offset,
					WORD_LIST_BUFFER_LENGTH - offset
					);
		if(readed < 0)
		{
			eg_debug("---> Error while reading file for searching"
				" matched FilePart in dictionary to word: %s",
				word
				);
			break;
		}
		else
		{
			further = (readed == (WORD_LIST_BUFFER_LENGTH-offset));
		}
		tmp = buffer;

		guint i = 0;
		gint min = _MIN(readed, WORD_LIST_BUFFER_LENGTH-270);
		while(i<min)
		{
			// check next words
			if(g_ascii_strncasecmp(tmp,word,pl) == 0 &&
			   strlen(tmp) == pl)
			{
				// we have found word in dictionary
				guint val = 0;
				result = 
				(FilePart*)g_try_malloc0(sizeof(FilePart));
				len = strlen(tmp)+1;
				tmp += len;
				memcpy(&val,tmp,sizeof(guint));
				result->offset = ntohl(val);
				tmp = tmp + sizeof(guint);
				memcpy(&val,tmp,sizeof(guint));
				result->length = ntohl(val);
				further = FALSE;
				break;
			}
			// calculate length of word
			len = strlen(tmp)+1;
			// jump in buffer over this word, and ...
			tmp += len;
			// ...skip offset and length of translation
			tmp += 2 * sizeof(guint);
			// update i value
			i = i + len + 2 * sizeof(guint);
		}
		if(!further) break;
		offset = WORD_LIST_BUFFER_LENGTH - i;
		g_memmove(buffer,buffer+WORD_LIST_BUFFER_LENGTH-offset, offset);
	}
	while(further);

	timer(TIMER_STOP,(gchar*)__FUNCTION__);
	if(result)
	{
		eg_debug("<- %s() return: OFFSET=%d LENGTH=%d.\n",
			__FUNCTION__,
			result->offset,
			result->length
			);
	}
	else
	{
		eg_debug("<- %s() didn't find proper FilePart!\n",__FUNCTION__);
	}	
	return result;
}
//------------------------------------------------------------------------------
/** \brief Close and clean unused dictionary.
 *
 * This function is called whenever you call macro dict_eng_destroy(). Every 
 * engine could have some memory allocated for its, so it is obligatory to call 
 * this function every time when You do not need engine no more.
 *
 * @param engine <b>Engine*</b> to close and clean
 */
void sd_engine_close(Engine* engine)
{
	eg_debug("StarDict/%s->%s() called.\n-->PARAM: engine adress=%p\n",
	     __FILE__,
	     __FUNCTION__,
	     engine
	    );

	g_assert(engine != NULL);
	SDData* data = (SDData*)(engine->engine_data);

	data->sd_close_idx(data->idx_file);
	g_free(data->idx_file);
	g_free(data->dict_path);
	g_free(data->idx_file_name);
	g_free(data->ifo_file_name);
	g_free(data->dic_file_name);
	g_free(data->icon);
	g_free(data->lang_from);
	g_free(data->lang_to);
	g_free(data->title);
	g_free(data);
	g_free(engine); 
	eg_debug("StarDict/%s->%s() engine at adress=%p is deleted.\n",
	     __FILE__,
	     __FUNCTION__,
	     engine
	    );
	engine = NULL;
}
//------------------------------------------------------------------------------
/** \brief Create engine to dictionary in given directory.
 *
 * Construct and initialize new engine from concrete location.
 *
 * @param location <b>gchar*</b> from where we are trying to load StarDict 
 * dictionary
 * @param auto_cache <b>gboolean</b> flag defining if we want automaticlly turn
 * on the optimization of newly created engine. In StarDict dictionaries this 
 * flag is unused.
 * @param progress_handler <b>gpointer</b> is a function which will be called 
 * from time to time
 * while optimization proccess is running. As far as there is no optimization
 * for StarDict dictionaries, this flag is unused either.
 * @param progress_data <b>gpointer</b> given as a parameter to progress_handler
 * @param seed <b>gdouble</b> tells how often to call progress_handler
 */
Engine* sd_engine_create(gchar* location,
        		EngineOptimizationFlag auto_cache,
        		cb_progress progress_handler,
        		gpointer progress_data,
        		gdouble seed
			)
{
	eg_debug("StarDict/%s->%s() called.\n"
		"-->PARAM:location=\'%s\'\n"
		"-->PARAM:auto_cache=%d\n",
		__FILE__,
		__FUNCTION__,
		location,
		(guint)auto_cache
               );
	g_assert(location != NULL);

	sd_timer(TIMER_START,(gchar*)__FUNCTION__); 

	gchar* tmp = g_strdup(location);
	string_to_path(&tmp);

	Engine* result = (Engine*)g_try_malloc0(sizeof(Engine));
	result->engine_location = sd_engine_location;
	result->engine_is_optimized = sd_engine_is_optimized;
	result->engine_optimize = sd_engine_optimize;
	result->engine_search_word_list = sd_engine_search_word_list;
	result->engine_search_word_translation = 
		sd_engine_search_word_translation;       
	result->engine_close = sd_engine_close;
	result->engine_status = sd_engine_status;
	result->engine_status_message = sd_engine_status_message;
	result->engine_set_callback = sd_engine_set_callback;
	result->engine_set_progress_seed = sd_engine_set_progress_seed;
	result->engine_set_auto_free = sd_engine_set_auto_free;
	result->engine_get_lang_to = sd_engine_get_lang_to;
	result->engine_get_lang_from = sd_engine_get_lang_from;
	result->engine_get_title = sd_engine_get_title;
	result->engine_get_icon_path = sd_engine_get_icon_path;
	result->engine_add_word = sd_engine_add_word;
	result->engine_remove_word = sd_engine_remove_word;

 	SDData* data = (SDData*)g_try_malloc0(sizeof(SDData));
 	result->engine_data = (gpointer)data;
	data->dict_path = g_strdup(tmp);
	data->cb_progress_caching = progress_handler;
	data->cb_progress_caching_data = progress_data; 
	data->cb_progress_caching_seed = seed; 
	data->cb_progress_word_list = NULL;
	data->cb_progress_word_list_data = NULL;
	data->cb_progress_word_list_seed = 0.01;
	data->cb_progress_word_trans = NULL;
	data->cb_progress_word_trans_data = NULL;
	data->cb_progress_word_trans_seed = 0.01;
	data->cb_search_word_list = NULL;
	data->cb_search_word_list_data = NULL;
	data->cb_search_word_trans = NULL;
	data->cb_search_word_trans_data = NULL;
	data->auto_free = FALSE;

	if(!sd_read_files_names(data))
	{
		eg_warning("Error while loading dictionaries filenames!\n");	
		result->engine_close(result);
		result = NULL;
		sd_timer(TIMER_STOP,(gchar*)__FUNCTION__);
		return result;
	}
	if(!sd_parse_ifo_file(data))
	{
		eg_warning("Error while reading *.ifo file!\n");	
		result->engine_close(result);
		result = NULL;
		sd_timer(TIMER_STOP,(gchar*)__FUNCTION__);
		return result;
	}
	// there are no information about this parameter of dictionaries in
	// StarDict format
	data->lang_from = NULL;
	data->lang_to = NULL;
	// StarDict dictionaries do NOT support different icons, so set global
	data->icon = g_strdup(ICON_PATH);
	
	sd_timer(TIMER_STOP,(gchar*)__FUNCTION__);
	eg_debug("StarDict/%s->%s() returned Engine at adress=%p\n",
		__FILE__,
		__FUNCTION__,
		result
	       );
	g_free(tmp); tmp = NULL;
	return result;
}
//------------------------------------------------------------------------------
/** \brief Search for word list in dictionary.
 *
 * @param engine <b>Engine*</b> to search in
 * @param pattern <b>gchar*</b> to search for
 */
void sd_engine_search_word_list(Engine* engine,
                                gchar* pattern,
                                gpointer cb_data)
{
	eg_debug("-> %s() called. Searching words list\n"
		"--->PARAM:engine at adress=%p\n"
		"--->PARAM:pattern=\"%s\"\n",
		__FUNCTION__,
		engine,
		pattern
		);
	g_assert(engine != NULL);
	g_assert(pattern != NULL);
	pattern = g_utf8_casefold(pattern, -1);
	sd_timer(TIMER_START,(gchar*)__FUNCTION__);
	SDData* data = (SDData*)(engine->engine_data);
	if(data->cb_search_word_list == NULL) {
		eg_warning("---> %s() callback for Word List not set. "
			"Searching aborted.\n",
			__FUNCTION__
			);
		sd_timer(TIMER_STOP,(gchar*)__FUNCTION__);
		return;
	};

	data->sd_seek_idx(data->idx_file, 0, _FILES_WRAPPER_BEG);
	gchar buffer[WORD_LIST_BUFFER_LENGTH + 1];
	gint readed =0;
	gint offset = 0;
	gchar* tmp = NULL;
	gchar* tmp1 = NULL;
	gchar* tmp2 = NULL;
	guint len = 0;
// 	guint pl = strlen(pattern);
 	GPatternSpec* regex;
	
	tmp = g_utf8_casefold (pattern, -1);
 	regex = g_pattern_spec_new (tmp);
	g_free (tmp);
	
	gboolean further = FALSE;
	guint count = 0;
	GArray* result = g_array_new(TRUE, TRUE, sizeof(gchar*));
	do
	{
		readed =  data->sd_read_idx(data->idx_file,
					buffer + offset,
					WORD_LIST_BUFFER_LENGTH - offset
					);
		if(readed < 0)
		{
			eg_debug("---> Error while reading file for searching"
				" matched words in dictionary to pattern: %s",
				pattern
				);
			break;
		}
		else
		{
			further = (readed == (WORD_LIST_BUFFER_LENGTH-offset));
		}
		tmp = buffer;
		
		guint i = 0;
		gint min = _MIN(readed, WORD_LIST_BUFFER_LENGTH-270);
		while(i<min)
		{
			tmp1 = g_strconcat (tmp, " ", NULL);
			tmp2 = g_utf8_casefold(tmp1, -1);
			g_free (tmp1);
			tmp1 = g_utf8_casefold(tmp, -1);
			
			// check next words
			if ((g_pattern_match_string (regex, tmp1) == TRUE) ||
				(g_pattern_match_string (regex, tmp2) == TRUE))
			{
				gchar* new = g_strdup(tmp);
				g_array_append_val(result,new);
				++count;
				eg_debug("New word %d. \'%s\' matched to "
					"pattern \'%s\'\n",
					count,
					new,
					pattern
					);
			}
			
			g_free (tmp1);
			g_free (tmp2);
			// calculate length of word
			len = strlen(tmp)+1;
			// jump in buffer over this word, and ...
			tmp += len;
			// ...skip offset and length of translation
			tmp += 2 * sizeof(guint);
			// update i value
			i = i + len + 2 * sizeof(guint);
		}
		
		offset = WORD_LIST_BUFFER_LENGTH - i;
		g_memmove(buffer,buffer+WORD_LIST_BUFFER_LENGTH-offset, offset);
	}
	while(further);

	sd_timer(TIMER_STOP,(gchar*)__FUNCTION__);
        sd_timer(TIMER_START,"callback for returning words LIST START");
	if (NULL == cb_data )
	{
		cb_data = data->cb_search_word_list_data;
	}
        data->cb_search_word_list(result,
				pattern,
				cb_data,
				ENGINE_NO_ERROR
				);
        sd_timer(TIMER_STOP,"callback for returning word LIST END");
	guint i = 0;
	for (i=0; i<result->len; i++)
	{
		g_free(g_array_index(result, gchar*, i));
	}
	g_array_free(result, TRUE);
	g_pattern_spec_free (regex);
}
//------------------------------------------------------------------------------
/** \brief Search translation of word in dictionary.
 *
 * @param engine <b>Engine*</b> to search in
 * @param word <b>gchar*</b> to search for
 */
void sd_engine_search_word_translation(Engine* engine,
                                       gchar* word,
                                       gpointer cb_data)
{
	eg_debug("-> %s() called.\n"
		"-->PARAM:engine at adress=%p\n"
		"-->PARAM:word=\'%s\'\n",
		__FUNCTION__,
		engine,
		word);
	g_assert(engine != NULL);
	g_assert(word != NULL);
	// start sd_timer for this function
	sd_timer(TIMER_START, (gchar*)__FUNCTION__);
	SDData* data = (SDData*)(engine->engine_data);
	gchar* trans0 = NULL;

	if (NULL == cb_data )
	{
		cb_data = data->cb_search_word_list_data;
	}

	FilePart* part = sd_find_file_part(data, word);
	if(!part)
	{
		sd_timer(TIMER_STOP,(gchar*)__FUNCTION__);
		eg_debug("<- %s did not found any article for word: %s",
			__FUNCTION__,
			word
			);

		data->cb_search_word_trans(NULL,
					word,
					cb_data,
					ENGINE_NO_ERROR
					);
		return;
	}

	trans0 = data->sd_read_dic_part(part,data->dic_file_name);
	
	if(!trans0)
	{
		sd_timer(TIMER_STOP,(gchar*)__FUNCTION__);
		eg_debug("<- %s could not read from *.dict[.dz] file: %s",
			__FUNCTION__,
			data->dic_file_name
			);

		data->cb_search_word_trans(NULL,
					word,
					cb_data,
					ENGINE_NO_ERROR
					);
		return;
	}

	// "parse" returned part of file with article about word
	gchar* trans = sd_parse_stardict_article(trans0,
						data->types,
						part->length
						);
	g_free(trans0); trans0 = NULL;

	// check if returned article has apprioprate format (only text)
	if(!trans)
	{
		sd_timer(TIMER_STOP,(gchar*)__FUNCTION__);
		eg_debug("<- %s could not parse stardict article!",
			__FUNCTION__
			);

		data->cb_search_word_trans(NULL,
					word,
					cb_data,
					ENGINE_NO_ERROR
					);
		return;
	}

	sd_timer(TIMER_STOP,(gchar*)__FUNCTION__);
	sd_timer(TIMER_START,"callback for returning word's translation START");
	// calling callback for word translation

	data->cb_search_word_trans(trans,
				word,
				cb_data,
				ENGINE_NO_ERROR
				);
	
	
	sd_timer(TIMER_STOP,"callback for returning word's translation END");

	/*if(data->auto_free) {
		eg_debug("---> %s() deleting all dynamic data because "
			"AUTO_FREE=TRUE\n",
			__FUNCTION__
		);
		g_free(trans);
	}*/
	g_free(trans);
	trans = NULL;
}
//------------------------------------------------------------------------------
/** \brief Print only beggining end end of given buffer
 *
 * @param buffer to present
 */
static void sd_print_buffer_partial(gchar* buffer)
{
#ifndef NOLOGS
#define PRINT_LEN 100
#define EDGE "\n-------------------------------------------------------------\n"
	gchar tmp = 0;
	guint len = strlen(buffer);
	if(!buffer)
	{
		eg_debug("Buffer is empty!\n");
	}
	else if(PRINT_LEN >= len)
	{
		eg_debug("Buffer(%p):whole=%s%s%s", buffer, EDGE, buffer, EDGE);
	}
	else
	{
		// printf first 100 bytes
		tmp = buffer[PRINT_LEN+1];
		buffer[PRINT_LEN+1] = '\0';
		eg_debug("Buffer(%p): first %db=%s%s (....)\n\n",
			buffer,
			PRINT_LEN,
			EDGE,
			buffer
			);
		buffer[PRINT_LEN+1] = tmp; 

		//print last 100 bytes
		eg_debug("Buffer(%p): last %db=\n(....)%s%s",
			buffer,
			PRINT_LEN,
			buffer+len-PRINT_LEN,
			EDGE
			);
	}
#endif
}

/** \brief Check and parse pure stardict data containing whole article.
 *
 * @param buf <b>gchar*</b> buffer with pure data get from stardict 
 * *.dict[.gz] file
 * @param type <b>gchar*</b> sametypesequence defined in *.ifo filed - 
 * defines what kind of data will be in *.dict[.dz] or NULL if *.ifo did not 
 * define this
 * @param length <b>guint</b> length of buffer buf
 * @return <b>gchar*</b> new allocated buffer with proper translation or NULL 
 * if there were no good data for engine (e.g. if engine do not handle PNG, WAV 
 * file etc.)
 */
gchar* sd_parse_stardict_article(gchar* buf, gchar* type, guint length)
{
	eg_debug("-> %s()\n--->PARAM:type=%p\n--->PARAM:length=%d",
		__FUNCTION__,
		type,
		length
		);
	sd_print_buffer_partial(buf);
	sd_timer(TIMER_START, (gchar*)__FUNCTION__);
	g_assert(buf != NULL);
	g_assert(length > 0);

	gchar* result = (gchar*)g_try_malloc0(sizeof(gchar));
	result[0] = '\0';
	gchar next_type = 0;
	gchar* ptr = buf;

	while(length > 0)
	{
		if(type == NULL)
		{
			memcpy(&next_type, ptr, sizeof(gchar));
			++ptr;
			--length;
		}
		else
		{
			next_type = type[0];
			++type;
		}
	
		// check if this is a text or binary data
		// all text data are signaled by low level letter
		if(g_ascii_islower(next_type))
		{
			// text, so add this to translation
			gchar* new_txt = sd_get_buffer_from_article(&ptr,
								&length
								);
			gchar* new_tra = g_strconcat(result,
						new_txt,
						"\n<br/>",
						NULL
						);
			g_free(result);
			result = new_tra;
			new_tra = NULL;
			g_free(new_txt);
			new_txt = NULL;
		}
		else
		{
			// binary, so skip it
			// read hom many bytes to skip
			guint len = 0;
			g_memmove(&len, ptr, sizeof(guint));
			//omit in buf and len - field containting size of data
			length -= sizeof(guint);
			ptr += sizeof(guint);
			// conver network byte oredered guint to host ordered
			len = ntohl(len);
			// skip uneeded bytes
			length -= len;
			ptr += len;
		}
	}

	sd_timer(TIMER_STOP, (gchar*)__FUNCTION__);
	eg_debug("<- %s() returned buffer at %p\n",__FUNCTION__,result);
	return result;
}
//------------------------------------------------------------------------------
/** \brief Get string from buffer.
 *
 * Function gets from buffer string of char NULL-terminated, but no more than
 * len. Additionaly it update start position of the buffer and lenght of buffer
 * left.
 *
 * @param buffer <b>gchar**</b> with part of article
 * @param len <b>guint</b> number of bytes which could be maximally readed
 * @return <b>gchar*</b> next string available in the buffer
 */
gchar* sd_get_buffer_from_article(gchar** buffer, guint* len)
{
	eg_debug("-> %s\n",__FUNCTION__);
	g_assert(len > 0);

	guint buf_len = 0;
	gchar* tmp = *buffer;
	while(tmp && *len > 0)
	{
		++buf_len;
		++tmp;
		--(*len);
	}

	gchar* result = (gchar*)g_try_malloc0(buf_len+1);
	g_memmove(result, *buffer, buf_len);
	result[buf_len] = '\0';

	*buffer = tmp;
	eg_debug("<- %s\n",__FUNCTION__);
	return result;
}

//------------------------------------------------------------------------------
//==============================================================================
//==============================================================================
//==============================================================================
//----------------------------------------------------------- FINISHED FUNCTIONS
//==============================================================================
/** \brief Trying to find file names of default StarDict dictionary files
 *
 * @param data <b>SDData*</b> this is a structure from which function gets
 * directory to search in (data->dict_path) for files. Also puts there founded
 * (if any) filenames (data->[idx|ifo|dic]_file_name)
 * @return <b>gboolean</b> telling us are there all of obligatory StarDict files
 * in directory: data->dict_path
 */
gboolean sd_read_files_names(SDData* data) {
	eg_debug("-> %s()\n",__FUNCTION__);
	GError *dir_err = NULL;
	gchar* tmp = data->dict_path;
	GDir* dir = g_dir_open (tmp, 0, &dir_err);
	if(!dir) 
	{
		eg_debug("---> Could not open a dir:\n%s\n",data->dict_path);
		return FALSE;
	}
	gboolean ifo = FALSE;
	gboolean idx = FALSE;
	gboolean dic = FALSE;
	const gchar* fn = g_dir_read_name(dir);

	while((fn != NULL) && !(ifo && idx && dic)) 
	{
		if(g_str_has_suffix(fn, ".ifo")) 
		{
			data->ifo_file_name = g_strconcat(tmp,"/",fn,NULL);
			ifo = TRUE;
		};
		if(g_str_has_suffix(fn, ".idx")) 
		{
			data->idx_file_name = g_strconcat(tmp,"/",fn,NULL);
			data->idx_compressed = FALSE;
			data->sd_open_idx = sd_open;
			data->sd_read_idx = sd_read;
			data->sd_seek_idx = sd_seek;
			data->sd_close_idx = sd_close;
			idx = TRUE;
		};
		if(g_str_has_suffix(fn, ".idx.gz")) 
		{
			data->idx_file_name = g_strconcat(tmp,"/",fn,NULL);
			data->idx_compressed = TRUE;
			data->sd_open_idx = sd_open_z;
			data->sd_read_idx = sd_read_z;
			data->sd_seek_idx = sd_seek_z;
			data->sd_close_idx = sd_close_z;
			idx = TRUE;
		};
		if(g_str_has_suffix(fn, ".dict")) 
		{
			data->dic_file_name = g_strconcat(tmp,"/",fn,NULL);
			data->dic_compressed = FALSE;
			data->sd_read_dic_part = sd_read_file_part;
			dic = TRUE;
		};
		if(g_str_has_suffix(fn, ".dict.dz")) 
		{
			data->dic_file_name = g_strconcat(tmp,"/",fn,NULL);
			data->dic_compressed = TRUE;
			data->sd_read_dic_part = sd_read_file_part_dz;
			dic = TRUE;
		};
		fn = g_dir_read_name(dir);
	}
	g_dir_close(dir);

	if(ifo && idx && dic)
	{
		eg_debug("---> Dictionary files :\nifo=%s\nidx=%s\ndict=%s\n",
			data->idx_file_name,
			data->idx_file_name,
			data->dic_file_name
			);
		data->idx_file = data->sd_open_idx(data->idx_file_name);
		if(data->idx_file == NULL)
		{
				g_free(data->idx_file_name);
				data->idx_file_name = NULL;
				g_free(data->ifo_file_name);
				data->ifo_file_name = NULL;
				g_free(data->dic_file_name);
				data->dic_file_name = NULL;
				eg_debug("---> Could no open *.idx file!");
				return FALSE;
		}
		eg_debug("<- %s()\n",__FUNCTION__);
		return TRUE;
	}
	else 
	{
		if(idx) 
		{
			g_free(data->idx_file_name);
			data->idx_file_name = NULL;
		}
		if(ifo) 
		{
			g_free(data->ifo_file_name);
			data->ifo_file_name = NULL;
		}
		if(dic) 
		{
			g_free(data->dic_file_name);
			data->dic_file_name = NULL;
		}
	}
	eg_debug("<- %s()\n",__FUNCTION__);
	return FALSE;
}
// -----------------------------------------------------------------------------
/** \brief Check if location is a proper location of StarDict dictionary.
 *
 * @param location <b>gchar*</b> location to check
 * @return <b>gboolean</b> telling if this location is location of some 
 * StarDict dictionary
 */
gboolean sd_engine_check(gchar* location) 
{
	eg_debug("-> %s()\n--->PARAM:location=%s\n",
		__FUNCTION__,
		location
		);
	g_assert(location);

	gboolean result = TRUE; 
	gchar* filepath = g_strdup(location);
	string_to_path(&filepath);
	if (filepath == NULL)
	{
		result = FALSE;
		eg_debug("---> %s is not a proper path!\n",location);
	}
	else
	{
		SDData* tmp = g_try_malloc0(sizeof(SDData));
		tmp->dict_path = filepath;

		result = sd_read_files_names(tmp);

		g_free(tmp->idx_file_name);
		g_free(tmp->ifo_file_name);
		g_free(tmp->dic_file_name);
		g_free(tmp); tmp = NULL;
	};
	g_free(filepath);

	eg_debug("<- %s() returned bool statement=%s.\n",
		__FUNCTION__,
		PRINT_STATE(result)
		);
	return result;
}
//------------------------------------------------------------------------------
/** \brief Simple measuring of time needed for functions to work.
 *
 * @param start <b>gboolean</b> if we pass flag TIMER_START timer will start 
 * to work and if we pass TIMER_STOP timer will stop working and print short 
 * summary of working time
 * @return <b>gdouble</b> time in seconds for which function has been working
 * or (if we pass TIMER_START as a flag) 0.0.
 */
static double timer(gboolean start, gchar* message)
{
	static GArray* stack = NULL;
	static gboolean first_run = TRUE;
	static struct timeval actual_time;
	static struct timeval last_time;
	static struct timeval result;
	static double seconds = 0.0;
	if(first_run)
	{
		first_run = FALSE;
		stack = g_array_new(TRUE, TRUE, sizeof(struct timeval));
	};

	if (start)
	{
		eg_debug("TIMER:function %s()'s timer has started.\n",message);
		g_array_prepend_val(stack, actual_time);
		gettimeofday(&g_array_index(stack, struct timeval, 0),NULL);
		return -1.0;
	}
	// we just want to end some sd_timer - print some information about 
	// working time;
	else {
		gettimeofday(&actual_time,NULL);
		last_time = g_array_index(stack, struct timeval, 0);
		g_array_remove_index(stack, 0);

		if (actual_time.tv_usec < last_time.tv_usec)
		{
			int nsec = (last_time.tv_usec - actual_time.tv_usec) / 
					(1000000 + 1);
			last_time.tv_usec -= 1000000 * nsec;
			last_time.tv_sec += nsec;
		}
		if (actual_time.tv_usec - last_time.tv_usec > 1000000)
		{
			int nsec = (last_time.tv_usec - actual_time.tv_usec) / 
					1000000;
			last_time.tv_usec += 1000000 * nsec;
			last_time.tv_sec -= nsec;
		}
		result.tv_sec = actual_time.tv_sec - last_time.tv_sec;
		result.tv_usec = actual_time.tv_usec - last_time.tv_usec;
		seconds = (((double)(result.tv_usec)) / 1e6) +
			((double)(result.tv_sec));

		eg_debug("TIMER:Function \'%s()\' was working for: %g "
			"[s] or %ld [us].\n",
			message,
			seconds,
			((long)(result.tv_sec*1e6)+(result.tv_usec))
			);
		// stack is empty so we delete everything
		if(stack->len == 0)   
		{
			g_array_free(stack, TRUE);
			first_run = TRUE;
		}
	}
	return seconds;
}
//------------------------------------------------------------------------------
/** \brief Cleans string and check if this is a proper directory path.
 *
 * @param string <b>gchar**</b> it is pointer to string which we want to check
 * if it is proper directory/file path
 * @return <b>gchar*</b> this is cleaned string (without filename at the end, 
 * spaces etc.) or NULL if passed string under pointer do not describe existing 
 * directory/file. On to this string points also passed pointer.
 *
 */
static gchar* string_to_path(gchar** string)
{
	eg_debug("-> %s()\n-->PARAM:string=%s\n",__FUNCTION__,string[0]);

	gchar* arg = string[0];
	gchar* new = NULL;
 	// cleaning from leading and trailing whitespaces
 	g_strstrip(arg); 
	// add current directory if this is not absolute directory
	if (!g_path_is_absolute(arg))
	{
		gchar* tmp = g_get_current_dir();
		new = g_strconcat(tmp,"/",arg,NULL);
		g_free(arg); arg = new; new = NULL;
	};
	// this is not a directory
	if (!g_file_test(arg, G_FILE_TEST_IS_DIR))
	{
		// if this is wrong filepath, string was wrong
		if (!g_file_test(arg, G_FILE_TEST_IS_REGULAR))
		{
			g_free(arg);
			new = NULL;
		}
		//if this is a file, remove filename
		else
		{
			new = g_path_get_dirname (arg);
			g_free(arg);
		}
	}
 	// this is a directory
	else
	{
		// remove suffix "/" if neded...     
		if (g_str_has_suffix(arg,"/") )
		{
			new = g_path_get_dirname (arg);
			g_free(arg);
		}
		else
		{
			new = arg;
		}
	};
	// now in new should be proper filepath, if not, string was wrong
	if (!g_file_test(new, G_FILE_TEST_IS_DIR))
	{
		// if that directory does not exist, passed string wasn't proper
		g_free(new);
		new = NULL;
	};
	// replace string under passed address
	string[0] = new;
	eg_debug("<- %s() returned string=%s\n",__FUNCTION__,string[0]);
	return new;
}
//------------------------------------------------------------------------------
/** \brief Get full filename (with path) to the icon file for this dictionary
 *
 * @param engine <b>Engine*</b> pointer to engine to which we want to get icon
 * @return <b>gchar*</b> string containing full filepath with filename to icon
 * file. If auto_free=FALSE of engine You should free this string if you do not
 * need it any more. If auto_free is set to TRUE you should remember to not
 * modyfing returned string!
 */
gchar* sd_engine_get_icon_path(Engine* engine)
{
	eg_debug("-> %s\n",__FUNCTION__);
	
	gchar* result;
	SDData* data = (SDData*)(engine->engine_data);
	if(data->auto_free)
	{
		result = data->icon;
	}
	else
	{
		result = g_strdup(data->icon);
	}

	eg_debug("<- %s return string = \"%s\"\n",__FUNCTION__,result);
	return result;
}
//------------------------------------------------------------------------------
/** \brief Function not supported.
 */
gboolean sd_engine_add_word(Engine* engine, gchar*  word, gchar*  translation)
{
	eg_debug("<->%s Operation not supported in this engine\n",__FUNCTION__);
	return FALSE;
}
//------------------------------------------------------------------------------
/** \brief Function not supported.
 */
gboolean sd_engine_remove_word(Engine* engine, gchar*  word)
{
	eg_debug("<->%s Operation not supported in this engine\n",__FUNCTION__);
	return FALSE;
} 
//------------------------------------------------------------------------------
/** \brief Get language FROM which this dictionary translates.
 *
 * @param engine <b>Engine*</b> to check
 */
gchar* sd_engine_get_lang_from(Engine* engine)
{
	eg_debug("-> %s\n",__FUNCTION__);

	gchar* result;
	SDData* data = (SDData*)(engine->engine_data);
	if(data->auto_free)
	{
		result = data->lang_from;
	}
	else
	{
		result = g_strdup(data->lang_from);
	}

	eg_debug("<- %s return string = \"%s\"\n",__FUNCTION__,result);
	return result;	
}
//------------------------------------------------------------------------------
/** \brief Get language TO which this dictionary translates.
 *
 * @param engine <b>Engine*</b> to check
 */
gchar* sd_engine_get_lang_to(Engine* engine)
{
	eg_debug("-> %s\n",__FUNCTION__);

	gchar* result;
	SDData* data = (SDData*)(engine->engine_data);
	if(data->auto_free)
	{
		result = data->lang_to;
	}
	else
	{
		result = g_strdup(data->lang_to);
	}

	eg_debug("<- %s return string = \"%s\"\n",__FUNCTION__,result);
	return result;	
}
//------------------------------------------------------------------------------
/** \brief Gets title of dictionary.
 *
 * @param engine <b>Engine*</b> which title we want to get
 */
gchar* sd_engine_get_title(Engine* engine)
{
	eg_debug("-> %s\n",__FUNCTION__);

	gchar* result;
	SDData* data = (SDData*)(engine->engine_data);
	if(data->auto_free)
	{
		result = data->title;
	}
	else
	{
		result = g_strdup(data->title);
	}

	eg_debug("<- %s return string = \"%s\"\n",__FUNCTION__,result);
	return result;	
}
//------------------------------------------------------------------------------
/** \brief Sets callbacks for this dictionary.
 *
 * @param engine <b>Engine*</b>which callback we want to set
 * @param signal <b>gchar*</b> define which callback we want to set
 * @param c_handler <b>gpointer</b> function which we want to handle signal 
 * callback
 * @param user_data <b>gpointer</b> which will be passed always as a argument 
 * to callback
 * \sa dict_eng_set_callback
 */
gpointer sd_engine_set_callback(Engine* engine,
			     gchar* signal,
			     gpointer c_handler,
			     gpointer user_data)
{
	eg_debug("-> %s().\n",__FUNCTION__);
	g_assert(engine != NULL);
	g_assert(signal != NULL);
	g_assert(c_handler != NULL);
	SDData* data = (SDData*)(engine->engine_data);
	if(g_ascii_strcasecmp(signal,ENGINE_PROGRESS_OPTIMIZING_SIGNAL) == 0)
	{
		gpointer result = data->cb_progress_caching;
		data->cb_progress_caching = c_handler;
		data->cb_progress_caching_data = user_data;
		eg_debug("---> %s() sets handler for signal \"%s\".\n",
			__FUNCTION__,
			signal
		       );
		eg_debug("---> %s() Function at adress =  %p.\n",
			__FUNCTION__,
			c_handler
		       );
		eg_debug("---> %s()     Data at adress =  %p.\n",
			__FUNCTION__,
			user_data
		       );
		eg_debug("<- %s().\n",__FUNCTION__);
		return result;
	}
	else if(g_ascii_strcasecmp(signal, ENGINE_WORD_LIST_SIGNAL) == 0)
	{
		gpointer result = data->cb_search_word_list;
		data->cb_search_word_list = c_handler;
		data->cb_search_word_list_data = user_data;
		eg_debug("---> %s() sets handler for signal \"%s\".\n",
			__FUNCTION__,
			signal
		       );
		eg_debug("---> %s() Function at adress =  %p.\n",
			__FUNCTION__,
			c_handler
		       );
		eg_debug("---> %s()     Data at adress =  %p.\n",
			__FUNCTION__,
			user_data
		       );
		eg_debug("<- %s().\n",__FUNCTION__);
		return result;
	}
	else if(g_ascii_strcasecmp(signal, ENGINE_WORD_TRANSLATION_SIGNAL) == 0)
	{
		gpointer result = data->cb_search_word_trans;
		data->cb_search_word_trans = c_handler;
		data->cb_search_word_trans_data = user_data;
		eg_debug("---> %s() sets handler for signal \"%s\".\n",
			__FUNCTION__,
			signal
			);
		eg_debug("---> %s() Function at adress =  %p.\n",
			__FUNCTION__,
			c_handler
			);
		eg_debug("---> %s()     Data at adress =  %p.\n",
			__FUNCTION__,
			user_data
			);
		eg_debug("<- %s().\n",__FUNCTION__);
		return result;   
	}
	else
	{
		eg_warning("---> %s() unsupported signal: %s.\n",
				__FUNCTION__,
				signal
				);
		eg_debug("<- %s().\n",__FUNCTION__);
		return NULL;
	}
}
//------------------------------------------------------------------------------
// for macro: dict_eng_set_progress_seed(engine, signal, val)
/** \brief Sets how often should be called concrete progress calback.
 *
 * @param engine <b>Engine*</b> for which we want set seed for progress proccess
 * @param signal <b>gchar*</b> for which we want to set seed
 * @param seed <b>gdouble</b> which we want to set
 */
void sd_engine_set_progress_seed(Engine* engine, gchar* signal, gdouble seed)
{
	eg_debug("-> %s().\n",__FUNCTION__);
	SDData* data = (SDData*)(engine->engine_data);
	if(g_ascii_strcasecmp(signal,ENGINE_PROGRESS_OPTIMIZING_SIGNAL) == 0)
	{
		data->cb_progress_caching_seed = seed;
		eg_debug("--->%s() sets new seed=%0.2f for for signal %s.\n",
			__FUNCTION__,
			seed,
			signal
			);
	}
	else
	{
		g_warning("--->%s() unsupported signal for progress: %s.\n",
			__FUNCTION__,
			signal
			);
	};
	eg_debug("<- %s().\n",__FUNCTION__);
}
//------------------------------------------------------------------------------
/** \brief Get the last status of engine.
 *
 * @param engine <b>Engine*</b> which status we want to get
 * @return <b>EngineStatus</b> status of the engine
 * \sa dict_eng_get_last_status
 */
EngineStatus sd_engine_status(Engine* engine) 
{
	eg_debug("-> %s()\n--->PARAM:engine at adress=%p\n",
		__FUNCTION__,
		engine
		);
	SDData* data = (SDData*)(engine->engine_data);
	eg_debug("<- %s() returned error code: %d\n",
		__FUNCTION__,
		(gint)(data->last_error)
		);
	return data->last_error;
}
//------------------------------------------------------------------------------
/** \brief Tells if engine is optimized or not - not supported in StarDict.
 */
gboolean sd_engine_is_optimized(Engine* engine) 
{
	eg_debug("-> %s() for engine at adress=%p\n",
		__FUNCTION__,
		engine
		);
	g_assert(engine != NULL);

	gboolean result = TRUE;

	eg_debug("<- %s() returned bool statement=%s.\n",
		__FUNCTION__,
		PRINT_STATE(result)
		);
	return result;
}
//------------------------------------------------------------------------------
/** \brief Optimize dictionary - not supported in StarDict dictionaries.
 */
void sd_engine_optimize(Engine* engine)
{
	eg_debug("-> %s() called for engine at adress=%p\n",
		__FUNCTION__,
		engine
		);
	eg_debug("---> Unsupported method for this engine\n");
	eg_debug("<- %s()\n",__FUNCTION__);
}
//------------------------------------------------------------------------------
/** \brief Get structure and access to all module functions.
 *
 * @return <b>EngineModule</b> structure containing all functions which are 
 * needed to handle new dictionary's format
 */
EngineModule engine_global_functions()
{
	eg_debug("-> %s() called.\n",__FUNCTION__);
	EngineModule result; 
	result.engine_check      = sd_engine_check;
	result.engine_description= sd_engine_description;
	result.engine_format     = sd_engine_format;
	result.engine_version    = sd_engine_version;
	result.engine_create     = sd_engine_create;
	eg_debug("<- %s() returned EngineModule at adress=%p.\n",
		__FUNCTION__,
		&result
		);
	return result;
}
//------------------------------------------------------------------------------
/** \brief Translate status code to meaningful, human-readable information.
 *
 * @param status <b>EngineStatus</b> to translates
 * @return <b>gchar*</b> message describing status (e.g. its reason etc.)
 * \sa dict_eng_status_message
 */
gchar* sd_engine_status_message(EngineStatus status) 
{
	eg_debug("<-> %s() called.\n",__FUNCTION__);
	switch (status)
	{
		case ENGINE_NO_ERROR:
			return "No error.";
		case ENGINE_WRONG_FILE:
			return "File, You are trying to use, is wrong type.";
		case ENGINE_COULDNT_READ:
			return "Could not read from file.";
		case ENGINE_NO_FILE:
			return "There is no such a file.";
		case ENGINE_OUT_OF_MEMORY:
			return "There were no enough memory for this action.";
		default:
			return "Wrong engine's status identifier!";
	}
}
//------------------------------------------------------------------------------
/** \brief Gets version of module.
 * @return <b>gchar*</b> version of this module.
 * \sa dict_eng_module_get_version
 */
gchar* sd_engine_version() 
{
	eg_debug("-> %s()\n",__FUNCTION__);
	gchar* result = g_strdup(DIC_ENG_VERSION);
	eg_debug("<- %s() return string=%s\n",
		__FUNCTION__,
		result
		);
	return result;
}
//------------------------------------------------------------------------------
/** \brief Get information about what kind of dictionaries this engine support.
 *
 * @return <b>gchar*</b> telling about supported dictionaries (e.g. XDXF, 
 * StarDict etc.)
 * \sa dict_eng_module_get_format
 */
gchar* sd_engine_format() 
{
	eg_debug("-> %s()\n",__FUNCTION__);
	gchar* result = g_strdup(DIC_ENG_FORMAT);
	eg_debug("<- %s() return string=%s\n",
		__FUNCTION__,
		result
		);
	return result;
}
//------------------------------------------------------------------------------
/** \brief Get short description of the module - dictionary engine.
 *
 * @return <b>gchar*</b> description of the module
 * \sa dict_eng_module_get_description
 */
gchar* sd_engine_description() 
{
	eg_debug("-> %s()\n",__FUNCTION__);
	gchar* result = g_strdup(DIC_ENG_DESCRIPTION);
	eg_debug("<- %s() return string=%s\n",
		__FUNCTION__,
		result
		);
	return result;
}
//------------------------------------------------------------------------------
/** \brief Get path to the dictionary - location of its files.
 *
 * @param engine <b>Engine*</b> which directory path we want to get
 * @return <b>gchar*</b> location of the dictionary
 * \sa dict_eng_get_location
 */
gchar* sd_engine_location(Engine* engine)
{
	eg_debug("->%s() called.\n-->PARAM: engine adress=%p\n",
		__FUNCTION__,
		engine
		);
	g_assert(engine != NULL);
	SDData* data = (SDData*)(engine->engine_data);
 
	gchar* result;
	if(data->auto_free)
	{
		result = data->dict_path;
	}
	else
	{
		result = g_strdup(data->dict_path);
	}

	eg_debug("<- %s() returned string=%s\n",
		__FUNCTION__,
		result
		);
	return result;
}
//------------------------------------------------------------------------------
/** \brief Set auto free status for engine.
 *
 * When auto flag is set to TRUE, user of engine do not have to worry about 
 * freeing any of buffer returned by the engine. Engine will clean up
 * everything. When auto free is set as FALSE, user has to remember that every-
 * thing returned from engine should be deallocated.
 *
 * @param engine <b>Engine*</b> which auto free flag we want to sets
 * @param state <b>gboolean</b> new value for auto free flag
 * \sa dict_eng_set_auto_free
 */
void sd_engine_set_auto_free(Engine* engine, gboolean state) 
{
	eg_debug("-> %s()\n"
		"--->PARAM:engine at adress=%p\n--->PARAM:state=%s\n",
		__FUNCTION__,
		engine,
		PRINT_STATE(state)
		);
	g_assert(engine != NULL);
	SDData* data = (SDData*)(engine->engine_data);
 
	data->auto_free = state;
	eg_debug("<- %s() Current auto_free is %s\n",
		__FUNCTION__,
		PRINT_STATE(data->auto_free)
	       );
}
//------------------------------------------------------------------------------
/** \brief Parse single record from *.ifo file from StarDict dictionary
 *
 * @param data <b>SDData*</b> structure which we will try to complete with
 * information from new record
 * @param key <b>gchar*</b> string representing key of record
 * @param value <b>gchar*</b> value of the key in record
 */
void sd_parse_record(SDData* data,gchar* key, gchar* value)
{
	if(g_ascii_strncasecmp(key,"version",100) == 0)
	{
	}
	else if(g_ascii_strncasecmp(key,"bookname",100) == 0)
	{
		data->title = g_strdup(value);
		eg_debug("%s set title to: %s\n",__FUNCTION__, data->title);
	}
	else if(g_ascii_strncasecmp(key,"wordcount",100) == 0)
	{
		data->word_count = (gint)g_ascii_strtoull( value, NULL, 10);
		eg_debug("%s set word count to: %d\n",
			__FUNCTION__,
			data->word_count
			);
	}
	else if(g_ascii_strncasecmp(key,"idxfilesize",100) == 0)
	{
		data->idx_file_length = (gint)g_ascii_strtoull(value, NULL, 10);
		eg_debug("%s set idx file length to: %d\n",
			__FUNCTION__,
			data->idx_file_length
			);
	}
	else if(g_ascii_strncasecmp(key,"sametypesequence",100) == 0)
	{
		data->types = g_strdup(value);
		eg_debug("%s set data types to: %s\n",__FUNCTION__,data->types);
	}
	else
	{
		eg_debug("%s trying to set usupported field\n",__FUNCTION__);
	}
	/* ----- as far unsupported field from *.ifo files
	else if(g_ascii_strncasecmp(key,"author",100))
	{
	}
	else if(g_ascii_strncasecmp(key,"email",100))
	{
	}
	else if(g_ascii_strncasecmp(key,"website",100))
	{
	}
	else if(g_ascii_strncasecmp(key,"description",100))
	{
	}
	else if(g_ascii_strncasecmp(key,"date",100))
	{
	} */
}
//------------------------------------------------------------------------------
/** \brief Try to read and parse whole *.ifo file from StarDict Dictionary
 *
 * @param data <b>SDData*</b> structure to which we try save information about
 * dictionary in data->dict_path path
 * @return <b>gboolean</b> if data->dict_path is good path and it contains all 
 * obligatory StarDict files, function return TRUE. Otherwise return FALSE.
 */
gboolean sd_parse_ifo_file(SDData* data)
{
	GnomeVFSHandle* fd = NULL;
	GnomeVFSResult  fr = 0;
	GnomeVFSFileSize n = 0;
	GnomeVFSFileSize k = 0;

	if(!gnome_vfs_initialized ())
	{
		gnome_vfs_init ();                       
	};
	// open the file
	fr = gnome_vfs_open (&fd, data->ifo_file_name, GNOME_VFS_OPEN_READ);
	if(fr != GNOME_VFS_OK)
	{
		eg_warning("Error while trying to open file:\n%s\n",
			data->ifo_file_name
			);
		return FALSE;
	}
	gnome_vfs_seek(fd, GNOME_VFS_SEEK_END, 0);
	fr = gnome_vfs_tell(fd, &n);
	gnome_vfs_seek(fd, GNOME_VFS_SEEK_START, 0);
	// read whole content
	gchar* tmp = g_try_malloc0(sizeof(gchar)*((gint)(n)));
	fr = gnome_vfs_read(fd, tmp, n, &k);
	if(fr != GNOME_VFS_OK) {
		eg_warning("Error while trying to read file:\n%s\n",
			data->ifo_file_name
			);
		gnome_vfs_close(fd);
		return FALSE;
	}
	gnome_vfs_close(fd);	
	// now we have in tmp whole content of file, we have to "parse it"
	gchar** table = g_strsplit(tmp, "\n", 100);
	gint i;
	for(i=0; table[i]!=NULL; ++i)
	{
		gchar* eq = g_strstr_len (table[i], 0xffff, "=");
		if(eq == NULL) continue;
		*eq = '\0';
		++eq;
		eg_debug("SD *.ifo record -> Key=%s => Value=%s\n",table[i],eq);
		sd_parse_record(data, table[i], eq);
	}

	g_free(tmp); tmp = NULL;
	g_strfreev(table); table = NULL;
	if(data->title)
	{
		return TRUE;
	}
	else 
	{
		return FALSE;
	}
}
//------------------------------------------------------------------------------
//-------------------------------------------------------------- STANDARD ------
//------------------------------------------------------------------------------
/** \brief Opening file not compressed
 * 
 * @param filename <b>gchar*</b> full filename (with path) to file to open
 * @return <b>gpointer</b> pointer to memory which is needed for other functions
 * from this wrapper. Under this pointer could be simple gint or more
 * complicated structure - depends on what kind of file we are trying to open.
 * If opening failed - function return NULL.
 */
static gpointer sd_open(gchar* filename)
{
	gint tmp = open(filename, O_RDONLY);
	if(tmp == -1)
	{
		eg_debug("---> Error while trying to open file %s\n",filename);
		return NULL;
	}
	gint* result = (int*)g_try_malloc0(sizeof(gint));
	*result = tmp;
	return (gpointer)result;
}
//------------------------------------------------------------------------------
/** \brief Reading from not compressed file
 * 
 * @param f <b>gpointer</b> pointer returned from sd_open()
 * @param buffer <b>gchar*</b> buffer where data should be put after reading
 * @param l <b>gint</b> maximum bytes to read. Effective number of bytes readed
 * could be different from this argument and is returned from function.
 * @return <b>gint</b> effective number of readed bytes
 */
static gint sd_read(gpointer f, gchar* buffer, gint l)
{
	gint tmp = *((gint*)(f));
	return (gint)read(tmp, buffer, l);
}
//------------------------------------------------------------------------------
/** \brief Seek throught not compressed files
 *
 * @param f <b>gpointer</b>  pointer returned from sd_open()
 * @param l <b>glong</b> offset which we want to seek
 * @param from <b>gchar</b> postion from which we want to seek. Possible values
 * are:\n _FILES_WRAPPER_BEG for the beggining of file \n
 * _FILES_WRAPPER_CUR for the current position \n
 * _FILES_WRAPPER_END for the end of file.
 * @return <b>glong</b> previous position in file if operation was successful,
 * or non positive value if there was an error.
 */
static glong sd_seek(gpointer f, glong l, gchar from)
{
	gint tmp = *((gint*)(f));
	switch(from)
	{
		case _FILES_WRAPPER_BEG:
			return (glong)lseek(tmp, l, SEEK_SET);
		case _FILES_WRAPPER_CUR:
			return (glong)lseek(tmp, l, SEEK_CUR);
		case _FILES_WRAPPER_END:
			return (glong)lseek(tmp, l, SEEK_END);
		default:
			eg_debug("---> Wrong relative position for sd_seek!\n");
			return -2;
	}
}
//------------------------------------------------------------------------------
/** \brief Close uncompressed file
 *
 * @param f <b>gpointer</b> pointer returned from sd_open() - file to close.
 */
static void sd_close(gpointer f)
{
	gint tmp = *((gint*)(f));
	close(tmp);
}
//------------------------------------------------------------------------------
//------------------------------------------------------- GZIPPED --------------
//------------------------------------------------------------------------------
/** \brief Opening file compressed with standard gzip
 * 
 * @param filename <b>gchar*</b> full filename (with path) to file to open
 * @return <b>gpointer</b> pointer to memory which is needed for other functions
 * from this wrapper. Under this pointer could be simple gint or more
 * complicated structure - depends on what kind of file we are trying to open.
 * If opening failed - function return NULL.
 */
static gpointer sd_open_z(gchar* filename)
{
	gzFile* out = (gzFile*)g_try_malloc0(sizeof(gzFile));
	*out = gzopen(filename, "rb9");
	if(out == NULL)
	{
		eg_debug("Error while opening compressed file with gzip!\n");
	}
	return (gpointer)out;
}
//------------------------------------------------------------------------------
/** \brief Reading from compressed file with gzip
 * 
 * @param f <b>gpointer</b> pointer retunred from sd_open()
 * @param buffer <b>gchar*</b> buffer where data should be put after reading
 * @param l <b>gint</b> maximum bytes to read. Effective number of bytes readed
 * could be different from this argument and is returned from function.
 * @return <b>gint</b> effective number of readed bytes
 */
static gint sd_read_z(gpointer f, gchar* buffer, gint l)
{
	gzFile tmp = *((gzFile*)f);
	return (gint)gzread(tmp, buffer, l);
}
//------------------------------------------------------------------------------
/** \brief Seek throught compressed files with gzip
 *
 * @param f <b>gpointer</b>  pointer returned from sd_open()
 * @param l <b>glong</b> offset which we want to seek
 * @param from <b>gchar</b> postion from which we want to seek. Possible values
 * are:\n _FILES_WRAPPER_BEG for the beggining of file \n
 * _FILES_WRAPPER_CUR for the current position \n
 * _FILES_WRAPPER_END for the end of file.
 * @return <b>glong</b> previous position in file if operation was successful,
 * or non positive value if there was an error.
 */
static glong sd_seek_z(gpointer f, glong l, gchar from)
{
	gzFile tmp = *((gzFile*)f);
	switch(from)
	{
		case _FILES_WRAPPER_BEG:
			return (glong)gzseek(tmp, l, SEEK_SET);
		case _FILES_WRAPPER_CUR:
			return (glong)gzseek(tmp, l, SEEK_CUR);
		default:
			eg_debug("---> Wrong position for sd_seek_z!\n");
			return -2;
	}
}
//------------------------------------------------------------------------------
/** \brief Close compressed file with gzip
 *
 * @param f <b>gpointer</b> pointer returned from sd_open() - file to close.
 */
static void sd_close_z(gpointer f)
{
	gzFile tmp = *((gzFile*)f);
	gzclose(tmp);
}
/*@}*/
