/*******************************************************************************
This file is part of mdictionary.

mdictionary is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

mdictionary is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License 
along with mdictionary; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Copyright 2006 ComArch S.A.
*******************************************************************************/
/** \defgroup StarDictEngine Dictionary Engine - StarDict format
 * \brief StarDict-based dictionary engine.
 *
 * This is library with dictionary engine supporting StarDict dictionaries. 
 */
/*@{*/
/** \file engine_stardict.h
 * \brief Header for XDXF-based dictionary engine.
 */

#ifndef _DICTIONARY_ENGINE_STARDICT
#define _DICTIONARY_ENGINE_STARDICT

#ifdef __cplusplus
        extern "C" {
#endif

#ifndef NOLOGS
	#define sd_timer timer
#else
	#define sd_timer(flag,name) while(FALSE)
#endif

//______________________________________________________________________________
// *****************************************************************************
//************************************************************* HEADERS SECTION:
//------------------------------------------------------------------------------
// headers with unix types/functions - only for timers and files operations
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <unistd.h>
//------------------------------------------------------------------------------
// header with GLIB definitions/functions/types
#include <glib.h>
//------------------------------------------------------------------------------
// header with gnome-vfs - recommended I/O API for maemo
#include <libgnomevfs/gnome-vfs.h>
//------------------------------------------------------------------------------
// header with engine API
#include <dictionary_engine.h>
//------------------------------------------------------------------------------
// header with standard gzip support library
#include <zlib.h>
//------------------------------------------------------------------------------
// header with dict zip support library
//#include "dictzip.h"
#include <data.h>
//#include "defs.h"
//------------------------------------------------------------------------------
// headero for ntohl conversion - network byte order to local host order
#include <netinet/in.h>
//------------------------------------------------------------------------------
// header to g_memmove only
#include <string.h>

//______________________________________________________________________________
// *****************************************************************************
//********************************************************* DEFINITIONS SECTION:
//------------------------------------------------------------------------------
// definitions for timer function - flag telling if we want to start or stop
// timing
#define TIMER_START     TRUE
#define TIMER_STOP      FALSE

#define WORD_LIST_BUFFER_LENGTH 16*1024

#define _MIN(a,b)	((a)<(b))?(a):(b)
//------------------------------------------------------------------------------
// definitions of version and format which engine handles
static const gchar* DIC_ENG_VERSION = "0.1"; 
static const gchar* DIC_ENG_FORMAT = "StarDict";
static const gchar* DIC_ENG_DESCRIPTION = "This module operates on StarDict dictionaries. Version 0.1.";

//------------------------------------------------------------------------------
// macro for "printing" gboolean statement - "TRUE" or "FALSE"
#define PRINT_STATE(state) ( (state) ? "TRUE" : "FALSE" )
//------------------------------------------------------------------------------

#define ICON_PATH "/usr/share/pixmaps/stardict_icon.png"

//______________________________________________________________________________
// *****************************************************************************
//****************************************** DATA STRUCTURE DEFINITIONS SECTION:
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
/** \brief Internal data structure for representing part of file.
 */
struct _FilePart {
        guint offset;
        guint length;
};
typedef struct _FilePart        FilePart;
//------------------------------------------------------------------------------
/** \brief Internal data structure of XDXF Engine.
 */
struct _SDData {
        GnomeVFSHandle*         dictionary;
        GnomeVFSHandle*         index;

        gchar*                  dict_path;
	gchar*			ifo_file_name;
	gchar*			idx_file_name;
	gboolean		idx_compressed;
	gchar*			dic_file_name;
	gboolean		dic_compressed;
	gchar*			lang_from;
	gchar*			lang_to;
	gchar*			title;
	gint			word_count;
	gint			idx_file_length;
	gchar*			types;
	gchar*			icon;

        EngineStatus            last_error;
        gboolean                auto_free;

	/* wrapper functions to access idx and dict files without checking
	 * it they are compressed or not - we decide about this while creating
	 * engine i function sd_engine_create()*/
	gpointer	(*sd_open_idx)(gchar* filename);
	//gpointer	(*sd_open_dic)(gchar* filename);
	gint		(*sd_read_idx)(gpointer f, gchar* buffer, gint l);
	//gint		(*sd_read_dic)(gpointer f, gchar* buffer, gint l);
	glong		(*sd_seek_idx)(gpointer f, glong l, gchar from);
	//glong		(*sd_seek_dic)(gpointer f, glong l, gchar from);
	void		(*sd_close_idx)(gpointer f);
	//void		(*sd_close_dic)(gpointer f);
	gchar*		(*sd_read_dic_part)(FilePart* part, gchar* file);
	gpointer	idx_file;

        cb_progress             cb_progress_caching;
        gpointer                cb_progress_caching_data;
        gdouble                 cb_progress_caching_seed;

        cb_progress             cb_progress_word_list;
        gpointer                cb_progress_word_list_data;
        gdouble                 cb_progress_word_list_seed;

        cb_progress             cb_progress_word_trans;
        gpointer                cb_progress_word_trans_data;
        gdouble                 cb_progress_word_trans_seed;

        cb_word_list            cb_search_word_list;
        gpointer                cb_search_word_list_data;

        cb_word_translation     cb_search_word_trans;
        gpointer                cb_search_word_trans_data;      
};
typedef struct _SDData        SDData;
//------------------------------------------------------------------------------


//______________________________________________________________________________
// *****************************************************************************
//************************************************ ADDITIONAL FUNCTIONS SECTION:
//------------------------------------------------------------------------------
// returning concrete part of file
//static gchar*           read_file_part(FilePart* part, GnomeVFSHandle* file);
//------------------------------------------------------------------------------
// convert string to proper path name (no filename, no "/" at the ned, file
// exist)
static gchar*           string_to_path(gchar** string);
//------------------------------------------------------------------------------
// fill out filenames of dictionary from directory in data->dict_path
gboolean 		sd_read_files_names(SDData* data);
//------------------------------------------------------------------------------
// fill out few information from IFO file
gboolean 		sd_parse_ifo_file(SDData* data);
//------------------------------------------------------------------------------
// parse concrete record from *.ifo files - used by sd_parse_ifo_file() function
void 			sd_parse_record(SDData* data,gchar* key, gchar* value);
//------------------------------------------------------------------------------
// start/stop timers -  return -1.0 if we start or seconds passed from start 
// if we want to stop timer
static double           timer(gboolean start, gchar* message);
//------------------------------------------------------------------------------
// find proper FilePart structure for concrete word in dictionary
FilePart* 		sd_find_file_part(SDData* data, gchar* word);
//------------------------------------------------------------------------------
// parse whole article to proper word translation
gchar*			sd_parse_stardict_article(gchar* buf,
						gchar* type,
						guint length
						);
//------------------------------------------------------------------------------
// find and return only one, next field from buffer and update length variable
gchar*			sd_get_buffer_from_article(gchar** buffer,
						guint* length
						);
//------------------------------------------------------------------------------
// return size of files
//static guint64          get_file_size(GnomeVFSHandle* file);
//------------------------------------------------------------------------------
// these are wrapper functions to acces different type of files: standard or
// compressed with gzip for *.idx or dictzip for *.dict files.

#define _FILES_WRAPPER_BEG 'b'
#define _FILES_WRAPPER_CUR 'c'
#define _FILES_WRAPPER_END 'e'

// standard
static gpointer		sd_open(gchar* filename);
static gint		sd_read(gpointer f, gchar* buffer, gint l);
static glong		sd_seek(gpointer f, glong l, gchar from);
static void		sd_close(gpointer f);
// gzipped
static gpointer		sd_open_z(gchar* filename);
static gint		sd_read_z(gpointer f, gchar* buffer, gint l);
static glong		sd_seek_z(gpointer f, glong l, gchar from);
static void		sd_close_z(gpointer f);
// files *.dict[.dz]
//------------------------------------------------------------------------------
// read concrete file part from dictionary - *.dict* file
static gchar*		sd_read_file_part_dz(FilePart* part, gchar* file);
static gchar*		sd_read_file_part(FilePart* part, gchar* file);
//------------------------------------------------------------------------------




//______________________________________________________________________________
// *****************************************************************************
//****************************************************** MAIN FUNCTIONS SECTION:
//------------------------------------------------------------------------------
gboolean        sd_engine_add_word(Engine* engine,
                                 gchar*  word,
                                 gchar*  translation);
//------------------------------------------------------------------------------
gboolean        sd_engine_remove_word(Engine* engine,
                                     gchar*  word);  
//------------------------------------------------------------------------------      
gchar*          sd_engine_get_lang_from(Engine* engine);
//------------------------------------------------------------------------------
gchar*          sd_engine_get_lang_to(Engine* engine);
//------------------------------------------------------------------------------
gchar*          sd_engine_get_title(Engine* engine);
//------------------------------------------------------------------------------
gchar*          sd_engine_get_icon_path(Engine* engine);
//------------------------------------------------------------------------------



//------------------------------------------------------------------------------
// implementation of dict_eng_module_check(module,location) function
gboolean        sd_engine_check(gchar* location);
//------------------------------------------------------------------------------
// implementation of dict_eng_module_get_description(module) function
gchar*          sd_engine_description();
//------------------------------------------------------------------------------
// implementation of dict_eng_module_get_format(module) function
gchar*          sd_engine_format();
//------------------------------------------------------------------------------
// implementation of dict_eng_module_get_version(module) function
gchar*          sd_engine_version();
//------------------------------------------------------------------------------
// implementation of dict_eng_module_create(module,location,flags) and
// dict_eng_module_create_ext(module,location,flags) functions
Engine*         sd_engine_create(gchar* location, 
                              EngineOptimizationFlag flags,
                              cb_progress progress_handler,
                              gpointer progress_data,
                              gdouble seed);
//------------------------------------------------------------------------------
// implementation of dict_eng_destroy(engine) function
void            sd_engine_close(Engine* engine);
//------------------------------------------------------------------------------
// implementation of dict_eng_get_location(engine) function
gchar*          sd_engine_location(Engine* engine);
//------------------------------------------------------------------------------
// implementation of dict_eng_optimize(engine) function
void            sd_engine_optimize(Engine* engine);
//------------------------------------------------------------------------------
// implementation of dict_eng_is_optimized( engine ) function
gboolean        sd_engine_is_optimized(Engine* engine);
//------------------------------------------------------------------------------
// implementation of dict_eng_set_auto_free(engine, state) function
void            sd_engine_set_auto_free(Engine* engine, gboolean state);
//------------------------------------------------------------------------------
// implementation of dict_eng_set_callback(engine,signal,c_handler,data) 
// function
gpointer        sd_engine_set_callback(Engine* engine,
                                     gchar* event,
                                     gpointer c_handler,
                                     gpointer user_data);
//------------------------------------------------------------------------------
// implementation of dict_eng_set_progress_seed(engine, signal, val) function
void            sd_engine_set_progress_seed(Engine* engine,
                                         gchar* signal,
                                         gdouble seed);
//------------------------------------------------------------------------------
// implementation ofdict_eng_search_word_list(engine,pattern) function
void            sd_engine_search_word_list(Engine* engine,
                                           gchar* pattern,
                                           gpointer cb_data);
//------------------------------------------------------------------------------
// implementation of dict_eng_search_word_translation(engine,word) function
void            sd_engine_search_word_translation(Engine* engine,
                                                  gchar* word,
                                                  gpointer cb_data);
//------------------------------------------------------------------------------
// implementation of dict_eng_get_last_state(engine) function
EngineStatus    sd_engine_status(Engine* engine);
//------------------------------------------------------------------------------
// implementation of dict_eng_state_message(error) function
gchar*          sd_engine_status_message(EngineStatus status);
//------------------------------------------------------------------------------
// implementation of engine_global_functions(void) function
EngineModule    engine_global_functions();

#ifdef __cplusplus
}
#endif
#endif /* #ifndef _DICTIONARY_ENGINE_STARDICT */
/*@}*/
