/*******************************************************************************
This file is part of mdictionary.

mdictionary is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

mdictionary is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License 
along with mdictionary; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Copyright 2006-2008 ComArch S.A.
*******************************************************************************/
/** \addtogroup Manager
 */
/*@{*/
/** \file ws_mng_threads_utils.c
 * \brief Thread utilities - code.
 */
#include <ws_mng_threads_utils.h>


/* added by Dariusz Wiechecki 
 * trying to lock mutext and tells if it was locked */
/**
 * \param data manager data
 * \param fun string to display in logs - it intentionaly this should be caller
 * function name
 */
gboolean try_lock_was_locked(WSMngSearchData* data, gchar* fun)
{
	gboolean res = !g_static_mutex_trylock(data->action_working);
	if (res)
	{
		g_debug("[M-action] %s - FAILED - locked already!", fun);
	}
	else
	{
		g_debug("[M-action] %s - SUCCESS - locked!", fun);
	}

	return res;
}

/* added by Dariusz Wiechecki
 * check if recursive mutex is already locked 
 * must be call from critical section!!! */
/**
 * If mutex was not locked it is not locked by this function. It only check
 * if given mutex is already locked.
 *
 * \param m mutex to be checked
 */
gboolean is_rec_locked(GStaticRecMutex* m)
{
	gboolean r = !g_static_rec_mutex_trylock(m);
	if (FALSE == r)
	{
		g_static_rec_mutex_unlock(m);
		g_debug("[M-stop] STOP mutex is UNLOCKED!");
	}
	else
	{
		g_debug("[M-stop] STOP mutex is LOCKED");
	}
	return r;
}

/* added by Dariusz Wiechecki
 * stop current thread if there is such a need - newer thread is waiting */
/**
 * \param data search atom data for current thread
 */
void stop_if_needed(WSMngSearchAtom* data)
{
	static GStaticMutex _loc;
	static GStaticMutex* loc = NULL;
	if(NULL == loc && NULL == data)
	{
		g_debug("Initializing static mutex. function:%s\n",__FUNCTION__);
		g_static_mutex_init (loc);
		loc = &_loc;
		return;
	}

	WSMngSearchData* app_data = data->data;
	/* critical section for calling is_rec_locked() function*/
	g_static_mutex_lock(loc);
	if ( is_rec_locked(app_data->action_stop) )
	{
		g_debug("[T-leaving] <---- Leaving thread (not finished).\n");
		ws_dbus_notify(app_data->dbus_data, WS_DBUS_WORDS_LIST_FINISHED);
		ws_dbus_notify(app_data->dbus_data, WS_DBUS_TRANSLATION_FINISHED);
		free_search_atom(data);
		g_static_mutex_unlock(app_data->action_working);
		g_static_mutex_unlock(loc);
		g_thread_exit (NULL);
	}
	g_static_mutex_unlock(loc);
}

/**
 * \param data search atom data for current thread
 */
void free_search_atom(WSMngSearchAtom* data)
{
	g_debug("[T-clear] Cleaning after thread\n");
	g_assert(NULL != data);
	if (NULL != data->word)
	{
		g_free(data->word);
	}
	if (NULL != data->trans)
	{
		g_free(data->trans);
	}
	if (NULL != data->word_list)
	{
		/* free memory used by each word from word list */
		gint i = 0;
		for (; i < data->word_list->len; i++)
		{
			g_free(g_array_index(data->word_list,gchar* ,i));
		}
		/* free memory used by GArray */
		g_array_free(data->word_list, TRUE);
		data->word_list = NULL;
	}
	g_free(data);
}

/**
 * \param app_data  manager data
 * \param word word to search
 */
WSMngSearchAtom *create_search_atom(WSMngSearchData* app_data, gchar* word)
{
	WSMngSearchAtom* search_data = NULL;
	search_data = (WSMngSearchAtom*)g_try_malloc(sizeof(WSMngSearchAtom));
	if (NULL == search_data)
	{
		g_debug("[L-*] allocatting memmory for data failed!");
		return NULL;
	}
        /* get the word sended by dbus */
        search_data->word = g_strdup(word);
	search_data->data = app_data;
	search_data->word_list = NULL;
	search_data->trans = NULL;

	return search_data;
}

/*@}*/
