/*******************************************************************************
This file is part of mdictionary.

mdictionary is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

mdictionary is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License 
along with mdictionary; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Copyright 2006-2008 ComArch S.A.
*******************************************************************************/
/** \addtogroup Manager
 */
/*@{*/
/** \file ws_mng_searching_threads.c
 * \brief Thread generation and thread functions - code.
 */
#include <ws_mng_searching_threads.h>
#include <glib/gstdio.h>

/**
 * \param data required data for search word event
 * \return gpointer return value 
 */
gpointer ws_mng_search_word (gpointer user_data)
{
	g_debug("[T-word] %s: Entering thread...", __FUNCTION__);
	WSMngSearchAtom *search_atom = (WSMngSearchAtom*)user_data;
	WSMngSearchData *search = search_atom->data;

	/* enter into CRITICAL SECTION */
	if (try_lock_was_locked(search,(gchar*)__FUNCTION__))
	{
		g_printf("[T-word] STOP mutex is locked! Aborting others!");
		g_static_rec_mutex_lock(search->action_stop);
		g_static_mutex_lock(search->action_working);
	}
	g_static_rec_mutex_unlock(search->action_stop);

	/* if another thread was run after this one - exit */
	stop_if_needed(search_atom);

	g_debug( "[T-word] %s - from now this thread is \'singleton\' ",
	         __FUNCTION__ );
	ws_dbus_notify(search->dbus_data, WS_DBUS_WORDS_LIST_STARTED);
	/* creating new GArray for word list */
	search_atom->word_list = g_array_new(TRUE, TRUE, sizeof(gchar*));

	g_debug("[T-word] %s - start searching... ",__FUNCTION__);
        if (search->bookmark_mode)
	{
		/* search only in user bookmarks */
		dict_eng_search_word_list( search->bookmark,
		                           search_atom->word,
		                           search_atom );
	}
	else
	{
		/* search for word in each dictionary */
		gint i = 0;
		for (i = 0; i < search->dict->len; i++)
        	{
			if (NULL == g_array_index(search->dict, Engine *, i))
			{
				continue;
			}

			stop_if_needed(search_atom);
			Engine* dict = g_array_index(search->dict,Engine *,i);
			dict_eng_search_word_list( dict,
			                           search_atom->word,
			                           search_atom );
        	}
	}
	g_debug("[T-word] %s - searching finished.",__FUNCTION__);

	/* if another thread was run after this one - exit */
	stop_if_needed(search_atom);

	/* sort and cleaning words list - only if there were more than one
	 * dictionary loaded */
	if ((FALSE == search->bookmark_mode) || (1 < search->dict->len))
	{
		g_array_sort(search_atom->word_list, ws_mng_compare_strings);
		ws_remove_multiple_words(search_atom);
	}

	/* if another thread was run after this one - exit */
	stop_if_needed(search_atom);

	ws_dbus_server_return_words(search->dbus_data, search_atom->word_list);
	ws_dbus_notify(search->dbus_data, WS_DBUS_WORDS_LIST_FINISHED);

	/* free memory used by each word from word list */
	gint i = 0;
	for (; i < search_atom->word_list->len; ++i)
	{
		g_free(g_array_index(search_atom->word_list,gchar* ,i));
	}

	/* free memory used by GArray */
	g_array_free(search_atom->word_list, TRUE);
	search_atom->word_list = NULL;

	g_free(search_atom->word);
	g_free(search_atom);
	g_debug("[T-word] %s - leaving thread!", __FUNCTION__);
	g_static_mutex_unlock(search->action_working);
	return NULL;
}

/**
 * \param list word list found in dictionaries
 * \param pattern a word which is being search for in dictionaries
 * \param user_data data passed to function
 * \param error engine status information
 */
void ws_mng_on_found_word(GArray* list,
                          gchar* pattern,
                          gpointer user_data,
                          EngineStatus error)
{
	g_debug("[T-word-ret]-> %s", __FUNCTION__);

	WSMngSearchAtom* search_atom = (WSMngSearchAtom*)user_data;
	static gint i = 0;
	for (i = 0; i < list->len; i++)
	{
		/* copy word found by search engine */
		gchar* new_word = g_strdup(g_array_index(list, gchar*, i));
		g_array_append_val(search_atom->word_list, new_word); 
		
	}

	g_debug("[T-word-ret]<- %s", __FUNCTION__);
}

/**
 * \param data required data for search translation
 * \return gpointer return value 
 */
gpointer ws_mng_search_translation (gpointer data)
{
	g_debug("[T-tran] %s: Entering thread...", __FUNCTION__);
	WSMngSearchAtom* search_atom = (WSMngSearchAtom*)data;
	WSMngSearchData* search = search_atom->data;

	/* ---> CRITICAL SECTION for this function */
	if (try_lock_was_locked(search,(gchar*)__FUNCTION__))
	{
		g_printf("[T-tran] STOP mutex is locked! Aborting others!");
		g_static_rec_mutex_lock(search->action_stop);
		g_static_mutex_lock(search->action_working);
	}
	g_static_rec_mutex_unlock(search->action_stop);
	/* if another thread was run after this one - exit */
	stop_if_needed(search_atom);

	g_debug( "[T-tran] %s - from now this thread is \'singleton\' ",
	         __FUNCTION__ );
	ws_dbus_notify(search->dbus_data, WS_DBUS_TRANSLATION_STARTED);

	/* run search for translation for every dictionary */
	if (search->bookmark_mode)
	{
		dict_eng_search_word_translation( search->bookmark,
		                                  search_atom->word,
		                                  search_atom );
	}
	else
	{
		gint i;
		for (i = 0; i < search->dict->len; i++)
		{
			stop_if_needed(search_atom);
			if (NULL == g_array_index(search->dict, Engine*, i) )
			{
				continue;
			}
			dict_eng_search_word_translation(
			                g_array_index(search->dict, Engine*, i),
			                search_atom->word,
			                search_atom);
		}
	}
	g_debug("[T-tran] %s - searching finished.",__FUNCTION__);

	/* if another thread was run after this one - exit */
	stop_if_needed(search_atom);

	/* send translation to gui */
	ws_dbus_server_return_translations(search->dbus_data, search_atom->trans);
	ws_dbus_notify(search->dbus_data, WS_DBUS_TRANSLATION_FINISHED);

	g_free(search->trans);
	search->trans = NULL;

	g_debug("[T-word] %s - leaving thread!", __FUNCTION__);
	g_static_mutex_unlock(search->action_working);
	return NULL;
}

/**
* \param translation translation of word found in dictionaries
* \param pattern a word which is being serch for in dictionaries
* \param user_data data passed to function
* \param error engine status information
*/
void ws_mng_on_found_translation(gchar* translation,
                                 gchar* pattern,
                                 gpointer user_data,
                                 EngineStatus error
                                 )
{
	g_debug("->%s", __FUNCTION__);
	WSMngSearchAtom* search_atom = (WSMngSearchAtom*)user_data;

	/* we get already the first translation */
	if ((NULL != translation) && (NULL == search_atom->trans))
	{
		/* concatenate tags and searched word and translation */
		search_atom->trans = g_strconcat("<PATTERN_OPEN>",
		                          pattern,
		                          "<PATTERN_CLOSED><TRANSLATION_OPEN>",
		                          translation,
		                          "<TRANSLATION_CLOSED>",
		                          NULL
		                         );
	}
	else if (NULL != translation)
	{
		/* if there was stored translation
		 * copy stored translation to temporary variable */
		gchar* tmp = g_strconcat(search_atom->trans,
		                         "<TRANSLATION_OPEN>",
		                         translation,
		                         "<TRANSLATION_CLOSED>",
		                         NULL);
		/* free memory used by stored old translation */
		gchar* loc_tmp = search_atom->trans;
		search_atom->trans = tmp;
		g_free(loc_tmp);
		tmp = loc_tmp = NULL;
	}

	g_debug("<-%s", __FUNCTION__);
}

/**
 * \param a first argument to compare
 * \param b second argument to compare
 * \return result of compare <0 if the second is greater than first 0 if the 
 *       strings are the same >0 if the first string is greater than second  
 */
gint ws_mng_compare_strings (gconstpointer a, gconstpointer b)
{
	gchar** str1 = (gchar**)(a);
	gchar** str2 = (gchar**)(b);

	gchar* stra = g_utf8_strdown(str1[0], -1);
	gchar* strb = g_utf8_strdown(str2[0], -1);

	gint result = g_utf8_collate(stra, strb);

	g_free(stra); 
	g_free(strb);
	return result;
}

/**
 * The same word could be existed in few dictionaries. Each of this dictionaries
 * will return this word and manager will add it to the wrods list. But finally
 * there should not be repeating words on the list, so we have to remove
 * repeated one (leave only one).
 * \param user_data WSMngSearchAtom with array of all words returned from all
 * dictionaries
 */
void ws_remove_multiple_words(WSMngSearchAtom* user_data)
{
	WSMngSearchAtom* search = (WSMngSearchAtom*)user_data;
	gint j = 0;
	gint i = 0;
	gint result = -1;
	gint temp = 256;
	gchar* tmp1 = NULL;
	gchar* tmp2 = NULL;

	/* check if words list is longer than 256 words */
	if (search->word_list->len < 256)
	{
		temp = search->word_list->len;
		if (temp >0)
		{
			ws_dbus_notify( search->data->dbus_data,
			                WS_DBUS_WORDS_LIST_FILLED_NOT_FULL );
		}
	}
	else
	{
		ws_dbus_notify( search->data->dbus_data,
		                WS_DBUS_WORDS_LIST_FULL );
	}

	/* remove repeating words in the first 256 places in the array - words
	 * in places further than 256 are not being sent to UI so we do not need
	 * to filter them for searching repeating words */
	for (i = 0; i < temp-1; i++)
	{
		tmp1 = g_utf8_casefold(
		               g_array_index(search->word_list,gchar*,i),
		               -1     );
		for (j = i + 1; j < temp; j++)
		{
			/* search if there is a word on word list */
			tmp2 = g_utf8_casefold(
			            g_array_index(search->word_list,gchar*,j),
			            -1 );
			result = g_utf8_collate(tmp1,tmp2);
			g_free(tmp2);
			tmp2 = NULL;

			/* if there is a word on the word list 
			 * remove that word */
			if (result == 0)
			{
				g_array_remove_index(search->word_list, j);
				--j;
				--temp;
				if (search->word_list->len >= 256)
				{
					temp = 256;
				}
			}
			else {
				/* there is no possiblity that further
				 * will be the same word, check next word */
				break;
			}
		}
		g_free(tmp1);
		tmp1 = NULL;
	}
}

/*@}*/
