/*******************************************************************************
This file is part of mdictionary.

mdictionary is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

mdictionary is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License 
along with mdictionary; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Copyright 2006-2008 ComArch S.A.
*******************************************************************************/
/** \addtogroup Manager
 */
/*@{*/
/** \file ws_mng_gconf_utils.c
 * \brief Utilities connected with GConf configuration - coder.
 */
#include <ws_mng_gconf_utils.h>


/**
 *
 * Function read from GConf locations of every dictionary and return them in
 * GArray to make it possible to load them.
 *
 * \return path to dictionaries location
 */
GArray* ws_mng_read_gconf()
{
	g_debug("--%s", __FUNCTION__);
	GConfClient* client = NULL;
	client = gconf_client_get_default();
	GArray* path_to_dir = g_array_new(TRUE, TRUE, sizeof(gchar *));
	gchar* path = NULL;
	gboolean key_active = FALSE;
	gint i = 0;

	/* get location of every dictionary */
	GSList* list = gconf_client_all_dirs(client, GCONF_KEY, NULL);
	for (i = 0; i < g_slist_length(list); ++i)
	{
		path = (gchar* ) g_slist_nth_data(list, i);
		gchar* new_path = g_strconcat(path, "/path", NULL);
		gchar* new_active = g_strconcat(path, "/active", NULL);
		key_active = gconf_client_get_bool(client, new_active, NULL);
		if (key_active == TRUE)
		{
			gchar* key_value = gconf_client_get_string( client,
			                                            new_path,
			                                            NULL );
			if (key_value != NULL){
				g_array_append_val(path_to_dir, key_value);
			}
		}
		g_free(new_path);
		g_free(new_active);
	}

	/* free memmory */
	for (i=0; i<g_slist_length(list); i++)
	{
		g_free(g_slist_nth_data(list, i));
	}
	g_slist_free(list);
	g_object_unref (client);
	return path_to_dir;
}


/**
 * \return libraries with dictionary engines location
 */
GArray* ws_mng_get_engines_location()
{
	g_debug("->%s", __FUNCTION__);
	GConfClient* client = NULL;
	GArray* path_to_dir = g_array_new(TRUE, TRUE, sizeof(gchar *));
	gchar* path = NULL;
	gint i = 0;
	client = gconf_client_get_default();

	GSList* list = gconf_client_all_dirs(client, GCONF_LIB_PATH, NULL);
	for (i=0; i<g_slist_length(list); i++)
	{
		path = (gchar* ) g_slist_nth_data(list, i);
		gchar* new_path = g_strconcat(path, "/path",NULL);
		gchar* key_value = gconf_client_get_string( client,
		                                            new_path,
		                                            NULL);
		gchar** tmp = g_strsplit(key_value, "/", -1);
		guint i = 0;
		gboolean bookmark_engine = FALSE;
		while(NULL != tmp[i])
		{
			g_debug("Comapred string %s", tmp[i]);
			if (g_ascii_strcasecmp(tmp[i], "ws_bookmark.so") == 0)
			{
				bookmark_engine = TRUE;
				break;
			}
			i++;
		}
		g_strfreev(tmp);
		
		if (key_value != NULL)
		{
			g_debug("Added library path %s", key_value);
			g_array_append_val(path_to_dir, key_value);
		}
		g_free(new_path);
	}

	for (i=0; i<g_slist_length(list); i++)
	{
		g_free(g_slist_nth_data(list, i));
	}
	g_slist_free(list);
	g_object_unref (client);
	return path_to_dir;
}


/**
 * \return path to bookmark location
 */
gchar* ws_mng_get_boomark_location()
{
	g_debug("->%s", __FUNCTION__);
	GConfClient* client = NULL;
	gchar* path = NULL;
	guint i = 0;
	client = gconf_client_get_default();

	GSList* list = gconf_client_all_dirs(client, GCONF_KEY, NULL);
	for (i=0; i<g_slist_length(list); i++)
	{
		path = (gchar* ) g_slist_nth_data(list, i);
		gchar* name = g_strconcat(path, "/name",NULL);
		gchar* new_path = g_strconcat(path, "/path",NULL);
		gchar* key_value = gconf_client_get_string( client,
		                                            name,
		                                            NULL );

		if (key_value != NULL)
		{
			if (g_ascii_strcasecmp(key_value, "Bookmarks") == 0)
			{
				gchar* key_value_path = 
					gconf_client_get_string( client,
					                         new_path,
					                         NULL );

				if (key_value_path != NULL)
				{
					g_free(key_value);
					key_value = NULL;
					g_free(new_path);
					new_path = NULL;
					g_free(name); name = NULL;	
					for (i=0; i<g_slist_length(list); i++)
        				{
                				g_free(
						g_slist_nth_data(list, i));
					}
					g_slist_free(list);
					g_object_unref(client);
					return key_value_path;
				}
			}
		}
		g_free(name); name = NULL;
		g_free(key_value);
		key_value = NULL;
		g_free(new_path);
		new_path = NULL;
	}

	g_debug("<-%s", __FUNCTION__);
	return NULL;
}

/**
 * \param dict path to dictionary file
 * \return TRUE if dictionary is optimize, FALSE otherwise
 */
gboolean ws_mng_if_optimized(gchar* dict)
{
	g_debug("->%s", __FUNCTION__);

	GConfClient* client;
	gchar* path = NULL;
	gboolean key_optimized = FALSE;
	gboolean key_found = FALSE;
	gint i = 0;

	client = gconf_client_get_default();
	GSList* list = gconf_client_all_dirs(client, GCONF_KEY, NULL);
	for (i=0; i<g_slist_length(list); i++)
	{
		path = (gchar *) g_slist_nth_data(list, i);
		gchar* new_path = g_strconcat(path, "/path",NULL);
		gchar* new_optimized = g_strconcat(path, "/optimized",NULL);
		gchar* key_value = gconf_client_get_string( client,
		                                            new_path,
		                                            NULL );

		if (g_ascii_strcasecmp(key_value, dict) == 0)
		{
			key_optimized = gconf_client_get_bool( client,
			                                       new_optimized,
			                                       NULL );
			key_found = TRUE;
		}
		g_free(new_path);
		g_free(new_optimized);
		g_free(key_value);

		if ( TRUE == key_found )
		{
			break;
		}
	}

	for (i=0; i<g_slist_length(list); i++)
	{
		g_free(g_slist_nth_data(list, i));
	}
	g_slist_free(list);
	g_object_unref (client);

	g_debug("<-%s", __FUNCTION__);
	return key_optimized;
}

/*@}*/
