/*******************************************************************************
This file is part of mdictionary.

mdictionary is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

mdictionary is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License 
along with mdictionary; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Copyright 2006 ComArch S.A.
*******************************************************************************/
/** \addtogroup Manager */
/*@{*/
/** \file ws_mng_callbacks.c
 * \brief Callback functions called by other modules - code.
 */

#include <ws_mng_callbacks.h>
#include <glib/gstdio.h>

/**
 * \param progess how much file is cached
 * \param user_data data needed to send information to UI
 * \param error error messages
 */
void ws_mng_progress_bar( double progress,
                          gpointer user_data,
                          EngineStatus error )
{
	g_debug("<--> %s - progress=%f", __FUNCTION__, progress);
	WSMngSearchData *data = (WSMngSearchData *) user_data;
	ws_dbus_server_update_progressbar(data->dbus_data, progress);
}

/**
 * \param word word to search
 * \param error error messages
 * \param user_data user data passed to function
 */
void ws_mng_on_search_word( GError *error,
                            GArray *word,
                            gpointer user_data )
{
	g_debug("[L-word] -> %s: reading parameters...", __FUNCTION__);
	WSMngSearchData *search = (WSMngSearchData *) user_data;
	osso_rpc_t* osso_data = NULL;

	/* ---> CRITICAL SECTION for this function */
	g_static_mutex_lock( search->thread_creation );
	
	/* get the word passed by dbus */
	osso_data = &g_array_index (word, osso_rpc_t, 0);
	gchar* tmp = NULL;

	if (osso_data->value.s != NULL)
	{
		/* check if we need add wildcard '*' at the end of the word */
		if (( g_utf8_strchr(osso_data->value.s, -1, '*') == NULL ) && 
		    ( g_utf8_strchr(osso_data->value.s, -1, '?') == NULL ))
		{
			tmp = g_strconcat(osso_data->value.s, "*", NULL);
		}
		else
		{
			tmp = g_strdup(osso_data->value.s);
		}
	}
	else
	{
		tmp = g_strdup ("*");
	}

	/* create and init searching data - separate for each thread */
	WSMngSearchAtom* search_data = NULL;
	search_data = create_search_atom(search, tmp);
	g_free(tmp);
	if (NULL == search_data)
	{
		return;
	}

	g_debug("[L-word] creating GThread object...");
	search_data->thread = g_thread_create( ws_mng_search_word,
	                                       search_data,
	                                       TRUE,
	                                       NULL );
	g_debug("[L-word] GThread object created. Exiting from CREATOR.");

	g_static_mutex_unlock( search->thread_creation );
	/* <--- end of CRITICAL SECTION for this function */

	g_debug("[L-word] <-%s", __FUNCTION__);
}

/**
 * \param word word to search
 * \param error error messages
 * \param user_data user data passed to function
 */
void ws_mng_on_search_translation (GError *error,
                                   GArray *word,
                                   gpointer user_data)
{
	g_debug("[L-tran] ->%s", __FUNCTION__);

	WSMngSearchData *data = (WSMngSearchData *) user_data;

	/* ---> CRITICAL SECTION for this function */
	g_static_mutex_lock(data->thread_creation);

	/* get the data sended by dbus */
	osso_rpc_t *osso_data;
	osso_data = &g_array_index (word, osso_rpc_t, 0);

	/* create and init searching data - separate for each thread */
	WSMngSearchAtom* search_data = NULL;
	search_data = create_search_atom(data, osso_data->value.s);
	if (NULL == search_data)
	{
		return;
	}

	g_debug("[L-tran] creating GThread object...");
	search_data->thread = g_thread_create( ws_mng_search_translation,
	                                       search_data,
	                                       TRUE,
	                                       NULL );
	g_debug("[L-tran] GThread object created. Exiting from CREATOR.");


	g_static_mutex_unlock(data->thread_creation);
	/* <--- end of CRITICAL SECTION for this function */

	g_debug("[L-tran] <-%s", __FUNCTION__);
}


/**
* @param error error messages
* @param signal type of signal
* @param user_data data passed to function 
*/
void ws_mng_signal_handling(GError *error, GArray *signal, gpointer user_data)
{
	g_debug("->%s", __FUNCTION__);
	osso_rpc_t osss_data;
	osss_data = g_array_index (signal, osso_rpc_t, 0);
	WSMngSearchData *data = (WSMngSearchData *) user_data;
	gint i = 0;

	switch(osss_data.value.i)
	{
		case WS_DBUS_INFO_TERMINATE:
			/* added by Dariusz Wiechecki
			 * canceling GLib Threads if any is working */
			if (try_lock_was_locked(data,(gchar*)__FUNCTION__))
			{
				g_printf("[S] STOP ACTION! %s\n",__FUNCTION__);
				g_static_rec_mutex_lock(data->action_stop);
				g_static_mutex_lock(data->action_working);
			}
			g_static_rec_mutex_unlock(data->action_stop);
			g_static_mutex_unlock(data->action_working);
			/* end current proccess */
			g_main_loop_quit (data->loop);
		break;
		case WS_DBUS_INFO_STOP_SEARCH:
			/* added by Dariusz Wiechecki
			 * canceling GLib Threads if any is working */
			if (try_lock_was_locked(data,(gchar*)__FUNCTION__))
			{
				g_printf("[S] STOP ACTION! %s\n",__FUNCTION__);
				g_static_rec_mutex_lock(data->action_stop);
				g_static_mutex_lock(data->action_working);
			}
			g_static_rec_mutex_unlock(data->action_stop);
			ws_dbus_notify(data->dbus_data,
			               WS_DBUS_WORDS_LIST_FINISHED);
			ws_dbus_notify(data->dbus_data,
			               WS_DBUS_TRANSLATION_FINISHED);
			g_static_mutex_unlock(data->action_working);
		break;
		case WS_DBUS_INFO_CONFIG_CHANGED:
			ws_dbus_notify(data->dbus_data, WS_DBUS_INFO_CACHING);
			/* first remove all dictionaries */
			for (i=0; i<data->dict->len; i++)
			{
				if(g_array_index(data->dict, Engine*,i) != NULL)
				{
					dict_eng_destroy( g_array_index(
					                       data->dict,
					                       Engine*,
					                       i       ) );
				}
			}
			g_array_free(data->dict, TRUE);

			/* load dictionaries again in the new configuration */
			data->dict = g_array_new (TRUE, TRUE, sizeof(Engine*));
			GArray* dir_array = ws_mng_read_gconf();
			ws_mng_load_dict(dir_array, data);

			/* check if there was loaded any dictionary */
			if (data->dict->len <= 0)
			{
				ws_dbus_notify( data->dbus_data,
				                WS_DBUS_INFO_CACHING_FINISHED );
				ws_dbus_notify( data->dbus_data,
				                WS_DBUS_ERROR_FILE_NOT_FOUND);
			}
			else
			{
				/*if there was typed word search for word list*/
				if (data->last_search != NULL) 
				{
					data->word_list =
					        g_array_new( TRUE,
					                     TRUE,
					                     sizeof(gchar*));

					g_free(data->word); data->word = NULL;
					data->word =
					           g_strdup(data->last_search);
				}
				/* signal end of dictionary load to gui */
				ws_dbus_notify( data->dbus_data,
				                WS_DBUS_INFO_CACHING_FINISHED );
			}

			/* free memmory */
			for (i=0; i<dir_array->len; i++)
			{
				g_free(g_array_index(dir_array, gchar*, i));
			}
			g_array_free(dir_array, TRUE);
		break;
		case WS_DBUS_BOOKMARK_MODE_ON:
			data->bookmark_mode = TRUE;
		break;
		
		case WS_DBUS_BOOKMARK_MODE_OFF:
			data->bookmark_mode = FALSE;
		break;
	}
	g_debug("<-%s", __FUNCTION__);
}

/*@}*/
