/*******************************************************************************
This file is part of mdictionary.

mdictionary is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

mdictionary is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License 
along with mdictionary; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Copyright 2006-2008 ComArch S.A.
*******************************************************************************/
/** \addtogroup Manager */
/*@{*/
/** \file ws_mng_bookmarks_utils.c
 * \brief Utilities for Manager connected with managing bookmarks - code.
 */
#include <ws_mng_bookmarks_utils.h>

/**
 * Function used to add bookmarks.
 * \param error error messages
 * \param param word and translation of word to adding
 * \param user_data structure holding nedded parameters
 */
void ws_mng_add_bookmark(GError *error, GArray* param, gpointer user_data)
{
	g_debug("->%s", __FUNCTION__);

	WSMngSearchData* data = (WSMngSearchData *) user_data;
	osso_rpc_t* osso_data = NULL;
	osso_rpc_t* osso_data_trans = NULL;
	gchar* word = NULL;
	gchar* translation = NULL;

	if (data->bookmark != NULL)
	{
		/* get the word passed by dbus */
		osso_data = &g_array_index (param, osso_rpc_t, 0); 
		word = g_strdup(osso_data->value.s);
		osso_data_trans = &g_array_index (param, osso_rpc_t, 1); 
		translation = g_strdup(osso_data_trans->value.s);

		/* try to add word to bookmarks */
		gboolean is_added = dict_eng_add_word( data->bookmark,
		                                       word,
		                                       translation );
		if (is_added == TRUE) 
		{
			ws_dbus_notify( data->dbus_data,
			                WS_DBUS_BOOKMARKS_ADDED_OK );
		}
		else
		{
			ws_dbus_notify( data->dbus_data, 
			                WS_DBUS_BOOKMARKS_ADDED_FAIL );
		}
		/* free memmory */
		osso_rpc_free_val(osso_data);
		g_free(translation);
		g_free(word);
	}
	else
	{
		ws_dbus_notify( data->dbus_data,
		                WS_DBUS_BOOKMARKS_ADDED_FAIL );
		g_debug("-> %s - there is no bookmark engine!\n", __FUNCTION__);
	}

	g_debug("<-%s", __FUNCTION__);
}

/**
 * Function used to remove bookmarks.
 * \param error error messages
 * \param param word and translation of word to remove
 * \param user_data structure holding nedded parameters
 */
void ws_mng_remove_bookmark(GError *error, GArray* param, gpointer user_data)
{
	g_debug("->%s", __FUNCTION__);

	WSMngSearchData* data = (WSMngSearchData *) user_data;
	osso_rpc_t* osso_data = NULL;
	gchar* word = NULL;

        if (data->bookmark != NULL)
	{
		/* get the word passed by dbus */
		osso_data = &g_array_index(param, osso_rpc_t, 0); 
		word = g_strdup(osso_data->value.s);

		gboolean is_remove  = dict_eng_remove_word( data->bookmark,
		                                            word );
		if (TRUE == is_remove)
		{
			ws_dbus_notify( data->dbus_data, 
			                WS_DBUS_BOOKMARKS_REMOVED_OK );
		}
		else 
		{
			ws_dbus_notify( data->dbus_data, 
			                WS_DBUS_BOOKMARKS_REMOVED_FAIL );
		}

		osso_rpc_free_val(osso_data);
	}
	else
	{	
		ws_dbus_notify( data->dbus_data,
		                WS_DBUS_BOOKMARKS_ADDED_FAIL );
		g_debug("-> %s - there is no bookmark engine!\n", __FUNCTION__);
	}

	g_debug("<-%s", __FUNCTION__);
}


/**
 * Function used to load bookmark module 
 * \param data structure which holds the loaded module
 */
void ws_mng_load_bookmark(WSMngSearchData* data)
{
	g_debug("->%s", __FUNCTION__);
	guint i = 0;
	data->bookmark = NULL;
	gchar* current_directory = ws_mng_get_boomark_location();

	if (NULL != current_directory)
	{
		for (i = 0; i < data->modules->len; ++i)
		{
			/* check each engine module to be compatible with 
			 * bookmark dictionary - searching for bookmark module*/
			if (TRUE == dict_eng_module_check(
				 g_array_index(data->modules, EngineModule, i),
			         current_directory ))
			{
				/* create engine for handling bookmarks */
				data->bookmark =
				dict_eng_module_create_ext(
				                g_array_index(data->modules,
				                              EngineModule,
				                              i),
				               current_directory,
				               ENGINE_CREATE,
				               ws_mng_progress_bar,
				               data,
				               0.02 );
				/* set callbacks */
				dict_eng_set_callback( data->bookmark,
				                       ENGINE_WORD_LIST_SIGNAL,
				                       ws_mng_on_found_word,
				                       data );
				dict_eng_set_callback(data->bookmark,
				                 ENGINE_WORD_TRANSLATION_SIGNAL,
				                 ws_mng_on_found_translation,
				                 data);
				/* bookmark engine found - stop searching */
				break;
			}
		}
		g_free(current_directory);
	}
	g_debug("<-%s", __FUNCTION__);
}


/*@}*/
