/*******************************************************************************
This file is part of mdictionary.

mdictionary is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

mdictionary is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License 
along with mdictionary; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Copyright 2006-2008 ComArch S.A.
*******************************************************************************/
/** \addtogroup Manager
 */
/*@{*/
/** \file ws_manager.c
 * \brief mDictionary Manager main module code.
 *
 * \author Łukasz Pawlik \<lukasz.pawlik\@comarch.com\>
 */

#include <ws_manager.h>

#include <ws_mng_searching_threads.h>
#include <ws_mng_dictionary_utils.h>
#include <ws_mng_bookmarks_utils.h>
#include <ws_mng_threads_utils.h>
#include <ws_mng_gconf_utils.h>
#include <ws_mng_callbacks.h>


/**
 * \return new instance of WSMngSearchData structure
 */
WSMngSearchData* ws_manager_create() {
	g_debug("<--> %s", __FUNCTION__);
	WSMngSearchData* tmp = g_try_new(WSMngSearchData, 1);

	if (NULL == tmp)
	{
		g_debug("Not enough memory !");
	}
	return tmp;
}

/**
 * \param serach_data structure which contains all data of program
 * \return TRUE if successfully run main loop, FALSE otherwise 
 */
gboolean ws_mng_start_main_loop(WSMngSearchData* search_data)
{
	g_debug("<--> %s", __FUNCTION__);
	search_data->loop = g_main_loop_new (NULL, FALSE);
	if (search_data->loop == NULL)
	{
		g_debug("Couldn't create new g_main_loop for Manager!");
		return FALSE;
	}
	g_main_loop_run (search_data->loop);
	return TRUE;
}

/**
 * \param data structure stores variables which are need to comunicate by D-BUS.
 */
void ws_mng_init_dbus (WSMngSearchData *data)
{
	g_debug("->%s", __FUNCTION__);

	/* initialize GThread support if it is not initialized yet */
	if (!g_thread_supported ()) g_thread_init (NULL);

	/* create data structure needed to comunicate with dbus wrapper */
	data->dbus_data = ws_dbus_create ("mdictionaryManager", "v1.0");

	/* set data used to comunicate with gui */
	ws_dbus_config(data->dbus_data,
	               WS_DBUS_CONFIG_SERVICE,
	               MANAGER_SERVICE);
	ws_dbus_config(data->dbus_data,
	                WS_DBUS_CONFIG_OBJECT,
	                MANAGER_OBJECT);
	ws_dbus_config(data->dbus_data,
	                WS_DBUS_CONFIG_IFACE,
	                MANAGER_IFACE);
	ws_dbus_config(data->dbus_data, 
	                WS_DBUS_CONFIG_REMOTE_SERVICE,
	                GUI_SERVICE);
	ws_dbus_config(data->dbus_data, 
	                WS_DBUS_CONFIG_REMOTE_OBJECT,
	                GUI_OBJECT);
	ws_dbus_config(data->dbus_data, 
	                WS_DBUS_CONFIG_REMOTE_IFACE,
	                GUI_IFACE);

	/* defines all methods which could be called through D-Bus */
	ws_dbus_add_method ( data->dbus_data,
	                     "find_word",
	                     WS_DBUS_TYPE_STRING,
	                     WS_DBUS_TYPE_INVALID );
	ws_dbus_add_method ( data->dbus_data,
	                     "find_translation",
	                     WS_DBUS_TYPE_STRING,
	                     WS_DBUS_TYPE_INVALID );
	ws_dbus_add_method ( data->dbus_data,
	                     "signal",
	                     WS_DBUS_TYPE_SIGNAL,
	                     WS_DBUS_TYPE_INVALID );
	ws_dbus_add_method ( data->dbus_data,
	                     "add_bookmark",
	                     WS_DBUS_TYPE_STRING,
	                     WS_DBUS_TYPE_STRING,
	                     WS_DBUS_TYPE_INVALID );
	ws_dbus_add_method ( data->dbus_data,
	                     "remove_bookmark",
	                     WS_DBUS_TYPE_STRING,
	                     WS_DBUS_TYPE_INVALID );
	ws_dbus_add_method ( data->dbus_data,
	                     "extract_dictionary",
	                     WS_DBUS_TYPE_STRING,
	                     WS_DBUS_TYPE_INVALID );

	/* set callback for find word signal */
	ws_dbus_set_cb( data->dbus_data,
	                "find_word",
	                ws_mng_on_search_word,
	                data );

	/* set callback for find translation signal */
	 ws_dbus_set_cb( data->dbus_data,
	                 "find_translation",
	                 ws_mng_on_search_translation,
	                 data );

	/* set callback for close program signal */
	ws_dbus_set_cb( data->dbus_data,
	                "signal",
	                ws_mng_signal_handling,
	                data );

	 /* set callback for add bookmarks signal */
	ws_dbus_set_cb( data->dbus_data,
	                "add_bookmark",
	                ws_mng_add_bookmark,
	                data );

	/* set callback for remove bookmarks signal */
	ws_dbus_set_cb( data->dbus_data,
	                "remove_bookmark",
	                ws_mng_remove_bookmark,
	                data );

	/* set callback for extracting dictionary */
	ws_dbus_set_cb( data->dbus_data,
	                "extract_dictionary",
	                ws_mng_extract_dictionary,
	                data );

	/* initialize d-bus connection with remote service - UI */
	ws_dbus_connect(data->dbus_data);

	g_debug("<-%s", __FUNCTION__);
}


/**
 *
 * Fuction loads from modules pointers to functions used to service searching 
 * in dictionaries and stores them in WSMngSearchData structure
 *
 * \param data pointer to structure WSMngSearchData which stores variables
 * needed to service manager
 */
void ws_mng_init (WSMngSearchData *data)
{
	g_debug("->%s", __FUNCTION__);

	/* create and initialize all data in WSMngSearchData structure */
	data->dict = g_array_new(TRUE, TRUE, sizeof(Engine *));
	data->modules = g_array_new(TRUE, TRUE, sizeof(EngineModule));
	data->libraries = g_array_new(TRUE, TRUE, sizeof(GModule*));
	data->word_list = NULL;
	data->last_search = NULL;
	data->trans = NULL;
	data->search_in_history = FALSE;
	data->word = NULL;
	data->bookmark = NULL;
	data->bookmark_mode = FALSE;
	/* added by Dariusz Wiechecki
	 * mutex initialization */
	data->action_working =
	               (GStaticMutex*)g_try_malloc(sizeof(GStaticMutex));
	data->thread_creation =
	               (GStaticMutex*)g_try_malloc(sizeof(GStaticMutex));
	data->action_stop =
	               (GStaticRecMutex*)g_try_malloc(sizeof(GStaticRecMutex));
	g_assert(NULL != (data->action_working) );
	g_assert(NULL != (data->action_stop) );
	g_static_mutex_init (data->action_working);
	g_static_mutex_init (data->thread_creation);
	g_static_rec_mutex_init (data->action_stop);
	/* initialize static stop_if_needed function mutex*/
	stop_if_needed (NULL);

	/* send signal if there is not proper sqlite library */
	#ifdef SQLITE
	if( FALSE == g_file_test(LIBRARY, G_FILE_TEST_EXISTS) )
	{
		ws_dbus_notify(data->dbus_data, WS_DBUS_LOAD_BOOKMARK_FAILED);
	}
	#endif

	/* load GConf configuration */
	GArray* dict_directory = ws_mng_read_gconf();
	data->library_path = ws_mng_get_engines_location();

	gint i = 0;
	for (; i < data->library_path->len; ++i)
	{
		gchar* path= g_array_index(data->library_path, gchar*, i);
		GModule* library = g_module_open(path, G_MODULE_BIND_LAZY);
		g_array_append_val(data->libraries, library);
		g_debug("%p library pinter %d iter", library, i);
        }

	getting_additional_t get_functions = NULL;
	for (i=0; i<data->libraries->len; i++)
        {
		g_debug("%p", g_array_index(data->libraries, GModule*, i));
		g_module_symbol( g_array_index(data->libraries, GModule*, i),
		                 GLOBAL_FUNCTIONS_NAME,
		                 (gpointer)&get_functions );
        	g_debug("%d	%p", i, &get_functions);

		/* if function was not properly imported close manager */
		if (NULL == get_functions)
		{
			ws_dbus_notify( data->dbus_data,
			                WS_DBUS_ERROR_ENGINE_NOT_FOUND );
			for (i=0; i<dict_directory->len; i++)
			{
				g_free(g_array_index( dict_directory,
				                      gchar*,
				                      i ));
			}
			g_array_free(dict_directory, TRUE);
			ws_mng_close(data);
			exit(0);
		}

		/* get EngineModule struct and append it to the array*/
		EngineModule module =  get_functions();
		g_array_append_val(data->modules, module);
	}

	/* load each dictionary */
	ws_mng_load_bookmark(data);
	if (dict_directory->len > 0)
        {
		ws_mng_load_dict(dict_directory, data);
		guint i = 0;
		g_debug("Trace bookmark engine %p", data->bookmark);
		for (i=0; i<data->dict->len; i++)
		{
			g_debug( "dict engines at %p",
			         g_array_index(data->dict, Engine*, i) );
		}
	}
	else
	{
		ws_dbus_notify( data->dbus_data,
		                WS_DBUS_ERROR_FILE_NOT_FOUND );
	}

	/* free memory used by dictionaries path array */
	for (i=0; i<dict_directory->len; i++)
	{
		g_free(g_array_index(dict_directory, gchar*, i));
	}
	g_array_free(dict_directory, TRUE);

	g_debug("<-%s", __FUNCTION__);
}

/**
 * \param data structure holds pointers to data which need to be freed from
 * memory
 */
void ws_mng_close (WSMngSearchData *data)
{
	int i = 0;
	g_debug("->%s", __FUNCTION__);

	ws_dbus_destroy (data->dbus_data);
	if (data->bookmark != NULL)
	{
		dict_eng_destroy(data->bookmark);
	}

	for (i = 0; i < data->dict->len; i++) 
	{
		dict_eng_destroy(g_array_index (data->dict, Engine*,i));
	}
	g_array_free(data->dict, TRUE);

	g_free(data->last_search);
	g_free(data->loop);
	g_static_mutex_free(data->thread_creation);
	g_static_mutex_free(data->action_working);
	g_static_rec_mutex_free(data->action_stop);

	for (i=0; i<data->library_path->len; i++)
	{
		g_free(g_array_index(data->library_path, gchar*, i));
	}
	g_array_free(data->library_path, TRUE);	
	g_array_free(data->modules, TRUE); data->modules = NULL;
	
	for (i=0; i< data->libraries->len; i++)
	{
		if (g_array_index(data->libraries, GModule*, i) != NULL)
		{
			g_module_close(g_array_index( data->libraries,
			                              GModule*,
			                              i ));
                }
	}
	g_array_free(data->libraries, TRUE);
	g_free(data);
	g_debug("<-%s", __FUNCTION__);
}

/*@}*/
