/*******************************************************************************
This file is part of mDictionary

mDictionary is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

mDictionary is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License 
along with WhiteStork; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Copyright 2006-2008 ComArch S.A.
*******************************************************************************/
/** \addtogroup Manager
 */
/*@{*/

/** \file untar.c
 * \brief Exctracting *.tar.bz2 archives function code.
 */
#include <untar.h>
#include <string.h>

#define g_strlen(string) ( NULL == (string) ? 0 : strlen(string) )

/** \brief Unpack particular file from *.tar.bz2 archive
 *
 * \param bz2_file pointer to file with data (compressed with bz2 algorithm 
 * \param header *.tar header with information about data in bz2_file
 * \param out_name string specifying destination path for uncompressed data
 * \return 0 if exctracting was succesfull
 */
static gint unpack_file_contents( BZFILE *bz2_file,
                                  TarHeader *header,
                                  gchar *out_name )
{
	gchar* buffer = NULL;
	guint i = 0;
	gint im = 0;
	FILE *file_out;
	gulong length = 0;
	guint result = 0;
	gint bzerror = 0;

	file_out = fopen(out_name, "w");

	if (file_out == NULL) return -1;

	sscanf(header->size, "%12lo", &length);
	g_debug("File size: %ld\n", length);

	im = length / BLOCK_SIZE;
	for (i = 0; i < im; ++i)
	{
		buffer = (gchar*) g_try_malloc(BLOCK_SIZE * sizeof(gchar));
		
		result = BZ2_bzRead( &bzerror,
		                     bz2_file,
		                     buffer,
		                     BLOCK_SIZE * sizeof (gchar) );

		fwrite (buffer, BLOCK_SIZE * sizeof(gchar), 1, file_out);
		g_free (buffer);
	}

	i = length % BLOCK_SIZE;
	if (i != 0)
	{
		buffer = (gchar*) g_try_malloc (BLOCK_SIZE * sizeof(gchar));
		if (NULL == buffer)
		{
			g_debug("Memory not allocated");
			return -100;
		};
		
		result = BZ2_bzRead( &bzerror,
		                     bz2_file,
		                     buffer,
		                     BLOCK_SIZE * sizeof(gchar) );
		if (BZ_OK == bzerror)
		{
			fwrite (buffer, i * sizeof(gchar), 1, file_out);
		}
		else
		{
			g_debug("bzerror = %d", bzerror);
			return -200;
		};
		g_free(buffer);
	}

	fclose(file_out);
	return 0;
};

/**
 * If user is trying to add archive with dictionary (tar.bz2 file) then there is
 * need to uncompress this dictionary before use. Function decompress_file could
 * do this.
 *
 * Usage example:
 * \code
 * char* path = g_strdup("./");
 * decompress_file ("comn_dictd04_wn.tar.bz2", &path);
 * \endcode
 * The above example extracts a given archive, to the current directory
 *
 * \param in_file path to file to exctract
 * \param out_path pointer to path (string) to output directory. After function
 * finished work in this parameter function return path to the exctracted
 * dictionary
 * \return 0 if exctracting of whole archive was succesfull
 */
gint decompress_file (gchar *in_file, gchar **out_path)
{
	FILE *file;
	BZFILE	*bz2_file;
	guint result;
	gint ret = 100;
	TarHeader* header;
	gint bzerror;
	gchar* dest_dir = NULL;
	gchar* file_name = NULL;
	header = (TarHeader*) g_try_malloc (BLOCK_SIZE * sizeof(gchar));
	if (NULL == header)
	{
		g_debug("\nCould not allocate memory\n");
		return ret = -1;
	}

	file = fopen (in_file, "rb");
	if (NULL == file)
	{
		g_debug("There was an error while trying to read archive file");
		g_free(header); header = NULL;
		return ret = -2;
	};

	bz2_file = BZ2_bzReadOpen (&bzerror, file, 0, 0, NULL, 0);
	if ( BZ_OK != bzerror )
	{
		fclose(file);
		g_free(header); header = NULL;
		g_debug("There was an error while reading compressed file\n");
		return ret = -3;
	}

	/* read archive and exctract all files in it */
	while (TRUE)
	{
		/* get info about next file/directory in the archive */
		result = BZ2_bzRead( &bzerror,
		                     bz2_file,
		                     header,
		                     BLOCK_SIZE * sizeof (char) );
		if ( BZ_OK == bzerror )
		{
			if (strlen(header->name) == 0)
			{
				g_debug("\nFilename length is 0, exitting\n");
				break;
			};
			gchar *temp = g_strconcat( *out_path,
			                           header->name,
			                           NULL );

			/* exctract file or create new directory */
			switch (header->typeflag)
			{
				case File:
					unpack_file_contents( bz2_file,
					                      header,
					                      temp );
					/*get dictionary file name to pass it*/
					if (NULL == file_name)
					{
						file_name = g_path_get_basename(temp);
					}
					
					break;
				case Dir:
					if (0 !=
						mkdir( temp,
						       S_IRUSR|S_IWUSR|S_IXUSR )
					) {
						g_debug("Couldnt create dir:%s",
							 temp);
						g_free(header);
						BZ2_bzReadClose( &bzerror,
						                 bz2_file );
						fclose(file);
						return ret = -4;
					}
					else {
						/* get last created directory */
						if (NULL != dest_dir) {
							g_free(dest_dir);
						}
						/*there is another directory in compressed file*/
						if (NULL != file_name)
						{
							g_free(file_name); file_name = NULL;
							
						}
						dest_dir = g_strdup(temp);
						g_debug( "dest_dir %s \n",
						         dest_dir );
					}
				break;
				default:
					g_debug("Untar:Wrong type of content!");
				break;
			}
			g_free(temp);
			temp = NULL;
		}
		else if ( bzerror != BZ_STREAM_END )
		{
			g_debug("\nBZ2 READ_CLOSE(stream end) %d\n", bzerror);
			BZ2_bzReadClose ( &bzerror, bz2_file );
			break;
		}
		else
		{
			g_debug("\nExitting, error nr %d\n", bzerror);
			BZ2_bzReadClose ( &bzerror, bz2_file );
			ret = -5;
			break;
		};
	};
	
	/* put newly created directory path into out_path parameter */
	if ((ret > 0) && (dest_dir != NULL) && (file_name !=  NULL))
	{
		g_free(*out_path); *out_path = NULL;
		*out_path = g_strconcat(dest_dir, file_name, NULL);
		
		
		g_free(dest_dir); dest_dir = NULL;
		g_free(file_name); file_name = NULL;
	}

	g_free(header); header = NULL;
	fclose(file);
	return ret;
};

/*@}*/

