/*******************************************************************************
This file is part of mdictionary.

mdictionary is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

mdictionary is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License 
along with mdictionary; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Copyright 2006-2008 ComArch S.A.
*******************************************************************************/
/** \brief mDictionary Manager.
 * \defgroup Manager mDictionary Manager
 *
 * Manager module is managing plugins, and separate searching proccess from GUI
 * module. It handles all dictionary engines and all dictionaries.
 *
 * \author Łukasz Pawlik \<lukasz.pawlik\@comarch.com\>
 */
/*@{*/

/** \file ws_manager.h
 * \brief mDictionary Manager main module header.
 *
 * \author Łukasz Pawlik \<lukasz.pawlik\@comarch.com\>
 */

#ifndef _WS_MANAGER
#define _WS_MANAGER

#include <dictionary_engine.h>
#include <glib-object.h>
#include <gmodule.h>
#include <ws_services.h>
#include <ws_dbus.h>

#ifdef __cplusplus
extern "C" {
#endif

/** \brief SQLite library version.
 *
 * There were two versions of libsqlite: libsqlite (OS2006) and libsqlite3 
 * (>=OS2007). According to system to which You are going to build mDictionary
 * You should specyfy SQLITE to 0 or 3. LIBRARY specify destination path for
 * used library.
 */
#if SQLITE==0
	#define LIBRARY "/usr/lib/libsqlite.so.0"
#elif SQLITE==3
	#define LIBRARY "/usr/lib/libsqlite3.so.0"
#endif

/** \brief Calculate length of given string.
 *
 * \param string pointer to the array of chars (string)
 * \return integer telling how many chars there were in the given array. (if
 * string was NULL it returns 0)
 */
#define g_strlen(string) ( NULL == (string) ? 0 : strlen(string) )

struct _WSMngSearchData;
typedef struct _WSMngSearchData WSMngSearchData;

/** \brief Creates new instance of WSMngSearchData structure */
WSMngSearchData* ws_manager_create();

/** \brief Function used to initialize D-BUS */
void ws_mng_init_dbus (WSMngSearchData *data);

/** \brief Function used to initialize manager */
void ws_mng_init (WSMngSearchData *data);

/** \brief Function used to close dictionaries and modules */
void ws_mng_close (WSMngSearchData *data);

/** \brief Function used to run and initialize main loop. */
gboolean ws_mng_start_main_loop(WSMngSearchData* serach_data);

/** \brief Structure holding all data needed by manager to work.
 *
 *
 */
struct _WSMngSearchData
{
	WSDBusData* dbus_data;
	/**< a pointer to the DBUS data for manager module */

	GMainLoop* loop;
	/**< pointer to the main loop of manager proccess */

	GArray* modules;
	/**< pointer to GArray structure which stores pointers to fuction used 
	 * to service modules loading */

	Engine* bookmark;
	/**< pointer to bookmark engine */

	GArray* dict;
	/**< pointer to GArray structure which stores pointers to particular
	 * dictionaries */ 

	gchar* word;
	/**< word which will be used to search a dictionaries */

	GArray* word_list;
	/**< GArray for whole words list returned to GUI */

	gchar* trans;
	/**< found translation */

	gchar* last_search;
	/**< last word used for searching */

	gboolean search_in_history;
	/**< flag signalig when search in history is commited */

	/* thread connected data
	 * added by Dariusz Wiechecki
	 * we are moving to GLib threads */
	GStaticMutex* action_working;
	/**< pointer to mutex; while any action is being performed this mutex
	 * is locked - it is telling if there is any active action curently */

	GStaticMutex* thread_creation;
	/**< pointer to mutex; is locked while new thread is creating. After
	 * thread is created and start to work, this mutex is unclocked */

	GStaticRecMutex* action_stop;
	/**< pointer to mutex; is locked when current job (search) should be
	 * aborted */

	/* --- pthread structures are not needed anymore:
	 * pthread_t p_thread;
	 * pthread_t p_thread_trans;
	 */

	GArray* library_path;
	/**< path to library */

	GArray* libraries;
	/**< table used for storing modules libraries */

	gboolean bookmark_mode;
	/**< flag telling if manager is in bookmark mode */

	GModule* bookmark_library;
	/**< pointer to bookmark library */
};



#ifdef __cplusplus
}
#endif
#endif
/*@}*/
