/*******************************************************************************
This file is part of mdictionary.

mdictionary is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

mdictionary is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License 
along with mdictionary; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Copyright 2006 ComArch S.A.
*******************************************************************************/
/** \addtogroup GUI
 */
/*@{*/
/** \file ws_gui_layout.c
 * \brief GUI layout definitions
 *
 * Defines the way GUI dialogues look
 * \author Anna Gadomska \<anna.gadomska\@comarch.com\>
 * \date 05-11-2007
 * \version 1.0.0
 */

#include <ws_gui_layout.h>
#include <ws_gui.h>
#include <ws_gui_callbacks.h>
#include <ws_dbus.h>
#include <libintl.h>


/** this function creates main window and it components; moreover there are 
 * signal connections;
 *
 * @param user_data - data set when the signal handler was connected 
 */
void ws_gui_create_window(gpointer user_data) 
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;
	
	ws_gui_app->welcome_note = g_strconcat("<center><font size=+2>", 
				   _("ws_ni_welcome"), "</font><br><br><IMG SRC=file:/usr/share/pixmaps/mdictionary.png>", 
				   NULL);

/*	ws_gui_app->ws_gui_w_list->ws_gui_list_item_data_string = 
						     (ws_gui_app->welcome_note);*/

	GArray *array_tmp = g_array_new (TRUE, TRUE, sizeof(gchar*));
	ws_gui_app->raw_translation = g_string_new(" ");
	ws_gui_app->last_searched = NULL;
	ws_gui_app->last_searched_in_book = NULL;
	ws_gui_app->last_word = NULL;

	/*indicates that searching banner is displayed*/
	ws_gui_app->ws_gui_banner_flag = FALSE;
	/*indicates full screen mode*/
	ws_gui_app->ws_gui_full_screen_flag = FALSE;
	/*indicates that caching is in progress*/
	ws_gui_app->caching_flag = FALSE;
	/*used for doubleclick recognition*/
	ws_gui_app->ws_gui_double_click = FALSE;
	/*indicates a that a piece of text is selected*/
	ws_gui_app->ws_gui_sel_flag = FALSE;
	/*indicates whether bookmark mode is active or not*/
	ws_gui_app->bookmark_mode = FALSE;
	/*indicates if bookmarks database is available*/
	ws_gui_app->bookmark_avail = TRUE;
	/*indicates that no translation has been found*/
	ws_gui_app->html_flag = TRUE;
	/*indicates loading banner is displayed*/
	ws_gui_app->loading = FALSE;
	/*differentiates between tap and hold and ordinary tap*/
	ws_gui_app->stop_clicked = FALSE;
	/*if dictionary is just run*/
	ws_gui_app->first_run = TRUE; 
	/*indicates that no words match a given search pattern*/
	ws_gui_app->no_words_found = TRUE;
	/*indicates that no words match a given search pattern*/
	ws_gui_app->dictionary_selection_changed = FALSE;
	
	ws_gui_app->zoom = ZOOM_DEFAULT;

	ws_gui_app->timer = g_timer_new();
	g_timer_stop(ws_gui_app->timer);

	ws_gui_app->ws_gui_history = g_array_new(TRUE, TRUE, sizeof(gchar*));
	ws_gui_app->ws_gui_history_list = g_array_new(TRUE, TRUE, sizeof(gchar*));
	ws_gui_app->ws_gui_history_iter = g_array_new(TRUE, TRUE, sizeof(gchar*));
	ws_gui_app->history_flag = FALSE;
	ws_gui_app->ws_gui_history_cur_pos = -1;
	
	ws_gui_app->ws_gui_hildon_program = 
	HILDON_PROGRAM(hildon_program_get_instance());

	g_set_application_name(PACKAGE);

	ws_gui_app->ws_gui_hildon_window = HILDON_WINDOW(hildon_window_new());

	hildon_program_add_window(ws_gui_app->ws_gui_hildon_program,
				ws_gui_app->ws_gui_hildon_window);

	ws_gui_create_main_menu(ws_gui_app);
	ws_gui_create_find_toolbar(ws_gui_app);

	ws_gui_app->ws_gui_hpane = gtk_hpaned_new();

	ws_gui_app->ws_gui_scrolledwindow_left = 
		gtk_scrolled_window_new (NULL, NULL);

	gtk_scrolled_window_set_policy(
		GTK_SCROLLED_WINDOW (ws_gui_app->ws_gui_scrolledwindow_left), 
		GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC);

	gtk_paned_set_position(GTK_PANED(ws_gui_app->ws_gui_hpane), 100);

	gtk_scrolled_window_set_placement(GTK_SCROLLED_WINDOW
				(ws_gui_app->ws_gui_scrolledwindow_left),
				GTK_CORNER_TOP_LEFT);

	ws_gui_app->ws_gui_scrolledwindow_right = 
		gtk_scrolled_window_new (NULL, NULL);

	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW
				(ws_gui_app->ws_gui_scrolledwindow_right), 
				GTK_POLICY_NEVER,
				GTK_POLICY_AUTOMATIC);

	ws_gui_app->ws_gui_w_list->ws_gui_store = 
		gtk_list_store_new (NUM_COLS, G_TYPE_STRING, G_TYPE_UINT);

	ws_gui_app->ws_gui_w_list->ws_gui_view = 
		create_view_and_model(array_tmp, ws_gui_app);

	g_array_free(array_tmp, TRUE);

	gtk_widget_show_all(GTK_WIDGET(ws_gui_app->ws_gui_w_list->ws_gui_view));

	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW
				(ws_gui_app->ws_gui_scrolledwindow_left), 
					ws_gui_app->ws_gui_w_list->ws_gui_view);

	ws_gui_app->ws_gui_html = gtk_html_new();

	ws_gui_app->p = pango_font_description_from_string("Tahoma 12");
	gtk_widget_modify_font(GTK_WIDGET(
					ws_gui_app->ws_gui_html),ws_gui_app->p);

	g_signal_connect(GTK_OBJECT (ws_gui_app->ws_gui_html),
				"url_requested",
				G_CALLBACK(ws_gui_url_requested),
				ws_gui_app);

	gtk_html_load_from_string(GTK_HTML(ws_gui_app->ws_gui_html),
			ws_gui_app->welcome_note,
			-1);
	
	gtk_html_set_magnification(GTK_HTML(ws_gui_app->ws_gui_html), 
				ws_gui_app->zoom);

	gtk_html_set_editable(GTK_HTML(ws_gui_app->ws_gui_html), FALSE); 
	gtk_html_allow_selection(GTK_HTML(ws_gui_app->ws_gui_html), TRUE);

	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW
				(ws_gui_app->ws_gui_scrolledwindow_right),
				ws_gui_app->ws_gui_html);

	/*adding vbox & label - used for notifications */

	ws_gui_app->ws_gui_list_image = gtk_image_new_from_file("/usr/share/pixmaps/ws_warning_icon.png");
	ws_gui_app->ws_gui_list_eventbox = gtk_event_box_new ();
	ws_gui_app->ws_gui_list_vbox = gtk_vbox_new(FALSE, 0);
	ws_gui_app->ws_gui_list_hbox = gtk_hbox_new(FALSE, 5);
	ws_gui_app->ws_gui_list_label = gtk_label_new(_("ws_ti_list_full"));

	ws_gui_app->p = pango_font_description_from_string("Tahoma 10");
	gtk_widget_modify_font(GTK_WIDGET(ws_gui_app->ws_gui_list_label), ws_gui_app->p);

	/*HBox packing ....*/
	gtk_box_pack_start(GTK_BOX(ws_gui_app->ws_gui_list_hbox),
			ws_gui_app->ws_gui_list_image,
			FALSE,
			FALSE,
			0);
	
	gtk_box_pack_start(GTK_BOX(ws_gui_app->ws_gui_list_hbox),
			ws_gui_app->ws_gui_list_label,
			FALSE,
			FALSE,
			0);
	
	// gtk_widget_set_size_request (ws_gui_app->ws_gui_list_label, 150, -1);

	/*VBox packing ....*/
	gtk_box_pack_start(GTK_BOX(ws_gui_app->ws_gui_list_vbox),
			ws_gui_app->ws_gui_scrolledwindow_left,
			TRUE,
			TRUE,
			0);

	gtk_box_pack_end(GTK_BOX(ws_gui_app->ws_gui_list_vbox),
			ws_gui_app->ws_gui_list_eventbox,
			FALSE,
			FALSE,
			0);
	
	gtk_container_add (GTK_CONTAINER (ws_gui_app->ws_gui_list_eventbox), 
			ws_gui_app->ws_gui_list_hbox);
	/************************************************/


	gtk_paned_pack1(GTK_PANED(ws_gui_app->ws_gui_hpane),
		ws_gui_app->ws_gui_list_vbox,
		TRUE,
		TRUE);
	gtk_paned_pack2(GTK_PANED(ws_gui_app->ws_gui_hpane),
		ws_gui_app->ws_gui_scrolledwindow_right,
		TRUE,
		TRUE);

	gtk_container_add(GTK_CONTAINER(ws_gui_app->ws_gui_hildon_window), 
			ws_gui_app->ws_gui_hpane);

	gtk_widget_show_all(GTK_WIDGET(ws_gui_app->ws_gui_hildon_window));

	gtk_widget_hide_all(GTK_WIDGET(ws_gui_app->ws_gui_list_eventbox));
	/*gtk_widget_hide_all(GTK_WIDGET(ws_gui_app->ws_gui_list_hbox));*/

	ws_gui_app->ws_gui_w_list->ws_gui_selection = 
		gtk_tree_view_get_selection (GTK_TREE_VIEW
		(ws_gui_app->ws_gui_w_list->ws_gui_view));

	gtk_tree_selection_set_mode(
				ws_gui_app->ws_gui_w_list->ws_gui_selection,
				GTK_SELECTION_BROWSE);
	
	ws_gui_app->directories = g_slist_alloc();
	ws_gui_set_bookmarks_sensitivity(ws_gui_app);

	ws_gui_app->ws_gui_banner_list_searching = hildon_banner_show_animation(
				GTK_WIDGET(ws_gui_app->ws_gui_hildon_window),
				NULL,
				_("ws_ab_searching"));

	ws_gui_app->ws_gui_banner_translation_loading = hildon_banner_show_animation(
				GTK_WIDGET(ws_gui_app->ws_gui_hildon_window),
				NULL,
				_("ws_ab_loading"));
	
//	gtk_widget_set_sensitive(GTK_WIDGET
// 			(ws_gui_app->ws_gui_menu->ws_gui_popup_bookmark),
// 			FALSE);

	gtk_widget_hide(GTK_WIDGET(ws_gui_app->ws_gui_banner_list_searching));
	gtk_widget_hide(GTK_WIDGET(ws_gui_app->ws_gui_banner_translation_loading));
}

/** this function creates a find toolbar and connects signals to the tollbars 
 * components
 *
 * @param user_data - set when the signal handler was connected
 */
void ws_gui_create_find_toolbar(gpointer user_data)
{
	WSGuiApp* ws_gui_app = (WSGuiApp*)user_data;
	
	GtkToolItem *space;
	ws_gui_app->ws_gui_hildon_find_toolbar = 
	HILDON_FIND_TOOLBAR(hildon_find_toolbar_new(_("ws_me_search_find")));
	
	ws_gui_app->ws_gui_toobar_button_hide =
	gtk_toggle_tool_button_new_from_stock(GTK_STOCK_GOTO_FIRST);
	ws_gui_app->ws_gui_toobar_button_back = 
	gtk_tool_button_new_from_stock(GTK_STOCK_GO_BACK);
	ws_gui_app->ws_gui_toobar_button_forward =
	gtk_tool_button_new_from_stock(GTK_STOCK_GO_FORWARD);
	ws_gui_app->ws_gui_toobar_button_stop = 
	gtk_tool_button_new_from_stock(GTK_STOCK_STOP);
	space = gtk_separator_tool_item_new(); 
	
	ws_gui_app->ws_gui_toobar_button_close = 
		gtk_toolbar_get_nth_item(GTK_TOOLBAR(
					ws_gui_app->ws_gui_hildon_find_toolbar),
					4);

	ws_gui_app->ws_gui_toobar_button_start = gtk_toolbar_get_nth_item(GTK_TOOLBAR(
					ws_gui_app->ws_gui_hildon_find_toolbar),
					2);

	gtk_tool_item_set_visible_horizontal(
				ws_gui_app->ws_gui_toobar_button_close,
				FALSE);
	
	gtk_toolbar_insert(GTK_TOOLBAR(ws_gui_app->ws_gui_hildon_find_toolbar),
				ws_gui_app->ws_gui_toobar_button_hide,
				0);
	gtk_toolbar_insert(GTK_TOOLBAR(ws_gui_app->ws_gui_hildon_find_toolbar),
				ws_gui_app->ws_gui_toobar_button_back,
				-1);
	gtk_toolbar_insert(GTK_TOOLBAR(ws_gui_app->ws_gui_hildon_find_toolbar),
				ws_gui_app->ws_gui_toobar_button_forward,
				-1);
	gtk_toolbar_insert(GTK_TOOLBAR(ws_gui_app->ws_gui_hildon_find_toolbar),
				space,
				-1);
	gtk_toolbar_insert(GTK_TOOLBAR(ws_gui_app->ws_gui_hildon_find_toolbar),
				ws_gui_app->ws_gui_toobar_button_stop,
				-1);
	
	g_signal_connect(G_OBJECT(ws_gui_app->ws_gui_hildon_find_toolbar), 
			"search",
			G_CALLBACK(ws_gui_search),
			ws_gui_app);
	
	g_signal_connect(G_OBJECT(ws_gui_app->ws_gui_toobar_button_hide), 
			"toggled",
			G_CALLBACK(ws_gui_words_list_hide),
			ws_gui_app);
	
	g_signal_connect(G_OBJECT(ws_gui_app->ws_gui_toobar_button_back), 
			"clicked",
			G_CALLBACK(ws_gui_history_back),
			ws_gui_app);
	
	g_signal_connect(G_OBJECT(ws_gui_app->ws_gui_toobar_button_forward), 
			"clicked",
			G_CALLBACK(ws_gui_history_next),
			ws_gui_app);
	g_signal_connect(G_OBJECT(ws_gui_app->ws_gui_toobar_button_stop), 
			"clicked",
			G_CALLBACK(ws_gui_search_stop),
			ws_gui_app);
	
	gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_toobar_button_back),
			FALSE);
	gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_toobar_button_forward),
			FALSE);
	gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_toobar_button_stop),
			FALSE);
	
	hildon_find_toolbar_highlight_entry((HildonFindToolbar*)(ws_gui_app->ws_gui_hildon_find_toolbar),
                                             FALSE);
	//gtk_toolbar_set_tooltips(GTK_TOOLBAR(ws_gui_app->ws_gui_hildon_find_toolbar), TRUE);

	hildon_window_add_toolbar(ws_gui_app->ws_gui_hildon_window,
			GTK_TOOLBAR(ws_gui_app->ws_gui_hildon_find_toolbar));
}

/** this function creates application menu and its items and connect signals to 
 * them
 *
 * @param user_data - set when the signal handler was connected
 */
void ws_gui_create_main_menu (gpointer user_data) 
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;

	ws_gui_app->ws_gui_menu->ws_gui_submenu_0 = gtk_menu_new();
	ws_gui_app->ws_gui_menu->ws_gui_submenu_1 = gtk_menu_new();
	ws_gui_app->ws_gui_menu->ws_gui_submenu_2 = gtk_menu_new();
	ws_gui_app->ws_gui_menu->ws_gui_submenu_3 = gtk_menu_new();
	ws_gui_app->ws_gui_menu->ws_gui_submenu_4 = gtk_menu_new();
	ws_gui_app->ws_gui_menu->ws_gui_main_menu = gtk_menu_new();

	ws_gui_app->ws_gui_menu->ws_gui_menu_dictionaries =
		gtk_menu_item_new_with_label(_("ws_me_dictionaries"));
	ws_gui_app->ws_gui_menu->ws_gui_menu_bookmarks =
		gtk_menu_item_new_with_label(_("ws_me_bookmarks"));
	ws_gui_app->ws_gui_menu->ws_gui_menu_edit = 
		gtk_menu_item_new_with_label(_("ws_me_edit"));
	ws_gui_app->ws_gui_menu->ws_gui_menu_view = 
		gtk_menu_item_new_with_label(_("ws_me_view"));
	ws_gui_app->ws_gui_menu->ws_gui_menu_search = 
		gtk_menu_item_new_with_label(_("ws_me_search"));
	ws_gui_app->ws_gui_menu->ws_gui_menu_about = 
		gtk_menu_item_new_with_label(_("ws_me_about"));
	ws_gui_app->ws_gui_menu->ws_gui_menu_close = 
		gtk_menu_item_new_with_label(_("ws_me_close"));
	ws_gui_app->ws_gui_menu->ws_gui_separator = 
		gtk_separator_menu_item_new();
	ws_gui_app->ws_gui_menu->ws_gui_separator1 = 
		gtk_separator_menu_item_new();

	ws_gui_app->ws_gui_menu->ws_gui_menu_load_dict = 
		gtk_menu_item_new_with_label(_("ws_me_dictionaries_load"));
	ws_gui_app->ws_gui_menu->ws_gui_menu_select_dict = 
		gtk_menu_item_new_with_label(_("ws_me_dictionaries_select"));
	ws_gui_app->ws_gui_menu->ws_gui_menu_remove_dict = 
		gtk_menu_item_new_with_label(_("ws_me_dictionaries_remove"));
	ws_gui_app->ws_gui_menu->ws_gui_menu_optimize_dict = 
		gtk_menu_item_new_with_label(_("ws_me_dictionaries_optimize"));

	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_submenu_0,
			ws_gui_app->ws_gui_menu->ws_gui_menu_load_dict);
	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_submenu_0,
			ws_gui_app->ws_gui_menu->ws_gui_menu_select_dict);
	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_submenu_0,
			ws_gui_app->ws_gui_menu->ws_gui_menu_remove_dict);
	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_submenu_0,
			ws_gui_app->ws_gui_menu->ws_gui_menu_optimize_dict);

	ws_gui_app->ws_gui_menu->ws_gui_menu_open_bookmark = 
		gtk_menu_item_new_with_label(_("ws_me_bookmarks_open"));
	ws_gui_app->ws_gui_menu->ws_gui_menu_close_bookmark = 
		gtk_menu_item_new_with_label(_("ws_me_bookmarks_close"));
	ws_gui_app->ws_gui_menu->ws_gui_menu_add_bookmark = 
		gtk_menu_item_new_with_label(_("ws_me_bookmarks_add"));
	ws_gui_app->ws_gui_menu->ws_gui_menu_remove_bookmark = 
		gtk_menu_item_new_with_label(_("ws_me_bookmarks_remove"));

	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_submenu_4,
			ws_gui_app->ws_gui_menu->ws_gui_menu_open_bookmark);
	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_submenu_4,
			ws_gui_app->ws_gui_menu->ws_gui_menu_close_bookmark);
	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_submenu_4,
			ws_gui_app->ws_gui_menu->ws_gui_menu_add_bookmark);
	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_submenu_4,
			ws_gui_app->ws_gui_menu->ws_gui_menu_remove_bookmark);

	ws_gui_app->ws_gui_menu->ws_gui_menu_copy = 
		gtk_menu_item_new_with_label(_("ws_me_edit_copy"));
	ws_gui_app->ws_gui_menu->ws_gui_menu_paste = 
		gtk_menu_item_new_with_label(_("ws_me_edit_paste"));
	ws_gui_app->ws_gui_menu->ws_gui_menu_select_all = 
		gtk_menu_item_new_with_label(_("ws_me_edit_select_all"));
	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_submenu_1,
			ws_gui_app->ws_gui_menu->ws_gui_menu_copy);
	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_submenu_1,
			ws_gui_app->ws_gui_menu->ws_gui_menu_paste);
	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_submenu_1,
			ws_gui_app->ws_gui_menu->ws_gui_menu_select_all);
	
	ws_gui_app->ws_gui_menu->ws_gui_menu_hide_list = 
	gtk_check_menu_item_new_with_label(_("ws_me_view_hide_words_list"));
	ws_gui_app->ws_gui_menu->ws_gui_menu_zoom_in = 
		gtk_menu_item_new_with_label(_("ws_me_view_zoom_in"));
	ws_gui_app->ws_gui_menu->ws_gui_menu_zoom_out = 
		gtk_menu_item_new_with_label(_("ws_me_view_zoom_out"));
	ws_gui_app->ws_gui_menu->ws_gui_menu_full_screen = 
		gtk_check_menu_item_new_with_label(_("ws_me_view_fullscreen"));
	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_submenu_2,
			ws_gui_app->ws_gui_menu->ws_gui_menu_hide_list);
	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_submenu_2,
			ws_gui_app->ws_gui_menu->ws_gui_menu_zoom_in);
	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_submenu_2,
			ws_gui_app->ws_gui_menu->ws_gui_menu_zoom_out);
	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_submenu_2,
			ws_gui_app->ws_gui_menu->ws_gui_menu_full_screen);
	
	ws_gui_app->ws_gui_menu->ws_gui_menu_find_word = 
		gtk_menu_item_new_with_label(_("ws_me_search_find"));
	ws_gui_app->ws_gui_menu->ws_gui_menu_find_next = 
		gtk_menu_item_new_with_label(_("ws_me_search_find_next"));
	ws_gui_app->ws_gui_menu->ws_gui_menu_find_prev = 
	gtk_menu_item_new_with_label(_("ws_me_search_find_prev"));
	ws_gui_app->ws_gui_menu->ws_gui_menu_stop = 
	gtk_menu_item_new_with_label(_("ws_me_search_stop"));
	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_submenu_3,
		ws_gui_app->ws_gui_menu->ws_gui_menu_find_word);
	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_submenu_3,
		ws_gui_app->ws_gui_menu->ws_gui_menu_find_next);
	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_submenu_3,
		ws_gui_app->ws_gui_menu->ws_gui_menu_find_prev);
	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_submenu_3,
		ws_gui_app->ws_gui_menu->ws_gui_menu_stop);


	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_main_menu,
		ws_gui_app->ws_gui_menu->ws_gui_menu_dictionaries);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM
			(ws_gui_app->ws_gui_menu->ws_gui_menu_dictionaries),
			ws_gui_app->ws_gui_menu->ws_gui_submenu_0);

	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_main_menu,
		ws_gui_app->ws_gui_menu->ws_gui_menu_bookmarks);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM
			(ws_gui_app->ws_gui_menu->ws_gui_menu_bookmarks),
			ws_gui_app->ws_gui_menu->ws_gui_submenu_4);

	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_main_menu,
		ws_gui_app->ws_gui_menu->ws_gui_separator1);

	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_main_menu,
		ws_gui_app->ws_gui_menu->ws_gui_menu_edit);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM
				(ws_gui_app->ws_gui_menu->ws_gui_menu_edit), 
				ws_gui_app->ws_gui_menu->ws_gui_submenu_1);

	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_main_menu,
			ws_gui_app->ws_gui_menu->ws_gui_menu_view);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM
				(ws_gui_app->ws_gui_menu->ws_gui_menu_view), 
				ws_gui_app->ws_gui_menu->ws_gui_submenu_2);

	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_main_menu,
			ws_gui_app->ws_gui_menu->ws_gui_menu_search);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM
				(ws_gui_app->ws_gui_menu->ws_gui_menu_search), 
				ws_gui_app->ws_gui_menu->ws_gui_submenu_3);

	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_main_menu,
			ws_gui_app->ws_gui_menu->ws_gui_separator);
	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_main_menu,
			ws_gui_app->ws_gui_menu->ws_gui_menu_about);
	gtk_menu_append(ws_gui_app->ws_gui_menu->ws_gui_main_menu,
			ws_gui_app->ws_gui_menu->ws_gui_menu_close);

	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_load_dict),
			"activate",
			G_CALLBACK(ws_gui_dictionary_loader),
			ws_gui_app);
	g_signal_connect(G_OBJECT(
			ws_gui_app->ws_gui_menu->ws_gui_menu_select_dict),
			"activate",
			G_CALLBACK(ws_gui_dictionary_chooser),
			ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_remove_dict),
			"activate",
			G_CALLBACK(ws_gui_dictionary_remover),
			ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_optimize_dict),
			"activate",
			G_CALLBACK(ws_gui_dictionary_optimizer), ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_open_bookmark),
			"activate",
			G_CALLBACK(ws_gui_dictionary_open_bookmark), 
			ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_close_bookmark),
			"activate",
			G_CALLBACK(ws_gui_dictionary_close_bookmark), 
			ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_add_bookmark),
			"activate",
			G_CALLBACK(ws_gui_dictionary_add_bookmark), 
			ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_remove_bookmark),
			"activate",
			G_CALLBACK(ws_gui_dictionary_remove_bookmark), 
			ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_copy),
			"activate", 
			GTK_SIGNAL_FUNC(ws_gui_html_copy),
			ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_paste),
			"activate",
			GTK_SIGNAL_FUNC(ws_gui_html_paste),
			ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_select_all),
			"activate",
			GTK_SIGNAL_FUNC(ws_gui_html_select_all),
			ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_hide_list),
			"activate",
			GTK_SIGNAL_FUNC(ws_gui_words_list_hide_from_menu),
			ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_zoom_in),
			"activate",
			G_CALLBACK(ws_gui_html_zoom_in),
			ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_zoom_out),
			"activate",
			G_CALLBACK(ws_gui_html_zoom_out),
			ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_full_screen),
			"activate",
			GTK_SIGNAL_FUNC(ws_gui_full_screen),
			ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_find_word),
			"activate",
			GTK_SIGNAL_FUNC(ws_gui_search),
			ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_find_next),
			"activate",
			GTK_SIGNAL_FUNC(ws_gui_history_next),
			ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_find_prev),
			"activate",
			GTK_SIGNAL_FUNC(ws_gui_history_back),
			ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_stop),
			"activate",
			GTK_SIGNAL_FUNC(ws_gui_search_stop),
			ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_about), 
			"activate",
			G_CALLBACK(ws_gui_about_application),
			ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_close),
			"activate", 
			G_CALLBACK(ws_gui_menu_quit),
			ws_gui_app);

	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_edit),
			"activate",
			GTK_SIGNAL_FUNC(ws_gui_check_clipboard),
			ws_gui_app);

	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_dictionaries),
			"activate",
			GTK_SIGNAL_FUNC(ws_gui_dict_availablity),
			ws_gui_app);

	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_menu_bookmarks),
			"activate",
			GTK_SIGNAL_FUNC(ws_gui_set_bookmark_menu_items),
			ws_gui_app);

	hildon_window_set_menu(HILDON_WINDOW(ws_gui_app->ws_gui_hildon_window),
			GTK_MENU(
				ws_gui_app->ws_gui_menu->ws_gui_main_menu));

	gtk_widget_set_sensitive(GTK_WIDGET
				(ws_gui_app->ws_gui_menu->ws_gui_menu_stop),
				FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET
			(ws_gui_app->ws_gui_menu->ws_gui_menu_find_next),
			FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET
			(ws_gui_app->ws_gui_menu->ws_gui_menu_find_prev),
			FALSE);

	
	
	gtk_widget_show_all(ws_gui_app->ws_gui_menu->ws_gui_main_menu);
}


/** this function creates contex popup menu and its items and connect signals to 
 * them
 *
 * @param user_data - set when the signal handler was connected
 */
void ws_gui_create_popup_menu (gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;
	
	ws_gui_app->ws_gui_menu->ws_gui_popup_menu = gtk_menu_new();
	ws_gui_app->ws_gui_menu->ws_gui_popup_submenu = gtk_menu_new();
	ws_gui_app->ws_gui_menu->ws_gui_separator = 
						gtk_separator_menu_item_new();
	
	ws_gui_app->ws_gui_menu->ws_gui_popup_search = 
		gtk_menu_item_new_with_label(_("ws_mp_search"));
	gtk_menu_append((ws_gui_app->ws_gui_menu->ws_gui_popup_menu),
			ws_gui_app->ws_gui_menu->ws_gui_popup_search);
	gtk_menu_append ((ws_gui_app->ws_gui_menu->ws_gui_popup_menu),
			gtk_separator_menu_item_new());
	ws_gui_app->ws_gui_menu->ws_gui_popup_bookmark = 
		gtk_menu_item_new_with_label(_("ws_mp_add_bookmark"));
	gtk_menu_append ((ws_gui_app->ws_gui_menu->ws_gui_popup_menu),
			ws_gui_app->ws_gui_menu->ws_gui_popup_bookmark);
	
	ws_gui_app->ws_gui_menu->ws_gui_popup_copy = 
		gtk_menu_item_new_with_label(_("ws_mp_edit_copy"));
	gtk_menu_append((ws_gui_app->ws_gui_menu->ws_gui_popup_submenu),
			ws_gui_app->ws_gui_menu->ws_gui_popup_copy);
	
	ws_gui_app->ws_gui_menu->ws_gui_popup_paste = 
		gtk_menu_item_new_with_label(_("ws_mp_edit_paste"));
	gtk_menu_append((ws_gui_app->ws_gui_menu->ws_gui_popup_submenu),
			ws_gui_app->ws_gui_menu->ws_gui_popup_paste);
	
	ws_gui_app->ws_gui_menu->ws_gui_popup_select_all = 
		gtk_menu_item_new_with_label(_("ws_mp_edit_select_all"));
	gtk_menu_append((ws_gui_app->ws_gui_menu->ws_gui_popup_submenu),
			ws_gui_app->ws_gui_menu->ws_gui_popup_select_all);
	
	gtk_widget_show_all(ws_gui_app->ws_gui_menu->ws_gui_popup_submenu);
	
	ws_gui_app->ws_gui_menu->ws_gui_popup_edit = 
			gtk_menu_item_new_with_label (_("ws_mp_edit"));
	gtk_menu_append ((ws_gui_app->ws_gui_menu->ws_gui_popup_menu),
			ws_gui_app->ws_gui_menu->ws_gui_popup_edit);
	
	gtk_menu_append ((ws_gui_app->ws_gui_menu->ws_gui_popup_menu),
			ws_gui_app->ws_gui_menu->ws_gui_separator);
	
	ws_gui_app->ws_gui_menu->ws_gui_popup_zoom_in = 
		gtk_menu_item_new_with_label (_("ws_mp_zoom_in"));
	gtk_menu_append ((ws_gui_app->ws_gui_menu->ws_gui_popup_menu),
			ws_gui_app->ws_gui_menu->ws_gui_popup_zoom_in);
	
	ws_gui_app->ws_gui_menu->ws_gui_popup_zoom_out =
		gtk_menu_item_new_with_label (_("ws_mp_zoom_out"));
	gtk_menu_append ((ws_gui_app->ws_gui_menu->ws_gui_popup_menu),
			ws_gui_app->ws_gui_menu->ws_gui_popup_zoom_out);
	
	gtk_menu_item_set_submenu(GTK_MENU_ITEM
				(ws_gui_app->ws_gui_menu->ws_gui_popup_edit),
				ws_gui_app->ws_gui_menu->ws_gui_popup_submenu);

	struct _GtkHTML *tmp = (struct _GtkHTML *)(ws_gui_app->ws_gui_html);

	if (html_engine_is_selection_active(tmp->engine) == TRUE)
	{
		gtk_widget_set_sensitive(GTK_WIDGET
				(ws_gui_app->ws_gui_menu->ws_gui_popup_search),
				TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET
				(ws_gui_app->ws_gui_menu->ws_gui_popup_copy),
				TRUE);
	}
	else 
	{
		gtk_widget_set_sensitive(GTK_WIDGET
				(ws_gui_app->ws_gui_menu->ws_gui_popup_search),
				FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET
				(ws_gui_app->ws_gui_menu->ws_gui_popup_copy),
				FALSE);
	}
	
	if (ws_gui_app->html_flag == FALSE)
	{
		gtk_widget_set_sensitive(GTK_WIDGET
				(ws_gui_app->ws_gui_menu->ws_gui_popup_select_all),
				FALSE);
	}
	else
	{
		gtk_widget_set_sensitive(GTK_WIDGET
				(ws_gui_app->ws_gui_menu->ws_gui_popup_select_all),
				TRUE);
	}	
	
	gtk_widget_show_all(ws_gui_app->ws_gui_menu->ws_gui_popup_menu);
	
	
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_popup_search),
			"activate",
			G_CALLBACK(ws_gui_popup_search),
			ws_gui_app);
	
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_popup_bookmark),
			"activate",
			G_CALLBACK(ws_gui_dictionary_add_bookmark),
			ws_gui_app);
	
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_popup_copy),
			"activate", 
			G_CALLBACK(ws_gui_html_copy),
			ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_popup_paste),
			"activate",
			G_CALLBACK(ws_gui_html_paste),
			ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_popup_select_all),
			"activate",
			G_CALLBACK(ws_gui_html_select_all),
			ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_popup_zoom_in),
			"activate",
			G_CALLBACK(ws_gui_html_zoom_in), ws_gui_app);
	g_signal_connect(G_OBJECT
			(ws_gui_app->ws_gui_menu->ws_gui_popup_zoom_out),
			"activate",
			G_CALLBACK(ws_gui_html_zoom_out),
			ws_gui_app);
	
	if (gtk_clipboard_wait_for_text(ws_gui_app->ws_gui_clipboard) != NULL)
	{
		gtk_widget_set_sensitive(GTK_WIDGET
				(ws_gui_app->ws_gui_menu->ws_gui_popup_paste),
				TRUE);
	}
	else
	{
		gtk_widget_set_sensitive(GTK_WIDGET
				(ws_gui_app->ws_gui_menu->ws_gui_popup_paste),
				FALSE);
	}
	
	
	if (ws_gui_app->bookmark_avail == FALSE)
	{
	gtk_widget_set_sensitive(GTK_WIDGET
			(ws_gui_app->ws_gui_menu->ws_gui_popup_bookmark),
			FALSE);
	}
	
	else
	{
	if (ws_gui_app->bookmark_mode == TRUE)
	{
		gtk_widget_set_sensitive(GTK_WIDGET
			(ws_gui_app->ws_gui_menu->ws_gui_popup_bookmark),
			FALSE);
	}
	else
	{
		g_strstrip(ws_gui_app->last_word);
		//g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "\n\nlast_word: %s\n\n", ws_gui_app->last_word);
		
		if (ws_gui_app->last_word == NULL || 
		    (ws_gui_app->last_word[0] == '\0'))
		{
			gtk_widget_set_sensitive(GTK_WIDGET
			(ws_gui_app->ws_gui_menu->ws_gui_popup_bookmark),
			FALSE);
		}
		else
		{
			gtk_widget_set_sensitive(GTK_WIDGET
			(ws_gui_app->ws_gui_menu->ws_gui_popup_bookmark),
			TRUE);
		}
	}
	}
	gtk_menu_popup(GTK_MENU(ws_gui_app->ws_gui_menu->ws_gui_popup_menu),
			NULL,
			NULL,
			NULL,
			ws_gui_app,
			0,
			0);
}

/** this function creates dialog popup window and allows to select dictionaries
 *  which user wants to use
 *
 * @param menuitem - the object which received the signal
 * @param user_data - set when the signal handler was connected
 */
void ws_gui_dictionary_chooser(GtkMenuItem *menuitem, gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;
	GtkWidget *dialog1;
	GtkWidget *dialog_vbox1;
	GtkWidget *scrolledwindow1;
	GtkWidget *viewport1;
	GtkWidget *vbox1;

	GtkWidget *checkbutton;
	GtkWidget *cancel_button;
	GtkWidget *ok_button;
	guint i =0;
	gchar* string;
	gchar* path;
	const gchar* name;
	GArray *checkboxes;

	checkboxes = g_array_new (TRUE, TRUE, sizeof(GtkWidget*));

	if (gconf_client_dir_exists(ws_gui_app->client, 
				GCONF_PATH, NULL) == TRUE)
	{
	ws_gui_app->directories = 
		gconf_client_all_dirs(ws_gui_app->client,
					GCONF_PATH,
					NULL);
	}
	else 
	{
	g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO,
		"\n.::GUI::. /apps/mdictionary/dictionaries does not exist!!");
	}

	dialog1 = gtk_dialog_new ();
	gtk_window_set_title (GTK_WINDOW (dialog1), 
			(_("ws_ti_choose_dictionaries_title")));
	gtk_window_set_type_hint (GTK_WINDOW (dialog1),
				GDK_WINDOW_TYPE_HINT_DIALOG);
	gtk_window_set_default_size(GTK_WINDOW (dialog1), 300, 200);

	dialog_vbox1 = GTK_DIALOG (dialog1)->vbox;
	gtk_widget_show (dialog_vbox1);

	scrolledwindow1 = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_show (scrolledwindow1);
	gtk_box_pack_start (GTK_BOX (dialog_vbox1),
			scrolledwindow1,
			TRUE,
			TRUE,
			0);

	viewport1 = gtk_viewport_new (NULL, NULL);
	gtk_widget_show (viewport1);
	gtk_container_add (GTK_CONTAINER (scrolledwindow1), viewport1);

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox1);
	gtk_container_add (GTK_CONTAINER (viewport1), vbox1);

	for (i=0; i< g_slist_length(ws_gui_app->directories);i++)
	{
	string = (gchar*)g_slist_nth_data(ws_gui_app->directories, i); 
	name = g_path_get_basename(string);
	checkbutton = gtk_check_button_new_with_label(name);
	path = g_strconcat(string, "/active", NULL);

	if (gconf_client_get_bool(ws_gui_app->client, 
					path, 
					NULL) == TRUE)
	{
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(
						checkbutton),
						TRUE);
	}
	else 
	{
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(
						checkbutton),
						FALSE);
	}

		g_array_append_val(checkboxes, checkbutton);
		gtk_box_pack_start (GTK_BOX (vbox1), 
				checkbutton, 
				FALSE, 
				FALSE, 
				0);
		
		if (strcmp(name, "bookmarks") == 0)
		{
			if (ws_gui_app->bookmark_avail == FALSE) 
			{
			gtk_widget_set_sensitive(GTK_WIDGET(checkbutton),
						FALSE);
			}
		}

		gtk_widget_show (checkbutton);
	}

	cancel_button = gtk_button_new_from_stock (_("ws_db_cancel"));
	gtk_widget_show (cancel_button);
	gtk_dialog_add_action_widget (GTK_DIALOG (dialog1),
					cancel_button,
					GTK_RESPONSE_CANCEL);
	
	ok_button = gtk_button_new_from_stock (_("ws_db_ok"));
	gtk_widget_show (ok_button);
	gtk_dialog_add_action_widget(GTK_DIALOG (dialog1),
					ok_button,
					GTK_RESPONSE_OK);
	
	i = gtk_dialog_run(GTK_DIALOG(dialog1));
	
	if (i == GTK_RESPONSE_OK) 
	{
		for(i=0;i<(checkboxes->len);i++)
		{
			if (gtk_toggle_button_get_active(
				GTK_TOGGLE_BUTTON(g_array_index(checkboxes,
						GtkWidget *, i))) == TRUE)
			{
				name = gtk_button_get_label(GTK_BUTTON(
				g_array_index(checkboxes, GtkWidget *, i)));
				
				path=g_strconcat(GCONF_PATH,
						"/",
						name,
						"/active",
						NULL);
				gconf_client_set_bool(ws_gui_app->client,
							path,
							TRUE,
							NULL);
			}
	
			if (gtk_toggle_button_get_active(
				GTK_TOGGLE_BUTTON(g_array_index(checkboxes,
						GtkWidget *, i))) == FALSE)
			{
				name = gtk_button_get_label(GTK_BUTTON(
				g_array_index(checkboxes, GtkWidget *, i)));
				
				path=g_strconcat(GCONF_PATH,
						"/",
						name,
						"/active",
						NULL);
				gconf_client_set_bool(ws_gui_app->client,
							path,
							FALSE,
							NULL);
			}
		}
	
		ws_dbus_notify(ws_gui_app->dbus_data, 
				WS_DBUS_INFO_CONFIG_CHANGED);

		ws_gui_app->dictionary_selection_changed = TRUE;

		if(ws_gui_app->ws_gui_history->len >= 0)
		{
		ws_gui_clear_array(ws_gui_app->ws_gui_history);
		ws_gui_clear_array(ws_gui_app->ws_gui_history_list);
		ws_gui_clear_array(ws_gui_app->ws_gui_history_iter);
		ws_gui_app->ws_gui_history_cur_pos = -1;
		}
		ws_gui_check_history(ws_gui_app);
	
		gtk_widget_destroy (dialog1);
		ws_refresh_search_results(ws_gui_app); //refreshing words list
	}
	else if (i == GTK_RESPONSE_CANCEL) 
	{
		gtk_widget_destroy (dialog1);
	}
	
	g_slist_free(ws_gui_app->directories);
}

/** this function creates dialog popup window and allows optimize dictionaries
 *
 * @param menuitem - the object which received the signal
 * @param user_data - set when the signal handler was connected
 */
void ws_gui_dictionary_optimizer(GtkMenuItem *menuitem, gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;

	GtkWidget *dialog1;
	GtkWidget *dialog_vbox1;
	GtkWidget *scrolledwindow1;
	GtkWidget *viewport1;
	GtkWidget *vbox1;

	GtkWidget *checkbutton;
	GtkWidget *cancel_button;
	GtkWidget *ok_button;
	guint i =0;
	gchar* string;
	gchar* path;
	const gchar* name;
	GArray *checkboxes;

	checkboxes = g_array_new (TRUE, TRUE, sizeof(GtkWidget*));

	//check if a gconf directory exists
	if (gconf_client_dir_exists(ws_gui_app->client, 
				GCONF_PATH, 
				NULL) == TRUE)
	{
		ws_gui_app->directories = 
		gconf_client_all_dirs(ws_gui_app->client, GCONF_PATH, NULL);
	}
	else 
	{
		g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO,
		"\n.::GUI::. /apps/mdictionary/dictionaries does not exist!!");
	}

	for (i=0; i< g_slist_length(ws_gui_app->directories);i++)
	{
		string = (gchar*)g_slist_nth_data(ws_gui_app->directories, i); 
		name = g_path_get_basename(string);

		path = g_strconcat(string, "/optimized", NULL);
		if (gconf_client_get_bool (ws_gui_app->client,
					path, 
					NULL) == FALSE)
		{
			checkbutton = gtk_check_button_new_with_label(name);
			g_array_append_val(checkboxes, checkbutton);
		}
	}
	if (checkboxes->len <= 0)
	{
		ws_gui_app->ws_message_dialog = gtk_message_dialog_new(
				GTK_WINDOW(ws_gui_app->ws_gui_hildon_window),
				GTK_DIALOG_DESTROY_WITH_PARENT,
				GTK_MESSAGE_ERROR,
				GTK_BUTTONS_OK,
				_("ws_ni_no_dictionaries_to_optimize"));
	gtk_widget_show_all(ws_gui_app->ws_message_dialog);
	if (gtk_dialog_run (GTK_DIALOG (
			ws_gui_app->ws_message_dialog)) == GTK_RESPONSE_OK)
	{
			gtk_widget_destroy(ws_gui_app->ws_message_dialog);
	}
	}
	else
	{
		dialog1 = gtk_dialog_new ();
		gtk_window_set_title(GTK_WINDOW (dialog1),
				(_("ws_ti_choose_dictionaries_title")));
		gtk_window_set_type_hint(GTK_WINDOW (dialog1),
					GDK_WINDOW_TYPE_HINT_DIALOG);
		gtk_window_set_default_size(GTK_WINDOW (dialog1), 300, 200);

		dialog_vbox1 = GTK_DIALOG (dialog1)->vbox;
		gtk_widget_show (dialog_vbox1);

		scrolledwindow1 = gtk_scrolled_window_new (NULL, NULL);
		gtk_widget_show (scrolledwindow1);
		gtk_box_pack_start(GTK_BOX (dialog_vbox1),
				scrolledwindow1,
				TRUE,
				TRUE,
				0);

		viewport1 = gtk_viewport_new (NULL, NULL);
		gtk_widget_show (viewport1);
		gtk_container_add (GTK_CONTAINER (scrolledwindow1), viewport1);

		vbox1 = gtk_vbox_new (FALSE, 0);
		gtk_widget_show (vbox1);
		gtk_container_add (GTK_CONTAINER (viewport1), vbox1);
		for (i=0; i<checkboxes->len; i++)
		{
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(
				g_array_index(checkboxes, GtkWidget*, i)),
				FALSE);
			gtk_box_pack_start(GTK_BOX (vbox1),
				g_array_index(checkboxes, GtkWidget*, i),
				FALSE,
				FALSE,
				0);
			gtk_widget_show(g_array_index(checkboxes, 
						GtkWidget*, 
						i));
		}
		cancel_button = gtk_button_new_from_stock (_("ws_db_cancel"));
		gtk_widget_show (cancel_button);
		gtk_dialog_add_action_widget(GTK_DIALOG (dialog1),
					cancel_button,
					GTK_RESPONSE_CANCEL);
		ok_button = gtk_button_new_from_stock (_("ws_db_ok"));
		gtk_widget_show (ok_button);
		gtk_dialog_add_action_widget(GTK_DIALOG (dialog1),
					ok_button,
					GTK_RESPONSE_OK);
		i = gtk_dialog_run(GTK_DIALOG(dialog1));
		if (i == GTK_RESPONSE_OK) 
		{
			for(i=0;i<(checkboxes->len);i++)
			{
				if (gtk_toggle_button_get_active(
				GTK_TOGGLE_BUTTON(
					g_array_index(checkboxes,
						GtkWidget *, i))) == TRUE)
				{
					name = gtk_button_get_label(GTK_BUTTON(
						g_array_index(checkboxes,
							GtkWidget *,
							i)));
					path=g_strconcat(GCONF_PATH,
							"/",
							name,
							"/optimized",
							NULL);
					gconf_client_set_bool(
							ws_gui_app->client,
							path,
							TRUE,
							NULL);
				}
				if (gtk_toggle_button_get_active(
				GTK_TOGGLE_BUTTON(
				g_array_index(checkboxes,
						GtkWidget *, i))) == FALSE)
				{
					name = gtk_button_get_label(GTK_BUTTON
						(g_array_index(checkboxes,
								GtkWidget*,
								i)));
					path=g_strconcat(GCONF_PATH,
							"/",
							name,
							"/optimized",
							NULL);
					gconf_client_set_bool(
							ws_gui_app->client,
							path,
							FALSE,
							NULL);
				}
			}
			ws_dbus_notify(ws_gui_app->dbus_data,
				WS_DBUS_INFO_CONFIG_CHANGED);
			gtk_widget_destroy (dialog1);
		}
		else if (i == GTK_RESPONSE_CANCEL) 
		{
			gtk_widget_destroy (dialog1);
		}
	}
	g_slist_free(ws_gui_app->directories);
}

/** this function creates dialog popup window which allows to load a new
 *dictionary file into application
 *
 * @param menuitem - the object which received the signal
 * @param user_data - set when the signal handler was connected
 */
void ws_gui_dictionary_loader(GtkMenuItem *menuitem, gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;

	GtkWidget *dialog;
//      GtkWidget *dialog2;	

	dialog = hildon_file_chooser_dialog_new(
				GTK_WINDOW(ws_gui_app->ws_gui_hildon_window),
				GTK_FILE_CHOOSER_ACTION_OPEN);

	gtk_window_set_default_size(GTK_WINDOW (dialog), 200, 200);
	gtk_file_chooser_set_local_only (GTK_FILE_CHOOSER(dialog), FALSE);

	//gtk_widget_show(GTK_WIDGET(dialog));

	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
	{ 
		gchar *filename;
		gchar *name;
	//   gchar *path;
		gint len;
	//  gchar *filepath;

		gchar* tmp;
		
		filename = gtk_file_chooser_get_filename(
						GTK_FILE_CHOOSER (dialog));
		name = g_path_get_basename(filename);
		len = strlen(filename) - strlen(name) -1;
		tmp = g_strndup (filename, len);
				
		
		tmp = g_path_get_basename(tmp);
		
		if (g_str_has_suffix(name, ".xdxf") 
			|| g_str_has_suffix(name, ".idx") 
			|| g_str_has_suffix(name, ".ifo")
			|| g_str_has_suffix(name, ".idx.gz")
			|| g_str_has_suffix(name, ".dict")
			|| g_str_has_suffix(name, ".dict.dz"))
		{
			g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "\n:::GUUUUIII GUIII::: check");
			if (ws_gui_load_dict(filename, ws_gui_app) == TRUE)
			{
				gtk_widget_destroy(dialog);
				ws_refresh_search_results(ws_gui_app);
				hildon_banner_show_information(GTK_WIDGET(
					ws_gui_app->ws_gui_hildon_window),
					NULL,
				_("ws_ni_dictionary_added"));
			} 
		}
		else if (g_str_has_suffix(name, ".bz2"))
		{	
			ws_dbus_client_extract_dictionary(ws_gui_app->dbus_data, filename);
			gtk_widget_destroy(dialog);
			ws_refresh_search_results(ws_gui_app);
		}
		else
		{
			gtk_widget_destroy (dialog);
			hildon_banner_show_information(GTK_WIDGET(
					ws_gui_app->ws_gui_hildon_window),
					NULL,
				_("ws_ni_dictionary_wrong_file"));
		}

	}
	else 
	{
		gtk_widget_destroy (dialog);
	}
	
}

/** this function creates dialog popup window which allows to load dictionary
 *
 * @param filename - the object which received the signal
 * @param user_data - set when the signal handler was connected
 */
gboolean ws_gui_load_dict(gchar *filename, gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;

	GtkWidget *dialog1;
	GtkWidget *dialog2;
	gchar *name = g_path_get_basename(filename);
	gchar *path;
	gint len;
	gchar *filepath;

	len = strlen(filename) - strlen(name) -1;
	gchar *tmp = g_strndup (filename, len);;
	tmp = g_path_get_basename(tmp);
	

	filepath = g_strndup (filename, len);
	path = g_strconcat(GCONF_PATH, "/", tmp, "/path", NULL);
	gconf_client_set_string(ws_gui_app->client,
				path,
				filepath,
				NULL);
	g_free(path); 
	path = NULL;
	path = g_strconcat(GCONF_PATH,
				"/",
				tmp,
				"/active",
				NULL);
	
	dialog1 = gtk_message_dialog_new(GTK_WINDOW(
					ws_gui_app->ws_gui_hildon_window),
					GTK_DIALOG_DESTROY_WITH_PARENT,
					GTK_MESSAGE_QUESTION,
					GTK_BUTTONS_YES_NO,
				_("ws_ni_dictionaries_activation_question"));

	gtk_widget_show_all(GTK_WIDGET(dialog1));

	if (gtk_dialog_run (GTK_DIALOG (dialog1)) == GTK_RESPONSE_YES)
	{
		gconf_client_set_bool(ws_gui_app->client,
					path,
					TRUE,
					NULL);
	}
	else 
	{
		gconf_client_set_bool(ws_gui_app->client,
					path,
					FALSE,
					NULL);
	}
	gtk_widget_destroy (dialog1);
	g_free(path); 
	path = NULL;

	path = g_strconcat(GCONF_PATH, "/", tmp, "/optimized", NULL);

	dialog2 = gtk_message_dialog_new(
		GTK_WINDOW(ws_gui_app->ws_gui_hildon_window),
		GTK_DIALOG_DESTROY_WITH_PARENT,
		GTK_MESSAGE_QUESTION,
		GTK_BUTTONS_YES_NO,
		_("ws_ni_dictionaries_optimalization_question"));

	gtk_widget_show_all(GTK_WIDGET(dialog2));

	if (gtk_dialog_run (GTK_DIALOG (dialog2)) == GTK_RESPONSE_YES)
	{
		gconf_client_set_bool(ws_gui_app->client,
					path,
					TRUE,
					NULL);
	}
	else 
	{
		gconf_client_set_bool(ws_gui_app->client,
					path,
					FALSE,
					NULL);
	}
	gtk_widget_destroy (dialog2);

	if(ws_gui_app->ws_gui_history->len > 0)
	{
		ws_gui_clear_array(ws_gui_app->ws_gui_history);
		ws_gui_clear_array(ws_gui_app->ws_gui_history_list);
		ws_gui_clear_array(ws_gui_app->ws_gui_history_iter);
		ws_gui_app->ws_gui_history_cur_pos = -1;
	}
	ws_gui_check_history(ws_gui_app);

	ws_dbus_notify(ws_gui_app->dbus_data,
				WS_DBUS_INFO_CONFIG_CHANGED);
	
	return TRUE;
}

/** this function creates dialog popup window which allows to remove dictionary
 *  which is no longer used
 *
 * @param menuitem - the object which received the signal
 * @param user_data - set when the signal handler was connected
 */
void ws_gui_dictionary_remover(GtkMenuItem *menuitem, gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;

	GtkWidget *dialog1;
	GtkWidget *dialog_vbox1;
	GtkWidget *scrolledwindow1;
	GtkWidget *viewport1;
	GtkWidget *vbox1;
	GtkWidget *checkbutton;
	GtkWidget *cancel_button;
	GtkWidget *ok_button;
	guint i =0;
	gchar* string;
	gchar* path;
	const gchar* name;
	GArray *checkboxes;

	checkboxes = g_array_new (TRUE, TRUE, sizeof(GtkWidget*));

	if (gconf_client_dir_exists(ws_gui_app->client, GCONF_PATH, NULL) 
									== TRUE)
	{
		ws_gui_app->directories = gconf_client_all_dirs(
							ws_gui_app->client,
							GCONF_PATH,
							NULL);
	}
	else 
	{
		g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO,
		"\n.::GUI::. /apps/mdictionary/dictionaries does not exist!!");
	}

	dialog1 = gtk_dialog_new ();
	gtk_window_set_title (GTK_WINDOW (dialog1), 
			(_("ws_ti_remove_dictionaries_title")));
	gtk_window_set_type_hint (GTK_WINDOW (dialog1),
				GDK_WINDOW_TYPE_HINT_DIALOG);
	gtk_window_set_default_size(GTK_WINDOW (dialog1), 300, 200);

	dialog_vbox1 = GTK_DIALOG (dialog1)->vbox;
	gtk_widget_show (dialog_vbox1);

	scrolledwindow1 = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_show (scrolledwindow1);
	gtk_box_pack_start (GTK_BOX (dialog_vbox1),
			scrolledwindow1,
			TRUE,
			TRUE,
			0);

	viewport1 = gtk_viewport_new (NULL, NULL);
	gtk_widget_show (viewport1);
	gtk_container_add (GTK_CONTAINER (scrolledwindow1), viewport1);

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox1);
	gtk_container_add (GTK_CONTAINER (viewport1), vbox1);

	for (i=0; i< g_slist_length(ws_gui_app->directories);i++)
	{
		string = (gchar*)g_slist_nth_data(ws_gui_app->directories, i); 
		name = g_path_get_basename(string);
		checkbutton = gtk_check_button_new_with_label(name);

		g_array_append_val(checkboxes, checkbutton);
		gtk_box_pack_start (GTK_BOX (vbox1),
				checkbutton,
				FALSE,
				FALSE,
				0);
		
		if (strcmp(name, "bookmarks") == 0)
		{
		gtk_widget_set_sensitive(GTK_WIDGET(checkbutton), FALSE);
		}
		gtk_widget_show (checkbutton);
	}

	cancel_button = gtk_button_new_from_stock (_("ws_db_cancel"));
	gtk_widget_show (cancel_button);
	gtk_dialog_add_action_widget (GTK_DIALOG (dialog1),
				cancel_button,
				GTK_RESPONSE_CANCEL);

	ok_button = gtk_button_new_from_stock 
			(_("ws_bd_remove_dictionaries_remove_selected"));
	gtk_widget_show (ok_button);
	gtk_dialog_add_action_widget(GTK_DIALOG (dialog1),
				ok_button,
				GTK_RESPONSE_OK);

	i = gtk_dialog_run(GTK_DIALOG(dialog1));

	if (i == GTK_RESPONSE_OK) 
	{
		gboolean tmp = FALSE;
		for(i=0;i<(checkboxes->len);i++)
		{
			if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
			g_array_index(checkboxes, GtkWidget *, i))) == TRUE)
			{

				name = gtk_button_get_label(GTK_BUTTON(
				g_array_index(checkboxes, GtkWidget *, i)));
				path=g_strconcat(GCONF_PATH, "/", name, NULL);
				gconf_client_recursive_unset(ws_gui_app->client,
					path,
					GCONF_UNSET_INCLUDING_SCHEMA_NAMES,
					NULL);
					tmp = TRUE;
			} 
		}

		if(ws_gui_app->ws_gui_history->len >= 0)
		{
			ws_gui_clear_array(ws_gui_app->ws_gui_history);
			ws_gui_clear_array(ws_gui_app->ws_gui_history_list);
			ws_gui_clear_array(ws_gui_app->ws_gui_history_iter);
			ws_gui_app->ws_gui_history_cur_pos = -1;
		}
		ws_gui_check_history(ws_gui_app);		

		ws_dbus_notify(ws_gui_app->dbus_data, 
			WS_DBUS_INFO_CONFIG_CHANGED);
	
		ws_gui_app->dictionary_selection_changed = TRUE;

		if (tmp)
			{
			hildon_banner_show_information(GTK_WIDGET(
					ws_gui_app->ws_gui_hildon_window),
					NULL,
					_("ws_ib_dictionary_removed"));
			}
		else
			{
			hildon_banner_show_information(GTK_WIDGET(
					ws_gui_app->ws_gui_hildon_window),
					NULL,
					_("ws_ib_dictionary_nothing_to_remove"));
			}

		gtk_widget_destroy (dialog1);
		ws_refresh_search_results(ws_gui_app);
	}
	else if (i == GTK_RESPONSE_CANCEL) 
	{
		gtk_widget_destroy (dialog1);
	}

	g_slist_free(ws_gui_app->directories);
}

/** this function creates dialog popup window which displays information about
 * application (about window)
 *
 * @param menuitem - the object which received the signal
 * @param user_data - set when the signal handler was connected
 */
void ws_gui_about_application(GtkMenuItem *menuitem, gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;

	gtk_widget_set_sensitive(GTK_WIDGET
				(ws_gui_app->ws_gui_hildon_find_toolbar),
				FALSE);

	GtkWidget *about_dialog;
	GtkWidget *dialog_vbox1;
	GtkWidget *vbox1;
	GtkWidget *image;
	GtkWidget *label2;
	GtkWidget *label3;
	GtkWidget *label4;
	GtkWidget *scrolledwindow;
	GtkWidget *textview;
	GtkWidget *button1;

	const gchar *ws_version;

	about_dialog = gtk_dialog_new ();
	gtk_window_set_title (GTK_WINDOW (about_dialog), 
			_("ws_ti_about_title"));
	gtk_window_set_default_size(GTK_WINDOW (about_dialog), 350, 350);

	dialog_vbox1 = GTK_DIALOG (about_dialog)->vbox;
	gtk_widget_show (dialog_vbox1);

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox1);
	gtk_box_pack_start (GTK_BOX (dialog_vbox1), vbox1, TRUE, TRUE, 0);

	image = gtk_image_new_from_file ("/usr/share/pixmaps/mdictionary_about.png");
	gtk_box_pack_start (GTK_BOX (vbox1), image, TRUE, TRUE, 0);

	label2 = gtk_label_new ("mDictionary");
	ws_gui_app->p = pango_font_description_from_string("Tahoma 18");
	gtk_widget_modify_font(GTK_WIDGET(label2), ws_gui_app->p);
	gtk_label_set_pattern (GTK_LABEL(label2), "__________");
	gtk_box_pack_start (GTK_BOX (vbox1), label2, FALSE, FALSE, 5);

	ws_version = g_strconcat("Maemo Multilingual Dictionary\nver. ",
				WS_VERSION,
				NULL);
	label3 = gtk_label_new_with_mnemonic (_(ws_version));
	ws_gui_app->p = pango_font_description_from_string("Tahoma 12");
	gtk_widget_modify_font(GTK_WIDGET(label3), ws_gui_app->p);
	gtk_box_pack_start (GTK_BOX (vbox1), label3, FALSE, FALSE, 5);
	gtk_label_set_justify (GTK_LABEL (label3), GTK_JUSTIFY_CENTER);

	label4 = gtk_label_new ("Copyright 2006 - 2007, ComArch S.A\nAll rights " 
				"reserved");
	ws_gui_app->p = pango_font_description_from_string("Tahoma 12");
	gtk_widget_modify_font(GTK_WIDGET(label4), ws_gui_app->p);
	gtk_box_pack_start (GTK_BOX (vbox1), label4, FALSE, FALSE, 5);
	gtk_label_set_justify (GTK_LABEL (label4), GTK_JUSTIFY_CENTER);

	scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW
					(scrolledwindow),
					GTK_SHADOW_ETCHED_OUT);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW
					(scrolledwindow),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start (GTK_BOX (vbox1),
			scrolledwindow,
			TRUE,
			TRUE,
			0);

	textview = gtk_text_view_new ();
	ws_gui_app->p = pango_font_description_from_string("Tahoma 10");
	gtk_widget_modify_font(GTK_WIDGET(textview), ws_gui_app->p);
	gtk_text_view_set_editable(GTK_TEXT_VIEW(textview), FALSE);
	gtk_text_view_set_left_margin(GTK_TEXT_VIEW(textview), 10);
	gtk_text_view_set_right_margin(GTK_TEXT_VIEW(textview), 10);
	gtk_container_add (GTK_CONTAINER (scrolledwindow), textview);
	gtk_text_buffer_set_text(
			gtk_text_view_get_buffer(GTK_TEXT_VIEW (textview)),
 /*                       "The following third party\ncomponents may be\nincluded"
			" depending \non your system configuration:\n\nD-BUS "
			"- License information:\nhttp://opensource.org/"
			"licenses/academic.php",
*/
			"This program is distributed\n"
			" in the hope that it will be\n"
			" useful, but WITHOUT ANY \n"
			" WARRANTY; without even the\n"
			" implied warranty of\n"
			" MERCHANTABILITY or FITNESS\n"
			" FOR A PARTICULAR PURPOSE.\n"
			" See the GNU General Public\n"
			" License for more details.\n"
			
			"The following third party\ncomponents may be\nincluded"
			" depending \non your system configuration:\n\nD-BUS "
			"- License information:\nhttp://opensource.org/"
			" licenses/academic.php\n\n"
			" Finnish-English dictionary -\n"
			" Converted by swaj under GNU\n"
			" Public License, Version: 1.1\n\n"
			" English-Finnish dictionary -\n"
			" Converted by swaj under GNU\n"
			" Public License, Version: 1.0",
			-1);

	button1 = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
	gtk_dialog_add_action_widget(GTK_DIALOG(about_dialog),
				button1,
				GTK_RESPONSE_CLOSE);

	gtk_widget_show_all (GTK_WIDGET(about_dialog));

	if (gtk_dialog_run(GTK_DIALOG(about_dialog)) == GTK_RESPONSE_CLOSE)
	{
		gtk_widget_destroy(GTK_WIDGET(about_dialog));
	}

	gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_hildon_find_toolbar),
			TRUE);
}

void ws_gui_add_bookmark_dialog(gpointer user_data)
{

	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;
	
	GtkWidget *dialog1;
	GtkWidget *dialog_vbox1;
	GtkWidget *vbox1;
	GtkWidget *hbox1;
	GtkWidget *entry1;
	GtkWidget *label;
	GtkWidget *cancelbutton1;
	GtkWidget *okbutton1;
	
	const gchar *book_word;
	
	dialog1 = gtk_dialog_new ();
	gtk_window_set_title (GTK_WINDOW (dialog1), _("ws_ti_add_bookmark"));
	gtk_window_set_default_size(GTK_WINDOW (dialog1), 300, 200);
	
	dialog_vbox1 = GTK_DIALOG (dialog1)->vbox;
	gtk_widget_show (dialog_vbox1);
	
	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (dialog_vbox1), vbox1, TRUE, TRUE, 5);
	
	hbox1 = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox1), hbox1, TRUE, TRUE, 5);
	
	label = gtk_label_new(_("ws_ti_bookmarks_add_question"));
	gtk_box_pack_start (GTK_BOX (hbox1), label, TRUE, TRUE, 5);
	
	entry1 = gtk_entry_new ();
	gtk_box_pack_start (GTK_BOX (vbox1), entry1, TRUE, TRUE, 5);
	gtk_entry_set_text(GTK_ENTRY(entry1), ws_gui_app->last_word);
	gtk_entry_set_alignment (GTK_ENTRY(entry1), 0.5);
	gtk_entry_set_editable(GTK_ENTRY(entry1), TRUE);
	
	cancelbutton1 = gtk_button_new_from_stock (_("ws_db_ok"));
	gtk_dialog_add_action_widget (GTK_DIALOG (dialog1), 
			 	      cancelbutton1, 
				      GTK_RESPONSE_OK);
	
	okbutton1 = gtk_button_new_from_stock (_("ws_db_cancel"));
	gtk_dialog_add_action_widget (GTK_DIALOG (dialog1), 
				      okbutton1, 
          			      GTK_RESPONSE_CANCEL);
	
	gtk_widget_show_all (dialog1);
	
	if (gtk_dialog_run(GTK_DIALOG(dialog1)) == GTK_RESPONSE_OK)
	{
		book_word = gtk_entry_get_text(GTK_ENTRY(entry1));
		ws_dbus_client_add_bookmark(ws_gui_app->dbus_data, 
				     (gchar*)book_word, 
				     ws_gui_app->raw_translation->str);
		gtk_widget_destroy(GTK_WIDGET(dialog1));
		
	}
	else 
	{
		gtk_widget_destroy(GTK_WIDGET(dialog1));
	}
}

void ws_gui_remove_bookmark_dialog(gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;
	
	GtkWidget *dialog1;
	GtkWidget *dialog_vbox1;
	GtkWidget *vbox1;
	GtkWidget *hbox1;
	GtkWidget *entry1;
	GtkWidget *label;
	GtkWidget *cancelbutton1;
	GtkWidget *okbutton1;
	
	dialog1 = gtk_dialog_new ();
	gtk_window_set_title (GTK_WINDOW (dialog1), _("ws_ti_remove_bookmark"));
	gtk_window_set_default_size(GTK_WINDOW (dialog1), 300, 200);
	
	dialog_vbox1 = GTK_DIALOG (dialog1)->vbox;
	gtk_widget_show (dialog_vbox1);
	
	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (dialog_vbox1), vbox1, TRUE, TRUE, 5);
	
	hbox1 = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox1), hbox1, TRUE, TRUE, 5);
	
	label = gtk_label_new(_("ws_ni_remove_bookmark_question"));
	gtk_box_pack_start (GTK_BOX (hbox1), label, TRUE, TRUE, 5);
	
	entry1 = gtk_entry_new ();
	gtk_box_pack_start (GTK_BOX (vbox1), entry1, TRUE, TRUE, 5);
	gtk_entry_set_text(GTK_ENTRY(entry1), ws_gui_app->last_word);
	gtk_entry_set_alignment (GTK_ENTRY(entry1), 0.5);
	gtk_entry_set_editable(GTK_ENTRY(entry1), FALSE);
	
	cancelbutton1 = gtk_button_new_from_stock (_("ws_db_ok"));
	gtk_dialog_add_action_widget (GTK_DIALOG (dialog1), 
				      cancelbutton1, 
				      GTK_RESPONSE_OK);
	
	okbutton1 = gtk_button_new_from_stock (_("ws_db_cancel"));
	gtk_dialog_add_action_widget (GTK_DIALOG (dialog1), 
				      okbutton1, 
          			      GTK_RESPONSE_CANCEL);
	
	gtk_widget_show_all (dialog1);
	
	if (gtk_dialog_run(GTK_DIALOG(dialog1)) == GTK_RESPONSE_OK)
	{
		gtk_widget_destroy(GTK_WIDGET(dialog1));
		gchar* tmp;
		
		ws_dbus_client_remove_bookmark(ws_gui_app->dbus_data, 
					ws_gui_app->last_word);

		ws_gui_app->last_word = NULL;

		ws_gui_fill_html(" ", ws_gui_app);

		g_object_get(G_OBJECT(ws_gui_app->ws_gui_hildon_find_toolbar),
                     		"prefix",
                   		&tmp,
                     		NULL);
		
		if (strlen(tmp) != 0)
		{
			ws_dbus_client_find_word (ws_gui_app->dbus_data, tmp);
		}
		else 
		{	
			g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "strlen(tmp)=0");
			gtk_list_store_clear(
				       ws_gui_app->ws_gui_w_list->ws_gui_store);
		}
	}
	else 
	{
		gtk_widget_destroy(GTK_WIDGET(dialog1));
	}
}

void ws_gui_list_full_dialog(GtkWidget *widget, GdkEventButton *event, gpointer user_data) {
	
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;
	
	ws_gui_app->ws_message_dialog =  gtk_message_dialog_new(
				GTK_WINDOW(ws_gui_app->ws_gui_hildon_window),
				GTK_DIALOG_DESTROY_WITH_PARENT,
				GTK_MESSAGE_INFO,
				GTK_BUTTONS_OK,
				_("ws_ni_words_list_full"));

	if (gtk_dialog_run(
			GTK_DIALOG(ws_gui_app->ws_message_dialog)) 
				== GTK_RESPONSE_OK)
		{
			gtk_widget_destroy(
						ws_gui_app->ws_message_dialog);
		}
}
/*@}*/
