/*******************************************************************************
This file is part of mdictionary.

mdictionary is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

mdictionary is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License 
along with mdictionary; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Copyright 2006 ComArch S.A.
*******************************************************************************/
/** \addtogroup GUI
 */
/*@{*/
/** \file ws_gui_callbacks.c
 * \brief Declaration of GUI callback handler functions.
 *
 * Callbacks for GUI elements
 * \author Anna Gadomska \<anna.gadomska\@comarch.com\>
 * \date 05-11-2007
 * \version 1.0.0
 */

#include <ws_gui_callbacks.h>
#include <ws_gui.h>
#include <ws_gui_layout.h>
#include <ws_gui_utils.h>
#include <ws_dbus.h>

/** \brief show how much time did take a callback of another function
*
*/
static double timer(gboolean start, gchar* message)
{
	static GArray* stack = NULL;
	static gboolean first_run = TRUE;
	static struct timeval actual_time;
	static struct timeval last_time;
	static struct timeval result;
	static double seconds = 0.0;
	if(first_run)
	{
		first_run = FALSE;
		stack = g_array_new(TRUE, TRUE, sizeof(struct timeval));
	};
	// things to do on the beggining of function's work
	if (start)
	{
		g_debug("XDXF->%s() start counting time for function '%s()'.\n",
			__FUNCTION__,message);
		g_array_prepend_val(stack, actual_time);
		gettimeofday(&g_array_index(stack, struct timeval, 0),NULL);
		return -1.0;
	}
	// we just want to end some timer - print some information 
	//about working time;
	else {
		gettimeofday(&actual_time,NULL);
		last_time = g_array_index(stack, struct timeval, 0);
		g_array_remove_index(stack, 0);

		if (actual_time.tv_usec < last_time.tv_usec) {
			int nsec = (last_time.tv_usec - 
					actual_time.tv_usec) / 1000000 + 1;
			last_time.tv_usec -= 1000000 * nsec;
			last_time.tv_sec += nsec;
		}
		if (actual_time.tv_usec - last_time.tv_usec > 1000000) {
			int nsec = (last_time.tv_usec -
						actual_time.tv_usec) / 1000000;
			last_time.tv_usec += 1000000 * nsec;
			last_time.tv_sec -= nsec;
		}
		result.tv_sec = actual_time.tv_sec - last_time.tv_sec;
		result.tv_usec = actual_time.tv_usec - last_time.tv_usec;
		seconds = (((double)(result.tv_usec)) / 1e6) + 
						((double)(result.tv_sec));

		g_debug("XDXF->%s() function \'%s()\' was working for: %g [s] "
			"or %ld [us].\n",
			__FUNCTION__,
			message,seconds,
			((long)(result.tv_sec*1e6)+(result.tv_usec)));
	// stack is empty so we delete everything
	if(stack->len == 0) 
	{
			g_array_free(stack, TRUE);
			first_run = TRUE;
	}
	return seconds;
	}
	return -2.0;
}


/**  this function handles signals from dbus; it is called 
when there are any messages from other modules
*
* @param error - error message recived from DBUS
* @param words - array with recived data structure
* @param user_data - pointer to data structure
* @return
*/
void ws_gui_signal_hander (GError *error, GArray *words, gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;
	osso_rpc_t osss_data;
	osss_data = g_array_index (words, osso_rpc_t, 0);
	switch(osss_data.value.i)
	{
		case WS_DBUS_ERROR_ENGINE_NOT_FOUND:
		{
			ws_gui_app->ws_message_dialog = 
			gtk_message_dialog_new(
				GTK_WINDOW(ws_gui_app->ws_gui_hildon_window),
				GTK_DIALOG_DESTROY_WITH_PARENT,
				GTK_MESSAGE_ERROR,
				GTK_BUTTONS_OK,
				_("ws_ni_error_occured"));
			gtk_widget_show_all(ws_gui_app->ws_message_dialog);

			g_signal_connect_swapped(
				GTK_OBJECT (ws_gui_app->ws_message_dialog),
				"response",
				G_CALLBACK (gtk_main_quit),
				ws_gui_app);
			break;
		}

		case WS_DBUS_ERROR_FILE_NOT_FOUND:
		{
			ws_gui_app->ws_message_dialog = 
				gtk_message_dialog_new(
				GTK_WINDOW(ws_gui_app->ws_gui_hildon_window),
				GTK_DIALOG_DESTROY_WITH_PARENT,
				GTK_MESSAGE_ERROR,
				GTK_BUTTONS_OK,
				_("ws_ni_no_dictionary_available"));

			gtk_widget_show_all(ws_gui_app->ws_message_dialog);

			GArray *tmp;
			tmp = g_array_new(TRUE, TRUE, sizeof(gchar*));
			gtk_list_store_clear(
				ws_gui_app->ws_gui_w_list->ws_gui_store);
			ws_gui_app->ws_gui_w_list->ws_gui_model = 
				create_and_fill_model(tmp, ws_gui_app);
			ws_gui_fill_html(" ", ws_gui_app);
			ws_gui_app->html_flag = FALSE;

			gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_hildon_find_toolbar),
			FALSE);
			gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_search),
			FALSE);

			if (gtk_dialog_run(
				GTK_DIALOG(ws_gui_app->ws_message_dialog)) 
							== GTK_RESPONSE_OK)
			{
				gtk_widget_destroy(
						ws_gui_app->ws_message_dialog);
			}
			break;
		}

		case WS_DBUS_INFO_CACHING:
		{
			ws_gui_app->ws_gui_banner_caching =
				hildon_banner_show_progress(
				GTK_WIDGET(ws_gui_app->ws_gui_hildon_window),
				NULL,
				_("ws_pb_caching"));
		ws_gui_app->caching_flag = TRUE;
		hildon_banner_set_fraction(
			HILDON_BANNER(ws_gui_app->ws_gui_banner_caching),
			0.0);
		gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_hildon_find_toolbar),
			FALSE);
		gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_search),
			FALSE);
			gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_bookmarks),
			FALSE);
		gtk_widget_set_sensitive(
		GTK_WIDGET(ws_gui_app->ws_gui_w_list->ws_gui_view),
		FALSE);

		break;
		}

		case WS_DBUS_INFO_CACHING_FINISHED:
		{
		gtk_widget_destroy(
		GTK_WIDGET(ws_gui_app->ws_gui_banner_caching));
		ws_gui_app->caching_flag = FALSE;
		gtk_widget_set_sensitive(
		GTK_WIDGET(ws_gui_app->ws_gui_hildon_find_toolbar),
		TRUE);
		gtk_widget_set_sensitive(
		GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_search),
		TRUE);
		gtk_widget_set_sensitive(
		GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_bookmarks),
		TRUE);
		gtk_widget_set_sensitive(
		GTK_WIDGET(ws_gui_app->ws_gui_w_list->ws_gui_view),
		TRUE);

		break;
		}

		case WS_DBUS_ERROR_DICTIONARY_NOT_LOAD:
		{
			if (ws_gui_app->ws_gui_history_cur_pos >= 0 &&
			ws_gui_app->ws_gui_history_cur_pos <=HISTORY_LEN)
			{ 
				g_array_remove_index(ws_gui_app->ws_gui_history,
					ws_gui_app->ws_gui_history_cur_pos);
			}

			ws_gui_app->ws_message_dialog = gtk_message_dialog_new(
				GTK_WINDOW(ws_gui_app->ws_gui_hildon_window),
				GTK_DIALOG_DESTROY_WITH_PARENT,
				GTK_MESSAGE_ERROR,
				GTK_BUTTONS_OK,
				_("ws_ni_dictionary_unavailable"));
		gtk_widget_show_all(ws_gui_app->ws_message_dialog);
		if (gtk_dialog_run(
		GTK_DIALOG(ws_gui_app->ws_message_dialog)) 
							== GTK_RESPONSE_OK)
		{
			gtk_widget_destroy(ws_gui_app->ws_message_dialog);
		}
		break;
		}
		
		case  WS_DBUS_BOOKMARKS_ADDED_OK:
		{
			hildon_banner_show_information(GTK_WIDGET(
					ws_gui_app->ws_gui_hildon_window),
					NULL,
					_("ws_ni_bookmark_added"));
		break;
		}

		case WS_DBUS_BOOKMARKS_REMOVED_OK:
		{
			hildon_banner_show_information(GTK_WIDGET(
					ws_gui_app->ws_gui_hildon_window),
					NULL,
					_("ws_ni_bookmark_removed"));
		break;
		}
	
		case WS_DBUS_BOOKMARKS_ADDED_FAIL:
		{
			hildon_banner_show_information(GTK_WIDGET(
					ws_gui_app->ws_gui_hildon_window),
					NULL,
					_("ws_ni_bookmark_not_added"));
		break;
		}

		case WS_DBUS_BOOKMARKS_REMOVED_FAIL:
		{
			hildon_banner_show_information(GTK_WIDGET(
					ws_gui_app->ws_gui_hildon_window),
					NULL,
					_("ws_ni_bookmark_not_removed"));
		break;
		}

		case WS_DBUS_LOAD_BOOKMARK_FAILED:
		{
		gtk_widget_set_sensitive(
		GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_bookmarks),
		FALSE);
		ws_gui_app->bookmark_avail = FALSE;
				
			ws_gui_app->ws_message_dialog = 
			gtk_message_dialog_new(
				GTK_WINDOW(ws_gui_app->ws_gui_hildon_window),
				GTK_DIALOG_DESTROY_WITH_PARENT,
				GTK_MESSAGE_ERROR,
				GTK_BUTTONS_OK,
				_("ws_ni_bookmarks_unavailable"));

			if (gtk_dialog_run(
				GTK_DIALOG(ws_gui_app->ws_message_dialog)) 
							== GTK_RESPONSE_OK)
			{
				gtk_widget_destroy(
						ws_gui_app->ws_message_dialog);
			}

			
			break;	
		}
		case WS_DBUS_EXTRACT_FILE:
		{
			ws_gui_app->ws_gui_banner_extracting =
			hildon_banner_show_animation(
				GTK_WIDGET(ws_gui_app->ws_gui_hildon_window),
				NULL,
				_("ws_pb_extracting"));

			ws_gui_app->caching_flag = TRUE;
			
			gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_hildon_find_toolbar),
			FALSE);
			gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_search),
			FALSE);

		break;
		}

		case WS_DBUS_EXTRACT_FILE_FINISHED:
		{
			gtk_widget_destroy(
			GTK_WIDGET(ws_gui_app->ws_gui_banner_extracting));
			
			ws_gui_app->caching_flag = FALSE;
			
			hildon_banner_show_information(GTK_WIDGET(
					ws_gui_app->ws_gui_hildon_window),
					NULL,
				_("ws_ni_dictionary_added"));
			
			gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_hildon_find_toolbar),
			TRUE);
			gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_search),
			TRUE);

			break;
		}

		case  WS_DBUS_WORDS_LIST_FULL:
		{
			gtk_widget_show_all(GTK_WIDGET(ws_gui_app->ws_gui_list_eventbox));
			/*gtk_widget_show(GTK_WIDGET(ws_gui_app->ws_gui_list_hbox));*/
		break;
		}

		case  WS_DBUS_WORDS_LIST_FILLED_NOT_FULL:
		{
			gtk_widget_hide_all(GTK_WIDGET(ws_gui_app->ws_gui_list_eventbox));
			/*gtk_widget_hide(GTK_WIDGET(ws_gui_app->ws_gui_list_hbox));*/
		break;
		}

		case  WS_DBUS_WORDS_LIST_FINISHED:
		{
			gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_bookmarks),
			TRUE);
			gtk_widget_hide(GTK_WIDGET(ws_gui_app->ws_gui_banner_list_searching));
		break;
		}

		case  WS_DBUS_TRANSLATION_FINISHED:
		{
			gtk_widget_hide(ws_gui_app->ws_gui_banner_translation_loading);
		break;
		}

		case  WS_DBUS_WORDS_LIST_STARTED:
		{
			gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_bookmarks),
			FALSE);
			gtk_widget_show(GTK_WIDGET(ws_gui_app->ws_gui_banner_list_searching));
		break;
		}

		case  WS_DBUS_TRANSLATION_STARTED:
		{
			gtk_widget_show(ws_gui_app->ws_gui_banner_translation_loading);
		break;
		}

	}
}


/* added by Dariusz Wiechecki - HISA */
void ws_gui_search_home_handler(GError *error, GArray *word, gpointer user_data)
{
	g_debug("->%s", __FUNCTION__);
	WSGuiApp *data = (WSGuiApp*) user_data;
	osso_rpc_t* osso_data = NULL;

	/*get the word passed by dbus*/
	osso_data = &g_array_index (word, osso_rpc_t, 0); 

	/* check for special characters in text starting with % */	
	ws_gui_word_conversion( osso_data );

	/* make a copy of passed string */
	gchar* tmp = NULL;
	tmp = g_strdup(osso_data->value.s + 11);
	
	g_object_set(G_OBJECT(data->ws_gui_hildon_find_toolbar),
		"prefix",
		tmp,
		NULL);
	ws_gui_search(NULL, data);
}

/**  this function handles signals from dbus; it is called when progress bar
status has been changed
*
* @param error - error message recived from DBUS
* @param words - array with recived data structure
* @param user_data - pointer to data structure
* @return
*/
void ws_dbus_progress_bar(GError *error, GArray *words, gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;
	osso_rpc_t osss_data;
	osss_data = g_array_index (words, osso_rpc_t, 0);
	double progress = osss_data.value.d;
	if (ws_gui_app->caching_flag == TRUE)
	{
		hildon_banner_set_fraction(
			HILDON_BANNER(ws_gui_app->ws_gui_banner_caching),
			progress);
	}
}

/** this function clean GtkListStore row by row
*
* @param list - GtkListStore to be remoeved
* @return
*/
void ws_gui_clear_list (GtkListStore* list, gpointer user_data) 
{
	GtkTreeIter iter;
	gboolean tmp;
	
	tmp = gtk_tree_model_get_iter_first(
			GTK_TREE_MODEL(list),
			&iter);	

	while (tmp == TRUE)
	{
		tmp = gtk_list_store_remove(list, &iter);
		
	}
}


/** this function handles signal from dbus and transfer recived 
(found in a dictionary) words to the words list
*
* @param error - error message recived from DBUS
* @param words - array with recived data structure
* @param user_data - pointer to data structure
* @return
*/
void ws_gui_dbus_return_words (GError *error, GArray *words, gpointer user_data)
{
	timer(TIMER_START, (gchar*)__FUNCTION__);
	guint i;
	osso_rpc_t data;

	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;
	ws_gui_app->no_words_found = FALSE;
	
	g_signal_handlers_block_by_func(G_OBJECT 
			(ws_gui_app->ws_gui_w_list->ws_gui_view),
			G_CALLBACK (ws_gui_view_cursor_changed), 
			ws_gui_app);	


	GArray *tmp;
	tmp = g_array_new(TRUE, TRUE, sizeof(gchar*));
	gchar *tmp_word;

	for (i=0;i<words->len;++i)
	{
		data = g_array_index (words, osso_rpc_t, i);
		tmp_word = g_strdup(data.value.s);
		g_array_append_val(tmp, tmp_word);
	}
	
	ws_gui_app->loading = FALSE;
	ws_gui_set_toolbar_avail(ws_gui_app);

	ws_gui_app->ws_gui_w_list->ws_gui_model = 
					create_and_fill_model(tmp, ws_gui_app);
	
	if (ws_gui_app->history_flag == TRUE)
	{

	GValue value = { 0, };
	GtkTreeIter tmp_iter;
	gchar *pattern;
	gboolean keep_searching = TRUE;

	if (gtk_tree_model_get_iter_first(GTK_TREE_MODEL(
	ws_gui_app->ws_gui_w_list->ws_gui_model), &tmp_iter) == TRUE)
	{

		pattern = strdup(g_array_index(ws_gui_app->ws_gui_history,
					gchar*, 
					ws_gui_app->ws_gui_history_cur_pos));
		
		gtk_tree_model_get_value(GTK_TREE_MODEL(
				ws_gui_app->ws_gui_w_list->ws_gui_model), 
				&tmp_iter, 
				COL_WORD, 
				&value);

		if ((g_value_get_string (&value) != NULL) &&
				(strcmp(((gchar *)g_value_get_string (&value)), 
				pattern) == 0))
		{
			gtk_tree_selection_select_iter(
				ws_gui_app->ws_gui_w_list->ws_gui_selection,
				&tmp_iter);

		}
		else
		{
		while (gtk_tree_model_iter_next(GTK_TREE_MODEL
		(ws_gui_app->ws_gui_w_list->ws_gui_model), 
		&tmp_iter) == TRUE && keep_searching == TRUE)
		{
			gtk_tree_model_get_value(GTK_TREE_MODEL(
				ws_gui_app->ws_gui_w_list->ws_gui_model), 
				&tmp_iter, 
				COL_WORD, 
				&value);

			if ((g_value_get_string (&value) != NULL) &&
				(strcmp(((gchar *)g_value_get_string (&value)), 
				pattern) == 0))
				{
				gtk_tree_model_get_iter_from_string (
				GTK_TREE_MODEL(
				ws_gui_app->ws_gui_w_list->ws_gui_model),
				&tmp_iter,
				g_array_index(
				ws_gui_app->ws_gui_history_iter,
				gchar*, 
				ws_gui_app->ws_gui_history_cur_pos));

				gtk_tree_selection_select_iter(
				ws_gui_app->ws_gui_w_list->ws_gui_selection,
				&tmp_iter);
								
				keep_searching = FALSE;
				}
			}
		}
	}
	g_free(pattern);

	ws_gui_app->history_flag = FALSE;
	ws_dbus_client_find_translation(ws_gui_app->dbus_data, 
				g_array_index(ws_gui_app->ws_gui_history,
				gchar*, 
				ws_gui_app->ws_gui_history_cur_pos));
	
	ws_gui_app->html_flag = TRUE;
	g_value_unset (&value);
	}
		
	if ((tmp->len == 0 || tmp_word == NULL) && ws_gui_app->stop_clicked != TRUE )
	{
			hildon_banner_show_information(GTK_WIDGET(
					ws_gui_app->ws_gui_hildon_window),
					NULL,
			_("ws_ni_no_words_found"));
		
		if (ws_gui_app->dictionary_selection_changed == TRUE)
		{
			ws_gui_clear_list(ws_gui_app->ws_gui_w_list->ws_gui_store, ws_gui_app);
			ws_gui_fill_html("", ws_gui_app);
			ws_gui_app->dictionary_selection_changed = FALSE;
		}
		
		g_free(ws_gui_app->last_word);
		ws_gui_app->last_word = NULL;
		ws_gui_app->no_words_found = TRUE;
	}

	g_signal_handlers_unblock_by_func(G_OBJECT(
					ws_gui_app->ws_gui_w_list->ws_gui_view),
					G_CALLBACK (ws_gui_view_cursor_changed), 
					ws_gui_app);	
	
	for (i=0;i<tmp->len;++i)
	{
		g_free(g_array_index(tmp, gchar* , i));
	}
	g_array_free(tmp, TRUE);

	
	ws_gui_app->dictionary_selection_changed = FALSE;
	timer(TIMER_STOP, (gchar*)__FUNCTION__);
}

/** this function handles signal from dbus and send recived data to 
the translation area
*
* @param error - error message recived from DBUS
* @param words - array with recived data structure
* @param user_data - pointer to data structure
* @return
*/
void ws_gui_dbus_return_translation (GError *error,
				GArray *words,
				gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;
	gchar *html_content = NULL;
	osso_rpc_t data;

	data = g_array_index (words, osso_rpc_t, 0);

	html_content = format_html(data.value.s, ws_gui_app);
	ws_gui_fill_html(html_content, ws_gui_app);
	g_free(html_content);
	ws_gui_app->html_flag = TRUE;
	
	ws_gui_app->loading = FALSE;
	ws_gui_set_toolbar_avail(ws_gui_app);

}

/** this function handles signal from dbus and send recived data to 
the translation area
*
* @param error - error message recived from DBUS
* @param words - array with recived data structure
* @param user_data - pointer to data structure
* @return
*/
void ws_dbus_server_return_extracted_bzip(GError *error,
				GArray *words,
				gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;

	osso_rpc_t data;

	data = g_array_index (words, osso_rpc_t, 0);

	if (data.value.s[0] == '\0')
	{
			hildon_banner_show_information(GTK_WIDGET(
					ws_gui_app->ws_gui_hildon_window),
					NULL,
				_("ws_ni_dictionary_not_added"));
	}
	else
	{
	if (ws_gui_load_dict(data.value.s, ws_gui_app) == TRUE)
	{
		
	}
	}
}

/**
* this function allows to free allocated memory
*
* @param user_data - pointer to data structure
*/
void ws_gui_free_memory(gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;
	guint i = 0;
	for (i = 0; i<ws_gui_app->ws_gui_history->len; ++i)
	{
		g_free(g_array_index(ws_gui_app->ws_gui_history, gchar *, i));
	}
	g_array_free(ws_gui_app->ws_gui_history, TRUE);

	for (i = 0; i<ws_gui_app->ws_gui_history_list->len; ++i)
	{
		g_free(g_array_index(ws_gui_app->ws_gui_history_list, gchar *, i));
	}
	g_array_free(ws_gui_app->ws_gui_history_list, TRUE);

	for (i = 0; i<ws_gui_app->ws_gui_history_iter->len; ++i)
	{
		g_free(g_array_index(ws_gui_app->ws_gui_history_iter, gchar *, i));
	}
	g_array_free(ws_gui_app->ws_gui_history_iter, TRUE);


	pango_font_description_free(ws_gui_app->p);

	g_free(ws_gui_app->welcome_note);
	ws_gui_app->welcome_note = NULL;

	g_string_free(ws_gui_app->raw_translation, TRUE);
	g_free(ws_gui_app->last_word);
	g_free(ws_gui_app->ws_gui_w_list);
	g_free(ws_gui_app->ws_gui_menu);
	g_free(ws_gui_app);
}

/** this function handle press signals (keyboard)
* 
* @param widget
* @param keyevent
* @param user_data - ponter to data structure
* @return TRUE to stop other handlers from being invoked for the event. 
FALSE to propagate the event further.
*/
gboolean hildon_key_press_listener (GtkWidget * widget,
				GdkEventKey * keyevent,
				gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;

	switch ((guint)(keyevent->keyval)) {
		case HILDON_HARDKEY_UP: 
		{
			gtk_container_set_focus_vadjustment(
			GTK_CONTAINER(ws_gui_app->ws_gui_scrolledwindow_left),
			gtk_scrolled_window_get_vadjustment(
			GTK_SCROLLED_WINDOW(
				ws_gui_app->ws_gui_scrolledwindow_left)));
		ws_gui_app->v_new_value =
			gtk_adjustment_get_value(
			GTK_ADJUSTMENT(
				ws_gui_app->ws_gui_vadj)) - ws_gui_app->v_delta;
		if (ws_gui_app->v_new_value > 
			ws_gui_app->ws_gui_vadj->lower) 
		{
			gtk_adjustment_set_value(
					GTK_ADJUSTMENT(ws_gui_app->ws_gui_vadj),
					ws_gui_app->v_new_value);
		}
			
			break;
		}

		case HILDON_HARDKEY_DOWN: 
		{
		gtk_container_set_focus_vadjustment(
			GTK_CONTAINER(ws_gui_app->ws_gui_scrolledwindow_left),
			gtk_scrolled_window_get_vadjustment(
			GTK_SCROLLED_WINDOW(
				ws_gui_app->ws_gui_scrolledwindow_left)));
		ws_gui_app->v_new_value = gtk_adjustment_get_value(
					GTK_ADJUSTMENT(ws_gui_app->ws_gui_vadj))
					+ ws_gui_app->v_delta;

		if (ws_gui_app->v_new_value < 
				(ws_gui_app->ws_gui_vadj->upper - 
					ws_gui_app->ws_gui_vadj->page_size)) 
		{
				gtk_adjustment_set_value(
					GTK_ADJUSTMENT(ws_gui_app->ws_gui_vadj),
					ws_gui_app->v_new_value);
		}
		break;
		}

		case HILDON_HARDKEY_LEFT:
		{
		gtk_container_set_focus_hadjustment(
		GTK_CONTAINER(ws_gui_app->ws_gui_scrolledwindow_left),
		gtk_scrolled_window_get_hadjustment(
		GTK_SCROLLED_WINDOW(
				ws_gui_app->ws_gui_scrolledwindow_left)));

		ws_gui_app->h_new_value = gtk_adjustment_get_value(
		GTK_ADJUSTMENT(ws_gui_app->ws_gui_hadj)) 
		- ws_gui_app->h_delta;

		if (ws_gui_app->h_new_value > 
			ws_gui_app->ws_gui_hadj->lower) 
		{
				gtk_adjustment_set_value(
					GTK_ADJUSTMENT(ws_gui_app->ws_gui_hadj),
					ws_gui_app->h_new_value);
		}
		}
		break;

		case HILDON_HARDKEY_RIGHT: 
		{
		gtk_container_set_focus_hadjustment(
		GTK_CONTAINER(ws_gui_app->ws_gui_scrolledwindow_left),
		gtk_scrolled_window_get_hadjustment(
		GTK_SCROLLED_WINDOW(
		ws_gui_app->ws_gui_scrolledwindow_left)));

		ws_gui_app->h_new_value = gtk_adjustment_get_value(
		GTK_ADJUSTMENT(ws_gui_app->ws_gui_hadj)) 
		+ ws_gui_app->h_delta;

		if (ws_gui_app->h_new_value < 
				(ws_gui_app->ws_gui_hadj->upper - 
					ws_gui_app->ws_gui_hadj->page_size)) 
		{
		gtk_adjustment_set_value(
					GTK_ADJUSTMENT(ws_gui_app->ws_gui_hadj),
					ws_gui_app->h_new_value);
		}
		}
		break;

		case HILDON_HARDKEY_SELECT: 
			ws_gui_search(NULL, ws_gui_app);
		break;

		case HILDON_HARDKEY_FULLSCREEN: 
		ws_gui_full_screen(NULL, ws_gui_app);
		break;

		case HILDON_HARDKEY_INCREASE: 
		ws_gui_html_zoom_in(NULL, ws_gui_app);
		break;

		case HILDON_HARDKEY_DECREASE:
		ws_gui_html_zoom_out(NULL, ws_gui_app);
		break;

		case HILDON_HARDKEY_ESC: 
		ws_gui_search_stop(NULL, ws_gui_app);
		break;

		default:
		return FALSE;
		break;
	}
	return TRUE;
}

/** this function allow to hide words list using menu item from application menu
*
* @param checkmenuitem - the object which received the signal
* @param user_data - user data set when the signal handler was connected
* @return
*/
void ws_gui_words_list_hide_from_menu(GtkCheckMenuItem *checkmenuitem,
				gpointer user_data)
{
	WSGuiApp* ws_gui_app = (WSGuiApp *)user_data;
	if (gtk_check_menu_item_get_active(
	GTK_CHECK_MENU_ITEM(ws_gui_app->ws_gui_menu->ws_gui_menu_hide_list)))
	{
		gtk_widget_hide(ws_gui_app->ws_gui_list_vbox);
		gtk_toggle_tool_button_set_active(
		GTK_TOGGLE_TOOL_BUTTON(ws_gui_app->ws_gui_toobar_button_hide),
		TRUE);
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
				ws_gui_app->ws_gui_menu->ws_gui_menu_hide_list),
				TRUE);
	}
	else 
	{
		gtk_widget_show(ws_gui_app->ws_gui_list_vbox);
		gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(
					ws_gui_app->ws_gui_toobar_button_hide),
					FALSE);
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
				ws_gui_app->ws_gui_menu->ws_gui_menu_hide_list),
				FALSE);
	}
}

/** this function allow to hide words list using toggle button placed in 
the find toolbar
*
* @param toolbar - the object which received the signal
* @param user_data - user data set when the signal handler was connected
* @return
*/
void ws_gui_words_list_hide(GtkToggleButton *togglebutton, gpointer user_data)
{
	WSGuiApp* ws_gui_app = (WSGuiApp *)user_data;

	if (gtk_toggle_tool_button_get_active(
		GTK_TOGGLE_TOOL_BUTTON(ws_gui_app->ws_gui_toobar_button_hide)))
	{
		gtk_widget_hide(ws_gui_app->ws_gui_list_vbox);
		gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(
					ws_gui_app->ws_gui_toobar_button_hide),
					TRUE);
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
				ws_gui_app->ws_gui_menu->ws_gui_menu_hide_list),
				TRUE);
	}
	else 
	{
		gtk_widget_show(ws_gui_app->ws_gui_list_vbox);
		gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(
					ws_gui_app->ws_gui_toobar_button_hide),
					FALSE);
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
				ws_gui_app->ws_gui_menu->ws_gui_menu_hide_list),
				FALSE);
	}
}

/** add word to the history
*
* @param new_word - word which is going to be append to the history array
* @param user_data - user data set when the signal handler was connected
* @return
*/
void ws_gui_history_add(char *new_word, gpointer user_data)
{
	WSGuiApp* ws_gui_app = (WSGuiApp *)user_data;

	if (ws_gui_app->history_flag != TRUE)
	{

		guint i;
		gchar *tmp_word;
		gchar *tmp_last_searched;
		gchar *tmp_iter = NULL;
		gchar *previous_word = " ";
	
		if (ws_gui_app->ws_gui_history_cur_pos > -1 &&
		g_array_index(ws_gui_app->ws_gui_history, 
				gchar*, 
				ws_gui_app->ws_gui_history_cur_pos) != NULL)
		{
			previous_word = NULL;
			previous_word = g_array_index(
					ws_gui_app->ws_gui_history, 
					gchar*, 
					ws_gui_app->ws_gui_history_cur_pos);
		}
	
		i = ws_gui_app->ws_gui_history_cur_pos + 1;
		gchar *tmp = g_array_index(ws_gui_app->ws_gui_history, 
					gchar*, 
					i);
	
		if (previous_word != NULL && 
		strcmp(previous_word, new_word) != 0)
		{
			while (tmp != NULL) 
			{
				g_array_remove_index(
				ws_gui_app->ws_gui_history_list, 
				i);
				g_array_remove_index(
				ws_gui_app->ws_gui_history_iter, 
				i);

				
				g_array_remove_index(
				ws_gui_app->ws_gui_history, 
				i);
				
				tmp = g_array_index(
				ws_gui_app->ws_gui_history, 
				gchar*, 
				i);
			}
	
		i = 0;
		ws_gui_app->ws_gui_history_cur_pos ++;
	
		if (ws_gui_app->bookmark_mode == FALSE)
		{
			tmp_last_searched = g_strdup(ws_gui_app->last_searched);
		}
		else
		{
		tmp_last_searched = g_strdup(ws_gui_app->last_searched_in_book);
		}

		g_array_append_val(ws_gui_app->ws_gui_history_list, 
				tmp_last_searched);
				
		tmp_word = g_strdup(new_word);
		
		g_array_append_val(ws_gui_app->ws_gui_history, tmp_word);
		g_array_append_val(ws_gui_app->ws_gui_history_iter, tmp_iter);
		
	
		if(ws_gui_app->ws_gui_history->len > HISTORY_LEN)
		{
			g_array_remove_index(ws_gui_app->ws_gui_history, 0);
			g_array_remove_index(ws_gui_app->ws_gui_history_list, 
					0);
			g_array_remove_index(ws_gui_app->ws_gui_history_iter, 
					0);
			
			ws_gui_app->ws_gui_history_cur_pos--;
		}
	
		i = 0;
		
		tmp = g_array_index(ws_gui_app->ws_gui_history, gchar*, i);
		}
		
	}

	ws_gui_check_history(ws_gui_app);
}

/** display previously choosen word (previous from the history array)
if avaible, sets current position in the history array
*
* @param button - button which recived a signal
* @param user_data - user data set when the signal handler was connected
* @return 
*/
void ws_gui_history_back(GtkButton *button, gpointer user_data) 
{
	WSGuiApp* ws_gui_app = (WSGuiApp *)user_data;
	
	ws_gui_app->history_flag = TRUE;

	if (ws_gui_app->ws_gui_history_cur_pos > -1) 
	{
		ws_gui_app->ws_gui_history_cur_pos = 
					ws_gui_app->ws_gui_history_cur_pos - 1;

		/*set object's property*/
		g_object_set(G_OBJECT(ws_gui_app->ws_gui_hildon_find_toolbar),
			"prefix",
			(g_array_index(ws_gui_app->ws_gui_history_list,
			gchar*, 
			ws_gui_app->ws_gui_history_cur_pos)),
			NULL);
	
		ws_dbus_client_find_word (ws_gui_app->dbus_data, 
					g_strstrip(g_array_index(ws_gui_app->ws_gui_history_list,
					gchar*, 
					ws_gui_app->ws_gui_history_cur_pos))
					);
		ws_gui_app->loading = TRUE;
		ws_gui_set_toolbar_avail(ws_gui_app);
	}
	else 
	{
		ws_gui_app->loading = FALSE;
		ws_gui_set_toolbar_avail(ws_gui_app);
	}

	ws_gui_check_history(ws_gui_app);
}

/** display choosen word, next in the history array (if avaible), 
sets current position in the history array
*
* @param button - button which recived a signal
* @param user_data - user data set when the signal handler was connected
* @return 
*/
void ws_gui_history_next(GtkButton *button, gpointer user_data) 
{
	WSGuiApp* ws_gui_app = (WSGuiApp *)user_data;
	
	ws_gui_app->history_flag = TRUE;

	gchar *tmp = g_array_index(ws_gui_app->ws_gui_history,
				gchar*,
				ws_gui_app->ws_gui_history_cur_pos+1);

	if ((ws_gui_app->ws_gui_history_cur_pos < HISTORY_LEN-1) 
	&& (tmp != NULL)) 
	{
		ws_gui_app->ws_gui_history_cur_pos =
			ws_gui_app->ws_gui_history_cur_pos + 1;


		/*set object's property*/
		g_object_set(G_OBJECT(ws_gui_app->ws_gui_hildon_find_toolbar),
			"prefix",
			(g_array_index(ws_gui_app->ws_gui_history_list,
			gchar*, 
			ws_gui_app->ws_gui_history_cur_pos)),
			NULL);
	
		
		ws_dbus_client_find_word(ws_gui_app->dbus_data, 
					g_strstrip(g_array_index(ws_gui_app->ws_gui_history_list,
					gchar*, 
					ws_gui_app->ws_gui_history_cur_pos))
					);
		
		ws_gui_app->loading = TRUE;
		ws_gui_set_toolbar_avail(ws_gui_app);
	}
	else 
	{
		/*make forward button inactive*/
		gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_toobar_button_forward),
			FALSE);

		/*make find next button inactive*/
		gtk_widget_set_sensitive(
		GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_find_next),
		FALSE);
	}

	ws_gui_check_history(ws_gui_app);
}

/** check current position in the history array and sets sensitivity of buttons 
/ menu items, depends on availablity of words in the history
*
* @param user_data - user data set when the signal handler was connected
* @return 
*/
void ws_gui_check_history(gpointer user_data) 
{
	WSGuiApp* ws_gui_app = (WSGuiApp *)user_data;

	gchar *tmp = g_array_index(ws_gui_app->ws_gui_history,
				gchar*,
				ws_gui_app->ws_gui_history_cur_pos+1);

	if ((ws_gui_app->ws_gui_history_cur_pos+1 < HISTORY_LEN) 
	&& (tmp != NULL))
	{
		/*make forward button active*/
		gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_toobar_button_forward),
			TRUE);

		/*make find next button active*/
		gtk_widget_set_sensitive(
		GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_find_next),
		TRUE);
	}
	else 
	{
		/*make forward button inactive*/
		gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_toobar_button_forward),
			FALSE);

		/*make find next button inactive*/
		gtk_widget_set_sensitive(
		GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_find_next),
		FALSE);
	}

	tmp = g_array_index(ws_gui_app->ws_gui_history,
			gchar*,
			ws_gui_app->ws_gui_history_cur_pos-1);
	if ((ws_gui_app->ws_gui_history_cur_pos > 0) && (tmp != NULL))
	{
		/*make back button active*/
		gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_toobar_button_back),
			TRUE);

		/*make find previous button active*/
		gtk_widget_set_sensitive(
		GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_find_prev),
		TRUE);
}
else
{
		gtk_widget_set_sensitive (
			GTK_WIDGET(ws_gui_app->ws_gui_toobar_button_back),
			FALSE);
		gtk_widget_set_sensitive(
		GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_find_prev),
		FALSE);
}
}

/** create TreeView Model, which allows to display words list
*
* @param words_list - array with words(found in a dictionary), recived from 
* DBUS;
* @param user_data - user data set when the signal handler was connected
* @return 
*/
GtkTreeModel * create_and_fill_model (GArray *words_list, gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;

	g_signal_handlers_block_by_func(G_OBJECT (ws_gui_app->ws_gui_w_list->ws_gui_view),
					G_CALLBACK (ws_gui_view_cursor_changed), 
					ws_gui_app);	

	guint i = 0;
	gchar *tmp = g_strdup(g_array_index(words_list, gchar*, i));

	if (tmp != NULL)
	{
		ws_gui_clear_list(ws_gui_app->ws_gui_w_list->ws_gui_store, ws_gui_app);
	}

	gboolean valid;
	valid = gtk_tree_model_get_iter_first(
			GTK_TREE_MODEL(ws_gui_app->ws_gui_w_list->ws_gui_store),
			&ws_gui_app->ws_gui_w_list->ws_gui_iter);

	/* Append a row and fill in some data */
	while (tmp != NULL)
	{
		gtk_list_store_append (ws_gui_app->ws_gui_w_list->ws_gui_store,
				&ws_gui_app->ws_gui_w_list->ws_gui_iter);

		gtk_list_store_set (ws_gui_app->ws_gui_w_list->ws_gui_store,
			&ws_gui_app->ws_gui_w_list->ws_gui_iter, 
			COL_WORD, tmp,
			-1);
		i=i+1;
		g_free(tmp);
		tmp = NULL;
		tmp = g_strdup(g_array_index(words_list, gchar*, i));
	};

	g_free(tmp);
	tmp = NULL;
	tmp = g_strdup(g_array_index(words_list, gchar*, 0));

		
	if (ws_gui_app->history_flag == FALSE && tmp != NULL)
	{
		ws_dbus_client_find_translation(ws_gui_app->dbus_data, tmp);
		ws_gui_history_add(tmp, ws_gui_app);

		g_free(ws_gui_app->last_word);
		ws_gui_app->last_word = NULL;
		ws_gui_app->last_word = g_strdup (tmp);
		ws_gui_app->history_flag = FALSE;
	}

	g_signal_handlers_unblock_by_func(G_OBJECT(
					ws_gui_app->ws_gui_w_list->ws_gui_view),
					G_CALLBACK (ws_gui_view_cursor_changed), 
					ws_gui_app);

	g_free(tmp);
	return GTK_TREE_MODEL(ws_gui_app->ws_gui_w_list->ws_gui_store);
	
}

/** create TreeView and TreeModel using create_and_fill_model() function;
it is necessary to display found words in a words list;
*
* @param words_list - array with words(found in a dictionary), 
* recived from DBUS;
* @param user_data - user data set when the signal handler was connected
* @return 
*/
GtkWidget * create_view_and_model (GArray *words_list, gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;

	ws_gui_app->ws_gui_w_list->ws_gui_view = gtk_tree_view_new ();

	ws_gui_app->ws_gui_w_list->ws_gui_renderer=gtk_cell_renderer_text_new();
	gtk_tree_view_insert_column_with_attributes(
			GTK_TREE_VIEW (ws_gui_app->ws_gui_w_list->ws_gui_view),
			-1,
			"Name",
			ws_gui_app->ws_gui_w_list->ws_gui_renderer,
			"text",
			COL_WORD,
			NULL);
	ws_gui_app->ws_gui_w_list->ws_gui_model = 
				create_and_fill_model(words_list, ws_gui_app);

	gtk_tree_view_set_model(
			GTK_TREE_VIEW (ws_gui_app->ws_gui_w_list->ws_gui_view),
			ws_gui_app->ws_gui_w_list->ws_gui_model);
	g_object_unref (ws_gui_app->ws_gui_w_list->ws_gui_model);
	
	return ws_gui_app->ws_gui_w_list->ws_gui_view;
}

/** switch application between full screen and normal mode
*
* @param menuitem - object which recived the signal
* @param user_data - user data set when the signal handler was connected 
* @return
*/
void ws_gui_full_screen(GtkMenuItem *menuitem, gpointer user_data)
{

	WSGuiApp* ws_gui_app = (WSGuiApp*)user_data;
	if (ws_gui_app->ws_gui_full_screen_flag == FALSE) {

		gtk_window_fullscreen(
				GTK_WINDOW(ws_gui_app->ws_gui_hildon_window));
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
			ws_gui_app->ws_gui_menu->ws_gui_menu_full_screen),
			TRUE);
		ws_gui_app->ws_gui_full_screen_flag = TRUE;
		hildon_banner_show_information(GTK_WIDGET(
				ws_gui_app->ws_gui_hildon_window),
				NULL,
				_("ws_ib_fullscreen_on"));
	}
	else 
	{
		gtk_window_unfullscreen(
				GTK_WINDOW(ws_gui_app->ws_gui_hildon_window));
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
			ws_gui_app->ws_gui_menu->ws_gui_menu_full_screen),
			FALSE);
		ws_gui_app->ws_gui_full_screen_flag = FALSE;
			hildon_banner_show_information(GTK_WIDGET(
					ws_gui_app->ws_gui_hildon_window),
					NULL,
				_("ws_ib_fullscreen_off"));
	}
}

/** search for selected text in a dictionary
*
* @param menuitem - object which recived the signal
* @param user_data - user data set when the signal handler was connected 
* @return
*/
void ws_gui_popup_search(GtkMenuItem *menuitem, gpointer user_data)
{
	WSGuiApp* ws_gui_app = (WSGuiApp*)user_data;
	gchar *temp;
	ws_gui_app->ws_gui_clipboard_primary = 
		gtk_widget_get_clipboard(ws_gui_app->ws_gui_html,
					GDK_SELECTION_PRIMARY);
	temp = gtk_clipboard_wait_for_text(
					ws_gui_app->ws_gui_clipboard_primary);
	/*strip leading and trailing spaces*/
	g_strstrip(temp);
	if (temp != NULL || strcmp(temp, " "))
	{
		/*set object's property*/
		g_object_set(G_OBJECT(ws_gui_app->ws_gui_hildon_find_toolbar),
				"prefix",
				temp,
				NULL);
		ws_dbus_client_find_word (ws_gui_app->dbus_data, temp);
	}
	else
	{
		hildon_banner_show_information(
				GTK_WIDGET(ws_gui_app->ws_gui_hildon_window),
				NULL,
				_("ws_ni_no_text_selected"));
	}
}

/** select whole text in the translation (html) area
*
* @param menuitem - object which recived the signal
* @param user_data - user data set when the signal handler was connected 
* @return
*/
void ws_gui_html_select_all(GtkMenuItem *menuitem, gpointer user_data) 
{
	WSGuiApp* ws_gui_app = (WSGuiApp*)user_data;
	gtk_html_select_all(GTK_HTML(ws_gui_app->ws_gui_html));
}

/** copy selected text to the clipoard from context popup menu
*
* @param menuitem - object which recived the signal
* @param user_data - user data set when the signal handler was connected 
* @return
*/
void ws_gui_html_copy(GtkMenuItem *menuitem, gpointer user_data) 
{
	WSGuiApp* ws_gui_app = (WSGuiApp*)user_data;
	gtk_html_copy(GTK_HTML(ws_gui_app->ws_gui_html));
	hildon_banner_show_information(GTK_WIDGET(
				ws_gui_app->ws_gui_hildon_window),
				NULL,
				_("ws_ib_copied"));
}

/** paste copied text into toolbar entry
*
* @param menuitem - object which recived the signal
* @param user_data - user data set when the signal handler was connected 
* @return
*/
void ws_gui_html_paste(GtkMenuItem *menuitem, gpointer user_data)
{
	WSGuiApp* ws_gui_app = (WSGuiApp*)user_data;

	gchar *temp;
	gchar *temp2;
	temp = gtk_clipboard_wait_for_text(ws_gui_app->ws_gui_clipboard);

	/*get object's property value*/
	g_object_get(G_OBJECT(ws_gui_app->ws_gui_hildon_find_toolbar),
		"prefix",
		&temp2,
		NULL);

	/*concatenate two strings*/
	temp = g_strconcat(temp2, temp, NULL);
	/*replace old delimiter with a new delimiter*/
	temp = g_strdelimit(temp, "\n", ' ');

	/*set object's property*/
	g_object_set(G_OBJECT(ws_gui_app->ws_gui_hildon_find_toolbar),
		"prefix",
		temp,
		NULL);
	hildon_banner_show_information(GTK_WIDGET(
			ws_gui_app->ws_gui_hildon_window),
			NULL,
			_("ws_ib_pasted"));
}

/** zoom in text in translation (html) area
*
* @param menuitem - object which recived the signal
* @param user_data - user data set when the signal handler was connected 
* @return
*/
void ws_gui_html_zoom_in(GtkMenuItem *menuitem, gpointer user_data)
{
	WSGuiApp* ws_gui_app = (WSGuiApp*)user_data;
	ws_gui_app->zoom = ws_gui_app->zoom + ZOOM_STEP;
	if (ws_gui_app->zoom > ZOOM_MAX)
	{
		ws_gui_app->zoom = ws_gui_app->zoom - ZOOM_STEP;
		hildon_banner_show_information(GTK_WIDGET(
			ws_gui_app->ws_gui_hildon_window),
			NULL,
			_("ws_ib_max_zoom"));
	}
	else
	{
		gtk_html_set_magnification(GTK_HTML(ws_gui_app->ws_gui_html), 
						ws_gui_app->zoom);

		if ((WS_GUI_ABS(ws_gui_app->zoom - ZOOM_DEFAULT)) < 0.000001)
		{
			hildon_banner_show_information(GTK_WIDGET(
					ws_gui_app->ws_gui_hildon_window),
					NULL,
					_("ws_ib_zoom_default"));
		}
		else
		{
			hildon_banner_show_information(GTK_WIDGET(
					ws_gui_app->ws_gui_hildon_window),
					NULL,
					_("ws_ib_zoom_in"));
		}
	}
}

/** zoom out text in translation (html) area
*
* @param menuitem - object which recived the signal
* @param user_data - user data set when the signal handler was connected 
* @return
*/
void ws_gui_html_zoom_out(GtkMenuItem *menuitem, gpointer user_data)
{
	WSGuiApp* ws_gui_app = (WSGuiApp*)user_data;
	ws_gui_app->zoom = ws_gui_app->zoom - ZOOM_STEP;
	if (ws_gui_app->zoom < ZOOM_MIN)
	{
		ws_gui_app->zoom = ws_gui_app->zoom + ZOOM_STEP;
			hildon_banner_show_information(GTK_WIDGET(
					ws_gui_app->ws_gui_hildon_window),
					NULL,
				_("ws_ib_min_zoom"));
	}
	else
	{
		gtk_html_set_magnification(GTK_HTML(ws_gui_app->ws_gui_html), 
						ws_gui_app->zoom);
		
		if ((WS_GUI_ABS(ws_gui_app->zoom - ZOOM_DEFAULT)) < 0.000001)
		{
			hildon_banner_show_information(GTK_WIDGET(
					ws_gui_app->ws_gui_hildon_window),
					NULL,
					_("ws_ib_zoom_default"));
		}
		else
		{
			hildon_banner_show_information(GTK_WIDGET(
					ws_gui_app->ws_gui_hildon_window),
					NULL,
					_("ws_ib_zoom_out"));
		}
	}
}

/** start searching, send typed word to DBUS and query for words
*
* @param widget - object which recived the signal
* @param user_data - user data set when the signal handler was connected 
* @return
*/
void ws_gui_search(GtkWidget * widget, gpointer user_data) 
{
	WSGuiApp* ws_gui_app = (WSGuiApp*)user_data;
	
	ws_gui_app->stop_clicked = FALSE;

	gchar* ws_gui_text = NULL;
	g_object_get(G_OBJECT(ws_gui_app->ws_gui_hildon_find_toolbar),
		"prefix",
		&ws_gui_text,
		NULL);

	if (ws_gui_app->bookmark_mode == FALSE) 
	{
		if (ws_gui_app->first_run == TRUE)
		{
			ws_gui_app->first_run = FALSE;
		}

		g_free(ws_gui_app->last_searched);
		ws_gui_app->last_searched = NULL;
		ws_gui_app->last_searched = g_strdup(ws_gui_text);
	}
	else
	{
		g_free(ws_gui_app->last_searched_in_book);
		ws_gui_app->last_searched_in_book = NULL;
		ws_gui_app->last_searched_in_book = g_strdup(ws_gui_text);
	}
	
	g_strstrip(ws_gui_text);
	if (strlen(ws_gui_text) != 0) 
	{
		ws_gui_app->loading = TRUE;
		ws_gui_set_toolbar_avail(ws_gui_app);
		ws_dbus_client_find_word (ws_gui_app->dbus_data, ws_gui_text);
		
		g_free(ws_gui_text);
	}
	else 
	{
		hildon_banner_show_information(GTK_WIDGET(
				ws_gui_app->ws_gui_hildon_window),
				NULL,
			_("ws_ni_no_word_typed"));
	}
	g_free(ws_gui_app->last_word);
	ws_gui_app->last_word=NULL;
}

/** stop search process
*
* @param button - object which recived the signal
* @param user_data - user data set when the signal handler was connected 
* @return
*/
void ws_gui_search_stop(GtkButton *button, gpointer user_data) 
{
	WSGuiApp* ws_gui_app = (WSGuiApp *)user_data;
	ws_gui_app->stop_clicked = TRUE;
	if (ws_gui_app->loading == TRUE) 
	{
		ws_gui_app->loading = FALSE;
		ws_gui_set_toolbar_avail(ws_gui_app);		

		ws_dbus_notify(ws_gui_app->dbus_data, WS_DBUS_INFO_STOP_SEARCH);
		hildon_banner_show_information(GTK_WIDGET(
				ws_gui_app->ws_gui_hildon_window),
				NULL,
			_("ws_ni_search_aborted"));
	}
}

/** this function is called just before closing application; 
it sends signal to DBUS and destroys it; 
*
* @param widget - object which recived the signal
* @param event - 
* @param user_data - user data set when the signal handler was connected 
* @return
*/
void ws_gui_on_exit (GtkWidget *widget, GdkEvent *event, gpointer user_data) 
{
	WSGuiApp* ws_gui_app = (WSGuiApp *)user_data;
	ws_gui_app->bookmark_mode = FALSE;
	ws_gui_set_bookmarks_sensitivity(ws_gui_app);
	g_timer_destroy(ws_gui_app->timer);
	ws_dbus_notify(ws_gui_app->dbus_data, WS_DBUS_INFO_TERMINATE);
	ws_dbus_destroy (ws_gui_app->dbus_data);
	ws_gui_free_memory(ws_gui_app);
	gtk_main_quit();
	exit (0);
}

/** this function is called just before closing application, 
from application menu; it sends signal to DBUS and destroys it;
*
* @param menuitem - object which recived the signal
* @param user_data - user data set when the signal handler was connected 
* @return
*/
void ws_gui_menu_quit(GtkMenuItem *menuitem, gpointer user_data)
{
	WSGuiApp* ws_gui_app = (WSGuiApp *)user_data;
	ws_gui_app->bookmark_mode = FALSE;
	ws_gui_set_bookmarks_sensitivity(ws_gui_app);
	g_timer_destroy(ws_gui_app->timer);
	ws_dbus_notify(ws_gui_app->dbus_data, WS_DBUS_INFO_TERMINATE);
	ws_dbus_destroy (ws_gui_app->dbus_data);
	ws_gui_free_memory(ws_gui_app);
	gtk_main_quit();
	exit (0);
}

/** fill translation area with text (html) recived from DBUS
*
* @param html_context - text which is going to be displayed; it should be html
* @param user_data - user data set when the function was called
* @return
*/
void ws_gui_fill_html(char *html_context, gpointer user_data) 
{
	WSGuiApp* ws_gui_app = (WSGuiApp*)user_data;
	
	gtk_html_set_editable (GTK_HTML(ws_gui_app->ws_gui_html), FALSE);

	/*Load welcome screen or translation string*/
	if (ws_gui_app->first_run == TRUE && ws_gui_app->bookmark_mode==FALSE)
	{
	gtk_html_load_from_string(GTK_HTML(ws_gui_app->ws_gui_html),
				ws_gui_app->welcome_note,
				-1);
	g_object_set(G_OBJECT(ws_gui_app->ws_gui_hildon_find_toolbar),
				"prefix",
				"",
				NULL);
	}
	else
	{
	gtk_html_load_from_string(GTK_HTML(ws_gui_app->ws_gui_html),
				html_context,
				-1);
	}
}

/** read adjustment of left scrollwindow, which is necessary to navigate with 
arrow keys inside words list
*
* @param user_data - user data set when the signal handler was connected 
* @return
*/
void ws_gui_read_adjustment(gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;

	ws_gui_app->ws_gui_hadj = 
		gtk_scrolled_window_get_hadjustment(
		GTK_SCROLLED_WINDOW(
		ws_gui_app->ws_gui_scrolledwindow_left));

	ws_gui_app->h_delta = (ws_gui_app->ws_gui_hadj->upper -
				ws_gui_app->ws_gui_hadj->lower)/SCROLL_STEP_H;

	ws_gui_app->ws_gui_vadj = gtk_scrolled_window_get_vadjustment(
	GTK_SCROLLED_WINDOW(ws_gui_app->ws_gui_scrolledwindow_left));

	ws_gui_app->v_delta = (ws_gui_app->ws_gui_vadj->upper -
				ws_gui_app->ws_gui_vadj->lower)/SCROLL_STEP_V;
	ws_gui_app->v_new_value =
	gtk_adjustment_get_value(GTK_ADJUSTMENT(ws_gui_app->ws_gui_vadj)) +
							ws_gui_app->v_delta;

	gtk_container_set_focus_vadjustment(
	GTK_CONTAINER(ws_gui_app->ws_gui_scrolledwindow_left), 
	gtk_scrolled_window_get_vadjustment(
		GTK_SCROLLED_WINDOW(ws_gui_app->ws_gui_scrolledwindow_left)));

}

/** allows to display image in html area
*
* @param html - object which received a signal 
* @param url - url to the image
* @param stream - html stream 
* @return
*/
void ws_gui_url_requested (
			GtkHTML *html, 
			const char *url, 
			GtkHTMLStream *stream)
{
	int fd;

	if (url && !strncmp (url, "file:", 5)) {
		url += 5;
		fd = open (url, O_RDONLY);
		if (fd != -1) {
			gchar *buf;
			size_t size;
			buf = alloca (8192);
			while ((size = read (fd, buf, 8192)) > 0) {
				gtk_html_stream_write (stream, buf, size);
			}
		gtk_html_stream_close(stream, size == -1
					? GTK_HTML_STREAM_ERROR
					: GTK_HTML_STREAM_OK);
		close (fd);

		return;
		}
	}

	gtk_html_stream_close (stream, GTK_HTML_STREAM_ERROR);
}

/** handles button press event and examines what kind of event was it 
	suppose to be
*
* @param widget - object which received a signal
* @param event - type of event which has been performed
* @param user_data - user data set when the signal handler was connected 
* @return TRUE to stop other handlers from being invoked for the event. 
FALSE to propagate the event further
*/
gboolean ws_gui_button_press(GtkWidget *widget,
			GdkEventButton *event,
			gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;
	if (ws_gui_app->ws_gui_sel_flag == FALSE)
	{
		if (event->type == GDK_2BUTTON_PRESS) 
		{
			ws_gui_app->ws_gui_double_click = TRUE;
			g_timer_stop(ws_gui_app->timer);
			g_timer_reset(ws_gui_app->timer);
			return FALSE;
		}

		g_signal_stop_emission_by_name(G_OBJECT(
						ws_gui_app->ws_gui_html),
					"button-press-event");
		g_timer_start(ws_gui_app->timer);
		gtk_timeout_add((guint)(PRESS_TIME*1000),
				(GtkFunction)(ws_gui_show_popup),
				ws_gui_app);
		return FALSE;
	}
	else
	{
		ws_gui_app->ws_gui_sel_flag = FALSE;
		return FALSE;
	}

}

/** handles button release event and examines whether 'click' or 'tap and hold' 
event it supposed to be
*
* @param widget - object which received a signal
* @param event - type of event which has been performed
* @param user_data - user data set when the signal handler was connected 
* @return TRUE to stop other handlers from being invoked for the event. 
FALSE to propagate the event further
*/
gboolean ws_gui_button_release (GtkWidget *widget,
				GdkEventButton *event,
				gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;
	gdouble tmp = g_timer_elapsed(ws_gui_app->timer, NULL);
	g_timer_stop(ws_gui_app->timer);
	g_timer_reset(ws_gui_app->timer);
	if (ws_gui_app->ws_gui_double_click == TRUE)
	{
		ws_gui_app->ws_gui_double_click = FALSE;
		return FALSE;
	}
	else if (tmp < PRESS_TIME)
	{
		struct _GtkHTML *tmp = (struct _GtkHTML *)
					(ws_gui_app->ws_gui_html);
		html_engine_unselect_all(tmp->engine);
		return TRUE;
	}
		return FALSE;
}

/** displays popup menu if user holds a stylus more than PRESS_TIME seconds
*
* @param user_data - user data set when the signal handler was connected 
* @return 
*/
guint ws_gui_show_popup (gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;
	gdouble tmp = g_timer_elapsed(ws_gui_app->timer, NULL);
	if (tmp > PRESS_TIME)
	{
		ws_gui_create_popup_menu(ws_gui_app);
	}
	return (guint)(FALSE);

}

/** checks clipboard content and sets sensitivity of widgets 
*
* @param widget - object which recived a signal
* @param user_data - user data set when the signal handler was connected 
* @return 
*/
void ws_gui_check_clipboard (GtkWidget *widget, gpointer user_data) 
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;
	if (gtk_clipboard_wait_for_text(ws_gui_app->ws_gui_clipboard) != NULL)
	{
		gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_paste),
			TRUE);
	}
	else
	{
		gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_paste),
			FALSE);
	}

	struct _GtkHTML *tmp = (struct _GtkHTML *)(ws_gui_app->ws_gui_html);

	if (html_engine_is_selection_active(tmp->engine) == TRUE)
	{
		gtk_widget_set_sensitive(GTK_WIDGET
				(ws_gui_app->ws_gui_menu->ws_gui_menu_copy),
				TRUE);
	}
	else 
	{
		gtk_widget_set_sensitive(GTK_WIDGET
				(ws_gui_app->ws_gui_menu->ws_gui_menu_copy),
				FALSE);
	}

	if (ws_gui_app->html_flag == FALSE)
	{
		gtk_widget_set_sensitive(GTK_WIDGET
			(ws_gui_app->ws_gui_menu->ws_gui_menu_select_all),
			FALSE);
	}
	else
	{
		gtk_widget_set_sensitive(GTK_WIDGET
			(ws_gui_app->ws_gui_menu->ws_gui_menu_select_all),
			TRUE);
	}	

}

/** gets received string (with translation), formats it and sends it to be 
* displayed
*
* @param received_string - content received from Manager
* @param user_data - user data set when the function has been called
* @return 
*/
gchar * format_html (gchar * received_string, gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;
	GString *str_final;
	GString *str_tmp;
	str_final = g_string_new(received_string);
	str_tmp = g_string_new(received_string);
	gchar * tmp;
	gchar * tmp2;
	gchar * returned_value;
	
	ws_gui_app->raw_translation=g_string_erase(ws_gui_app->raw_translation, 
						0, 
						-1);

	while (strstr(str_final->str, "<PATTERN_OPEN>") != NULL)
	{
		tmp = strstr(str_final->str, "<PATTERN_OPEN>");
		str_final =  g_string_erase(str_final,
					(long)(tmp - str_final->str),
					14);
		str_final =  g_string_insert(str_final,
		(long)(tmp - str_final->str),
		"<html><head></head><body><center><table width=\"95%\"><tr>"
		"<td><table border=1 width=100%><tr>"
		"<th background=\"file:/usr/share/pixmaps/ws_top.png\">"
		"<img align=left src=\"file:/usr/share/pixmaps/"
		"engine_xdxf_icon.png\"><font color=\"#eeeeee\">");
		
		tmp2 = strstr(str_tmp->str, "<PATTERN_OPEN>");
		if (ws_gui_app->last_word != NULL)
		{
			str_tmp =  g_string_erase(str_tmp,
					(long)(tmp2 - str_tmp->str),
					14 + strlen(ws_gui_app->last_word));
		}
	}

	while (strstr(str_final->str, "<PATTERN_CLOSED>") != NULL)
	{
		tmp = strstr(str_final->str, "<PATTERN_CLOSED>");
		str_final =  g_string_erase(str_final,
					(long)(tmp - str_final->str),
					16);
		str_final =  g_string_insert(str_final,
					(long)(tmp - str_final->str),
					"</font></th></tr></table>");

		tmp2 = strstr(str_tmp->str, "<PATTERN_CLOSED>");
		str_tmp =  g_string_erase(str_tmp,
					(long)(tmp2 - str_tmp->str),
					16);
	}

	while (strstr(str_final->str, "<TRANSLATION_OPEN>") != NULL)
	{
		tmp = strstr(str_final->str, "<TRANSLATION_OPEN>");
		str_final =  g_string_erase (str_final,
					(long)(tmp - str_final->str),
					18);
		str_final =  g_string_insert (str_final,
		(long)(tmp - str_final->str),
		"<table border=1 width=100%><tr><td background=\"file:/usr/"
		"share/pixmaps/ws_tra.png\">");

		tmp2 = strstr(str_tmp->str, "<TRANSLATION_OPEN>");
		str_tmp =  g_string_erase(str_tmp,
					(long)(tmp2 - str_tmp->str),
					18);
	}

	while (strstr(str_final->str, "<TRANSLATION_CLOSED>") != NULL)
	{
		tmp = strstr(str_final->str, "<TRANSLATION_CLOSED>");
		str_final =  g_string_erase(str_final,
					(long)(tmp - str_final->str),
					20);
		str_final =  g_string_insert(str_final,
					(long)(tmp - str_final->str),
					"</td></tr></table></center>");

		tmp2 = strstr(str_tmp->str, "<TRANSLATION_CLOSED>");
		str_tmp =  g_string_erase(str_tmp,
					(long)(tmp2 - str_tmp->str),
					20);
	}

	str_final = g_string_append(str_final,
				"</td></tr></table></body></html>");

	ws_gui_app->raw_translation = g_string_insert(
						ws_gui_app->raw_translation,
						0,
						str_tmp->str);
	
	returned_value = g_strdup(str_final->str);
	g_string_free(str_final, TRUE);
	g_string_free(str_tmp, TRUE);
	return returned_value;
}

/** open bookmarks database 
*
* @param menuitem - object which recived the signal
* @param user_data - user data set when the signal handler was connected 
* @return
*/
void ws_gui_dictionary_open_bookmark(GtkMenuItem *menuitem, gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;

	ws_gui_read_active_dictionaries(ws_gui_app);
	
	/*list has to be cleared if we're gonna use a completely different dict*/
	ws_gui_clear_list(ws_gui_app->ws_gui_w_list->ws_gui_store, ws_gui_app);
	g_free(ws_gui_app->last_word);
	ws_gui_app->last_word = NULL;

	ws_gui_app->bookmark_mode = TRUE;
	ws_gui_set_bookmarks_sensitivity(ws_gui_app);
	g_object_set(G_OBJECT(ws_gui_app->ws_gui_hildon_find_toolbar),
				"prefix",
				"*",
				NULL);
	ws_dbus_notify(ws_gui_app->dbus_data, WS_DBUS_BOOKMARK_MODE_ON);
	
	if(ws_gui_app->ws_gui_history->len >= 0)
	{
	ws_gui_clear_array(ws_gui_app->ws_gui_history);
	ws_gui_clear_array(ws_gui_app->ws_gui_history_list);
	ws_gui_clear_array(ws_gui_app->ws_gui_history_iter);
	ws_gui_app->ws_gui_history_cur_pos = -1;
	}
	ws_gui_check_history(ws_gui_app);
	ws_gui_fill_html(" ", ws_gui_app);
	ws_gui_app->html_flag = FALSE;
	ws_dbus_client_find_word(ws_gui_app->dbus_data, "*");
	gtk_widget_hide(ws_gui_app->ws_gui_list_hbox);
	
	ws_gui_search(NULL, ws_gui_app);
}

/** closes bookmarks database 
*
* @param menuitem - object which recived the signal
* @param user_data - user data set when the signal handler was connected 
* @return
*/
void ws_gui_dictionary_close_bookmark(GtkMenuItem *menuitem, gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;

	ws_gui_read_active_dictionaries(ws_gui_app);
	
	g_free(ws_gui_app->last_word);
	ws_gui_app->last_word = NULL;

	ws_gui_app->bookmark_mode = FALSE;
	ws_gui_set_bookmarks_sensitivity(ws_gui_app);	
	ws_dbus_notify(ws_gui_app->dbus_data, WS_DBUS_BOOKMARK_MODE_OFF);

	if(ws_gui_app->ws_gui_history->len >= 0)
	{
	ws_gui_clear_array(ws_gui_app->ws_gui_history);
	ws_gui_clear_array(ws_gui_app->ws_gui_history_list);
	ws_gui_clear_array(ws_gui_app->ws_gui_history_iter);
	ws_gui_app->ws_gui_history_cur_pos = -1;
	}
	ws_gui_check_history(ws_gui_app);

	ws_gui_fill_html(" ", ws_gui_app);
	ws_gui_app->html_flag = TRUE;
	if (g_strstrip(ws_gui_app->last_searched) != NULL && ws_gui_app->first_run == FALSE) 
	{
	g_object_set(G_OBJECT(ws_gui_app->ws_gui_hildon_find_toolbar),
				"prefix",
				ws_gui_app->last_searched,
				NULL);

	ws_gui_search(NULL, ws_gui_app);
	}
	else
	{
		gtk_list_store_clear(ws_gui_app->ws_gui_w_list->ws_gui_store);
	}

}

/** adds bookmark to bookmarks database 
*
* @param menuitem - object which recived the signal
* @param user_data - user data set when the signal handler was connected 
* @return
*/
void ws_gui_dictionary_add_bookmark(GtkMenuItem *menuitem, gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;

	if (ws_gui_app->last_word != NULL)
	{
		ws_gui_add_bookmark_dialog(ws_gui_app);
	} 
	else
	{
		hildon_banner_show_information(GTK_WIDGET(
				ws_gui_app->ws_gui_hildon_window),
				NULL,
				_("ws_ni_select_word_to_add"));
	}
}

/** removes bookmark from bookmarks database 
*
* @param menuitem - object which recived the signal
* @param user_data - user data set when the signal handler was connected 
* @return
*/
void ws_gui_dictionary_remove_bookmark(GtkMenuItem *menuitem, 
				gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;

	if (ws_gui_app->last_word != NULL)
	{
		ws_gui_remove_bookmark_dialog(ws_gui_app);

		ws_gui_clear_list(ws_gui_app->ws_gui_w_list->ws_gui_store, ws_gui_app);
		gchar *temp;
		g_object_get(G_OBJECT(ws_gui_app->ws_gui_hildon_find_toolbar),
				"prefix",
				&temp,
				NULL);
		ws_dbus_client_find_word (ws_gui_app->dbus_data, temp);
		
	}
	else
	{
		hildon_banner_show_information(GTK_WIDGET(
				ws_gui_app->ws_gui_hildon_window),
				NULL,
			_("ws_ni_select_word_to_remove"));
	}
	
	
}

/** checks wheather dictionaries are available or not; according to the result
* dimmes proper menuitem 
*
* @param menuitem - object which recived the signal
* @param user_data - user data set when the signal handler was connected 
* @return
*/
void ws_gui_dict_availablity(GtkMenuItem *menuitem, gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;
	if (gconf_client_dir_exists(ws_gui_app->client, 
				GCONF_PATH, NULL) == TRUE)
	{
		ws_gui_app->directories = gconf_client_all_dirs(
							ws_gui_app->client,
							GCONF_PATH,
							NULL
							);
		
	
		if(g_slist_length(ws_gui_app->directories) != 0)
		{
			gtk_widget_set_sensitive(GTK_WIDGET
			(ws_gui_app->ws_gui_menu->ws_gui_menu_remove_dict),
			TRUE);
			gtk_widget_set_sensitive(GTK_WIDGET
			(ws_gui_app->ws_gui_menu->ws_gui_menu_select_dict),
			TRUE);
			gtk_widget_set_sensitive(GTK_WIDGET
			(ws_gui_app->ws_gui_menu->ws_gui_menu_optimize_dict),
			TRUE);
		}

	}
	else 
	{
	gtk_widget_set_sensitive(GTK_WIDGET
			(ws_gui_app->ws_gui_menu->ws_gui_menu_remove_dict),
			FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET
			(ws_gui_app->ws_gui_menu->ws_gui_menu_select_dict),
			FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET
			(ws_gui_app->ws_gui_menu->ws_gui_menu_optimize_dict),
			FALSE);
	}

}

/** sets sensitivity of menu items due to bookmark's mode
*
* @param user_data - user data set when the function was called 
* @return
*/
void ws_gui_set_bookmarks_sensitivity(gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;
	int i;
	gchar *string;
	gchar *name;

	if (ws_gui_app->bookmark_mode == TRUE)
	{
		g_slist_free(ws_gui_app->directories_last);
		ws_gui_app->directories_last = NULL;
		ws_gui_app->directories_last = g_slist_alloc();
		g_slist_free(ws_gui_app->directories);
		ws_gui_app->directories = g_slist_alloc();

		ws_gui_app->directories = gconf_client_all_dirs(
							ws_gui_app->client, 
							GCONF_PATH, NULL);
		
	for (i=0; i<g_slist_length(ws_gui_app->directories); i++)
	{
		string = (gchar*)g_slist_nth_data(ws_gui_app->directories, i);
		name = g_path_get_basename(string);
		string = g_strconcat(string, "/active", NULL);
		
		if (gconf_client_get_bool(ws_gui_app->client, string, NULL) == TRUE) 
		{
			ws_gui_app->directories_last = 
				g_slist_append(ws_gui_app->directories_last, 
					string);
		}

		if (strcmp(name, "bookmarks") == 0)
		{
			gconf_client_set_bool(ws_gui_app->client, 
					string, 
					TRUE, 
					NULL);
		}
		else
		{
			gconf_client_set_bool(ws_gui_app->client, 
					string, 
					FALSE, 
					NULL);
		}
	}

	/*setting sensitivity of components*/
	gtk_widget_set_sensitive(
		GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_open_bookmark),
		FALSE);
	gtk_widget_set_sensitive(
		GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_close_bookmark),
		TRUE);
	gtk_widget_set_sensitive(
	GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_remove_bookmark),
	TRUE);
	gtk_widget_set_sensitive(
		GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_add_bookmark),
		FALSE);
	gtk_widget_set_sensitive(
		GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_dictionaries),
		FALSE);
	}
	else 
	{
	gchar *bookmarks_path = g_strconcat(GCONF_PATH, "/bookmarks/active", NULL);
	if (g_slist_length(ws_gui_app->directories_last) > 0)
	{
	gconf_client_set_bool(ws_gui_app->client, bookmarks_path, FALSE, NULL);
	}
	
	for (i=0; i<g_slist_length(ws_gui_app->directories_last); i++)
	{
		string = (gchar*)g_slist_nth_data(ws_gui_app->directories_last, 
						i);
		
		if (string != NULL)
		{
		gconf_client_set_bool(ws_gui_app->client, 
				string, 
				TRUE, 
				NULL);
		}
	}

	/*setting sensitivity of components*/
	gtk_widget_set_sensitive(
		GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_open_bookmark),
		TRUE);
	gtk_widget_set_sensitive(
		GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_close_bookmark),
		FALSE);
	gtk_widget_set_sensitive(
	GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_remove_bookmark),
	FALSE);
	gtk_widget_set_sensitive(
		GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_add_bookmark),
		TRUE);
	gtk_widget_set_sensitive(
		GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_dictionaries),
		TRUE);

	g_free(bookmarks_path);
	}

}

/** sets sensitivity of menu items due to user selection made
*
* @param user_data - user data set when the function was called 
* @return
*/
void ws_gui_set_bookmark_menu_items(GtkMenuItem *menuitem, gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;
	if (ws_gui_app->bookmark_mode == TRUE && ws_gui_app->last_word != NULL)
	{
	gtk_widget_set_sensitive(
	GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_remove_bookmark),
	TRUE);
	}
	else
	{
	gtk_widget_set_sensitive(
	GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_remove_bookmark),
	FALSE);
	}

	if (ws_gui_app->last_word == NULL || 
	(ws_gui_app->last_word[0] == '\0') ||
	ws_gui_app->bookmark_mode == TRUE)
	{
	gtk_widget_set_sensitive(
	GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_add_bookmark),
	FALSE);
	}
	else
	{
	gtk_widget_set_sensitive(
	GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_add_bookmark),
	TRUE);
	}
}
/** reads gconf entries and sets variables with a proper data; it is necessary
* for changing the bookmarks mode
*
* @param user_data - user data set when the function was called
* @return
*/
void ws_gui_read_active_dictionaries(gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;

	
	ws_gui_app->directories = gconf_client_all_dirs(ws_gui_app->client,
					GCONF_PATH,
					NULL);
}

/** Deletes all elements from a GArray
*
* @param history - a GArray to be acted upon
* @return
*/
void ws_gui_clear_array(GArray *history)
{
	if(history->len >= 0)
	{
		history = g_array_remove_range(history, 0, history->len);
	}
}

/** Fixes a bug in hildon GtkTreeView(check if the GtkTreeView widget isn't
 * unnecessarily cleared and initialized 
 *
 * @param treeview - a treeview to act upon
 * @param user_data - user data set when the function was called
 * @return
 */
void ws_gui_view_cursor_changed(GtkTreeView *treeview, gpointer user_data)
{
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;
	gchar *string;
	if (gtk_tree_selection_get_selected(ws_gui_app->ws_gui_w_list->ws_gui_selection,
				&ws_gui_app->ws_gui_w_list->ws_gui_model,
				&ws_gui_app->ws_gui_w_list->ws_gui_iter))
	{
	gtk_tree_model_get(ws_gui_app->ws_gui_w_list->ws_gui_model,
				&ws_gui_app->ws_gui_w_list->ws_gui_iter,
				COL_WORD,
				&string,
				-1);
	if (string != NULL)
		{
		ws_gui_app->loading = TRUE;
		ws_gui_set_toolbar_avail(ws_gui_app);

		ws_dbus_client_find_translation(ws_gui_app->dbus_data, string);
		
		
		g_free(ws_gui_app->last_word);
		ws_gui_app->last_word = NULL;
		ws_gui_app->last_word = g_strdup (string);

		if (ws_gui_app->history_flag == FALSE)
		{
		ws_gui_history_add(string, ws_gui_app);
		
		}
		g_free (string);
		}
		
	}
	
	
}

/** Sets toolbar availability
*
* @param user_data - user data set when the function was called
* @return
*/
void ws_gui_set_toolbar_avail(gpointer user_data) {
	
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;
	
	if (ws_gui_app->loading == TRUE)
	{
		gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_toobar_button_stop),
			TRUE);
	
		gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_search),
			FALSE);

	}
	else 
	{
		gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_toobar_button_stop),
			FALSE);
	
		gtk_widget_set_sensitive(
			GTK_WIDGET(ws_gui_app->ws_gui_menu->ws_gui_menu_search),
			TRUE);
	}
}

/** refreshes search results
*
* @param user_data - user data set when the function was called
* @return
*/
void ws_refresh_search_results(gpointer user_data) {
	
	WSGuiApp* ws_gui_app=(WSGuiApp*)user_data;

	/*efreshing the words list and translation pane*/
	gchar *temp;
	g_object_get(G_OBJECT(ws_gui_app->ws_gui_hildon_find_toolbar),
				"prefix",
				&temp,
				NULL);
	
	if (ws_gui_app->first_run == FALSE)
	{
		ws_dbus_client_find_word (ws_gui_app->dbus_data, temp);
	}
}
/*@}*/
