/*******************************************************************************
This file is part of mdictionary.

mdictionary is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

mdictionary is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License 
along with mdictionary; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Copyright 2006 ComArch S.A.
*******************************************************************************/
#include <pc-instances.h>

//------------------------------------------------------------------------------
/** \brief Get comandline by which process in /proc tree was invoked.
 *
 * @param process describes which process from /proc tree we want to check. It
 * should be only concrete directory from /proc nothing else. e.g.: /proc/self,
 * /proc/8321 etc. Something like /proc or /proc/self is not good.
 * @return string containing comandline which has invoked process. If NULL there
 * were some problems, possibly parameter were wrong or ther is no such
 * a process.
 */
char* get_process_cmdline(char* process)
{
	// build final filename to read
	const char* postfix = "/cmdline";
	int arg_len = strlen(process);
	char* tmp = (char*)malloc((size_t)(arg_len+9));
	tmp[arg_len+8] = '\0';
	sprintf(tmp,"%s%s",process,postfix);
	//printf("Process to check: %s\n",tmp);

	// open file to read
	int file = open(tmp, O_RDONLY);
	free(tmp); tmp = 0;
	if(-1 == file)
	{
		return (char*)0;
	}

	int length = 512;	

	// buffer for data from /proc/self/cmdline file
	char* buffer = (char*)malloc(length + 1);

	// read file to buffer
	int readed = read(file, buffer, length);
	close(file);
	if(readed == length)
	{
		free(buffer);
		return (char*)0;
	}
	buffer[length] = '\0';

	//printf("%s() returned finaly: %s\n",__FUNCTION__,buffer);
	return buffer;
}
//------------------------------------------------------------------------------
/** \brief Cut from commandline only program file name.
 *
 * @param cmdline commandline which from which we want to remove options.
 * @return string with only program file name, or NULL if there was any problem
 */
char* get_process_program_file(char* cmdline)
{
	int start = 0;
	int stop  = 0;
	// find first space in buffer (to remove optional options etc.)
	while(cmdline[stop] && cmdline[stop] != ' ')
	{
		++stop;
	}
	cmdline[stop] = '\0';

	// find last "/" to remove directories 
	start = stop;
	while((start>0) && (cmdline[start] != '/'))
	{
		--start;
	}
	if(cmdline[start] == '/')
	{
		++start;
	}

	// cut only needed fragment of buffer
	if(start == stop)
	{
		return (char*)0;
	}

	char* result = (char*)malloc(stop - start + 1);
	memcpy(result, cmdline + start, stop - start + 1);
	return result;
}
//------------------------------------------------------------------------------
/** \brief Get file name containing program executing by current process.
 *
 * @return file name of program or NULL if there were some problems.
 */
char* get_current_process_program_file()
{
	char* cmd = get_process_cmdline("/proc/self");
	if(!cmd)
	{
		return (char*)0;
	}
	char* name = get_process_program_file(cmd);
	free(cmd); cmd = 0;

	return name;
}
//------------------------------------------------------------------------------
/** \brief Check if there is another instance of current programm.
 *
 * @return integer telling how many processes were initiated with the same
 * program as curent process. If it is 1, it means there is no other procces.
 * If > 1 there is another instance.
 */
int is_already_running_this_application()
{
	char* app = get_current_process_program_file();
	int   count = 0;

	struct dirent* entry;
	DIR*   dir = opendir("/proc");
	struct stat st;

	char   path[512];
	sprintf(path,"%s","/proc/");
	char*  tmp = path + 6;

	while((entry = readdir(dir)) != NULL)
	{
		sprintf(tmp,"%s",entry->d_name);
		lstat(path,&st);

		if(S_ISDIR(st.st_mode))
		{
			char* cmd = get_process_cmdline(path);
			if(!cmd) continue;
			char* name = get_process_program_file(cmd);
			free(cmd); cmd = 0;
			if(!name) continue;
			//printf("%s: next programm: %s\n",app,name);

			if(strcmp(app,name) == 0) 
			{
				++count;
			}

			free(name); name = 0;
		}
	}
	closedir(dir); dir = NULL;
	free(app); app = 0;
	return count;
}
//------------------------------------------------------------------------------
