/*******************************************************************************
This file is part of mdictionary.

mdictionary is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

mdictionary is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License 
along with mdictionary; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Copyright 2006 ComArch S.A.
*******************************************************************************/
/** \addtogroup GUI
 */
/*@{*/
/** \file ws_gui_callbacks.h
 * \brief Declaration of structures used by ws_gui_callbacks.c
 *
 * \author Anna Gadomska \<anna.gadomska\@comarch.com\>
 * \date 05-11-2007
 * \version 1.0.0
 */
#ifndef _WS_GUI_CALLBACKS
#define _WS_GUI_CALLBACKS 

#ifdef __cplusplus
extern "C" {
#endif


#include <ws_gui.h>


#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>     
#include <unistd.h>   

#define TIMER_START TRUE
#define TIMER_STOP  FALSE


/* added by Dariusz Wiechecki - HISA */
void ws_gui_search_home_handler(GError *error, GArray *word, gpointer user_data);

/** \brief handle dbus signals
 *
 */
void ws_gui_signal_hander (GError *error, GArray *words, gpointer user_data);

/** \brief handle signal from dbus and updates progress bar(in order to caching)
 *
 */
void ws_dbus_progress_bar(GError *error, GArray *words, gpointer user_data);

/** \brief clean words list
 *
 */
void ws_gui_clear_list (GtkListStore* list, gpointer user_data);

/** \brief handle dbus signal and transfer words to the words list
 *
 */
void ws_gui_dbus_return_words (GError *error, 
			       GArray *words, 
 			       gpointer user_data);

/** \brief handle dbus signal and convert message into readable text 
 *
 */
void ws_gui_dbus_return_translation (GError *error, 
                                     GArray *words,
                                     gpointer user_data);

/** \brief handle dbus signal and load extracted dictionary 
 *
 */
void ws_dbus_server_return_extracted_bzip(GError *error, 
					  GArray *words, 
					  gpointer user_data);

/**  \brief is used to free memory, which was allocated
*
*/
void ws_gui_free_memory(gpointer user_data);

/**  \brief handle keys press signals (hardware keys in device)
*
*/
gboolean hildon_key_press_listener (GtkWidget * widget,
                                    GdkEventKey * keyevent,
                                    gpointer user_data);

/**  \brief allow to hide words list from menu
*
*/
void ws_gui_words_list_hide_from_menu(GtkCheckMenuItem *checkmenuitem,
                                      gpointer user_data);

/**  \brief allow to hide words list from toolbar
*
*/
void ws_gui_words_list_hide(GtkToggleButton *togglebutton, gpointer user_data);

/**  \brief add word to the words history
*
*/
void ws_gui_history_add(char *new_word, gpointer user_data);

/**  \brief display translation of word stored in the history (if available)
*
*/
void ws_gui_history_back(GtkButton *button, gpointer user_data);

/**  \brief display translation of word stored in the history (if available)
*
*/
void ws_gui_history_next(GtkButton *button, gpointer user_data);

/** \brief check history and disable buttons or uncheck checkbuttons 
 *
 */
void ws_gui_check_history(gpointer user_data);

/** \brief creates model and view of words list
 *
 */
GtkWidget * create_view_and_model (GArray *words_list, gpointer user_data);

/** \brief creates words list and fill it with content
 *
 */
GtkTreeModel * create_and_fill_model (GArray *words_list, gpointer user_data);

/**  \brief switch view between full screen and normal mode 
*
*/
void ws_gui_full_screen(GtkMenuItem *menuitem, gpointer user_data);

/**  \brief search for selected text in a dictionary
*
*/
void ws_gui_popup_search(GtkMenuItem *menuitem, gpointer user_data);

/**  \brief allows to select whole text in translation (html) area
*
*/
void ws_gui_html_select_all(GtkMenuItem *menuitem, gpointer user_data);

/**  \brief allows to copy text from translation (html) area into clipboard
*
*/
void ws_gui_html_copy(GtkMenuItem *menuitem, gpointer user_data);

/**  \brief allows to paste text from the clipboard into find toolbar entry
*
*/
void ws_gui_html_paste(GtkMenuItem *menuitem, gpointer user_data);

/**  \brief magnify translation (html) area
*
*/
void ws_gui_html_zoom_in(GtkMenuItem *menuitem, gpointer user_data);

/**  \brief zoom out translation (html) area.
*
*/
void ws_gui_html_zoom_out(GtkMenuItem *menuitem, gpointer user_data);

/**  \brief start search (sends typed word to the engine)
*
*/
void ws_gui_search(GtkWidget * widget, gpointer user_data);

/**  \brief  abort search
*
*/
void ws_gui_search_stop(GtkButton *button, gpointer user_data);

/**  \brief used when user closes the application
*
*/
void ws_gui_on_exit (GtkWidget *widget, GdkEvent *event, gpointer user_data);

/**  \brief used when user closes the application from menu
*
*/
void ws_gui_menu_quit(GtkMenuItem *menuitem, gpointer user_data);

/**  \brief sets content of translation (html) area
*
*/
void ws_gui_fill_html(char *html_context, gpointer user_data);

/** \brief read current adjustment value
 *
 */
void ws_gui_read_adjustment(gpointer user_data);

/** \brief necessary to load images into translation (html) area
 *
 */
void ws_gui_url_requested (GtkHTML *html, const char *url,
                           GtkHTMLStream *stream);

/** \brief handles button press event
 *
 */
gboolean ws_gui_button_press (GtkWidget *widget,
                              GdkEventButton *event,
                              gpointer user_data);

/** \brief handles button release event
 *
 */
gboolean ws_gui_button_release (GtkWidget *widget,
                                GdkEventButton *event,
                                gpointer user_data);

/** \brief shows popup when user clicks and holds
 *
 */
guint ws_gui_show_popup(gpointer user_data);

/** \brief checks clipboard content
 *
 */
void ws_gui_check_clipboard (GtkWidget *widget, gpointer user_data);

/** \brief checks dictionaries availablity
 *
 */
void ws_gui_dict_availablity(GtkMenuItem *menuitem, gpointer user_data);

/** \brief allows to format html content
 *
 */
gchar * format_html (gchar * received_string, gpointer user_data);

/** \brief sets sensitivity of Bookmarks menu items
 *
 */
void ws_gui_set_bookmarks_sensitivity(gpointer user_data);

/** \brief sets sensitivity of Bookmarks menu items due to selection made
 *
 */
void ws_gui_set_bookmark_menu_items(GtkMenuItem *menuitem, gpointer user_data);

/** \brief sets sensitivity of menu items due to dictionary mode 
 *
 */
void ws_gui_read_active_dictionaries(gpointer user_data);

/** \brief free a GArray and its elements.
 *
 */
void ws_gui_clear_array(GArray *history);

/** \brief escapes a bug in Hildon GtkTreeList
 *
 */
void ws_gui_view_cursor_changed(GtkTreeView *treeview, gpointer user_data);

/** \brief sets toolbar availability
 *
 */
void ws_gui_set_toolbar_avail(gpointer user_data);

/** \brief refreshes search results
 *
 */
void ws_refresh_search_results(gpointer user_data);
/*@}*/

#ifdef __cplusplus
}
#endif

#endif
/*@}*/
