/*******************************************************************************
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation;
version 2.1 of the License.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

Copyright 2006-2008 ComArch S.A.
*******************************************************************************/
/** \addtogroup DBusWrapperAPI
 */
/*@{*/
/** \file ws_dbus.c
 * \brief API for D-Bus Wrapper Library - code.
 *
 * Bodies of methods used to communicate between
 * dictionary UI and searching module
 * \author Krzysztof Sasiak \<krzysztof.sasiak\@comarch.com\>
 * \date 05-11-2007
 * \version 1.0.0
 * \see DBusWrapperAPI
 */
#include <ws_dbus.h>


struct _WSDBusMethodData
{
	GQuark id;
	gchar *name;
	gpointer handler;
	gpointer user_data;
	gchar* data_types;
};


typedef struct _WSDBusMethodData WSDBusMethodData;

static gchar* ws_dbus_get_ptr2method (GArray *methods, gchar* name );
static void ws_dbus_libosso_errors (osso_return_t result);
static void ws_dbus_fill_message (DBusMessage *msg, void *data);

/**This function will be used in future version of DBUS as soon as introspection
 * is implemented
 */
static gchar* ws_dbus_get_ptr2method (GArray *methods, gchar* name)
{
	guint i;
	gchar* temp;

	g_debug ("%d", methods->len);
	
	return NULL;
	
        for (i = 0; i < methods->len; i+=2)
        {
		temp = g_array_index (methods, gchar *,i);

		if (strcmp (name, temp) == 0) {
			temp = g_array_index (methods, gchar *,i+1);
			g_debug ("\nData types: %s\n", temp);
			return temp;
		}

        };

        return NULL;
};

static void ws_dbus_libosso_errors (osso_return_t result)
{
	switch (result) 
	{
		case OSSO_OK: 
			g_debug ("All OK\n");
		break;
		case OSSO_ERROR: 
			g_debug ("Ordinary Error\n");
		break;
		case OSSO_INVALID: 
			g_debug ("At least one parameter is invalid\n");
		break;
		case OSSO_RPC_ERROR: 
			g_debug ("Osso RPC method returned an error\n");
		break;
		case OSSO_ERROR_NAME: 
			g_debug ("Error Name\n");
		break;
		case OSSO_ERROR_NO_STATE: 
			g_debug ("No state file found to read\n");
		break;
		case OSSO_ERROR_STATE_SIZE: 
			g_debug("The size of the given structure");
			g_debug(" is different from the saved size\n");
		break;
	};
};


static WSDBusStatus ws_dbus_run_cb( WSDBusMethodData *method_data,
                                    gpointer data )
{
	g_debug ( "%s: Running cb for method %s\n",
	          __FUNCTION__,
	          method_data->name);

	ws_dbus_cb cb;
	osso_rpc_t osso_data;
	GArray *args = (GArray *) data;

	cb = (ws_dbus_cb) method_data->handler;
	g_debug ("cb pointer address %p\n", cb);

	if (cb != NULL) 
	{
		if (args->len == 0)
		{
			osso_data.type = WS_DBUS_TYPE_STRING;
			osso_data.value.s = NULL;
			g_array_append_val (args, osso_data);
		}
			
		cb (NULL, data, method_data->user_data); 
		return OSSO_OK;
	}
	else
	{
		g_debug ("DBUS: No callback defined for this function\n");
		return OSSO_ERROR;
	};
};

/* A dispatcher function */
static gint ws_dbus_cb_handler (const gchar * interface,
                                const gchar * method,
                                GArray * arguments,
                                gpointer data,
                                osso_rpc_t * retval)
{
	WSDBusMethodData *method_data;
	GArray *dbus_method_data;
	GQuark temp;
	guint i;
	dbus_method_data = (GArray *) data;
	retval = NULL;
	g_debug ("DBUS: Method:  %s\n", method);

	/* check if a unique identifier has been generated for this string */
	temp = g_quark_try_string (method); 
	if (temp != 0)
	{
		/*run through all available methods and run the one which was 
		called remotely*/

		for (i=0; i < dbus_method_data->len; ++i)
		{
			method_data = g_array_index( dbus_method_data,
			                             WSDBusMethodData*,
			                             i );

			if (method_data->id == temp) 
			{
				g_debug( "DBUS: Running callback for %s\n",
				         method);
				ws_dbus_run_cb (method_data, arguments);
			};
		};
	}
	else
	{
		g_debug( "Error in function %s, couldn't find the signal %s\n",
		         __FUNCTION__,
		         method );
		return OSSO_ERROR;
	};

	return OSSO_OK;
};


/** This function adds words stored in a GArray as arguments to a DBUS message 
 * \param msg - DBusMessage structure
 * \param data - GArray of osso_rpc_t containing arguments and its types
 */
static void ws_dbus_fill_message (DBusMessage *msg, void *data) 
{
	guint i;
	osso_rpc_t *data_unit;
	GArray *data_bundle;
	gpointer buffer;

	data_bundle = (GArray *) data;
	
	g_debug ("%s, number of fields: %d", __FUNCTION__, data_bundle->len);
	
	if (data_bundle->len > 255)
	{
		g_debug ("DBUS: Number of fields in message exceeds 255\n");
	}

	for (i = 0; ((i < data_bundle->len) && (i < 255)); ++i)
	{
		data_unit = g_array_index (data_bundle, osso_rpc_t *, i);
		g_debug ("%s, type: %c, value: ", __FUNCTION__, data_unit->type);

		switch (data_unit->type)
		{
			case DBUS_TYPE_STRING:
				buffer = &data_unit->value.s;
				g_debug ("%s", data_unit->value.s);
			break;
			case DBUS_TYPE_INT32:
				buffer = &data_unit->value.i;
				g_debug ("%d", data_unit->value.i);
			break;
			case DBUS_TYPE_DOUBLE:
				buffer = &data_unit->value.d;
				g_debug ("%f", data_unit->value.d);
			break;
			case DBUS_TYPE_BOOLEAN:
				buffer = &data_unit->value.b;
				g_debug ("%d", data_unit->value.b);
			break;
		};
		
                if (data_unit->value.s != NULL) {
			dbus_message_append_args( msg,
			                          data_unit->type,
			                          buffer,
			                          DBUS_TYPE_INVALID );
		}
		else
		{
			if (data_bundle->len > 1)
			{
				g_debug ("One argument ommited");
			}
		}
	};

	g_debug ("DBUS: Added %d words\n", i);
};

/**Function will be used in future version of dbus wrapper library, as soon as
 *introspection is implemented
 *\param error - a GError error
 *\param data - a GArray of methods (listed in a string form)
 *\param user_data - a pointer WSDBusData structure
*/
void ws_dbus_add_remote_methods (GError *error,
                                 gpointer data,
                                 gpointer user_data)
{
	WSDBusData *ws_dbus_data = (WSDBusData *) user_data;
	GArray *methods_list = (GArray *) data;
	guint i=0;

	osso_rpc_t *temp;

	for (i = 0; i < methods_list->len; ++i)
	{
		temp = g_array_index (methods_list, osso_rpc_t *, i);
		g_array_append_val( ws_dbus_data->remote_method_data,
		                    temp->value );
	};

	return;
};

/** This function is used for allocating memory for the WSDBusData structure and 
 * setting some basic parameters. 
 * \param name - name of the application
 * \param version - app's version
 * \return pointer to WSDBusData structure\n
 * \sa WSDBusData
 */
WSDBusData * ws_dbus_create (gchar *name, gchar *version)
{
	WSDBusData *temp;
	temp = (gpointer) g_try_malloc (sizeof (WSDBusData));
	g_debug ("\nDBUS: ");

	if (temp == NULL)
	{
		g_debug ("Error in function %s - cannot allocate memory\n",
			__FUNCTION__);
		g_assert_not_reached();
	}
	else
	{
		g_debug ("Memory allocation successful\n");
	}

	temp->name = g_strconcat (name, NULL);
	temp->version = g_strconcat (version, NULL);

	temp->method_data = g_array_new( TRUE,
	                                 TRUE,
	                                 sizeof(WSDBusMethodData *) );
	return temp;
};

/** Setting DBUS wrapper configuration.
 * \param ws_dbus_data - pointer to a structure uniquely identifying 
 * the application for DBus. The possible values of 
 * this parameter are listed here: ::WSDBusConfig
 * \param field - the name of the variable, this function is supposed to set. 
 * \param value - the value of the variable
 * \return WS_STATUS_OK - on success \n
 * WS_STATUS_ERROR - on error
 * \sa WSDBusConfig
 */
WSDBusStatus ws_dbus_config (WSDBusData * ws_dbus_data,
                             WSDBusConfig field,
                             gchar *value)
{
	if (ws_dbus_data == NULL) 
	{
		g_debug ("DBUS: Error in function %s - ws_dbus_data is NULL\n",
		         __FUNCTION__);
		return WS_DBUS_STATUS_ERROR;
	};

	switch (field)
	{
		case WS_DBUS_CONFIG_SERVICE :
			ws_dbus_data->service = g_strconcat(value, NULL);
		break;
		case WS_DBUS_CONFIG_OBJECT :
			ws_dbus_data->object = g_strconcat(value, NULL);
		break;
		case WS_DBUS_CONFIG_IFACE :
			ws_dbus_data->iface = g_strconcat(value, NULL);
		break;
		case WS_DBUS_CONFIG_REMOTE_SERVICE :
			ws_dbus_data->remote_service = g_strconcat(value, NULL);
		break;
		case WS_DBUS_CONFIG_REMOTE_OBJECT :
			ws_dbus_data->remote_object = g_strconcat(value, NULL);
		break;
		case WS_DBUS_CONFIG_REMOTE_IFACE :
			ws_dbus_data->remote_iface = g_strconcat(value, NULL);
		break;
	};

	return WS_DBUS_STATUS_OK;
};

/** It should be the called after <B>ws_dbus_create</B> and after setting all 
 * DBUS parameters with the <B>ws_dbus_config</B> function.
 * The structure it returns is necessary to call every function implemented in
 * this wrapper library.
 * \param ws_dbus_data - it's a structure containing the unique paths to objects 
 * and interfaces to be registered in the DBus session daemon
 * \return WS_STATUS_OK - on success \n
 * WS_STATUS_ERROR - on error
 * \sa WSDBusConfig | ws_dbus_create | ws_dbus_config
 */
WSDBusStatus ws_dbus_connect (WSDBusData * ws_dbus_data)
{
	osso_return_t result;
	/* initialize osso_rpc */
	ws_dbus_data->context = osso_initialize (ws_dbus_data->name,
	                                         ws_dbus_data->version,
	                                         FALSE,
	                                         NULL);
	osso_rpc_set_timeout (ws_dbus_data->context, 100);

	/* set a dispatcher function which runs local methods when asked via 
	 * DBUS */
	result = osso_rpc_set_cb_f(ws_dbus_data->context,
	                           ws_dbus_data->service,
	                           ws_dbus_data->object,
	                           ws_dbus_data->iface,
	                           ws_dbus_cb_handler,
	                           ws_dbus_data->method_data);

	g_debug ("\nSending method data\n");
	/*sends methods list to remote receiver
	 * ws_dbus_send_method_data (ws_dbus_data); */

	if (result == OSSO_OK)
	{
		return WS_DBUS_STATUS_OK;
	}
	else
	{
		return WS_DBUS_STATUS_ERROR;
	}
};

/** Before the program exits, all memory allocated by DBUS wrapper should be 
 * freed.
 * \param ws_dbus_data - pointer to a structure uniquely identifying the 
 * application for DBus
 * \return \li WS_STATUS_OK - on success \li WS_STATUS_ERROR - on error
 */
void ws_dbus_destroy (WSDBusData * ws_dbus_data)
{
	guint i;

	if (ws_dbus_data == NULL) 
	{
		g_debug ("DBUS: Error in fun %s - cannot free osso_context\n",
		         __FUNCTION__);
		g_free (ws_dbus_data);
		g_assert_not_reached();
	};

	g_debug( "DBUS deinitialization by %s:\n---------------------------\n",
	         ws_dbus_data->service);
	osso_deinitialize (ws_dbus_data->context);
	g_debug ("| Deinitializing osso context    |\n");
	if (ws_dbus_data->method_data != NULL) 
	{
		for (i = 0; i < ws_dbus_data->method_data->len; ++i)
		{
			WSDBusMethodData *temp = 
					g_array_index(ws_dbus_data->method_data,
					WSDBusMethodData *, i);
			g_free (temp->data_types);
			g_free (temp->name);
			g_free (temp);
		};

		g_array_free (ws_dbus_data->method_data, TRUE);
		g_debug ("| Freeing callback pointers list |\n");
	};

	g_free (ws_dbus_data->name);
	g_free (ws_dbus_data->version);
	g_free (ws_dbus_data->service);
	g_free (ws_dbus_data->object);
	g_free (ws_dbus_data->iface);

	g_free (ws_dbus_data->remote_service);
	g_free (ws_dbus_data->remote_object);
	g_free (ws_dbus_data->remote_iface);
	
	/* Introspection not implemented yet. This piece of code will be used 
	 *in future version.*/
	/* if (ws_dbus_data->remote_method_data != NULL) 
	 * {
	 *	for (i = 0; i < ws_dbus_data->remote_method_data->len; ++i)
	 *	{
	 *		g_free (g_array_index(ws_dbus_data->remote_method_data,
	 *			gchar*, i));
	 *	};
	 *	g_array_free (ws_dbus_data->remote_method_data, TRUE);
	 * };
         */

	g_free (ws_dbus_data);
	g_debug ("| Freeing WSDBusData structure   |\n----------------------\n");
};


/** \param ws_dbus_data - pointer to a structure uniquely identifying the 
 * application for D-BUS
 * \param detailed_signal - string containing the signal name which will trigger 
 * the function passed in the c_func parameter.
 * \param c_func - is a pointer to the signal handling function.
 * \param user_data - is the data passed directly to the signal handling
 *  function
 * \return \li WS_STATUS_OK - on success \li WS_STATUS_ERROR - on error
 * \sa WSDBusConfig | ws_dbus_cb
 */
WSDBusStatus ws_dbus_set_cb (WSDBusData * ws_dbus_data,
                             gchar * method,
                             gpointer c_func,
                             gpointer user_data)
{
	WSDBusMethodData *method_data;
	GQuark temp;
	guint i;

	/* check if a unique GQuark id for this method has been generated */
	temp = g_quark_try_string  (method);

	if (temp == 0) 
	{
		g_debug ("DBUS:%s- method not defined or invalid method name\n",
		         method);
		return WS_DBUS_STATUS_ERROR;
	}
	else
	{
		for (i = 0; i < ws_dbus_data->method_data->len; ++i)
		{
			method_data = g_array_index (ws_dbus_data->method_data,
			                             WSDBusMethodData *,
			                             i);

			/* Add method data (callback pointer and data) for a 
			 * given method */
			if (method_data->id == temp)
			{
				method_data->handler = c_func;
				method_data->user_data = user_data;
				g_array_insert_val(ws_dbus_data->method_data,
				                   i, method_data);
				g_array_remove_index (ws_dbus_data->method_data,
				                      i+1);
			};
		};
	};

	/* ws_dbus_send_method_data (ws_dbus_data); */
	return WS_DBUS_STATUS_OK;
};

/** \param ws_dbus_data - pointer to a structure uniquely identifying the 
 * application for DBus
 * \param method - method name\n
 * \param arglist - a comma separated list of method's arguments (actually just
 * their datatypes) finished with WS_DBUS_TYPE_INVALID
 * \return \li WS_STATUS_OK - on success \li WS_STATUS_ERROR - on error
 * \sa WSDBusDataType
 */
WSDBusStatus ws_dbus_add_method (WSDBusData * ws_dbus_data, gchar *method, ...)
{
	WSDBusMethodData *method_data;
	WSDBusDataType data_type;
	GQuark temp;
	gchar *buffer;
	va_list arglist;

	/* Check if a unique GQuark id exists for a given string */
	temp = g_quark_try_string (method);

	if (temp != 0)
	{
		g_debug( "DBUS:\"%s\"- method has already been registered\n",
		         method );
		return WS_DBUS_STATUS_ERROR;
	}

	method_data = g_try_malloc (sizeof (WSDBusMethodData));
	if (method_data != NULL)
	{
		va_start (arglist, method);

		method_data->id = g_quark_from_string (method);
		method_data->name = g_strdup (method);
		data_type = va_arg (arglist, WSDBusDataType);
		method_data->data_types = g_strdup("");

		/* Add all datatypes in arglist as arguments to method */
		while (data_type != WS_DBUS_TYPE_INVALID)
		{
			buffer = method_data->data_types;

			method_data->data_types = 
				g_strdup_printf( "%s%c",
				                 method_data->data_types,
				                 (gchar) data_type );
			g_free (buffer);
			data_type = va_arg (arglist, WSDBusDataType);
		};

		g_debug( "%d, %s, %s\n",
		         method_data->id,
		         method_data->name,
		         method_data->data_types );
		va_end (arglist);
		g_array_append_val( ws_dbus_data->method_data,
		                    method_data );
	}
	else
	{
		g_debug( "DBUS: Failed to allocate memory for method data");
		return WS_DBUS_STATUS_ERROR;
	}

	return WS_DBUS_STATUS_OK;
};

/** Converts an array of strings to an array of osso_rpc_t which is necessary
 * then sending data via DBUS
 * \param data_bundle - a preallocated GArray of osso_rpc_t
 * \param strings - a GArray of strings to be converted
 * \return 
 */
WSDBusStatus ws_dbus_add_garray (GArray *data_bundle, GArray *strings)
{
	osso_rpc_t *temp;
	guint i;

	if (strings != NULL)
	{
		for (i = 0; i < strings->len; ++i)
		{
			temp = g_try_malloc (sizeof (osso_rpc_t));
			temp->type = WS_DBUS_TYPE_STRING;
			temp->value.s = g_array_index (strings, gchar *, i);
			g_debug ("%d, %s", temp->type, temp->value.s);
			g_array_append_val (data_bundle, temp);
		};
		return WS_DBUS_STATUS_OK;
	}
	else
	{
		printf ("\n%s - Error - GArray is NULL", __FUNCTION__);
		return WS_DBUS_STATUS_ERROR;
	};
};

/**Runs a remote method with specific name
 *
 * \param ws_dbus_data - pointer to a structure uniquely identifying the 
 * application for DBus
 * \param method - method name
 * \param arglist - a comma separated list of pairs: datatype, value finished 
 * with WS_DBUS_TYPE_INVALID
 * \return \li WS_STATUS_OK - on success \li WS_STATUS_ERROR - on error
 * \sa WSDBusDataType
 */
WSDBusStatus ws_dbus_call_method (WSDBusData * ws_dbus_data, gchar *method, ...)
{
	va_list arglist;
	GArray *data_to_send;
	osso_rpc_t *temp, *retval;
        osso_return_t result;
	GArray *temp_garray;
	
	data_to_send = g_array_new (TRUE, TRUE, sizeof (osso_rpc_t *));
	va_start (arglist, method);

	/* convert all given arguments to a GArray of osso_rpc_t */
	while (TRUE)
	{
		temp = g_try_malloc (sizeof (osso_rpc_t));
		temp->type = va_arg (arglist, WSDBusDataType);
		/* g_debug ("Type: %c ", temp->type); */
		if (temp->type == WS_DBUS_TYPE_INVALID) 
		{
			break;
		}

		switch (temp->type)
		{
			case WS_DBUS_TYPE_STRING:
				temp->value.s = va_arg (arglist, gchar *);
				g_debug ("Value: %s\n", temp->value.s);
			break;
			case WS_DBUS_TYPE_INT:
				temp->value.i = va_arg (arglist, gint);
				g_debug ("Value: %d\n", temp->value.i);
			break;
			case WS_DBUS_TYPE_UINT:
				temp->value.u = va_arg (arglist, guint);
				g_debug ("Value: %d\n", temp->value.u);
			break;
			case WS_DBUS_TYPE_DOUBLE:
				temp->value.d = va_arg (arglist, gdouble);
				g_debug ("Value: %f\n", temp->value.d);
			break;
			case WS_DBUS_TYPE_BOOLEAN:
				temp->value.b = va_arg (arglist, gboolean);
				g_debug ("Value: %d\n", temp->value.b);
			break;
			case WS_DBUS_TYPE_GARRAY:
				temp_garray = va_arg (arglist, GArray *);
				ws_dbus_add_garray (data_to_send, 
							temp_garray);
				g_debug ("Value: GArray of strings");
			break;
		};
		g_debug ("Type: %c %d", temp->type, temp->value.i);

		if (temp->type != WS_DBUS_TYPE_GARRAY)
		{
			g_array_append_val (data_to_send, temp);
		}
	};

	va_end (arglist);
	g_debug( "%s, method %s, added %d arguments",
	         __FUNCTION__,
	         method, data_to_send->len);
	
	/* allocate memory for return value */
	retval = g_try_malloc (sizeof (osso_rpc_t));
	if (retval == NULL) 
	{
		g_debug("DBUS: Error in function %s:",__FUNCTION__);
		g_debug("Couldn't allocate memory for message return value\n");
	};

	result = osso_rpc_run_with_argfill (ws_dbus_data->context,
	                                    ws_dbus_data->remote_service,
	                                    ws_dbus_data->remote_object,
	                                    ws_dbus_data->remote_iface,
	                                    method,
	                                    retval,
	                                    ws_dbus_fill_message,
	                                    data_to_send);

	g_debug ("\nDBUS: %s: ", __FUNCTION__);
	ws_dbus_libosso_errors (result);

	guint i;
	for (i = 0; i < data_to_send->len; ++i)
	{
		g_free (g_array_index(data_to_send, osso_rpc_t*,i));
	};

	g_array_free (data_to_send, TRUE);
	if (result != OSSO_OK)
	{
		g_debug ("Error message: %s\n", retval->value.s);
		osso_rpc_free_val (retval);
		g_free (retval);
		return WS_DBUS_STATUS_ERROR;
	};

	osso_rpc_free_val (retval);
	g_free (retval);
	return WS_DBUS_STATUS_OK;
};

/**This function will be used in future version of DBUS as soon as introspection
 * is implemented
 */
WSDBusStatus ws_dbus_call_registered_method (WSDBusData * ws_dbus_data,
                                             gchar *method, ...)
{
	va_list arglist;
	guint i;
	GArray *data_to_send;
	osso_rpc_t *temp;
	osso_return_t result;
	osso_rpc_t *retval;

	gchar* data_types = ws_dbus_get_ptr2method (
	                          ws_dbus_data->remote_method_data,
	                          method );
	g_debug ("\n%s\n", data_types);

	data_to_send = g_array_new (TRUE, TRUE, sizeof (osso_rpc_t *));

	va_start (arglist, method);

	for (i = 0; i < strlen (data_types); ++i)
	{
		temp = g_try_malloc (sizeof (osso_rpc_t));
		
		switch ((char) data_types[i])
		{
			case WS_DBUS_TYPE_STRING:
				temp->value.s = va_arg (arglist, gchar *);
			break;
			case WS_DBUS_TYPE_INT:
				temp->value.i = va_arg (arglist, gint);
			break;
			case WS_DBUS_TYPE_UINT:
				temp->value.u = va_arg (arglist, guint);
			break;
			case WS_DBUS_TYPE_DOUBLE:
				temp->value.d = va_arg (arglist, gdouble);
			break;
			case WS_DBUS_TYPE_BOOLEAN:
				temp->value.b = va_arg (arglist, gboolean);
			break;
		};
		g_array_append_val (data_to_send, temp);
	};
	va_end (arglist);

	retval = g_try_malloc (sizeof (osso_rpc_t));

	if (retval == NULL) 
	{
		g_debug("DBUS: Error in function %s:",__FUNCTION__);
		g_debug("Couldn't allocate memory for message return value\n");
	};

	result = osso_rpc_run_with_argfill (ws_dbus_data->context,
	                                    ws_dbus_data->remote_service,
	                                    ws_dbus_data->remote_object,
	                                    ws_dbus_data->remote_iface,
	                                    method,
	                                    retval,
	                                    ws_dbus_fill_message,
	                                    data_to_send);
	g_debug ("\nDBUS: %s: ", __FUNCTION__);

	ws_dbus_libosso_errors (result);

	if (result != OSSO_OK)
	{
		g_debug ("Error message: %s\n", retval->value.s);
		osso_rpc_free_val (retval);
		g_free (retval);
		return WS_DBUS_STATUS_ERROR;
	};

	osso_rpc_free_val (retval);
	g_free (retval);
	return WS_DBUS_STATUS_OK;
};

/*@}*/
