/******************************************************************************
This file is part of mdictionary.

mdictionary is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

mdictionary is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License 
along with mdictionary; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Copyright 2006-2008 ComArch S.A.
******************************************************************************/
/** \addtogroup BookmarkEngine
 */
/*@{*/
/** \file engine_bookmark.c
 * \brief Code of bookmark dictionary engine.
 */
#include "engine_bookmark.h"

#ifndef NOLOGS
	#include <glib/gstdio.h>
	#define LOGS g_debug
#else
	#define LOGS(frm,...) while(FALSE)
#endif


static gint bm_compare_key_trans(const DBT *a, const DBT *b)
{
	guint tmpa;// = (guint*)(a->data);
	guint tmpb;// = (guint*)(b->data);
	memcpy(&tmpa, a->data, sizeof(guint));
	memcpy(&tmpb, b->data, sizeof(guint));

	if(tmpa == tmpb) return 0;
	if(tmpa  > tmpb) return 1;
	else return -1;
}

static gint bm_compare_key_words(const DBT *a, const DBT *b)
{
	gchar* tmpa = g_utf8_casefold((gchar*)(a->data),-1);
	gchar* tmpb = g_utf8_casefold((gchar*)(b->data),-1);
	gint result = g_utf8_collate(tmpa,tmpb);
	g_free(tmpa); tmpa = NULL;
	g_free(tmpb); tmpb = NULL;
	return result;
}

static void bm_save_freeID(BookData* data)
{
	LOGS("Saveing new freeID=%u...\n",data->freeID);
	guint temp = 0;
	DBT key = {     &temp       , sizeof(guint)};
	DBT val = { &(data->freeID) , sizeof(guint) };

	gint res = data->db_trans->del(data->db_trans, &key, 0);
	if(-1 == res)
	{
		data->last_error = ENGINE_INTERNAL_ERROR;
		LOGS("Error while trying to delete old freeID!\n");
		return;
	}
	else
	{
		LOGS("Old freeID=%u deleted successfully!\n",data->freeID-1);
	}

	res = data->db_trans->put(data->db_trans, &key, &val, R_NOOVERWRITE);
	if(-1 == res || 1 == res)
	{
		data->last_error = ENGINE_INTERNAL_ERROR;
		LOGS("Error while trying to write new value for freeID!\n");
	}
	else
	{
		LOGS("New freeID=%u written successfully!\n",data->freeID);
	}

	res = data->db_trans->sync(data->db_trans, 0);
	if(-1 == res || 1 == res)
	{
		data->last_error = ENGINE_INTERNAL_ERROR;
		LOGS("Error while trying to write data to fuile!\n");
	}
	else
	{
		LOGS("New data saved successfully to file!\n");
	}	
}

static void bm_load_freeID(BookData* data)
{
	guint temp = 0;
	DBT key = { &temp , sizeof(guint) };
	DBT val = { NULL  , 0 };

	gint res = data->db_trans->get(data->db_trans, &key, &val, 0);
	if(-1 == res)
	{
		data->last_error = ENGINE_INTERNAL_ERROR;
		LOGS("Bookmark/%s->%s() Error while getting access to trans "
		     "database!",
		     (gchar*)__FILE__,
		     (gchar*)__FUNCTION__
		    );
	}
	else if( 1 != res)
	{
		memcpy(&(data->freeID), val.data, sizeof(guint));
		//data->freeID = *((guint*)(val.data));
		LOGS("Bookmark/%s->%s() Available next free ID is equal = %d",
		     (gchar*)__FILE__,
		     (gchar*)__FUNCTION__,
		     data->freeID
		    );
	}
	else
	{
		LOGS("Bookmark/%s->%s() Could not load the minimal, available"
		     " ID for next record - translation!",
		     (gchar*)__FILE__,
		     (gchar*)__FUNCTION__
		    );
		data->freeID = 1;
		data->last_error = ENGINE_INTERNAL_ERROR;
	}
}



gboolean bm_engine_add_word( Engine* engine,
                             gchar*  word,
                             gchar*  translation)
{
        LOGS("Bookmark/%s->%s() called. Param\nEngine at address: %p\n"
             "word: %s\ntranslation address: %p\n",
             (gchar*)__FILE__,
             (gchar*)__FUNCTION__,
             engine,
             word,
             translation
            );
        g_assert(engine != NULL);
        g_assert(word != NULL);
        g_assert(translation != NULL);

        /* start bm_timer for this function */
        /* bm_timer(TIMER_START, (gchar*)(gchar*)__FUNCTION__); */

	gboolean result = TRUE;
	BookData* data = (BookData*)(engine->engine_data);
	guint length = strlen(word) + 1;
	DBT key = { word , length };
	DBT val = { NULL , 0 };
		

	gint db_res = data->db_words->get(data->db_words, &key, &val, 0);
	if ( 0 == db_res )
	{
	// there is already entry for this key - check if this translation is in
	// database - check string hash equality
		LOGS("Bookmark/%s->%s() updating entry for key: %s",
             		(gchar*)__FILE__,
             		(gchar*)__FUNCTION__,
			(gchar*)(key.data)
            		);

		guint hash = g_str_hash(translation);
		guint* values = (guint*)(val.data);
		guint  N = values[0];
		memcpy(&N, values, sizeof(guint));
		guint  i = N;
		guint  tmp_hash = 0;
		gboolean exist = FALSE;
		++values;
		while(i--)
		{
			memcpy(&tmp_hash, values + (i*2+1), sizeof(guint));
			if( tmp_hash == hash)
			{
				exist = TRUE;
				break;
			}
		}
		if(!exist)
		{
			LOGS("Bookmark/%s->%s() Adding new translation to "
				"already exist word in dictionary.\n",
				(gchar*)__FILE__,
				(gchar*)__FUNCTION__
				);
			values = (guint*)(val.data);
			guint* tmp = g_malloc(val.size + sizeof(guint)*2);
			g_memmove(tmp, values, val.size);
			tmp[0]++;
			tmp[N*2+1] = data->freeID;
			tmp[N*2+2] = hash;
			val.data = tmp;
			val.size += 2*sizeof(guint);
			gint o = data->db_words->del(data->db_words,&key,0);
			if(0 != o)
			{
				LOGS("Bookmark/%s->%s() Error while removing!",
             			     (gchar*)__FILE__,
             			     (gchar*)__FUNCTION__
            			    );				
			}
			o = data->db_words->sync(data->db_words, 0);
			if(0 != o) {
				LOGS("Error while 1st synchronizing file with data!\n");
			}
			
			o = data->db_words->put(data->db_words,
					    &key,
					    &val,
					    R_NOOVERWRITE);
			if(0 != o) {
				LOGS("Error while putting new info about word!\n");
			}
			o = data->db_words->sync(data->db_words, 0);
			if(0 != o) {
				LOGS("Error while 2nd synchronizing file with data!\n");
			}
			bm_add_only_translation(data,translation,data->freeID);
			(data->freeID)++;
			bm_save_freeID(data);
			if(NULL != tmp) {
				g_free(tmp);
				tmp = NULL;
			}
		}
		else
		{
			LOGS("Bookmark/%s->%s() This translation already exist!",
				(gchar*)__FILE__,
				(gchar*)__FUNCTION__
				);
		}
	}
	else if ( 1 == db_res )
	{
	// there is no such a key - word in database - create new entry
		LOGS("Bookmark/%s->%s() adding new entry for key: %s",
             		(gchar*)__FILE__,
             		(gchar*)__FUNCTION__,
			(gchar*)(key.data)
            		);
		result = bm_add_new_entry(word,translation,data);
	}
	else {
	// there was an error, while accessing to the database
		LOGS("Bookmark/%s->%s() Error while trying to add new word: %s",
             		(gchar*)__FILE__,
             		(gchar*)__FUNCTION__,
			(gchar*)(key.data)
            		);
		data->last_error = ENGINE_COULDNT_READ;
		result = FALSE;
	}

        /* bm_timer(TIMER_STOP, (gchar*)(gchar*)__FUNCTION__); */
        return result;
}

static gboolean bm_add_only_translation(BookData* data,
					gchar* translation,
					guint id)
{
	DBT key = { &id , sizeof(id) };
	DBT val = { translation , strlen(translation) + 1 };
	gint res = data->db_trans->put(data->db_trans,
					&key,
					&val,
					R_NOOVERWRITE);
	if(-1 == res) {
		LOGS("Error while adding only translation!\n");
		return FALSE;
	}
	res = data->db_trans->sync(data->db_trans, 0);
	if(-1 == res) {
		LOGS("Error while synchronizing file with data\n");
		return FALSE;
	}
	return TRUE;
}

static gboolean bm_add_new_entry(gchar* word,gchar* translation,BookData* data)
{
	guint hash = g_str_hash(translation);
	gboolean result = TRUE;

	DBT new_key = { &(data->freeID) , sizeof(guint) };
	DBT new_val = { translation , strlen(translation) + 1};
	gint db_res = data->db_trans->put(data->db_trans,
					  &new_key,
					  &new_val,
					  R_NOOVERWRITE
					 );
	if(-1 == db_res)
	{
		data->last_error = ENGINE_COULDNT_WRITE;
		result = FALSE;
	}
	else
	{
		new_key.data = word;
		new_key.size = strlen(word) + 1;

		// { number of entries , id of the first entry , hash #1 }
		guint temp[3] = { 1 , data->freeID, hash };
		new_val.data = temp;
		new_val.size = sizeof(guint) * 3;

		db_res = data->db_words->put(data->db_words,
					     &new_key,
					     &new_val,
					     R_NOOVERWRITE
					    );
		if(-1 == db_res)
		{
			new_key.data = &(data->freeID);
			new_key.size =  sizeof(guint);
			data->db_trans->del(data->db_trans, &new_key, 0);
			result = FALSE;
			data->last_error = ENGINE_INTERNAL_ERROR;
		}
		else
		{
			result = TRUE;
			(data->freeID)++;
			bm_save_freeID(data);
		}
	}
	db_res = data->db_words->sync(data->db_words,0);
	db_res |= data->db_trans->sync(data->db_trans,0);
	if(0 == db_res)
	{
		LOGS("Bookmark/%s->%s() adding new bookmark successful.\n",
		      (gchar*)__FILE__,(gchar*)__FUNCTION__);
	}
	else
	{
		LOGS("Bookmark/%s->%s() adding new bookmark failed.\n",
		      (gchar*)__FILE__,(gchar*)__FUNCTION__);
	}
	return result;	
}

gboolean bm_engine_remove_word(Engine* engine,
                             gchar*  word)
{
        gboolean result = TRUE;

        LOGS("Bookmark/%s->%s() called. Param\nEngine at address: %p\n"
             "word: %s\n",(gchar*)__FILE__,(gchar*)__FUNCTION__,engine,word);
        g_assert(engine != NULL);
        g_assert(word != NULL);
        /* bm_timer(TIMER_START, (gchar*)(gchar*)__FUNCTION__); */

        BookData* data = (BookData*)(engine->engine_data);

	DBT key = { word , strlen(word) + 1 };
	DBT val = { NULL , 0 };
		
	gint db_res = data->db_words->get(data->db_words, &key, &val, 0);
	if ( 0 == db_res )
	{
		guint* t = (guint*)(val.data);
		guint  N = t[0];
		guint  id = 0;
		memcpy(&N, t, sizeof(guint));
		++t;
		guint  i = 0;
		key.size = sizeof(guint);
		key.data = &id;
		while( i < N ) {
			memcpy(&id, t + i*2, sizeof(guint));
			db_res = data->db_trans->del(data->db_trans, &key, 0);
			if(0 != db_res) {
				LOGS("Error while removing translation # %u for word: %s",i+1,word);
			}
			++i;
		}
	}
	else if( -1 == db_res) {
		LOGS("Bookmark/%s->%s() Error while removing word: %s!",
			(gchar*)__FILE__,
			(gchar*)__FUNCTION__,
			word
			);
		return FALSE;
	}
	else {
		LOGS("Bookmark/%s->%s() Ther is no such a word!",
			(gchar*)__FILE__,
			(gchar*)__FUNCTION__
			);
		return TRUE;
	};

	// all data from trnaslation database has been deleted - now delete
	// record in words database.

	DBT key_del = { word , strlen(word) + 1 };
	db_res = data->db_words->del(data->db_words,&key_del,0);
	if(-1 == db_res)
	{
		LOGS("Bookmark/%s->%s() Error while removing!\n",
			(gchar*)__FILE__,
			(gchar*)__FUNCTION__
			);				
	}
	else if(1 == db_res)
	{
		LOGS("Bookmark/%s->%s() There is no such a word!\n",
			(gchar*)__FILE__,
			(gchar*)__FUNCTION__
			);
	}
	else if(0 == db_res)
	{
		LOGS("Bookmark/%s->%s() word deleted successfully!\n",
			(gchar*)__FILE__,
			(gchar*)__FUNCTION__
			);
	}
	db_res = data->db_words->sync(data->db_words, 0);

	if((0 != db_res) || (NULL == data->db_words) || (NULL == data->db_trans)) {
		LOGS("Error while 2nd synchronizing file with data!\n");
	}


        /* bm_timer(TIMER_STOP, (gchar*)(gchar*)__FUNCTION__); */
        LOGS("Bookmark/%s->%s() finished work.\n",
                (gchar*)__FILE__,
                (gchar*)__FUNCTION__
               );
        return result;
}

gchar* bm_engine_get_lang_from(Engine* engine)
{
        LOGS("Bookmark/%s->%s() called.\n",(gchar*)__FILE__,(gchar*)__FUNCTION__);
        gchar* result = g_strdup("any");
        LOGS("Bookmark/%s->%s() return string=%s\n",
                (gchar*)__FILE__,
                (gchar*)__FUNCTION__,
                result
               );
        return result;
}

gchar* bm_engine_get_lang_to(Engine* engine)
{
        LOGS("Bookmark/%s->%s() called.\n",(gchar*)__FILE__,(gchar*)__FUNCTION__);
        gchar* result = g_strdup("any");
        LOGS("Bookmark/%s->%s() return string=%s\n",
                (gchar*)__FILE__,
                (gchar*)__FUNCTION__,
                result
               );
        return result;
}

gchar* bm_engine_get_title(Engine* engine)
{
        LOGS("Bookmark/%s->%s() called.\n",(gchar*)__FILE__,(gchar*)__FUNCTION__);
        gchar* result = g_strconcat(g_get_user_name(),"s' bookmarks",NULL);
        LOGS("Bookmark/%s->%s() return string=%s\n",
                (gchar*)__FILE__,
                (gchar*)__FUNCTION__,
                result
               );
        return result;
}

gchar* bm_engine_get_icon_path(Engine* engine)
{
        LOGS("Bookmark/%s->%s() called.\n",(gchar*)__FILE__,(gchar*)__FUNCTION__);
        gchar* result = g_strdup("/usr/share/pixmaps/ws_eng_bookmark_icon.png");
        LOGS("Bookmark/%s->%s() return string=%s\n",
                (gchar*)__FILE__,
                (gchar*)__FUNCTION__,
                result
               );
        return result;
}

void bm_engine_search_word_translation(Engine* engine,
                                       gchar* word,
                                       gpointer cb_data)
{
        LOGS("Bookmark/%s->%s() called.\n-->PARAM:engine at adress=%p\n"
             "-->PARAM:word=\'%s\'\n",
             (gchar*)__FILE__,
             (gchar*)__FUNCTION__,
             engine,
             word
            );
        g_assert(engine != NULL);
        g_assert(word != NULL);
        // start bm_timer for this function
        /* bm_timer(TIMER_START, (gchar*)(gchar*)__FUNCTION__); */
	BookData* data = (BookData*)(engine->engine_data);

	gchar* down_word = g_utf8_strdown(word,-1);
	DBT search = { down_word , strlen(down_word) + 1 };
	DBT info   = {    NULL   ,        0              };
	DBT trans  = {    NULL   ,        0              };
	gchar* tran = NULL;
	gchar* tran_ = NULL;

	gint tmp = data->db_words->get(data->db_words, &search, &info, 0);
	if(0 == tmp)
	{
		// plugin found some information about this word
		
		guint* records = (guint*)(info.data);
		guint count = records[0];
		memcpy(&count, records, sizeof(guint));
		//printf("Buffer (memcpy): %#x-%#x-%#x\n",count,records[1],records[2]);
		++records;
		guint id = 0;
		search.data = &id;
		search.size = sizeof(guint);
		
		while(count-- > 0 && count < 100)
		{
			//printf("--> LOOP:count = %#x\n",count);
			memcpy(search.data, records, sizeof(guint));
			records += 2;
			tmp = data->db_trans->get(data->db_trans, &search, &trans, 0);
			if(0 == tmp)
			{
				if(NULL == tran)
				{
					tran = g_strdup(trans.data);
				}
				else
				{
					tran_ = 
					g_strconcat(tran,"<br />",trans.data,NULL);
					g_free(tran);
					tran = tran_;
					tran_ = NULL;
				}
			}
		}
	};
	g_free(down_word);
        /* bm_timer(TIMER_STOP,(gchar*)(gchar*)__FUNCTION__);
        bm_timer(TIMER_START,"callback for returning word's translation START");
	*/
        // calling callback for word translation

	if ( NULL == cb_data )
	{
		cb_data = data->cb_search_word_trans_data;
	}
        data->cb_search_word_trans(tran, word, cb_data, ENGINE_NO_ERROR);
		
        /* bm_timer(TIMER_STOP,"callback for returning word's translation END");
	*/
//         if(data->auto_free) {
//                 LOGS("Bookmark/%s->%s() deleting all dynamic data because "
//                      "AUTO_FREE=TRUE\n",
//                      (gchar*)__FILE__,
//                      (gchar*)__FUNCTION__
//                     );
//                 g_free(tran);
//         }
	g_free(tran);
        tran = NULL;
}

void bm_engine_close(Engine* engine)
{
        LOGS("Bookmark/%s->%s() called.\n-->PARAM: engine adress=%p\n",
                (gchar*)__FILE__,
                (gchar*)__FUNCTION__,
                engine);
        g_assert(engine != NULL);

        BookData* data = (BookData*)(engine->engine_data);
        data->db_words->close(data->db_words);
	data->db_trans->close(data->db_trans);

        LOGS("Bookmark/%s->%s() engine at adress=%p is deleted.\n",
                (gchar*)__FILE__,
                (gchar*)__FUNCTION__,
                engine);
	g_free(data->dict_path); 
	g_free(data);
	data = NULL;
        g_free(engine);
        engine = NULL;
}

Engine* bm_engine_create(gchar* location,
                      EngineOptimizationFlag auto_cache,
                      cb_progress progress_handler,
                      gpointer progress_data,
                      gdouble seed)
{
        LOGS("Bookmark/%s->%s() called.\n"
             "-->PARAM:location=\'%s\'\n"
             "-->PARAM:auto_cache=%d\n",
             (gchar*)__FILE__,
             (gchar*)__FUNCTION__,
             location,
             (guint)auto_cache
            );
        /* bm_timer(TIMER_START,(gchar*)(gchar*)__FUNCTION__); */

        gchar* tmp = g_strdup(location);
        string_to_path(&tmp);

        Engine* result = (Engine*)g_try_malloc(sizeof(Engine));
        result->engine_location = bm_engine_location;
        result->engine_is_optimized = bm_engine_is_optimized;
        result->engine_optimize = bm_engine_optimize;
        result->engine_search_word_list = bm_engine_search_word_list;
        result->engine_search_word_translation = 
                        bm_engine_search_word_translation;        
        result->engine_close = bm_engine_close;
        result->engine_status = bm_engine_status;
        result->engine_status_message = bm_engine_status_message;
        result->engine_set_callback = bm_engine_set_callback;
        result->engine_set_progress_seed = bm_engine_set_progress_seed;
        result->engine_set_auto_free = bm_engine_set_auto_free;
        // 0.2 API:
        result->engine_add_word = bm_engine_add_word;
        result->engine_remove_word = bm_engine_remove_word;
        result->engine_get_lang_from = bm_engine_get_lang_from;
        result->engine_get_lang_to = bm_engine_get_lang_to;
        result->engine_get_title = bm_engine_get_title;
        result->engine_get_icon_path = bm_engine_get_icon_path;


        BookData* data = (BookData*)g_try_malloc(sizeof(BookData));
        result->engine_data = (gpointer)data;

        LOGS("Bookmark/%s->%s() opening file...\'%s\'.\n",
             (gchar*)__FILE__,
             (gchar*)__FUNCTION__,
             location
            );

	u_int32_t flags = O_CREAT | O_RDWR;
        gchar* tmp_w = g_strconcat(tmp,"/bm_words.db",NULL);
	gchar* tmp_t = g_strconcat(tmp,"/bm_trans.db",NULL);

	BTREEINFO inf = {
	  0,		/* permit duplicate keys? */
	  0,			/* cache size; 0 - default size */
	  0,			/* page size; 0 - default */
	  0,			/* byte order; 0 - use host order */
	  0,			/* min page number per page; 0 - default=2 */
	  bm_compare_key_words,	/* comparision function */
	  NULL			/* prefix comparision function */
	};
	data->info_words = inf;
	inf.compare = bm_compare_key_trans;
	data->info_trans = inf;

        data->db_words = 
		dbopen(tmp_w,       /* On-disk file that holds the database. */
		       flags,       /* flags, like O_CREAT etc. */
		       0755,        /* mode same as flags <hmmm> ? */
		       DB_BTREE,    /* type */
		       &(data->info_words)
		      );
        if(data->db_words == NULL)
	{
                g_free(data);
                g_free(result);
                result = NULL;
        }
	else {
		data->db_trans = 
			dbopen(tmp_t,
			flags,
			0755,
			DB_BTREE,
			&(data->info_trans)
			);

		if(data->db_trans == NULL)
		{
			data->db_words->close(data->db_words);
			g_free(data);
			g_free(result);
			result = NULL;
		}

	}
        g_free(tmp_w); tmp_w = NULL;
	g_free(tmp_t); tmp_t = NULL;
        if(result == NULL) {
                LOGS("Bookmark/%s->%s() opening bookmark file failed.\n",
                     (gchar*)__FILE__,
                     (gchar*)__FUNCTION__
                    );
        }
        else {
                LOGS("Bookmark/%s->%s()opening dictionary file successed.\n",
                        (gchar*)__FILE__,
                        (gchar*)__FUNCTION__
                       );
                data->dict_path = g_strdup(tmp);
                data->cb_progress_caching = progress_handler;
                data->cb_progress_caching_data = progress_data;        
                data->cb_progress_caching_seed = seed;        
                data->cb_progress_word_list = NULL;
                data->cb_progress_word_list_data = NULL;
                data->cb_progress_word_list_seed = 0.01;
                data->cb_progress_word_trans = NULL;
                data->cb_progress_word_trans_data = NULL;
                data->cb_progress_word_trans_seed = 0.01;

                data->cb_search_word_list = NULL;
                data->cb_search_word_list_data = NULL;

                data->cb_search_word_trans = NULL;
                data->cb_search_word_trans_data = NULL;

                data->auto_free = FALSE;

		bm_load_freeID(data);

        }
        g_free(tmp); tmp = NULL;
        
        /* bm_timer(TIMER_STOP,(gchar*)(gchar*)__FUNCTION__); */
        LOGS("Bookmark/%s->%s() returned Engine at adress=%p\n",
             (gchar*)__FILE__,
             (gchar*)__FUNCTION__,
             result
            );
        return result;
}


static gboolean is_Bookmark_db_file(gchar* file)
{
        LOGS("Bookmark/%s->%s() called.\n\
                 -->PARAM:file=\'%s\'\n",
                 (gchar*)__FILE__,
                 (gchar*)__FUNCTION__,
                 file
               );

	u_int32_t flags = O_RDWR;
	DB* dbp = dbopen(file,flags,0755,DB_BTREE,NULL);

        if(NULL == dbp) 
	{
		LOGS("Could no open! Wrong database! Not a bookmark.\n");
                return FALSE;
        };
	
	DBT search = {"four",sizeof("four")};
	DBT result = {NULL, 0};

	int errCode = dbp->get(dbp,&search,&result,0);
	dbp->close(dbp);
	g_free(result.data);

	if(-1 == errCode)
	{
                LOGS("Could not read! Wrong database! Not a bookmark.\n");
                return FALSE;
        };
        return TRUE;
}


void bm_engine_optimize(Engine* engine)
{
        LOGS("Bookmark/%s->%s() called for engine at adress=%p\n",
             (gchar*)__FILE__,
             (gchar*)__FUNCTION__,
             engine
            );
        LOGS("Unsupported optimization mechanizm for this engine!\n");
        LOGS("Bookmark/%s->%s()'s work finished.\n",(gchar*)__FILE__,(gchar*)__FUNCTION__);
}

gboolean bm_engine_check(gchar* location) 
{
        LOGS("Bookmark/%s->%s() called.\n-->PARAM:location=\'%s\'\n",
             (gchar*)__FILE__,
             (gchar*)__FUNCTION__,
             location
            );
        /* bm_timer(TIMER_START,(gchar*)(gchar*)__FUNCTION__); */
        gboolean result = TRUE;
        gchar* filepath = g_strdup(location);
        gchar* tmp = NULL;
	gchar* tmp2 = NULL;

        string_to_path(&filepath);
        if (filepath == NULL) {
                result = FALSE;
                LOGS("Bookmark/%s->%s() location \'%s\' is not a proper "
                     "path!\n",
                     (gchar*)__FILE__,
                     (gchar*)__FUNCTION__,
                     location
                    );
        }
        else
	{
                tmp = g_strconcat(filepath,"/bm_words.db",NULL);
		tmp2 = g_strconcat(filepath,"/bm_trans.db",NULL);
                g_free(filepath);
                filepath = tmp;
                tmp = NULL;

                LOGS("Bookmark/%s->%s() finnal file to check is: %s\n",
                     (gchar*)__FILE__,
                     (gchar*)__FUNCTION__,
                     filepath
                    );
                if (!g_file_test(filepath, G_FILE_TEST_IS_REGULAR) || 
		    !g_file_test(tmp2, G_FILE_TEST_IS_REGULAR) 
		   ) {
                        LOGS("Bookmark/%s->%s() file \'%s\' does not exists!\n",
                             (gchar*)__FILE__,
                             (gchar*)__FUNCTION__,
                             filepath
                            );
                        result = FALSE;
                };
        };
        if (result != FALSE) {
                result = is_Bookmark_db_file(filepath) & 
			 is_Bookmark_db_file(tmp2);
        };

        g_free(filepath);  filepath = NULL;
	g_free(tmp2);      tmp2 = NULL;
        /* bm_timer(TIMER_STOP,(gchar*)(gchar*)__FUNCTION__); */
        LOGS("Bookmark/%s->%s() returned bool statement=%s.\n",
             (gchar*)__FILE__,
             (gchar*)__FUNCTION__,
             PRINT_STATE(result)
            );
        return result;
}

gboolean bm_engine_is_optimized(Engine* engine) 
{
        LOGS("Bookmark/%s->%s() called.\n-->PARAM: engine adress=%p\n",
             (gchar*)__FILE__,
             (gchar*)__FUNCTION__,
             engine
            );
        g_assert(engine != NULL);
        gboolean result = FALSE;
        LOGS("Bookmark/%s->%s() returned bool statement=%s.\n",
             (gchar*)__FILE__,
             (gchar*)__FUNCTION__,
             PRINT_STATE(result)
            );
        return result;
}

void bm_engine_search_word_list(Engine* engine,
                                gchar* pattern,
                                gpointer cb_data)
{
        LOGS("Bookmark/%s->%s() called. Searching words list\n"
             "-->PARAM:engine at adress=%p\n"
             "-->PARAM:pattern=\"%s\"\n",
             (gchar*)__FILE__,
             (gchar*)__FUNCTION__,
             engine,
             pattern
            );
        g_assert(engine != NULL);
        g_assert(pattern != NULL);

        /* bm_timer(TIMER_START,(gchar*)(gchar*)__FUNCTION__); */
        BookData* data = (BookData*)(engine->engine_data);
        if(data->cb_search_word_list == NULL) {
                LOGS("Bookmark/%s->%s() callback for Word List not set. "
                     "Searching aborted.\n",
                     (gchar*)__FILE__,
                     (gchar*)__FUNCTION__
                    );
                /* bm_timer(TIMER_STOP,(gchar*)(gchar*)__FUNCTION__); */
                return;
        };

	GArray* result = g_array_new(TRUE, TRUE, sizeof(gchar*) );
	guint a = G_MAXUINT32;
	DBT search = { &a , sizeof(a) };
	DBT reply  = { NULL , 0 };
	gchar* down_word = NULL;
	gchar *tmp;
	
 	GPatternSpec* regex;
 	regex = g_pattern_spec_new (g_utf8_casefold(pattern,-1));

	gint code = data->db_words->sync(data->db_words, 0);
	code = data->db_words->seq(data->db_words, &search, &reply, R_FIRST);
	
/*
	if((('*' == pattern[0]) && ('\0' == pattern[1])) || ('\0' == pattern[0]))
	{
		// especiall treat if user give only '*'
		while(0 == code) {
			gchar* cos = g_strdup(search.data);
			g_array_append_val(result,cos);
			code = data->db_words->seq(data->db_words, &search, &reply, R_NEXT);
			// we are giving to the user all words from dictionary
		}
	}
	else
	{*/
	while(0 == code) {
		tmp = g_strconcat ((gchar*)(search.data), " ", NULL);
		down_word = g_utf8_casefold(tmp,-1);
		g_free (tmp);
		tmp = g_utf8_casefold ((gchar*)(search.data),-1);
		
		if(( g_pattern_match_string( regex, down_word ) == TRUE ) ||
			( g_pattern_match_string( regex, tmp ) == TRUE ))
		{
			g_free (tmp);
			tmp = g_strdup(search.data);
			g_array_append_val(result, tmp );
		};

		/* eg_free(reply.data); */
		if(NULL != down_word)
		{
			g_free(down_word);
			down_word = NULL;
		}
		code = data->db_words->seq(data->db_words, &search, &reply, R_NEXT);
	}
	
        /* bm_timer(TIMER_STOP,(gchar*)(gchar*)__FUNCTION__); */
	g_pattern_spec_free (regex);


        /* bm_timer(TIMER_START,"callback for returning words LIST START"); */
        // calling callback for word translation

	if ( NULL == cb_data )
	{
		cb_data = data->cb_search_word_list_data;
	}
        data->cb_search_word_list(result, pattern, cb_data, ENGINE_NO_ERROR);


       /* if(data->auto_free) {
                LOGS("Bookmark/%s->%s() deleting all dynamic data because "
                     "AUTO_FREE=TRUE\n",
                     (gchar*)__FILE__,
                     (gchar*)__FUNCTION__
                    );
		len = 0;
		while(NULL != (tmp = g_array_index(result,gchar*,len++)))
		{
			g_free(tmp); tmp = NULL;
		}
		g_array_free(result, TRUE);
        }*/
	guint len = 0;
	
	while(NULL != (tmp = g_array_index(result,gchar*,len++)))
	{
		g_free(tmp); tmp = NULL;
	}
	g_array_free(result, TRUE);
	/* bm_timer(TIMER_STOP,"callback for returning word LIST END"); */
}

EngineModule engine_global_functions()
{
	LOGS("Bookmark/%s->%s() called.\n",(gchar*)__FILE__,(gchar*)__FUNCTION__);
	EngineModule result;        
	result.engine_check             = bm_engine_check;
	result.engine_description       = bm_engine_description;
	result.engine_format            = bm_engine_format;
	result.engine_version           = bm_engine_version;
	result.engine_create            = bm_engine_create;
	LOGS("Bookmark/%s->%s()returned EngineModule at adress=%p.\n",
	     (gchar*)__FILE__,
	     (gchar*)__FUNCTION__,
             &result
	    );
	return result;
}

gchar* bm_engine_status_message(EngineStatus error) 
{
	LOGS("Bookmark/%s->%s() called.\n",(gchar*)__FILE__,(gchar*)__FUNCTION__);
	switch (error) {
		case ENGINE_NO_ERROR:
			return "No error.";
		case ENGINE_WRONG_FILE:
			return "File which You are trying to use is wrong type.";
		case ENGINE_COULDNT_READ:
			return "Could not read from file.";
		case ENGINE_NO_FILE:
			return "There is no such a file.";
		case ENGINE_OUT_OF_MEMORY:
			return "There were no enough memory for this action.";
		default:
			return "Wrong engine's status identifier!";
	}
}

gchar* bm_engine_version() 
{
	LOGS("Bookmark/%s->%s() called.\n",(gchar*)__FILE__,(gchar*)__FUNCTION__);
	gchar* result = g_strdup(DIC_ENG_VERSION);
	LOGS("Bookmark/%s->%s() return string=%s\n",
		(gchar*)__FILE__,
		(gchar*)__FUNCTION__,
		result
	       );
	return result;
}

gchar* bm_engine_format() 
{
	LOGS("Bookmark/%s->%s() called.\n",(gchar*)__FILE__,(gchar*)__FUNCTION__);
	gchar* result = g_strdup(DIC_ENG_FORMAT);
	LOGS("Bookmark/%s->%s() return string=%s\n",
	     (gchar*)__FILE__,
	     (gchar*)__FUNCTION__,
	     result
	    );
	return result;
}

gchar* bm_engine_description() 
{
	LOGS("Bookmark/%s->%s() called.\n",(gchar*)__FILE__,(gchar*)__FUNCTION__);
	gchar* result = g_strdup(DIC_ENG_DESCRIPTION);
	LOGS("Bookmark/%s->%s() return string=%s\n",
	     (gchar*)__FILE__,
	     (gchar*)__FUNCTION__,
	     result
	    );
	return result;
}

gchar* bm_engine_location(Engine* engine)
{
	LOGS("Bookmark/%s->%s() called.\n-->PARAM: engine adress=%p\n",
	     (gchar*)__FILE__,
	     (gchar*)__FUNCTION__,
	     engine
	    );
	g_assert(engine != NULL);
	BookData* data = (BookData*)(engine->engine_data);
 
	gchar* result;
	if(data->auto_free) {
		result = data->dict_path;
	}
	else {
		result = g_strdup(data->dict_path);
	}

	LOGS("Bookmark/%s->%s() returned string=%s\n",
	     (gchar*)__FILE__,
	     (gchar*)__FUNCTION__,
	     result
	    );
	return result;
}

void bm_engine_set_auto_free(Engine* engine, gboolean state) 
{
	LOGS("Bookmark/%s->%s() called.\n"
	     "-->PARAM:engine at adress=%p\n"
	     "-->PARAM:state=%s\n",
	     (gchar*)__FILE__,
	     (gchar*)__FUNCTION__,
	     engine,
	     PRINT_STATE(state)
	    );
	g_assert(engine != NULL);
	BookData* data = (BookData*)(engine->engine_data);
        
	data->auto_free = state;
	LOGS("Bookmark/%s->%s() Current auto_free is %s\n",
	     (gchar*)__FILE__,
	     (gchar*)__FUNCTION__,
	     PRINT_STATE(data->auto_free)
	    );
}

EngineStatus bm_engine_status(Engine* engine) 
{
	LOGS("Bookmark/%s->%s() called.\n"
		"-->PARAM:engine at adress=%p\n",
		(gchar*)__FILE__,
		(gchar*)__FUNCTION__,
	       engine
	       );
	BookData* data = (BookData*)(engine->engine_data);
	LOGS("Bookmark/%s->%s() returned error code: %d\n",
	     (gchar*)__FILE__,
	     (gchar*)__FUNCTION__,
	     (gint)(data->last_error)
	    );
	return data->last_error;
}

void bm_engine_set_progress_seed(Engine* engine, gchar* signal, gdouble seed) {
	LOGS("Bookmark/%s->%s() called.\n",(gchar*)__FILE__,(gchar*)__FUNCTION__);
	BookData* data = (BookData*)(engine->engine_data);
	if(g_ascii_strcasecmp(signal,ENGINE_PROGRESS_OPTIMIZING_SIGNAL) == 0)  {
		data->cb_progress_caching_seed = seed;
		LOGS("Bookmark/%s->%s() sets new seed=%0.2f for for signal "
		     "\"%s\".\n",
		     (gchar*)__FILE__,
		     (gchar*)__FUNCTION__,
		     seed,
		     signal
		    );        
	} 
	else {
		LOGS("Bookmark/%s->%s() unsupported signalfor progress: %s.\n",
		     (gchar*)__FILE__,
		     (gchar*)__FUNCTION__,
		     signal
	            );
	};
}

gpointer bm_engine_set_callback(Engine* engine,
			     gchar* signal,
			     gpointer c_handler,
			     gpointer user_data)
{
	LOGS("Bookmark/%s->%s() called.\n",(gchar*)__FILE__,(gchar*)__FUNCTION__);
	g_assert(engine != NULL);
	g_assert(signal != NULL);
	g_assert(c_handler != NULL);
	BookData* data = (BookData*)(engine->engine_data);
	if(g_ascii_strcasecmp(signal,ENGINE_PROGRESS_OPTIMIZING_SIGNAL) == 0)  {
		gpointer result = data->cb_progress_caching;
		data->cb_progress_caching = c_handler;
		data->cb_progress_caching_data = user_data;
		LOGS("Bookmark/%s->%s() sets handler for signal \"%s\".\n",
			(gchar*)__FILE__,
			(gchar*)__FUNCTION__,
			signal
		       );
		LOGS("Bookmark/%s->%s() Function at adress =  %d.\n",
			(gchar*)__FILE__,
			(gchar*)__FUNCTION__,
			(guint)c_handler
		       );
		LOGS("Bookmark/%s->%s()     Data at adress =  %d.\n",
			(gchar*)__FILE__,
			(gchar*)__FUNCTION__,
			(guint)user_data
		       );
		return result;                
	}
	else if(g_ascii_strcasecmp(signal, ENGINE_WORD_LIST_SIGNAL) == 0) {
		gpointer result = data->cb_search_word_list;
		data->cb_search_word_list = c_handler;
		data->cb_search_word_list_data = user_data;
		LOGS("Bookmark/%s->%s() sets handler for signal \"%s\".\n",
			(gchar*)__FILE__,
			(gchar*)__FUNCTION__,
			signal
		       );
		LOGS("Bookmark/%s->%s() Function at adress =  %d.\n",
			(gchar*)__FILE__,
			(gchar*)__FUNCTION__,
			(guint)c_handler
		       );
		LOGS("Bookmark/%s->%s()     Data at adress =  %d.\n",
			(gchar*)__FILE__,
			(gchar*)__FUNCTION__,
			(guint)user_data
		       );
		return result;                        
	}
	else if(g_ascii_strcasecmp(signal,
		ENGINE_WORD_TRANSLATION_SIGNAL) == 0)  {
			gpointer result = data->cb_search_word_trans;
			data->cb_search_word_trans = c_handler;
			data->cb_search_word_trans_data = user_data;
			LOGS("Bookmark/%s->%s() sets handler for signal \"%s\".\n",
				(gchar*)__FILE__,
				(gchar*)__FUNCTION__,
				signal
			       );
			LOGS("Bookmark/%s->%s() Function at adress =  %d.\n",
				(gchar*)__FILE__,
				(gchar*)__FUNCTION__,
				(guint)c_handler
			       );
			LOGS("Bookmark/%s->%s()     Data at adress =  %d.\n",
				(gchar*)__FILE__,
				(gchar*)__FUNCTION__,
				(guint)user_data
			       );
			return result;                        
		}
		else {
			g_warning("Bookmark/%s->%s() unsupported signal: %s.\n",
				  (gchar*)__FILE__,
				  (gchar*)__FUNCTION__,
				  signal
				 );
			return NULL;
		}
}
/**
* Translate given string to proper full file path. Function works "in-situ".
* It means that string is translated and replaced by proper full file path if
* this file path exists, or this string is cleared and setted to NULL, if string
* was representing wrong path. This function is static - to use only 
* within this module.
*
* @param string pointer to to string 
*       representing file path - it will be replaced by a proper filepath.
*       It should be path to directory or file, ended with "/" or not, absolute 
*       or not
* @return pointer to new full file path, it is the same
*       as string[0] in fact and it is returned only for abillity of nesting of
* functions by pointer to string identyfying full file path
*/
static gchar* string_to_path(gchar** string) {
	g_debug("XDXF/%s->%s() called.\n"
	        "-->PARAM:string=\'%s\'\n",
	        __FILE__,
	        __FUNCTION__,
	        string[0] );
	gchar* arg = string[0];
	gchar* new = NULL;

	/* cleaning from leading and trailing whitespaces */
	g_strstrip(arg);

	/* add current directory if this is not absolute directory */
	if (!g_path_is_absolute(arg))
	{
		gchar* tmp = g_get_current_dir();
		new = g_strconcat(tmp,"/",arg,NULL);
		g_free(arg); arg = new; new = NULL;
	};

	/* this is not a directory */
	if (!g_file_test(arg, G_FILE_TEST_IS_DIR))
	{
		/* if this is wrong filepath, string was wrong */
		if (!g_file_test(arg, G_FILE_TEST_IS_REGULAR))
		{
			g_free(arg);
			new = NULL;
		}
		else
		{
			/* if this is a file, remove filename */
			new = g_path_get_dirname (arg);
			g_free(arg);
		}
	}
	else
	{
		/* this is a directory ... */
		/* remove suffix "/" if neded... */
		if (g_str_has_suffix(arg,"/") )
		{
			new = g_path_get_dirname (arg);
			g_free(arg);
		}
		else
		{
			new = arg;
		}
	};

	/* now in new should be proper filepath, if not, string was wrong */
	if (!g_file_test(new, G_FILE_TEST_IS_DIR))
	{
		/* if that directory does not exist, passed string wasn't proper
		 */
		g_free(new);
		new = NULL;
	};

	/* replace string under passed address */
	string[0] = new;
	g_debug("XDXF/%s->%s() returned string=\'%s\'\n",
	         __FILE__,
	         __FUNCTION__,
	         string[0] );
	return new;
}

/*
static double bm_timer(gboolean start, gchar* message)
{
	static GArray* stack = NULL;
	static gboolean first_run = TRUE;
	static struct timeval actual_time;
	static struct timeval last_time;
	static struct timeval result;
	static double seconds = 0.0;
	if(first_run) {
		first_run = FALSE;
		stack = g_array_new(TRUE, TRUE, sizeof(struct timeval));
	};

	if (start) {
		LOGS("Bookmark->%s() start bm_timer for function '%s()'.\n",
			(gchar*)__FUNCTION__,
			message
		       );
		g_array_prepend_val(stack, actual_time);
		gettimeofday(&g_array_index(stack, struct timeval, 0),NULL);
		return -1.0;
	}
	else
	{
		gettimeofday(&actual_time,NULL);
		last_time = g_array_index(stack, struct timeval, 0);
		g_array_remove_index(stack, 0);

		if (actual_time.tv_usec < last_time.tv_usec) {
			int nsec = (last_time.tv_usec - actual_time.tv_usec) / 
					(1000000 + 1);
			last_time.tv_usec -= 1000000 * nsec;
			last_time.tv_sec += nsec;
		}
		if (actual_time.tv_usec - last_time.tv_usec > 1000000) {
			int nsec = (last_time.tv_usec - actual_time.tv_usec) / 
					1000000;
			last_time.tv_usec += 1000000 * nsec;
			last_time.tv_sec -= nsec;
		}
		result.tv_sec = actual_time.tv_sec - last_time.tv_sec;
		result.tv_usec = actual_time.tv_usec - last_time.tv_usec;
		seconds = (((double)(result.tv_usec)) / 1e6) +
		           ((double)(result.tv_sec));

		LOGS("Bookmark->%s() function \'%s()\' was working for: %g "
		     "[s] or %ld [us].\n",
		     (gchar*)__FUNCTION__,
		     message,
		     seconds,
		     ((long)(result.tv_sec*1e6)+(result.tv_usec)) );
		if(stack->len == 0)
		{
			g_array_free(stack, TRUE);
			first_run = TRUE;
		}
	}
	return seconds;
}
*/
