/*******************************************************************************
This file is part of mdictionary.

mdictionary is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

mdictionary is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License 
along with mdictionary; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Copyright 2006-2008 ComArch S.A.
*******************************************************************************/
/** \defgroup BookmarkEngine Dictionary Engine - Bookmark format
 * \brief Bookmark Dictionary Engine.
 *
 * This is library with dictionary engine supporting bookmarks. It uses Berkeley
 * DB as a storage method.
 *
 * TODO: currently bookmark engine does not support all function from engine
 * API! It need implementation of API version 0.2 function, like getting icon
 * path etc.
 */
/*@{*/
/** \file engine_bookmark.h
 * \brief Header for bookmark dictionary engine.
 */

#ifndef _DICTIONARY_ENGINE_BOOKMARK
#define _DICTIONARY_ENGINE_BOOKMARK

#ifdef __cplusplus
        extern "C" {
#endif


/* headers with unix types/functions - only for timers */
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <unistd.h>

/* header with GLIB definitions/functions/types */
#include <glib.h>

/* header wit engine API */
#include <dictionary_engine.h>

/* header with Berkeley DB API */
#include <db/db.h>

/* operation on strings */
#include <string.h>


/** \name Timers */
/*@{*/
/** \brief Flags telling that we want to start timer. */
#define TIMER_START     TRUE
/** \brief Flags telling that we want to stop timer. */
#define TIMER_STOP      FALSE

/** \brief Start/stop timers.
 *
 * \param start do we want start new timer or end the last one
 * \param message string which shoul be printed by function while debbuging
 * \return -1.0 if we start or seconds passed from start if we want to 
 * stop timer
 */
/* in final releases timer capapilities should be removed for increase
 * performance */
/* static double bm_timer(gboolean start, gchar* message); */

/*@}*/

/** \brief Version of bookmark engine. */
static const gchar* DIC_ENG_VERSION = "0.2";

/** \brief Short description of format supported by the current engine. */
static const gchar* DIC_ENG_FORMAT = "Users' Bookmarks";

/** \brief Short description of engine */
static const gchar* DIC_ENG_DESCRIPTION = "This module handles users' book"
                                          "marks. Based on Berkeley DB 1.85";

/** \brief Translate boolean value into string. */
#define PRINT_STATE(state) ( (state) ? "TRUE" : "FALSE" )


/** \brief Internal data structure of Bookmark Engine.
 */
struct _BookData {
	DB*                     db_words;
	/**< \brief Pointer to the database with words */

	DB*                     db_trans;
	/**< \brief Pointer to the database with translations */

	BTREEINFO               info_words;
	/**< \brief Structure describing search method in words database */

	BTREEINFO               info_trans;
	/**<\brief Structure describing search method in translations database*/

	guint                   freeID;
	/**< \brief next free available id for word */

	/**< \brief pointer to cache file */
	gchar*                  dict_path;
	/**< \brief path to dictionary */
	EngineStatus            last_error;
	/**< \brief status of last taken action */
	gboolean                auto_free;
	/**< \brief auto free mechanism status: FALSE - off, TRUE - on */

	cb_progress             cb_progress_caching;
	/**< \brief pointer to callback function called while informing about 
	 * caching progress */
	gpointer                cb_progress_caching_data;
	/**< \brief pointer to data passed to callback function called while
	 * informing about caching progress */
	gdouble                 cb_progress_caching_seed;
	/**< \brief how often progress callback should be called. 0.01 mean 
	 * that after each 1% of work callback shoul be called */

	cb_progress             cb_progress_word_list;
	/**< \brief pointer to callback function called while informing about
	 * words list searching progress */
	gpointer                cb_progress_word_list_data;
	/**< \brief pointer to data passed to callback function called while
	 * informing about words list searching progress */
	gdouble                 cb_progress_word_list_seed;
	/**< \brief how often progress callback should be called. 0.01 mean
	 * that after each 1% of work callback shoul be called */

	cb_progress             cb_progress_word_trans;
	/**< \brief pointer to callback function called while informing about
	 * word's translation searching progress */
	gpointer                cb_progress_word_trans_data;
	/**< \brief pointer to data passed to callback function called while
	 * informing about word's translation searching progress */
	gdouble                 cb_progress_word_trans_seed;
	/**< \brief how often progress callback should be called. 0.01 mean
	 * that after each 1% of work callback shoul be called */

	cb_word_list            cb_search_word_list;
	/**< \brief pointer to callback function called after words list is
	 * found */
	gpointer                cb_search_word_list_data;
	/**< \brief pointer to data passed to callback function called after
	 * words list is found */

	cb_word_translation     cb_search_word_trans;
	/**< \brief pointer to callback function called after word's translation
	 * is found */
	gpointer                cb_search_word_trans_data;
	/**< \brief pointer to data passed to callback function called after
	 * word's translation is found */
};
typedef struct _BookData        BookData;


/** \brief Convert string to proper path name. */
static gchar* string_to_path(gchar** string);

/** \brief Tells if file is in Bookmarks format (file should exist). */
static gboolean is_Bookmark_db_file(gchar* file);

/** \brief Add new translation for word */
static gboolean bm_add_new_entry( gchar* word,
                                  gchar* translation,
                                  BookData* data );

/** \brief Get minimal free value available for word's id */
static void bm_load_freeID(BookData* data);

/** \brief Get minimal free value available for word's id */
static void bm_save_freeID(BookData* data);

/** \brief Add only new translation with id */
static gboolean bm_add_only_translation( BookData* data,
                                         gchar* translation,
                                         guint id );

/** \brief Compare method for searching in translation file */
static gint bm_compare_key_trans(const DBT *a, const DBT *b);

/** \brief Compare method for searching in words file */
static gint bm_compare_key_words(const DBT *a, const DBT *b);


/** \name Module functions */
/*@{*/

/** \brief dict_eng_module_check() function implementation. */
gboolean        bm_engine_check(gchar* location);

/** \brief dict_eng_module_get_description() function implementation. */
gchar*          bm_engine_description();

/** \brief dict_eng_module_get_format() function implementation. */
gchar*          bm_engine_format();

/** \brief dict_eng_module_get_version() function implementation. */
gchar*          bm_engine_version();

/** \brief dict_eng_module_create() function implementation. */
Engine*         bm_engine_create( gchar* location, 
                                  EngineOptimizationFlag flags,
                                  cb_progress progress_handler,
                                  gpointer progress_data,
                                  gdouble seed );
/*@}*/

/** \name Particular dictionary function */
/*@{*/

/** \brief dict_eng_add_word() function implementation. */
gboolean        bm_engine_add_word( Engine* engine,
                                    gchar*  word,
                                    gchar*  translation );

/** \brief dict_eng_remove_word() function implementation. */
gboolean        bm_engine_remove_word( Engine* engine, gchar*  word );

/** \brief dict_eng_get_lang_from() function implementation. */
gchar*          bm_engine_get_lang_from(Engine* engine);

/** \brief dict_eng_get_lang_to() function implementation. */
gchar*          bm_engine_get_lang_to(Engine* engine);

/** \brief dict_eng_get_title() function implementation. */
gchar*          bm_engine_get_title(Engine* engine);

/** \brief dict_eng_get_icon_path() function implementation. */
gchar*          bm_engine_get_icon_path(Engine* engine);

/** \brief dict_eng_destroy() function implementation. */
void            bm_engine_close(Engine* engine);

/** \brief dict_eng_get_location() function implementation. */
gchar*          bm_engine_location(Engine* engine);

/** \brief dict_eng_optimize() function implementation. */
void            bm_engine_optimize(Engine* engine);

/** \brief dict_eng_is_optimized() function implementation. */
gboolean        bm_engine_is_optimized(Engine* engine);

/** \brief dict_eng_set_auto_free() function implementation. */
void            bm_engine_set_auto_free(Engine* engine, gboolean state);

/** \brief dict_eng_set_callback() function implementation. */
gpointer        bm_engine_set_callback( Engine* engine,
                                        gchar* event,
                                        gpointer c_handler,
                                        gpointer user_data );

/** \brief dict_eng_set_progress_seed() function implementation. */
void            bm_engine_set_progress_seed( Engine* engine,
                                             gchar* signal,
                                             gdouble seed );

/** \brief dict_eng_search_word_list() function implementation. */
void            bm_engine_search_word_list( Engine* engine,
                                            gchar* pattern,
                                            gpointer data );

/** \brief dict_eng_search_word_translation() function implementation. */
void            bm_engine_search_word_translation( Engine* engine,
                                                   gchar* word,
                                                   gpointer data );

/** \brief dict_eng_get_last_status() function implementation. */
EngineStatus    bm_engine_status(Engine* engine);


/** \brief dict_eng_status_message() function implementation. */
gchar*          bm_engine_status_message(EngineStatus error);

/*@}*/

/** \brief implementation of engine_global_functions(void) function. */
EngineModule    engine_global_functions();

#ifdef __cplusplus
}
#endif
#endif /* #ifndef _DICTIONARY_ENGINE_STARDICT */

/*@}*/
