/*******************************************************************************
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation;
version 2.1 of the License.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

Copyright 2006-2008 ComArch S.A.
*******************************************************************************/
/** \defgroup DBusWrapperAPI mDictionary D-Bus Wrapper API
 * \brief Wrapper for GLib D-Bus bindings for mDictionary project.
 *
 */
/*@{*/
/** \file ws_dbus.h
 * \brief API for D-Bus Wrapper Library.
 *
 * Declaration of methods used to communicate between
 * dictionary UI and searching module
 * \author Krzysztof Sasiak \<krzysztof.sasiak\@comarch.com\>
 * \date 05-11-2007
 * \version 1.0.0
 * \see DBusWrapperAPI
 */

#ifndef _WS_DBUS
#define _WS_DBUS

#ifdef __cplusplus
extern "C" {
#endif

#include <libosso.h>
#include <dbus/dbus-protocol.h>
#include <string.h>
#include <dbus-1.0/dbus/dbus-protocol.h>
#include <stdio.h>
#include <stdlib.h>
#include <glib-2.0/glib.h>

/** \brief Callback function definition
 *
 * @param error - pointer to a GError structure. Currently not used
 * @param data - GArray of osso_rpc_t structures containing sent data
 * @param user_data - data passed by user while setting the callback 
 * function with ::ws_dbus_set_cb
 * \sa ws_dbus_set_cb
 */
typedef void (* ws_dbus_cb) (GError *error, gpointer data, gpointer user_data); 

/** \brief An enumeration of data types to be sent via DBUS*/
typedef enum 
{
        WS_DBUS_TYPE_STRING = DBUS_TYPE_STRING,
        WS_DBUS_TYPE_INT = DBUS_TYPE_INT32,
        WS_DBUS_TYPE_UINT  = DBUS_TYPE_UINT32,
        WS_DBUS_TYPE_BOOLEAN  = DBUS_TYPE_BOOLEAN,
        WS_DBUS_TYPE_DOUBLE  = DBUS_TYPE_DOUBLE,
        WS_DBUS_TYPE_INVALID  = DBUS_TYPE_INVALID,
        WS_DBUS_TYPE_GARRAY = ((int) 'a'),
        WS_DBUS_TYPE_SIGNAL = DBUS_TYPE_INT32
} WSDBusDataType;

/** \brief This structure contains data vital for DBUS communication.
 *
 * The fields shouldn't be modified directly. It should be done by
 * ws_dbus_config function
 * \sa ws_dbus_config
 */
struct _WSDBusData 
{
	osso_context_t * context; 
	/**< a pointer needed by osso_initiallize*/
	gchar *service; 
	/**< owner service path*/
	gchar *object; 
	/**< owner object path*/
	gchar *iface; 
	/**< owner interface path*/

	gchar *remote_service; 
	/**< remote service path*/
	gchar *remote_object; 
	/**< remote object path*/
	gchar *remote_iface; 
	/**< remote interface path*/

	gchar *name; 
	/**< program's name*/
	gchar *version; 
	/**< program's version*/

	GArray *method_data; 
	/**< A GArray of local methods*/
	GArray *remote_method_data; 
	/**< A GArray of remote methods 
	 *(obsolete as no introspection has been
	 * implemented
	 */
};

/** \brief An enumeration of return values.
 */
typedef enum 
{
	WS_DBUS_STATUS_OK = 0, 
	/**< Transmission succeeded*/
	WS_DBUS_STATUS_ERROR 
	/**< Transmission error*/
} WSDBusStatus;

/** \brief Specifies the possible kinds of event notifications that can be sent 
 * via DBus
 */
typedef enum
{
	WS_DBUS_INFO_TERMINATE = 1,
	/**<GUI signal - terminate the application*/
	WS_DBUS_INFO_STOP_SEARCH, 
	/**<GUI signal - stop the current search*/
	WS_DBUS_INFO_SEARCH_FINISHED, 
	/**<Manager signal - informs the GUI, that there will be no more results 
	 *returned
	 */
	WS_DBUS_INFO_CACHING, 
	/**<Manager signal - informs the GUI, that caching has started*/
	WS_DBUS_INFO_CACHING_FINISHED, 
	/**<Manager signal - informs the GUI, that caching has finished*/
	WS_DBUS_INFO_CONFIG_CHANGED, 
	/**<GUI signal - informs the Manager about changes in the GConf 
	 *configuration
	 */
	WS_DBUS_ERROR_INVALID_FILE_FORMAT, 
	/**<Parser error - dictionary file has invalid format*/
	WS_DBUS_ERROR_FILE_NOT_FOUND, 
	/**<Parser error - dictionary file not found*/
	WS_DBUS_ERROR_NO_PERMISSION, 
	/**<Parser error - no permission to read the dictionary file*/
	WS_DBUS_ERROR_UNKNOWN, 
	/**<Parser error - unknown error*/
	WS_DBUS_ERROR_OUT_OF_MEMORY, 
	/**<Either the UI or the manager is out of memory*/
	WS_DBUS_ERROR_ENGINE_NOT_FOUND, 
	/**<Manager error - dictionary engine module, couldn't be located*/

	WS_DBUS_ERROR_DICTIONARY_NOT_LOAD, 
	/**<Manager signal - informs the GUI,that history record can not be 
	 *found because dictionary is not activate
	 */
	WS_DBUS_BOOKMARKS_ADDED_OK, 
	/**<GUI signal - send if bookmark has been added succesfully*/
	WS_DBUS_BOOKMARKS_REMOVED_OK, 
	/**<GUI signal - send if bookmark has been removed succesfully*/
	WS_DBUS_BOOKMARKS_ADDED_FAIL, 
	/**<GUI signal - send if bookmark has not been added succesfully*/
	WS_DBUS_BOOKMARKS_REMOVED_FAIL, 
	/**<GUI signal - send if bookmark has not been removed succesfully*/
	WS_DBUS_LOAD_BOOKMARK_FAILED, 
	/**<GUI signal - blocks bookmarks*/
	WS_DBUS_BOOKMARK_MODE_ON, 
	/**<GUI signal - bookmark mode is ON*/
	WS_DBUS_BOOKMARK_MODE_OFF, 
	/**<GUI signal - bookmark mode is OFF*/
	WS_DBUS_EXTRACT_FILE, 
	/**<GUI signal - decompress a file*/
	WS_DBUS_EXTRACT_FILE_FINISHED, 
	/**<Manager signal - file decompression finished*/
	WS_DBUS_WORDS_LIST_STARTED, 
	/**<Manager signal - words' list started*/
	WS_DBUS_TRANSLATION_STARTED, 
	/**<Manager signal - translations' list started*/
	WS_DBUS_WORDS_LIST_FINISHED, 
	/**<Manager signal - words' list finished*/
	WS_DBUS_TRANSLATION_FINISHED, 
	/**<Manager signal - translations' list finished*/
	WS_DBUS_WORDS_LIST_FULL, 
	/**<Manager signal - words' list full*/
	WS_DBUS_WORDS_LIST_FILLED_NOT_FULL 
	/**<Manager signal - words' list filled but not full*/
} WSDBusNotify;


/** \brief This structure is used for specifying the field of WSDBusData
 * structure one would like to modify.
 *
 * It is used in the ws_dbus_config function. Its main purpose is to clarify
 * D-BUS configuration.
 */
typedef enum 
{
	WS_DBUS_CONFIG_SERVICE = 1,
	WS_DBUS_CONFIG_OBJECT,
	WS_DBUS_CONFIG_IFACE,
	WS_DBUS_CONFIG_REMOTE_SERVICE,
	WS_DBUS_CONFIG_REMOTE_OBJECT,
	WS_DBUS_CONFIG_REMOTE_IFACE
} WSDBusConfig;

typedef struct _WSDBusData WSDBusData;


/** \brief First function to be called in every program using this wrapper
 *
 * This function is used for allocating memory for the WSDBusData structure and 
 * setting some basic parameters. 
 * \param name - name of the application
 * \param version - app's version
 * \return pointer to WSDBusData structure\n
 * \sa WSDBusData
 */
WSDBusData * ws_dbus_create (gchar *name, gchar *version);

/** \brief This function is used for setting dbus service/client parameters.
 *
 * These include the addresses of both remote and local apps communicating
 * \param ws_dbus_data - pointer to a structure uniquely identifying 
 * the application for DBus. The possible values of 
 * this parameter are listed here: ::WSDBusConfig
 * \param field - the name of the variable, this function is supposed to set. 
 * \param value - the value of the variable
 * \return WS_STATUS_OK - on success \n
 * WS_STATUS_ERROR - on error
 * \sa WSDBusConfig
 */
WSDBusStatus ws_dbus_config (WSDBusData * ws_dbus_data,
                             WSDBusConfig field,
                             gchar *value);

/**
 * \brief Initialize D-BUS communication.
 *
 * It should be the called after <B>ws_dbus_create</B> and after setting all D-BUS
 * parameters with the <B>ws_dbus_config</B> function. 
 * The structure it returns is necessary to call out every function implemented in
 * this wrapper library.
 * \param ws_dbus_data - it's a structure containing the unique paths to objects 
 * and interfaces to be registered in the DBus session daemon
 * \return WS_STATUS_OK - on success \n
 * WS_STATUS_ERROR - on error
 * \sa WSDBusConfig | ws_dbus_create | ws_dbus_config
 */
WSDBusStatus ws_dbus_connect (WSDBusData * ws_dbus_data);

/** \brief Function deinitializing D-BUS wrapper library
 *
 * Before the program exits, all memory allocated by D-BUS should be freed.
 * \param ws_dbus_data - pointer to a structure uniquely identifying the 
 * application for DBus
 * \return \li WS_STATUS_OK - on success \li WS_STATUS_ERROR - on error
 */
void ws_dbus_destroy (WSDBusData * ws_dbus_data);

/** \brief This function is used for setting a callback function for a
 * specific method name
 *
 * \param ws_dbus_data - pointer to a structure uniquely identifying the 
 * application for D-BUS
 * \param detailed_signal - string containing the signal name which will trigger 
 * the function passed in the c_func parameter.
 * \param c_func - is a pointer to the signal handling function.
 * \param user_data - is the data passed directly to the signal handling
 *  function
 * \return \li WS_STATUS_OK - on success \li WS_STATUS_ERROR - on error
 * \sa WSDBusConfig | ws_dbus_cb
 */
WSDBusStatus ws_dbus_set_cb (WSDBusData * ws_dbus_data,
                             gchar * detailed_signal,
                             gpointer c_func,
                             gpointer user_data);

/** \brief Function registering a method to be called remotely
 *
 * \param ws_dbus_data - pointer to a structure uniquely identifying the 
 * application for DBus
 * \param method - method name\n
 * \param arglist - a comma separated list of method's arguments (actually just
 * their datatypes) finished with WS_DBUS_TYPE_INVALID
 * \return \li WS_STATUS_OK - on success \li WS_STATUS_ERROR - on error
 * \sa WSDBusDataType
 */
WSDBusStatus ws_dbus_add_method (WSDBusData * ws_dbus_data, gchar *method, ...);

/** \brief Function deinitializing D-BUS wrapper library
 *
 * Before the program exits, all memory allocated by D-BUS should be freed
 *
 * \param ws_dbus_data - pointer to a structure uniquely identifying the 
 * application for DBus
 * \param method - method name
 * \param arglist - a comma separated list of pairs: datatype, value finished 
 * with WS_DBUS_TYPE_INVALID
 * \return \li WS_STATUS_OK - on success \li WS_STATUS_ERROR - on error
 * \sa WSDBusDataType
 */
WSDBusStatus ws_dbus_call_method (WSDBusData * ws_dbus_data, gchar *method,...);

/** \name Functions to be called by client (GUI)*/
/*@{*/
/** \brief Instructs the manager to start search of words matching a given
 * pattern
 */
#define ws_dbus_client_find_word(structure,arg)\
		ws_dbus_call_method(structure,\
		                    "find_word",\
		                    WS_DBUS_TYPE_STRING,\
		                    arg,\
		                    WS_DBUS_TYPE_INVALID)

/** \brief Instructs the manager to start search of translations for a specific 
 * word
 */
#define ws_dbus_client_find_translation(structure,arg)\
		ws_dbus_call_method(structure,\
		                    "find_translation",\
		                    WS_DBUS_TYPE_STRING,\
		                    arg,\
		                    WS_DBUS_TYPE_INVALID)

/** \brief Instructs the manager to bookmark a given pair of word, translation
 */
#define ws_dbus_client_add_bookmark(structure,arg1,arg2)\
		ws_dbus_call_method(structure,\
		                    "add_bookmark",\
		                    WS_DBUS_TYPE_STRING,\
		                    arg1,\
		                    WS_DBUS_TYPE_STRING,\
		                    arg2,\
		                    WS_DBUS_TYPE_INVALID)

/** \brief Instructs the manager to remove the specified bookmark
 */
#define ws_dbus_client_remove_bookmark(structure,arg)\
		ws_dbus_call_method(structure,\
		                    "remove_bookmark",\
		                    WS_DBUS_TYPE_STRING,\
		                    arg,\
		                    WS_DBUS_TYPE_INVALID)

/** \brief Instructs the manager to extract a dictionary from a tar.bz2 archive
 */
#define ws_dbus_client_extract_dictionary(structure,arg)\
		ws_dbus_call_method(structure,\
		                    "extract_dictionary",\
		                    WS_DBUS_TYPE_STRING,\
		                    arg,\
		                    WS_DBUS_TYPE_INVALID)
/*@}*/

/** \name Functions to be called by server (thread's manager)*/
/*@{*/

/** \brief Informs GUI that a dictionary has been extracted
 */
#define ws_dbus_server_return_extracted_dict(structure,arg)\
		ws_dbus_call_method(structure,\
		                    "return_extracted_dict",\
		                    WS_DBUS_TYPE_STRING,\
		                    arg,\
		                    WS_DBUS_TYPE_INVALID)

/** \brief Returns a list of words to GUI
 */
#define ws_dbus_server_return_words(structure,arg)\
		ws_dbus_call_method(structure,\
		                    "return_words",\
		                    WS_DBUS_TYPE_GARRAY,\
		                    arg,\
		                    WS_DBUS_TYPE_INVALID)

/** \brief Returns to GUI a list of translations for a given word
 */
#define ws_dbus_server_return_translations(structure,arg)\
		ws_dbus_call_method(structure,\
		                    "return_translations",\
		                    WS_DBUS_TYPE_STRING,\
		                    arg,\
		                    WS_DBUS_TYPE_INVALID)

/** \brief Updates the search progressbar
 */
#define ws_dbus_server_update_progressbar(structure,arg)\
		ws_dbus_call_method(structure,\
		                    "update_progressbar",\
		                    WS_DBUS_TYPE_DOUBLE,\
		                    arg,\
		                    WS_DBUS_TYPE_INVALID)
/*@}*/

/** \name Functions to be called by both client and server*/
/*@{*/

/** \brief Sends a predefined signal
 */
#define ws_dbus_notify(structure,arg)\
		ws_dbus_call_method(structure,\
		                    "signal",\
		                    WS_DBUS_TYPE_SIGNAL,\
		                    arg,\
		                    WS_DBUS_TYPE_INVALID)

/*@}*/

#ifdef __cplusplus
}
#endif

#endif

/*@}*/
