/*******************************************************************************
This file is part of mDictionary

mDictionary is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

mDictionary is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License 
along with WhiteStork; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Copyright 2006-2008 ComArch S.A.
*******************************************************************************/
/** \file dictionary_engine.h
 * \brief API for dictionary engines.
 *
 * This file define engine API which should be implemented by each dictionary
 * and is being used by WhiteStork manager (or any other program that want to
 * use WhiteStork plug-in capabilities).
 * \author Dariusz Wiechecki \<dariusz.wiechecki\@comarch.com\>
 * \date 05-11-2007
 * \version 1.0.0
 * \see EngineAPI
 */
#ifndef   _DICTIONARY_ENGINE_BASE_API_
#define   _DICTIONARY_ENGINE_BASE_API_
#ifdef    __cplusplus
       extern "C" {
#endif

/** \brief API for dictionary engines.
 * \defgroup EngineAPI Dictionary Engine API
 *
 * This file define engine API which should be implemented by each dictionary
 * and is being used by WhiteStork manager (or any other program that want to
 * use WhiteStork plug-in capabilities).
 * \author Dariusz Wiechecki \<dariusz.wiechecki\@comarch.com\>
 * \date 05-11-2007
 * \version 1.0.0
 */
/*@{*/

/* GLib header - enigne API is based on GLib's basic types and functions */
#include <glib.h>

/** \name Names and Versions
 * These defines how should be called exported function and what is the version
 * of this Engine API.
 * */
/*@{*/

/** \brief ENGINE_API_VERSION tells what version of API this header define.
 *
 * Numbers in version are:
 * \li 1st - major version number
 * \li 2nd - minor version number
 * \li 3rd - patch number \n
 *
 * All version with the same major number should be compatible (it is possible
 * that versions with different major number will be compatiblle but this is not
 * guaranteed).\n Minor number is for extension of basic major version (it is
 * not allow that version with the same major number and different minor number
 * are not compatible!).\n Patch number is only for internal use (e.g. when some
 * new comments has been added, one could change patch number of current API)
 *
 * \WARNING Each module should remember what version of API it had been
 *          compiled with!!!
 *
 * \changes
 * \li version 1.0.0 - First release of engine API. It is not compatible with
 *                   previous engine's API versions.
 * \li version < 1.0.0 - First version of engine API, when there was no engine
 *                     version yet.
 */
#define ENGINE_API_VERSION "1.0.0"

/** \brief Global functions structure name.
 *
 * This name defines what should be called function used for getting global
 * functions structure. It helps in dynamical loading of particular engine.\n
 * When writing new engine, there should be always function called like this
 * and returning particular EngineModule structure.
 * \see engine_global_functions | EngineModule
 */
#define _GLOBAL_FUNCTIONS_ engine_global_functions

/* We do not want this two defines in output documentation. */
/** \cond */
#define ___TOSTRING(x) #x
#define __TOSTRING(x) ___TOSTRING(x)
/** \endcond */

/** \brief String containing global functions structure name.
 *
 * This string contains GLOBAL_FUNCTIONS name. It should be used while
 * trying to load global functions structure. It could look like this:
 * \code
 * getting_additional_t get_functions;
 * GModule *library = g_module_open("/path/to/module", G_MODULE_BIND_LAZY);
 * (...)
 * g_module_symbol(library, GLOBAL_FUNCTIONS_NAME, (gpointer)&get_functions);
 * \endcode
 * \see _GLOBAL_FUNCTIONS_ | getting_additional_t | engine_global_functions |
 * EngineModule
 */
#define GLOBAL_FUNCTIONS_NAME   __TOSTRING(_GLOBAL_FUNCTIONS_)
/*@}*/

/** \name Global Module Macros
 *
 * These macros can be used to make it easier to work with many dictionary
 * engine modules.
 * */
/*@{*/

/** \brief Checks if given path points to dictionary is in compatible format.
 *
 * If there is no knowledge what format dictionary has, it is possible to check
 * if particular engine will handle given dictionary.
 * \param module module which compatibility we want to check
 * \param location dictionary which compatibility with module we want to check
 * \return true if module and dictionary are compatible; otherwise it returns
 * false
 */
#define dict_eng_module_check(module, location)\
                            ((module).engine_check(location))

/** \brief Get description of the module.
 *
 * If user would like to present some information about particular module, he
 * can use this function. It returns string (char*) that should contain basic
 * information about module (engine). It could return, for example, something
 * similiar to:
 * \code "XDXF Engine Module. Copyright (C) 2007, foo. Version 1.0" \endcode
 * 
 *
 * \param module module which description we want to get
 * \return string describes given module
 */
#define dict_eng_module_get_description(module)\
                                      ((module).engine_description())

/** \brief Get version of the module.
 *
 * Each module (dictionary engine) shoul keep its version number. It could be
 * accessed by this function.
 * \Warning Module version and Engine API version are not the same !
 * \param module module which version we want to get
 * \return string containing version of given module
 */
#define dict_eng_module_get_version(module)\
                                  ((module).engine_version())

/** \brief Get version of Engine API with which module was compiled.
 *
 * Use this function to get know which version of Engine API this particular
 * engine support.
 * \Warning Module version and Engine API version are not the same !
 * \param module module which supported API version we want to get
 * \return string containing version of engine API given module was compiled 
 * with
 */
#define dict_eng_module_get_api_version(module)\
                                      ((module).engine_api_version())

/** \brief Get format of supported dictionaries.
 *
 * Call this function to get description of supported dictionary format.
 * \param module module which supported format we want to know
 * \return string describing supported format (e.g. XDXF, StarDict)
 * with
 */
#define dict_eng_module_get_format(module)\
                                 ((module).engine_format())

/** \brief Create engine instance connected with particular dictionary.
 *
 * It is simplified version of macro dict_eng_module_create_ext. It do the same,
 * the only difference is that it takes less number of parameter and disable 
 * progress_handler while function is working.
 * \param module module which we want to use to handle dictionary
 * \param location path to dictionary file
 * \param flags define if we want to optimize dictionary
 * \return pointer to Engine structure connected with dictionary or NULL if 
 * creating failed.
 */
#define dict_eng_module_create(module, location, flags)\
                            ( (module).engine_create((location),\
                                                     (flags),\
                                                      NULL,\
                                                      NULL,\
                                                      0.01) )

/** \brief Create engine instance connected with particular dictionary.
 *
 * To start using particular dictionary it is required to create concrete
 * instance of engine structure, connected with that dictionary (file). This
 * function allow to create such a structure. In addition to this, this function
 * set a callback for caching progress bar, while this proccess take place
 * before final engine structure is created. Setting other callbacks is possible
 * only after this function finished.
 * \param module module which we want to use to handle dictionary
 * \param location path to dictionary file
 * \param flags define if we want to optimize dictionary
 * \param progress_handler callback for progress bar
 * \param progress_data pointer to data which will be passed to progress_handler
 * \param seed defines how often will be progress_handler called (0.01 means
 * that progress_handler will be called at each 1% of progress)
 * \return pointer to Engine structure connected with dictionary or NULL if 
 * creating failed.
 * \see cb_progress
 */
#define dict_eng_module_create_ext(module, location, flags, progress_handler,\
                                   progress_data, seed)\
                                ( (module).engine_create((location),\
                                                         (flags),\
                                                         (progress_handler),\
                                                         (progress_data),\
                                                         (seed)) )
/*@}*/

/** \name Particular Dictionary Macros
 *
 * These macros are working on particular dictionary. It makes it easier to work
 * through these macros than by calling member of engine structure.
 * \see Engine
 */
/*@{*/


/** \brief Set seed value for particular progress callback.
 *
 * While engine is working it is possible to call progress callback sometimes.
 * Seed define after how big progress such a calback will be called.
 * \param engine pointer to engine (particular dictionary)
 * \param signal signal for which we want to set a seed
 * \param val progress step for progress callback
 */
#define dict_eng_set_progress_seed(engine, signal, val)\
                                 ((engine)->engine_set_progress_seed( (engine),\
                                                                      (signal),\
                                                                      (val) ))

/** \brief Set-up auto-freeing of memory allocated by engine.
 *
 * Auto-freeing mechanism allows user to not thinking about memory chunks
 * allocated by engine. If it is enabled user do not need to free received
 * GArray with words or translation in callback functions.
 * \param engine pointer to engine (particular dictionary)
 * \param state boolean value deciding if auto-freeing shoul be enabled
 */
#define dict_eng_set_auto_free(engine, state)\
                              (engine)->engine_set_auto_free((engine), (state))

/** \brief Activate optimization mechanizms for particular dictionary.
 *
 * Every engine could have some optimization methods. By calling this function 
 * we enabled this option. For the user of the engine it is not important what 
 * kind of optimization is used in current engine, how does it work etc.
 * It should be opaque for engine's users.
 *
 * \param engine pointer to engine which should be optimized
 * \see dict_eng_module_create() | dict_eng_is_optimized() | Engine
 */
#define dict_eng_optimize(engine) ((engine)->engine_optimize((engine)))

/** \brief Find out if specified engine is using optimization now.
 *
 * \param engine pointer to engine
 * \return boolean value telling if particular dictionary is optimized at this
 * moment
 */
#define dict_eng_is_optimized(engine) ((engine)->engine_is_optimized((engine)))

/** \brief To get know from what location was created current dictionary.
 *
 * If user of engine want to get know from what locatione exactly was created 
 * particular engine, he could call this function. It returns string, most
 * likely the same as the one passed earlier to dict_eng_module_create() 
 * function as a location argument.
 * \param engine pointer to the engine
 * \return gchar* containing path to dictionary path
 * \see dict_eng_module_create | dict_eng_module_create_ext
 */
#define dict_eng_get_location(engine) ((engine)->engine_location((engine)))

/** \brief Tells if last operation was sucesfull.
 *
 * If user want to be sure that everything was ok, during last operation 
 * (it has finished with success), he has to check if dict_eng_get_last_status()
 * is equal to ENGINE_NO_ERROR. If not, he can also find the reason 
 * why operation failed.
 * \param engine pointer to the engine
 * \return EngineStatus defining finish code of last operation
 * \see EngineStatus | dict_eng_status_message
 */
#define dict_eng_get_last_status(engine) ((engine)->engine_status((engine)))

/** \brief Translate pure state code to meaningfull message.
 *
 * If there was a error during engine was working, we can present to the user
 * the reason for this error in meaningfull form.
 * \param engine pointer to the engine
 * \param error code which we want to translate
 * \return gchar* message describing concrete error code
 * \see EngineStatus | dict_eng_get_last_status
 */
#define dict_eng_status_message(engine, status)\
                              ((engine)->engine_status_message((status)))

/** \brief Destroy gently and completely given dictionary engine.
 *
 * When user do not need anymore particular dictionary, he must destroy it to
 * free memory allocated by this dictionary. It will close every opened file
 * descriptor, free any additional memory block allocated while engine was
 * working etc. User are not allowed to call free() on dictionary without
 * calling before dict_eng_destroy().
 *
 * \param engine pointer to the engine which is not needed anymore
 * \see dict_eng_module_create | Engine
 */
#define dict_eng_destroy(engine) ((engine)->engine_close(engine))

/** \brief Start searching for words matching given pattern.
 *
 * \param engine pointer to the engine to search in
 * \param pattern string describing pattern to search
 * \param data pointer do data which should be passed to callback function after
 * searching has been completed. If it is NULL, plugin will use data pointer
 * given while setting callback function for searching words list
 * \see dict_eng_search_word_translation | Engine
 */
#define dict_eng_search_word_list(engine, pattern, data)\
                        ((engine)->engine_search_word_list( (engine),\
                                                            (pattern),\
                                                            (data) ))

/** \brief Start searching for word's translation.
 *
 * \param engine pointer to the engine to search in
 * \param word word which translation we are searching
 * \param data pointer do data which should be passed to callback function after
 * searching has been completed. If it is NULL, plugin will use data pointer
 * given while setting callback function for searching translation
 * \see dict_eng_search_word_list | Engine
 */
#define dict_eng_search_word_translation(engine, word, data)\
                        ((engine)->engine_search_word_translation( (engine),\
                                                                   (word),\
                                                                   (data) ))

/** \brief Set callback function for particular signal.
 *
 * \param engine pointer to the engine
 * \param signal defines for what signal we are setting callback
 * \param c_handler function which will be called on signal
 * \param data pointer to data which will be passed to c_handler. It could be
 * overwritten while starting search
 * \return gpointer to the previously set callback
 * \see dict_eng_search_word_list | dict_eng_search_word_translation  | Engine
 */
#define dict_eng_set_callback(engine, signal, c_handler, data)\
                            ((engine)->engine_set_callback( (engine),\
                                                            (signal),\
                                                            (c_handler),\
                                                            (data) ))

/** \brief Add new word to the dictionary.
 *
 * This method should be implemented by every engine, but it is in API only for
 * bookmark engine. Other enignes should do nothing but return FALSE.
 *
 * \param engine pointer to the engine
 * \param word string with word which we want to add
 * \param tran string containing translation of word
 * \return boolean telling if word has been added. FALSE value could be returned
 * for few reasons: engine do not support adding word, word already exists in
 * dictionary etc.
 */
#define dict_eng_add_word(engine,word,tran)\
                        ((engine)->engine_add_word( (engine),\
                                                    (word),\
                                                    (tran) ))


/** \brief Remove word from the dictionary.
 *
 * This method should be implemented by every engine, but it is in API only for
 * bookmark engine. Other enignes should do nothing but return FALSE.
 *
 * \param engine pointer to the engine
 * \param word string with word which we want to remove
 * \return boolean telling if word has been removed. FALSE value could be 
 * returned for few reasons: engine do not support adding word, word already 
 * exists in dictionary etc.
 */
#define dict_eng_remove_word(engine,word)\
                           ((engine)->engine_remove_word( (engine),\
                                                          (word) ))

/** \brief Get language from which this dictionary translates.
 *
 * Method returns language description from which given dictionary engine
 * translates.
 *
 * \param engine pointer to the engine
 * \return string describing language
 * \sa dict_eng_get_lang_to
 */
#define dict_eng_get_lang_from(engine) ((engine)->engine_get_lang_from(engine))

/** \brief Get language to which this dictionary translates.
 *
 * Method returns language description to which given dictionary engine
 * translates.
 *
 * \param engine pointer to the engine
 * \return string describing language
 * \sa dict_eng_get_lang_from
 */
#define dict_eng_get_lang_to(engine) ((engine)->engine_get_lang_to(engine))

/** \brief Get dictionary title.
 *
 * Each dictionary has title describing data in it, its sources etc. This
 * function returns this title which could be used in UI for the end user.
 *
 * \param engine pointer to the engine
 * \return string containing dictionary title
 * \sa dict_eng_get_lang_from | dict_eng_get_lang_to
 */
#define dict_eng_get_title(engine) ((engine)->engine_get_title(engine))

/** \brief Get dictionary icon path.
 *
 * Some dictionaries have own icons, which could be used in UI. If such a icon
 * exists, this function should return path to it. Otherwise it should return
 * path to the default icon.
 *
 * \param engine pointer to the engine
 * \return string containing path to icon
 */
#define dict_eng_get_icon_path(engine) ((engine)->engine_get_icon_path(engine))
/*@}*/

/** \name Callbacks' Signals
 *
 * These strings identify signals which will be emitted after dictionary engine
 * finished some work.
 *
 * Signals are defined as a strings to make it easier to keep compatibility and
 * to be more flexible for future changes.
 */
/*@{*/

/** \brief Signals that words list matching for given pattern is available.
 *
 * ENGINE_WORD_LIST_SIGNAL defines name for signal passed to 
 * dict_eng_set_callback() function as a signal parameter. Function set 
 * to handle this signal should be called only from dict_eng_search_word_list()
 * and have cb_word_list type.
 * \note Programmers must not use value of ENGINE_WORD_LIST_SIGNAL
 * directly!
 * \see cb_word_list | dict_eng_set_callback | dict_eng_search_word_list |
 *  ENGINE_WORD_TRANSLATION_SIGNAL
 */
#define ENGINE_WORD_LIST_SIGNAL                 "on_word_list_found"

/** \brief Signals that translation is available for given word.
 *
 * ENGINE_WORD_TRANSLATION_SIGNAL defines name for signal passed to
 * dict_eng_set_callback() function as a signal parameter. Function set
 * to handle this signal should be called only from
 * dict_eng_search_word_translation() and have cb_word_translation
 * \note Programmers must not use value of ENGINE_WORD_TRANSLATION_SIGNAL_SIGNAL
 * directly!
 * \see cb_word_translation | dict_eng_set_callback |
 * dict_eng_search_word_translation ENGINE_WORD_LIST_SIGNAL
 */
#define ENGINE_WORD_TRANSLATION_SIGNAL          "on_word_translation_found"

/** \brief Signals progress event for optimization proccess.
 *
 * ENGINE_PROGRESS_OPTIMIZING_SIGNAL defines name for signal passed to
 * dict_eng_set_callback() function as a signal parameter. Function set
 * to handle this signal should be called only while creating cache files and
 * other resources needed for optimization. This function should have
 * cb_word_translation type.
 * \note Programmers must not use value of ENGINE_PROGRESS_OPTIMIZING_SIGNAL
 * directly!
 * \see cb_progress | dict_eng_set_progress_seed
 */
#define ENGINE_PROGRESS_OPTIMIZING_SIGNAL       "on_progress_optimizing"
/*@}*/

/** \name Flags and Error Codes */
/*@{*/

/** \brief Flags defining how to use optimization for particular dictionary.
 *
 * This flags are used while calling dict_eng_module_create_ext and 
 * dict_eng_module_create functions. After instance of engine was created it is
 * possible to enable optimization by function dict_eng_optimize.
 * \see dict_eng_optimize | dict_eng_module_create | dict_eng_module_create_ext
 * | dict_eng_is_optimized
 */
typedef enum
{
	ENGINE_CREATE   = 1 << 0,
	/**< use optimization. If cache file does not exist - create it. */
	ENGINE_NO       = 1 << 1,
	/**< disables optimization at all */
	ENGINE_REFRESH  = 1 << 2
	/**< always recreate cache file and use optimization */
} EngineOptimizationFlag;

/** \brief Codes of possible errors which can occure while engine is working.
 * 
 * Enum type for errors' codes. One of this code is always in last error
 * variable (variable 'last error' is invisible for programmers - they should 
 * use function dict_eng_get_last_status() and optionally 
 * dict_eng_get_status_message() to get know what kind of error occured).
 * \see dict_eng_get_last_status | dict_eng_status_message
 */
typedef enum
{
	ENGINE_NO_ERROR        = 0,
	/**< there was no error - last action successed */
	ENGINE_WRONG_FILE      = 1 << 0,
	/**< file which engine tried to read, has wrong format or it is 
	 * corrupted.*/
	ENGINE_COULDNT_READ    = 1 << 1,
	/**< user do not have permission to read file which engine tried
	 * to use */
	ENGINE_COULDNT_WRITE   = 1 << 2,
	/**< engine could not write data into dictionary (also when engine does
	 * not support such a action) */
	ENGINE_INTERNAL_ERROR  = 1 << 3,
	/**< there was engine's internal error */
	ENGINE_NO_FILE         = 1 << 4,
	/**< file on which engine tried to operate, do not exist */
	ENGINE_OUT_OF_MEMORY   = 1 << 5
	/**< there was no sufficient available memory to execute last action */
}
EngineStatus;
/*@}*/


/** \name Callbacks' Types
 *
 * Types defining of what types should be particular callbacks to properly work
 * with Engine API.
 * \see dict_eng_set_callback
 */
/*@{*/

/** \brief Type of callback functions for retrieving word list.
 *
 * Function which is set by dict_eng_set_callback() to signal 
 * ENGINE_WORD_LIST_SIGNAL should be exactly this type.
 *
 * \param list GArray with all words matching given pattern
 * \param pattern string containing pattern for words
 * \param error error code; if everything was ok it is ENGINE_NO_ERROR
 * \param user_data pointer to data set by user to be passing to each callback
 * for ENGINE_WORD_LIST_SIGNAL signal
 */
typedef void (*cb_word_list)(GArray* list,
                             gchar* pattern,
                             gpointer user_data,
                             EngineStatus error);

/** \brief Type of callback functions for retrieving word's translation.
 *
 * Function which is set by dict_eng_set_callback() to signal
 * ENGINE_WORD_TRANSLATION_SIGNAL should be exactly this type.
 *
 * \param translation translation of given word
 * \param word word which translation we already retrieved
 * \param error error code; if everything was ok it is ENGINE_NO_ERROR
 * \param user_data pointer to data set by user to be passing to each callback
 * for ENGINE_WORD_TRANSLATION_SIGNAL signal
 */
typedef void (*cb_word_translation)(gchar* translation,
                                    gchar* word,
                                    gpointer user_data,
                                    EngineStatus error);

/** \brief Type of callback functions for handling progress of optimization.
 *
 * Function which is set by dict_eng_set_callback() to signal
 * ENGINE_PROGRESS_OPTIMIZING_SIGNAL should be exactly this type.
 *
 * \param value number telling what percentage (0.00 - 1.00) of optimization
 * proccess is finished.
 * \param error error code; if everything was ok it is ENGINE_NO_ERROR
 * \param user_data pointer to data set by user to be passing to each callback
 * for ENGINE_PROGRESS_OPTIMIZING_SIGNAL signal
 */
typedef void (*cb_progress)(gdouble value,
                            gpointer user_data,
                            EngineStatus error);
/*@}*/



struct _Engine;


typedef struct _Engine Engine;

/** \brief Structure describing particular dictionary.
 *
 */
struct _Engine
{
	/* dictionary independent data. REMEMBER: always at the beggining of
	   structure!!! */
	gpointer engine_data;
	/**< engine independent data - this should be always placed as the first
	 * field in engine structure. It must not be used by the user of engine
	 * and there should be all data needed for engine to work. Structure
	 * of data pointed by this pointer is free-form. */

        // 0.1 API specification
	void (*engine_set_auto_free)( Engine* engine,
	                              gboolean state );
	/**< pointer to dictionary 'dict_eng_set_auto_free' function
	 * \sa dict_eng_set_auto_free */

	gchar* (*engine_location)(Engine* engine);
	/**< pointer to dictionary 'dict_eng_get_location' function
	 * \sa dict_eng_get_location */

	gboolean (*engine_is_optimized)(Engine* engine);
	/**< pointer to dictionary 'dict_eng_is_optimized' function
	 * \sa dict_eng_is_optimized */

	void (*engine_optimize)(Engine* engine);
	/**< pointer to dictionary 'dict_eng_optimize' function
	 * \sa dict_eng_optimize */

	void (*engine_search_word_list)( Engine* engine,
	                                 gchar* pattern,
	                                 gpointer data );
	/**< pointer to dictionary 'dict_eng_search_word_list' function
	 * \sa dict_eng_search_word_list */

	void (*engine_search_word_translation)( Engine* engine,
	                                        gchar* word,
	                                        gpointer data );
	/**< pointer to dictionary 'dict_eng_search_word_translation' function
	 * \sa dict_eng_search_word_translation */

	void (*engine_close)(Engine* engine);
	/**< pointer to dictionary 'dict_eng_destroy' function
	 * \sa dict_eng_destroy */

	EngineStatus (*engine_status)(Engine* engine);
	/**< pointer to dictionary 'dict_eng_get_last_status' function
	 * \sa dict_eng_get_last_status */

	gchar* (*engine_status_message)(EngineStatus error);
	/**< pointer to dictionary 'dict_eng_get_status_message' function
	 * \sa dict_eng_get_status_message */

	gpointer (*engine_set_callback)( Engine* engine,
	                                 gchar* signal,
	                                 gpointer c_handler,
	                                 gpointer user_data );
	/**< pointer to dictionary 'dict_eng_set_callback' function
	 * \sa dict_eng_set_callback */

	void (*engine_set_progress_seed)( Engine* engine,
	                                  gchar* signal,
	                                  gdouble seed );
	/**< pointer to dictionary 'dict_eng_set_progress_seed' function
	 * \sa dict_eng_set_progress_seed */


	/* 0.2 API specification's functions */
	gboolean (*engine_add_word)( Engine* engine,
	                             gchar*  word,
	                             gchar*  translation );
	/**< pointer to dictionary 'dict_eng_add_word' function
	 * \sa dict_eng_add_word */

	gboolean (*engine_remove_word)( Engine* engine,
	                                gchar*  word );
	/**< pointer to dictionary 'dict_eng_remove_word' function
	 * \sa dict_eng_remove_word */

	gchar* (*engine_get_lang_from)(Engine* engine);
	/**< pointer to dictionary 'dict_eng_get_lang_from' function
	 * \sa dict_eng_get_lang_from */

	gchar* (*engine_get_lang_to)(Engine* engine);
	/**< pointer to dictionary 'dict_eng_get_lang_to' function
	 * \sa dict_eng_get_lang_to */

	gchar* (*engine_get_title)(Engine* engine);
	/**< pointer to dictionary 'dict_eng_get_title' function
	 * \sa dict_eng_get_title */

	gchar* (*engine_get_icon_path)(Engine* engine);
	/**< pointer to dictionary 'dict_eng_get_icon_path' function
	 * \sa dict_eng_get_icon_path */
};

//------------------------------------------------------------------------------
/** \brief Global functions structure.
 *
 * This structure holds pointers to helpful functions, that could be used for
 * creating engines for particular dictionary. It helps programmers to work with
 * dictionary API while they need only to import one function
 * (engine_global_functions) and it will return this structure. After that
 * programmers can start using this dictionary module by calling accesible
 * functions.
 */
typedef struct  {
	gboolean (*engine_check)(gchar* location);
	/**< pointer to 'dict_eng_module_check' function
	 * \sa dict_eng_module_check */

	gchar*   (*engine_description)(void);
	/**< pointer to 'dict_eng_module_get_description' function
	 * \sa dict_eng_module_get_description */

	gchar*   (*engine_format)(void);
	/**< pointer to 'dict_eng_module_get_format' function
	 * \sa dict_eng_module_get_format */

	gchar*   (*engine_version)(void);
	/**< pointer to 'dict_eng_module_get_version' function
	 * \sa dict_eng_module_get_version */

	gchar*   (*engine_api_version)(void);
	/**< pointer to 'dict_eng_module_get_api_version' function
	 * \sa dict_eng_module_get_api_version */

	Engine*  (*engine_create)( gchar* location,
	                           EngineOptimizationFlag flags,
	                           cb_progress progress_handler,
	                           gpointer progress_data,
	                           gdouble seed );
	/**< pointer to 'dict_eng_module_create' function
	 * \sa dict_eng_module_create */
}
EngineModule;
//------------------------------------------------------------------------------
typedef EngineModule (*getting_additional_t)(void);
//------------------------------------------------------------------------------
extern EngineModule engine_global_functions(void);
//------------------------------------------------------------------------------


/*@}*/

#ifdef __cplusplus
}
#endif
#endif /* _DICTIONARY_ENGINE_BASE_API_ */

/** \example test_dictionary_engine.c
 *
 * Example below show how Dictionary Engine API simplify proccess of searching
 * translation in different dictionaries.
 * \li At first we need to define callbacks which will be called after engine
 * sucesfully find words list or translation:
 * \code
 * // Callback for returning words list
 * void words_list_cb(GArray* list,
 *                    gchar* pattern,
 *                    gpointer user_data,
 *                    EngineStatus error)
 * {
 *	if ( ENGINE_NO_ERROR == error )
 *	{
 *		printf("Word matching to pattern: %s\n",pattern);
 *		int i = 0;
 *		while(g_array_index(list, gchar*, i) != NULL) 
 *		{
 *			printf("%d.: %s\n", i, g_array_index(list, gchar*, i));
 *			i++;
 *		}
 *	}
 * }
 *
 * // Callback for returning translation of word
 * void translation_cb(gchar* translation,
 *                     gchar* word,
 *                     gpointer user_data,
 *                     EngineStatus error)
 * {
 *	if ( ENGINE_NO_ERROR == error )
 *	{
 *		printf("Translation for word %s:\n",word);
 *		printf("%s\n",translation);
 *	}
 * }
 * \endcode
 *
 * \li Secondly we need to load module:
 * \code
 * getting_additional_t get_functions;
 * GModule *library = g_module_open(library_to_path, G_MODULE_BIND_LAZY);
 * g_module_symbol(library, GLOBAL_FUNCTIONS_NAME, (gpointer)&get_functions);
 * \endcode
 * \li Next, we need to get basic, global functions which will allow us to 
 * create Engine for particular dictionary:
 * \code
 * EngineModule module = get_functions();
 * \endcode
 * \li Now it is possbile to check if we can handle particular dictionary with
 * current engine:
 * \code
 * gchar* dictionary = "/path/to/dictionary/file"
 * gboolean compatible = dict_eng_module_check(module, dictionary);
 * if ( TRUE == compatible )
 * {
 * 	// we can use this engine to work with dictionary
 * 	...
 * }
 * else {
 * 	// we must use other engine to work with this dictionary
 * }
 * \endcode
 * \li If current module is compatible with dictionary we can create engine
 * instance, connected with particular dictionary (file). It is also recommended
 * to set-up callback as far as possible:
 * \code
 * Engine* xdxf;
 * xdxf = dict_eng_module_create(module, dictionary, ENGINE_CREATE);
 * dict_eng_set_callback(xdxf, ENGINE_WORD_LIST_SIGNAL, words_list_cb, NULL);
 * dict_eng_set_callback(xdxf,
 *                       ENGINE_WORD_TRANSLATION_SIGNAL,
 *                       translation_cb,
 *                       NULL);
 * \endcode
 * \li Now we can start working with particular dictionary. To search in
 * dictionary words matching pattern "pol", just call:
 * \code
 * dict_eng_search_word_list(xdxf, "pol");
 * \endcode
 * If You want to find out what is the translation of word "car" simple call:
 * \code
 * dict_eng_search_word_translation(xdxf, "car");
 * \endcode
 */
