/*
 * libtiltstick.c (c) 2008 by Till Harbaum <till@harbaum.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

/* the includes below ae required for the tiltstick interface */
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/select.h>
#include <linux/input.h>  /* linux kernel event interface */

// #include <libosso.h>      /* required for screen saver timeout */

#include "tiltstick.h"

static int tiltstick_evif = 0;
static char tiltstick_name[256];
// osso_context_t *osso_context = NULL;

/* The maemo 2008 kernel does not include the joystick subsystem. */
/* This program thus accesses the TiltStick via the even interface */
/* the linux kernel is providing for all input devices */
/* open the event interface to the TiltStick */
int tiltstick_evif_open(void) {
  DIR *dirp;
  struct dirent *dp;
  
  dirp = opendir("/dev/input");
  if(!dirp) {
    fprintf(stderr, "Unable to open directory /dev/input");
    return -1;
  }
  
  /* try to find TiltStick interface */
  while ((dp = readdir(dirp)) != NULL) {
    if(strncmp(dp->d_name, "event", 5) == 0) {

      sprintf(tiltstick_name, "/dev/input/%s", dp->d_name);
      if((tiltstick_evif = open(tiltstick_name, O_RDONLY)) >= 0) {
	struct input_id id;
	
	/* suck out some device information */
	if(ioctl(tiltstick_evif, EVIOCGID, &id)) {
	  perror("ioctl(EVIOCGID)");
	} else {
	  /* only the tiltstick uses these vendor ids */
	  if((id.vendor == 0x1c40) && (id.product == 0x0533)) {
	    printf("Found device at %s\n", tiltstick_name);
	    
	    if(ioctl(tiltstick_evif, EVIOCGNAME(sizeof(tiltstick_name)),
		     tiltstick_name) < 0) {
	      perror("ioctl(EVIOCGNAME)");
	      strcpy(tiltstick_name, "Unknown");
	    } else
	      printf("Device name: %s\n", tiltstick_name);
	    
	    closedir(dirp);
	    return 0;
	  }
	}
	
	close(tiltstick_evif);
	tiltstick_evif = -1;
      } else
	fprintf(stderr, "Unable to open %s: %s\n",
		tiltstick_name, strerror(errno));
    }
  }
  closedir(dirp);
  return -1;
}

char *tiltstick_get_name(void) {
  return tiltstick_name;
}

/* currently there are only two axis tiltsticks */
static const int tiltstick_axes = 2;

int tiltstick_get_axes(void) {
  return tiltstick_axes;
}

int tiltstick_evif_poll(int *axis, int *axis_num, int *buttons) {
  fd_set rfds;
  struct timeval tv;
  int i, retval;

  /* don't report more axes than we have */
  if(*axis_num > tiltstick_axes)
    *axis_num = tiltstick_axes;
 
  static int buf[TILTSTICK_MAX_AXIS] = { 0, 0, 0 };
  static int rem[TILTSTICK_MAX_AXIS] = { 0, 0, 0 };
  int cnt[TILTSTICK_MAX_AXIS] = { 1, 1, 1 };
  
  if(tiltstick_evif < 0) {
    /* report zero for all axes */
    for(i=0;i<*axis_num;i++)
      axis[i] = 0;

    return -1;
  }

  *buttons = 0;

  /* restore remainder from last call */
  for(i=0;i<*axis_num;i++)
    axis[i] = buf[i] + rem[i];
  
  do {
    FD_ZERO(&rfds);
    FD_SET(tiltstick_evif, &rfds);
    
    /* don't wait at all */
    tv.tv_sec = tv.tv_usec = 0;
    
    retval = select(FD_SETSIZE, &rfds, NULL, NULL, &tv);
    
    if (retval == -1)
      perror("select()");
    else if (retval) {
      if(FD_ISSET(tiltstick_evif, &rfds)) {
	struct input_event ev;
	
	if(read(tiltstick_evif, &ev, sizeof(struct input_event)) < 0) {
	  perror("read()");
	  return -1;
	}
	
	/* button press */
	if(ev.type == 1) *buttons |= 1<<ev.value;
	
	/* the TiltStick returns signed 12 bit values */
	if((ev.type == EV_ABS)&&(ev.code < TILTSTICK_MAX_AXIS)) {
	  buf[ev.code] = ev.value; 

	  if(ev.code < *axis_num) {
	    axis[ev.code] += buf[ev.code];
	    cnt[ev.code]++;
	  }
	}
      }
    }
  } while(retval > 0);  /* read until no more data available */
  
  /* save remainder */
  for(i=0;i<*axis_num;i++) {
    rem[i] = axis[i] % cnt[i];
    axis[i] /= cnt[i];
  }

  return 0;
}
