/*
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of libobd.
 *
 *  libobd is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  libobd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with libobd.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define _GNU_SOURCE /* For strcasestr */

#include <errno.h>
#include <stdio.h>
#include <poll.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>

#include "libobd.h"
#include "obd-chan-file.h"
#include "obd-sensor.h"
#include "obd-log.h"

#define PIPE_READ 0
#define PIPE_WRITE 1
#define LINE_LEN 256
#define BASE_DELAY 90
#define MAX_DELAY_DELTA 30


static const char* name = "OBD File";

struct obdchanfile {
	int closepipe[2];
	struct pollfd poll;
	char* filename;
};

static int obdchanfile_name(char* buffer, unsigned int len)
{
	strncpy(buffer, name, len - 1);
	buffer[len - 1] = '\0';

	return 0;
}

static void obdchanfile_parsedtc(FILE* pfd, struct obd_dtc_data* dtc_data)
{
	char line[LINE_LEN];
	char* equal;

	line[LINE_LEN - 1] = '\0';

	dtc_data->valid = -1;

	while(fgets(line, LINE_LEN - 1, pfd)) {

		if(strchr(line, '#'))
			continue;

		if((equal = strchr(line, '=')) == NULL)
			continue;

		equal[0] = '\0';

		if(!strstr(line, "DTC"))
			continue;

		equal++;

		while ((equal != line + LINE_LEN) && equal != '\0' &&
				(equal[0] != 'P' && equal[0] != 'C' &&
				equal[0] != 'B' && equal[0] != 'U'))
			equal++;


		if((equal == line + LINE_LEN) || equal[0] == '0')
			continue;

		if(equal + 1 == line + LINE_LEN ||
				equal + 2 == line + LINE_LEN ||
				equal + 3 == line + LINE_LEN ||
				equal + 4 == line + LINE_LEN)
			continue;

		if(!isdigit(equal[1]) || !isdigit(equal[2])
					|| !isdigit(equal[3])
					|| !isdigit(equal[4]))
			continue;

		equal[5] = '\0';

		if(dtc_data->valid == -1)
			dtc_data->valid = 1;
		else
			dtc_data->valid++;

		strcpy(dtc_data->code_list[dtc_data->valid - 1], equal);

		if(dtc_data->valid == MAX_DTCS)
			break;
	}
}

static void obdchanfile_parsepid(FILE* pfd, struct obd_sensor_data* sensor_data)
{
	char line[LINE_LEN];
	char* equal;
	unsigned int pid;
	double value;

	line[LINE_LEN - 1] = '\0';

	sensor_data->valid |= OBD_INVALID_DATA;

	while(fgets(line, LINE_LEN - 1, pfd)) {
		if(strchr(line, '#') || strcasestr(line, "DTC"))
			continue;

		if((equal = strchr(line, '=')) == NULL)
			continue;

		equal[0] = '\0';

		errno = 0;
		pid = (unsigned int) strtol(line, 0, 0);
		if(errno || (pid > 0x60)) {
			LOG_WARNING("Unable to parse \'%s\' - ignoring line", line);
			continue;
		}

		if(pid != sensor_data->pid)
			continue;

		if(sscanf(equal + 1, "%lf", &value) == 0) {
			LOG_WARNING("Unable to parse value of PID %02X" , pid);
			continue;
		}

		if(sensor_data->valid & OBD_INVALID_DATA) {
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			sensor_data->result[0] = value;
		} else {
			sensor_data->valid |= OBD_HAS_SECOND_VALUE | OBD_HAS_FIRST_VALUE;
			sensor_data->result[1] = value;
			break;
		}
	}
}

static int obdchanfile_sendpid(struct obdchan* chan, struct obd_sensor_data* sensor_data, unsigned int timeout)
{
	struct obdchanfile* file = NULL;
	FILE* pfd;
	int poll_ret;

	if(chan->type != OBDCHAN_FILE) {
		LOG_ERROR("channel is not a file (did you call chan_setup?)");
		return OBDCHANERR_NOT_SETUP;
	}

	file = chan->data;

	pfd = fopen(file->filename, "r");

	if(pfd == NULL) {
		LOG_ERROR("Unable to open file \"%s\" - %s", file->filename, strerror(errno));
		return OBDCHANERR_ERROR;
	}

	obdchanfile_parsepid(pfd, sensor_data);

	fclose(pfd);

	poll_ret = poll(&file->poll, 1, BASE_DELAY + rand() % MAX_DELAY_DELTA);

	return 0;
}

static int obdchanfile_senddtc(struct obdchan* chan, struct obd_dtc_data* dtc_data, unsigned int timeout)
{
	struct obdchanfile* file = NULL;
	FILE* pfd;
	int poll_ret;

	if(chan->type != OBDCHAN_FILE)
		return OBDCHANERR_NOT_SETUP;

	file = chan->data;

	pfd = fopen(file->filename, "r");

	if(pfd == NULL) {
		LOG_ERROR("Unable to open file \"%s\" - %s", file->filename, strerror(errno));
		return OBDCHANERR_ERROR;
	}

	obdchanfile_parsedtc(pfd, dtc_data);

	fclose(pfd);

	poll_ret = poll(&file->poll, 1, BASE_DELAY + rand() % MAX_DELAY_DELTA);

	return 0;
}

static int obdchanfile_close(struct obdchan* chan)
{
	struct obdchanfile* file = NULL;

	if(chan->type == OBDCHAN_UNDEFINED)
		return OBDCHANERR_NOT_SETUP;

	file = chan->data;

	write(file->closepipe[PIPE_WRITE], "c", 1);
	chan->type = OBDCHAN_UNDEFINED;

	close(file->closepipe[PIPE_WRITE]);
	close(file->closepipe[PIPE_READ]);

	chan->name = NULL;
	chan->sendpid = NULL;
	chan->senddtc = NULL;
	chan->close = NULL;

	free(file->filename);
	free(chan->data);
	chan->data = NULL;

	return 0;
}

int obdchanfile_setup(struct obdchan* chan, char* filename)
{
	struct obdchanfile* file = NULL;
	struct pollfd* pol = NULL;
	FILE* pfd;
	int ret = 0;

	if(chan->type != OBDCHAN_UNDEFINED) {
		LOG_ERROR("Chan was already setup: %d", chan->type);
		return OBDCHANERR_ALREADY_SETUP;
	}

	pfd = fopen(filename, "r");
	if(pfd == NULL) {
		LOG_ERROR("Unable to open file \"%s\" - %s", filename, strerror(errno));
		return OBDCHANERR_ERROR;
	}
	fclose(pfd);

	chan->data = malloc(sizeof(struct obdchanfile));

	if(chan->data == NULL) {
		LOG_ERROR("Could not allocate data for the simulator");
		return OBDCHANERR_ERROR;
	}

	file = chan->data;
	file->filename = strdup(filename);

	file->closepipe[PIPE_READ] = 0;
	file->closepipe[PIPE_WRITE] = 0;

	pol = &file->poll;

	if(pipe(file->closepipe)){
		LOG_ERROR("Could not create the close pipe");
		ret = OBDCONERR_ERROR;
		goto error;
	}

	pol->fd = file->closepipe[PIPE_READ];
	pol->events = POLLIN | POLLPRI;

	chan->type = OBDCHAN_FILE;
	chan->name = obdchanfile_name;
	chan->sendpid = obdchanfile_sendpid;
	chan->senddtc = obdchanfile_senddtc;
	chan->close = obdchanfile_close;

	srand((unsigned)(time(0)));

	return 0;

error:
	if(file) {
		if(file->filename)
			free(file->filename);
		if(file->closepipe[PIPE_READ])
			close(file->closepipe[PIPE_READ]);
		if(file->closepipe[PIPE_WRITE])
			close(file->closepipe[PIPE_WRITE]);

		free(chan->data);
		chan->data = NULL;
	}
	return OBDCHANERR_ERROR;
}

