/*
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of libobd.
 *
 *  libobd is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  libobd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with libobd.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <poll.h>
#include <unistd.h>

#include "obd-chan-sim.h"
#include "obd-log.h"

#define PIPE_READ 0
#define PIPE_WRITE 1

#define MIN_OBD_DELAY 90
#define MAX_OBD_DELAY 250

static const char* name = "OBD Simulator";

struct obdchansim {
	int closepipe[2];
	struct pollfd poll;
	time_t start;
};

/*
 * -------------------------------------------------------
 * Random releated funcitons 
 * -------------------------------------------------------
 */

static int obdchansim_irand(int min, int max)
{
	if (min>max)
		return max+(int)((min-max+1)*(rand()/(RAND_MAX+1.0)));
	else
		return min+(int)((max-min+1)*(rand()/(RAND_MAX+1.0)));
}

static double obdchansim_drand(double min, double max)
{
	double random = rand()/((double)(RAND_MAX)+1);
	if (min>max)
		return random*(min-max)+max;
	else
		return random*(max-min)+min;
}

/*
 * -------------------------------------------------------
 * End of random releated funcitons 
 * -------------------------------------------------------
 */

static int obdchansim_name(char* buffer, unsigned int len)
{
	strncpy(buffer, name, len - 1);
	buffer[len - 1] = '\0';

	return 0;
}

static void obdchansim_pidvalue(struct obdchan* chan, struct obd_sensor_data* sensor_data)
{
	struct obdchansim* sim = chan->data;

	switch (sensor_data->pid) {
		case 0x00:
		case 0x20:
		case 0x40:
			sensor_data->result[0] = 0xFFFFFFFF;
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			return;
		case 0x01:
			sensor_data->result[0] = 1 << 7;
			sensor_data->result[1] = 6;
			sensor_data->valid |= OBD_HAS_FIRST_VALUE | OBD_HAS_SECOND_VALUE;
			return;
		case 0x03:
			sensor_data->result[0] = 1 << 1; /* Closed loop, using oxygen sensor feedback to determine fuel mix */
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			return;
		case 0x04:
		case 0x11:
		case 0x2C:
		case 0x2E:
		case 0x2F:
		case 0x45:
		case 0x47:
		case 0x48:
		case 0x49:
		case 0x4A:
		case 0x4B:
		case 0x4C:
			sensor_data->result[0] =  obdchansim_drand(0, 100);
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			return;
		case 0x05:
		case 0x0F:
		case 0x46:
			sensor_data->result[0] =  obdchansim_irand(-40, 215);
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			sensor_data->rimperial[0] = ((sensor_data->result[0] * 9) / 5.0) - 32;
			sensor_data->valid |= OBD_HAS_FIRST_VALUE_IMPERIAL;
			return;
		case 0x06:
		case 0x07:
		case 0x08:
		case 0x09:
			sensor_data->result[0] =  obdchansim_drand(-100, 99);
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			return;
		case 0x0A:
		case 0x0B:
		case 0x33:
			sensor_data->result[0] =  obdchansim_irand(0, 765);
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			sensor_data->valid |= OBD_HAS_FIRST_VALUE_IMPERIAL;
			sensor_data->rimperial[0] = (sensor_data->result[0] * 9) / 100;
			return;
		case 0x0D:
			sensor_data->result[0] =  obdchansim_irand(0, 255);
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			sensor_data->valid |= OBD_HAS_FIRST_VALUE_IMPERIAL;
			sensor_data->rimperial[0] = (sensor_data->result[0] * 62) / 100;
			return;
		case 0x30:
			sensor_data->result[0] =  obdchansim_irand(0, 255);
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			sensor_data->valid |= OBD_HAS_FIRST_VALUE_IMPERIAL;
			sensor_data->rimperial[0] = (sensor_data->result[0] * 9) / 100;
			return;
		case 0x0C:
			sensor_data->result[0] =  obdchansim_drand(0, 16383);
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			return;
		case 0x0E:
			sensor_data->result[0] =  obdchansim_irand(-64, 64);
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			return;
		case 0x10:
			sensor_data->result[0] =  obdchansim_irand(0, 655);
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			sensor_data->valid |= OBD_HAS_FIRST_VALUE_IMPERIAL;
			sensor_data->rimperial[0] = (sensor_data->result[0] * 132) / 1000;
			return;
		case 0x12:
			sensor_data->result[0] =  1<<1;
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			return;
		case 0x13:
		case 0x1D:
			sensor_data->result[0] =  0xFF;
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			return;
		case 0x14:
		case 0x15:
		case 0x16:
		case 0x17:
		case 0x18:
		case 0x19:
		case 0x1A:
		case 0x1B:
			sensor_data->result[0] =  obdchansim_drand(0, 1.275);
			sensor_data->result[1] =  obdchansim_drand(0, 99.2);
			sensor_data->valid |= OBD_HAS_FIRST_VALUE | OBD_HAS_SECOND_VALUE;
			return;
		case 0x1C:
		case 0x1E:
			sensor_data->result[0] =  1;
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			return;
		case 0x1F:
			sensor_data->result[0] =  (time(0) - sim->start) % 65535;
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			return;

		case 0x21:
		case 0x31:
		case 0x4D:
		case 0x4E:
			sensor_data->result[0] =  obdchansim_irand(0, 65535);
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			return;

		case 0x22:
			sensor_data->result[0] =  obdchansim_drand(0, 5177.265);
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			sensor_data->valid |= OBD_HAS_FIRST_VALUE_IMPERIAL;
			sensor_data->rimperial[0] = (sensor_data->result[0] * 9) / 100;
			return;

		case 0x23:
			sensor_data->result[0] =  obdchansim_irand(0, 655350);
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			sensor_data->valid |= OBD_HAS_FIRST_VALUE_IMPERIAL;
			sensor_data->rimperial[0] = (sensor_data->result[0] * 9) / 100;
			return;

		case 0x24:
		case 0x25:
		case 0x26:
		case 0x27:
		case 0x28:
		case 0x29:
		case 0x2A:
		case 0x2B:
			sensor_data->result[0] =  obdchansim_drand(0, 2);
			sensor_data->result[1] =  obdchansim_drand(0, 8);
			sensor_data->valid |= OBD_HAS_FIRST_VALUE | OBD_HAS_SECOND_VALUE;
			return;

		case 0x2D:
			sensor_data->result[0] =  obdchansim_drand(-100, 99.22);
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			return;

		case 0x32:
			sensor_data->result[0] =  obdchansim_drand(-8.192, 8.192);
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			sensor_data->valid |= OBD_HAS_FIRST_VALUE_IMPERIAL;
			sensor_data->rimperial[0] = (sensor_data->result[0] * 29) / 100000;
			return;

		case 0x34:
		case 0x35:
		case 0x36:
		case 0x37:
		case 0x38:
		case 0x39:
		case 0x3A:
		case 0x3B:
			sensor_data->result[0] =  obdchansim_drand(0, 2);
			sensor_data->result[1] =  obdchansim_irand(-128, 128);
			sensor_data->valid |= OBD_HAS_FIRST_VALUE | OBD_HAS_SECOND_VALUE;
			sensor_data->valid |= OBD_HAS_FIRST_VALUE_IMPERIAL;
			sensor_data->rimperial[0] = (sensor_data->result[0] * 29) / 100000;
			return;

		case 0x3C:
		case 0x3D:
		case 0x3E:
		case 0x3F:
			sensor_data->result[0] =  obdchansim_drand(40, 6513.5);
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			sensor_data->valid |= OBD_HAS_FIRST_VALUE_IMPERIAL;
			sensor_data->rimperial[0] = ((sensor_data->result[0] * 9) / 5.0) - 32;
			return;

		case 0x42:
			sensor_data->result[0] =  obdchansim_drand(0, 65.535);
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			return;

		case 0x43:
			sensor_data->result[0] =  obdchansim_irand(0, 25696);
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			return;

		case 0x44:
			sensor_data->result[0] =  obdchansim_drand(0, 2);
			sensor_data->valid |= OBD_HAS_FIRST_VALUE;
			return;


		default:
			sensor_data->valid |= OBD_INVALID_DATA;
			return;
	}
}

static int obdchansim_sendpid(struct obdchan* chan, struct obd_sensor_data* sensor_data, unsigned int timeout)
{
	int poll_ret;
	int delay;
	struct obdchansim* sim = NULL;

	if(chan->type != OBDCHAN_SIMULATOR) {
		LOG_ERROR("channel is not a simulator (did you call chan_setup?)");
		return OBDCHANERR_NOT_SETUP;
	}

	sim = chan->data;


	obdchansim_pidvalue(chan, sensor_data);

	delay = obdchansim_irand(MIN_OBD_DELAY, MAX_OBD_DELAY);
	poll_ret = poll(&sim->poll, 1, delay);

	return 0;
}

static int obdchansim_senddtc(struct obdchan* chan, struct obd_dtc_data* dtc_data, unsigned int timeout)
{
	int poll_ret;
	int delay;
	struct obdchansim* sim = NULL;

	if(chan->type != OBDCHAN_SIMULATOR)
		return OBDCHANERR_NOT_SETUP;

	sim = chan->data;

	dtc_data->valid = 6;
	strncpy(dtc_data->code_list[0], "P0010", 6);
	strncpy(dtc_data->code_list[1], "P0130", 6);
	strncpy(dtc_data->code_list[2], "B0150", 6);
	strncpy(dtc_data->code_list[3], "C0030", 6);
	strncpy(dtc_data->code_list[4], "C0030", 6);
	strncpy(dtc_data->code_list[5], "U1090", 6);

	delay = obdchansim_irand(MIN_OBD_DELAY, MAX_OBD_DELAY);
	poll_ret = poll(&sim->poll, 1, delay);

	return 0;
}

static int obdchansim_close(struct obdchan* chan)
{
	struct obdchansim* sim = NULL;

	if(chan->type == OBDCHAN_UNDEFINED)
		return OBDCHANERR_NOT_SETUP;

	sim = chan->data;

	write(sim->closepipe[PIPE_WRITE], "c", 1);
	chan->type = OBDCHAN_UNDEFINED;

	close(sim->closepipe[PIPE_WRITE]);
	close(sim->closepipe[PIPE_READ]);

	chan->name = NULL;
	chan->sendpid = NULL;
	chan->senddtc = NULL;
	chan->close = NULL;

	free(chan->data);
	chan->data = NULL;

	return 0;
}

int obdchansim_setup(struct obdchan* chan)
{
	struct obdchansim* sim = NULL;
	struct pollfd* pol = NULL;
	int ret = 0;

	if(chan->type != OBDCHAN_UNDEFINED) {
		LOG_ERROR("Chan was already setup: %d", chan->type);
		return OBDCHANERR_ALREADY_SETUP;
	}

	chan->data = malloc(sizeof(struct obdchansim));

	if(chan->data == NULL) {
		LOG_ERROR("Could not allocate data for the simulator");
		return OBDCHANERR_ERROR;
	}

	sim = chan->data;
	sim->closepipe[PIPE_READ] = 0;
	sim->closepipe[PIPE_WRITE] = 0;

	pol = &sim->poll;

	if(pipe(sim->closepipe)){
		LOG_ERROR("Could not create the close pipe");
		ret = OBDCONERR_ERROR;
		goto error;
	}

	pol->fd = sim->closepipe[PIPE_READ];
	pol->events = POLLIN | POLLPRI;

	chan->type = OBDCHAN_SIMULATOR;
	chan->name = obdchansim_name;
	chan->sendpid = obdchansim_sendpid;
	chan->senddtc = obdchansim_senddtc;
	chan->close = obdchansim_close;

	srand((unsigned)(time(0)));
	sim->start = time(0);
	return 0;

error:
	if(sim) {
		if(sim->closepipe[PIPE_READ])
			close(sim->closepipe[PIPE_READ]);
		if(sim->closepipe[PIPE_WRITE])
			close(sim->closepipe[PIPE_WRITE]);

		free(chan->data);
		chan->data = NULL;
	}
	return OBDCHANERR_ERROR;
}

