/*
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of libobd.
 *
 *  libobd is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  libobd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with libobd.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <stdlib.h>
#include <time.h>

#include "obd-chan-ozen.h"
#include "obd-sensor.h"
#include "obd-log.h"

#define BUFFER_LEN 256
#define FRAME_LEN 32
#define READ_DEADLINE_FACTOR 4
#define NUM_CMDS sizeof(ozen_cmd_size)/sizeof(int)
#define MAX_RETRY 10

static const char* name = "Car-Pal";

static const int ozen_cmd_size[] = {0, 1, 1, 0, 0, 3, 3, 3 };

typedef enum {
	OBD_NOPROTOCOL = 0x0000,
	OBD_ISO9141_1 = 0x0001,
	OBD_ISO9141_2 = 0x0002,
	OBD_KWP2000_SLOW = 0x0004,
	OBD_KWP2000_FAST = 0x0008,
	OBD_J1850_PWM = 0x0010,
	OBD_J1850_VPWM = 0x0020,
	OBD_CAN11_250 = 0x0040,
	OBD_CAN11_500 = 0x0080,
	OBD_CAN29_250 = 0x0100,
	OBD_CAN29_500 = 0x0200,
	OBD_SAE_J1939 = 0x0400,
	OBD_KW1281 = 0x0800,
	OBD_KW71 = 0x0801,
	OBD_KW81 = 0x1000,
	OBD_KW82 = 0x1001,
	OBD_ALDL = 0x2000,
	OBD_OTHER = 0xFFFF
} obd_protocol;

struct obdchanozen {
	unsigned int blen;
	char buffer[BUFFER_LEN];
	unsigned int protocol;
};

static int obdchanozen_name(char* buffer, unsigned int len)
{
	strncpy(buffer, name, len - 1);
	buffer[len - 1] = '\0';

	return 0;
}

static inline unsigned int obdchanozen_calcprotocol(unsigned char p1, unsigned char p2)
{
	return (256 * p1) + p2;
}

static int obdchanozen_replypos(struct obdchan* chan)
{
	struct obdchanozen *pcar = chan->data;
	
	if(pcar->protocol == OBD_CAN29_250 || pcar->protocol == OBD_CAN29_500)
		return 7;

	return 5;
}

static int obdchanozen_readframe(struct obdchan* chan, char* buffer, int size, unsigned int timeout)
{
	int readed;
	int total_readed;
	time_t tread;
	int ret_cmd, ret_size;

	total_readed = readed = 0;

	if(size <= 0)
		return OBDCHANERR_ERROR;

	tread = time(0);

	ret_cmd = ret_size = -1;

	while ((ret_size < 0 || total_readed < ret_size + 2) &&
			time(0) - tread < READ_DEADLINE_FACTOR * timeout &&
			total_readed < size) {

		readed = obdcon_read(chan->con, buffer + total_readed,
				size - total_readed,
				timeout);
		if(readed < 0)
			return OBDCHANERR_ERROR;

		total_readed += readed;

		if(ret_cmd < 0 && total_readed > 1) {
			ret_cmd = (unsigned char) buffer[0];
			ret_size = (unsigned char) buffer[1];

			if(ret_cmd != '\x02') {
				LOG_ERROR("Sync byte is not correct!");
				return OBDCHANERR_ERROR;
			}
			if(ret_size == 0) {
				LOG_ERROR("Ozen device indicates a 0 length response!");
				return OBDCHANERR_ERROR;
			}
		}
	}

	if(total_readed != ret_size + 2) {
		LOG_ERROR("Ozen device data read timeout!");
		return OBDCHANERR_TIMEOUT;
	}

	return total_readed;
}

static int obdchanozen_sendinit(struct obdchan* chan, unsigned char cmd, unsigned int timeout)
{
	int readed;
	int total_readed;
	struct obdchanozen *pcar = chan->data;
	time_t tread;

	if(cmd > NUM_CMDS || !ozen_cmd_size[cmd]) {
		LOG_ERROR("Invalid Car-Pal command: %d/%d", cmd, NUM_CMDS);
		return OBDCHANERR_INVALID_ARGS;
	}

	pcar->buffer[0] = cmd;
	pcar->blen = 1;

	LOG_BINARY(OBDLOG_DEBUG, "Sending Ozen cmd ->", pcar->buffer, pcar->blen);

	if (obdcon_write(chan->con, pcar->buffer, pcar->blen) < 0) {
		LOG_ERROR("Unable to write to serial connection");
		return OBDCHANERR_ERROR;
	}

	total_readed = readed = 0;

	tread = time(0);

	while (total_readed < ozen_cmd_size[cmd] &&
			time(0) - tread < READ_DEADLINE_FACTOR * timeout &&
			total_readed < BUFFER_LEN) {

		readed = obdcon_read(chan->con, pcar->buffer + total_readed,
				BUFFER_LEN - total_readed,
				timeout);
		if(readed < 0)
			return OBDCHANERR_ERROR;

		total_readed += readed;
	}
	pcar->blen = total_readed;

	LOG_BINARY(OBDLOG_DEBUG, "Recived Ozen data ->", pcar->buffer, pcar->blen);

	if(pcar->blen !=  ozen_cmd_size[cmd])
		return OBDCHANERR_ERROR;

	return 0;
}

static int obdchanozen_sendcmd(struct obdchan* chan, unsigned int timeout)
{
	struct obdchanozen *pcar = chan->data;
	int ret;

	if(pcar->blen >= BUFFER_LEN) {
		LOG_ERROR("Trying to send more than the buffer size (%d)", BUFFER_LEN);
		return -1;
	}

	LOG_BINARY(OBDLOG_DEBUG, "Sending Ozen ->", pcar->buffer, pcar->blen);

	if (obdcon_write(chan->con, pcar->buffer, pcar->blen) < 0) {
		LOG_ERROR("Unable to write to serial connection");
		return OBDCHANERR_ERROR;
	}

	if((ret = obdchanozen_readframe(chan, pcar->buffer, BUFFER_LEN, timeout)) < 0) {
		pcar->blen = 0;
		LOG_ERROR("Unable to get an Ozen valid frame");
		return OBDCHANERR_ERROR;
	}

	pcar->blen = ret;

	return 0;
}

static int obdchanozen_sendpid(struct obdchan* chan, struct obd_sensor_data* sensor_data, unsigned int timeout)
{
	struct obdchanozen* pcar = chan->data;
	unsigned int retry = 0;
	unsigned int delta;

	if(pcar == NULL)
		return OBDCHANERR_NOT_SETUP;

	while(retry < MAX_RETRY) {
		if(obdchanozen_sendinit(chan, '\x02', timeout) < 0)
			return OBDCHANERR_TIMEOUT;

		if(pcar->buffer[0] == '\x06')
			break;

		LOG_ERROR("Ozen cmd: 02 reply: %02X", pcar->buffer[0]);
		retry++;
		continue;
	}

	if(retry == MAX_RETRY)
		return OBDCHANERR_ERROR;

	pcar->buffer[0] = '\x01';
	pcar->buffer[1] = (char) sensor_data->pid;
	pcar->blen = 2;

	if(obdchanozen_sendcmd(chan, timeout) < 0)
		return OBDCHANERR_ERROR;

	LOG_BINARY(OBDLOG_DEBUG, "Recived Ozen data ->", pcar->buffer, pcar->blen);

	delta = obdchanozen_replypos(chan);

	return obdsensor_process_obd((unsigned char*)pcar->buffer + delta, pcar->blen - delta, sensor_data);
}

/* FIXME: This method was tested for CAN-11 only. Probably will not work with
 * other protocols as the response structure is different. Anyhow, we need to
 * test on a car with other protocols (anyone wishing to contribute?).*/
static int obdchanozen_senddtc(struct obdchan* chan, struct obd_dtc_data* dtc_data, unsigned int timeout)
{
	struct obdchanozen* pcar = chan->data;
	unsigned int retry = 0;
	unsigned int delta;
	int obd_size;
	int obd_readed;
	int readed;

	if(pcar == NULL)
		return OBDCHANERR_NOT_SETUP;

	while(retry < MAX_RETRY) {
		if(obdchanozen_sendinit(chan, '\x02', timeout) < 0)
			return OBDCHANERR_TIMEOUT;

		if(pcar->buffer[0] == '\x06')
			break;

		LOG_ERROR("Ozen cmd 02 reply: %02X", pcar->buffer[0]);
		retry++;
		continue;
	}

	if(retry == MAX_RETRY)
		return OBDCHANERR_ERROR;

	pcar->buffer[0] = '\x03';
	pcar->blen = 1;

	if(obdchanozen_sendcmd(chan, timeout) < 0)
		return OBDCHANERR_ERROR;

	delta = obdchanozen_replypos(chan);

	if((unsigned char)pcar->buffer[delta] != 0x43 && (unsigned char)pcar->buffer[delta] < 0x70) { /*multiple responses */
		obd_size = pcar->buffer[delta];
		obd_readed = pcar->blen - delta - 1;

		memmove(pcar->buffer + delta, pcar->buffer + delta + 1, pcar->blen - delta - 1);
		pcar->blen--;

		while (obd_readed < obd_size && pcar->blen < BUFFER_LEN) {
			readed = obdchanozen_readframe(chan, pcar->buffer + pcar->blen, BUFFER_LEN - pcar->blen, timeout);
			if(readed < 0) {
				LOG_ERROR("Timeout or serial error reading the next frame");
				return readed;
			}
			memmove(pcar->buffer + pcar->blen, pcar->buffer + pcar->blen + delta, readed - delta);

			if(obd_readed + readed - delta < obd_size)
				pcar->blen += readed - delta;
			else
				pcar->blen += obd_size - obd_readed;

			obd_readed += readed - delta;
		}

	} else if((unsigned char)pcar->buffer[delta] != 0x43 && (unsigned char)pcar->buffer[delta] >= 0x70) {
		dtc_data->valid = -1;
		return 0;
	}

	LOG_BINARY(OBDLOG_DEBUG, "Recived Ozen data ->", pcar->buffer, pcar->blen);

	return obdsensor_process_dtc((unsigned char*) (pcar->buffer + delta), pcar->blen - delta, dtc_data);
}

static int obdchanozen_close(struct obdchan* chan)
{
	if(chan->type == OBDCHAN_UNDEFINED)
		return OBDCHANERR_NOT_SETUP;

	if(chan->data)
		free(chan->data);

	chan->data = NULL;
	chan->name = NULL;
	chan->sendpid = NULL;
	chan->senddtc = NULL;
	chan->close = NULL;

	return 0;
}

int obdchanozen_setup(struct obdchan* chan, struct obdcon* con)
{
	struct obdchanozen* pcar = NULL;
	int ret = 0;
	unsigned int ident;

	if(chan->type != OBDCHAN_UNDEFINED) {
		LOG_ERROR("Channel was already setup!");
		return OBDCHANERR_ALREADY_SETUP;
	}
	
	if(con == NULL) {
		LOG_ERROR("(BUG!!) Connection pointer is NULL!");
		return OBDCHANERR_INVALID_ARGS;
	}

	chan->con = con;

	chan->data = malloc(sizeof(struct obdchanozen));

	if(chan->data == NULL) {
		LOG_ERROR("Unable to alloc memory for ELM data!");
		ret =  OBDCHANERR_ERROR;
		goto error;
	}

	pcar = chan->data;

	if(obdchanozen_sendinit(chan, '\x06', 10) < 0) { /* Ozen chip id */
		LOG_INFO("Unable to send the command 06 to Ozen");
		ret = OBDCHANERR_ERROR;
		goto error;
	}

	if(pcar->buffer[0] != '\x06') {
		LOG_INFO("Unable to send the command 06 to Ozen");
		ret = OBDCHANERR_ERROR;
		goto error;
	}

	ident = pcar->buffer[1] << 8 | pcar->buffer[2];
	if(!(ident == 0x0A28 || ident == 0x0E10 || ident == 0x2600 ||
				ident == 0x2700 || ident == 0x2800)) {
		LOG_INFO("Ozen idenfitication failed");
		ret = OBDCHANERR_ERROR;
		goto error;
	}

	if(obdchanozen_sendinit(chan, '\x05', 2) < 0) { /* Protocol identification */
		LOG_INFO("Unable to send the command 05 to Ozen");
		ret = OBDCHANERR_ERROR;
		goto error;
	}


	if(pcar->buffer[0] != '\x05') {
		LOG_ERROR("Ozen reply for command 05 is not ok!");
		ret = OBDCHANERR_ERROR;
		goto error;
	}

	pcar->protocol = obdchanozen_calcprotocol((unsigned char)pcar->buffer[1],
					(unsigned char)pcar->buffer[2]);

	if(pcar->protocol == OBD_NOPROTOCOL) {

		if(obdchanozen_sendinit(chan, '\x07', 40) < 0) { /* Ozen chip id */
			LOG_INFO("Unable to send the command 07 to Ozen");
			ret = OBDCHANERR_ERROR;
			goto error;
		}

		if(pcar->buffer[0] == '\x07' && pcar->buffer[1] == '\x00' && 
			pcar->buffer[2] == '\x00') {
			LOG_INFO("Ozen is not able to connect to ECU");
			ret =  OBDCHANERR_ERROR;
			goto error;
		}
	}

	chan->type = OBDCHAN_OZEN;
	chan->name = obdchanozen_name;
	chan->sendpid = obdchanozen_sendpid;
	chan->senddtc = obdchanozen_senddtc;
	chan->close = obdchanozen_close;

	return 0;

error:
	if (pcar) {
		free(pcar);
		chan->data = NULL;
	}

	chan->con = NULL;
	return ret;
}
