/*
 * Copyright (C) 2008 Andrew Olmsted, all rights reserved.
 *
 * Contact: Andrew Olmsted <andrew.olmsted@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <glib-object.h>
#include <stdlib.h>
#include <libosso.h>

#include <hildon/hildon.h>
#include <libhildondesktop/libhildondesktop.h>
#include <stdio.h>
#include <string.h>
#include <alarmd/alarm_event.h>

#include "large-statusbar-clock-minutes.h"
#include "large-statusbar-clock-alarm.h"

#define HOME_DIR g_get_home_dir()
#define ICON_FILE "/usr/share/pixmaps/large-statusbar-clock-alarm.png"

/*HD_DEFINE_PLUGIN (LargeStatusbarClockMinutesPlugin, large_statusbar_clock_minutes_plugin, STATUSBAR_TYPE_ITEM);*/

/* Hildon Status Bar plugin API prototypes */
void *large_statusbar_clock_minutes_initialize(HildonStatusBarItem *item, GtkWidget **button);
void large_statusbar_clock_minutes_entry(HildonStatusBarPluginFn_st *fn);
void large_statusbar_clock_minutes_destroy(void *data);
gint large_statusbar_clock_minutes_get_priority(void *data);
void large_statusbar_clock_minutes_update(void *data, gint value1, gint value2, const gchar *str);

void minutes_button_clicked(GtkButton *button, gpointer data);
gboolean update_minutes(gpointer data);
void launch_clock(GtkMenuItem *item, gpointer data);
void configure_clock(GtkMenuItem *item, gpointer data);
void popup_place(GtkMenu *menu, gint *x, gint *y, gboolean *push_in, gpointer user_data);
gint dbus_req_handler(const gchar* interface, const gchar* method, GArray* arguments, gpointer data, osso_rpc_t* retval);

void *large_statusbar_clock_minutes_initialize(HildonStatusBarItem *item, GtkWidget **button)
{
      gchar* filename;
      gboolean fileExists;
      GKeyFile* keyFile;
      gchar* fileData;
      FILE* rcFile;
      gsize size;
      osso_return_t result;
      GError* error=NULL;
      LargeStatusbarClockMinutesPlugin *statusbar_plugin = g_new0(LargeStatusbarClockMinutesPlugin, 1);

      statusbar_plugin->sbitem=item;
      statusbar_plugin->minutesLabel=gtk_label_new("");
      statusbar_plugin->button=gtk_button_new();
      /*gtk_misc_set_padding(GTK_MISC(GTK_BIN(statusbar_plugin->button)->child),0,0);*/

      gtk_container_add(GTK_CONTAINER(statusbar_plugin->button),statusbar_plugin->minutesLabel);
      /*gtk_container_add(GTK_CONTAINER(statusbar_plugin), statusbar_plugin->button);*/
      gtk_misc_set_alignment(GTK_MISC(statusbar_plugin->minutesLabel),0.0,0.5);
      /*gtk_widget_show_all(GTK_WIDGET(statusbar_plugin));*/
      *button=statusbar_plugin->button;

      /* Get whether the user is using 24 hour or AM/PM format */
      keyFile=g_key_file_new();
      filename=g_strdup(HOME_DIR);
      filename=g_strconcat(filename,"/.statusbar_clock.rc",NULL);
      fileExists=g_key_file_load_from_file(keyFile,filename,G_KEY_FILE_KEEP_COMMENTS,NULL);
      if (!fileExists)
      {
	  g_key_file_set_comment(keyFile,NULL,NULL,"Configuration file for large statusbar clock",NULL);
	  g_key_file_set_boolean(keyFile,"initialize","stacked",FALSE);
	  g_key_file_set_boolean(keyFile,"initialize","ampm",TRUE);
	  g_key_file_set_string(keyFile,"initialize","font","Sans 18");
	  g_key_file_set_string(keyFile,"initialize","font colour","FFFFFFFFFFFF");
	  g_key_file_set_string(keyFile,"alarm","alarm 1 colour","FFFF00000000");
	  g_key_file_set_integer(keyFile,"alarm","alarm 1 time",24);
	  g_key_file_set_string(keyFile,"alarm","alarm 2 colour","0000FFFF0000");
	  g_key_file_set_integer(keyFile,"alarm","alarm 2 time",168);
	  g_key_file_set_boolean(keyFile,"alarm","alarms on",FALSE);
      }

      statusbar_plugin->ampm=g_key_file_get_boolean(keyFile,"initialize","ampm",&error);
      if (error)
      {
	  statusbar_plugin->ampm=TRUE;
	  g_key_file_set_boolean(keyFile,"initialize","ampm",TRUE);
	  g_error_free(error);
      }
      statusbar_plugin->stacked=g_key_file_get_boolean(keyFile,"initialize","stacked",&error);
      if (error)
      {
	  statusbar_plugin->stacked=FALSE;
	  g_key_file_set_boolean(keyFile,"initialize","stacked",FALSE);
	  g_error_free(error);
      }
      statusbar_plugin->fontDesc=g_key_file_get_string(keyFile,"initialize","font",&error);
      if (error)
      {
	  statusbar_plugin->fontDesc="Sans 18";
	  g_key_file_set_string(keyFile,"initialize","font","Sans 18");
	  g_error_free(error);
      }
      statusbar_plugin->fontColour=g_key_file_get_string(keyFile,"initialize","font colour",&error);
      if (error)
      {
	  statusbar_plugin->fontColour="FFFFFFFFFFFF";
	  g_key_file_set_string(keyFile,"initialize","font colour","FFFFFFFFFFFF");
	  g_error_free(error);
      }
      statusbar_plugin->alarm1Colour=g_key_file_get_string(keyFile,"alarm","alarm 1 colour",&error);
      if (error)
      {
	  statusbar_plugin->alarm1Colour="FFFF00000000";
	  g_key_file_set_string(keyFile,"alarm","alarm 1 colour","FFFF00000000");
	  g_error_free(error);
      }
      statusbar_plugin->alarm1Time=g_key_file_get_integer(keyFile,"alarm","alarm 1 time",&error);
      if (error)
      {
	  statusbar_plugin->alarm1Time=24;
	  g_key_file_set_integer(keyFile,"alarm","alarm 1 time",24);
	  g_error_free(error);
      }
      statusbar_plugin->alarm2Colour=g_key_file_get_string(keyFile,"alarm","alarm 2 colour",&error);
      if (error)
      {
	  statusbar_plugin->alarm2Colour="0000FFFF0000";
	  g_key_file_set_string(keyFile,"alarm","alarm 2 colour","0000FFFF0000");
	  g_error_free(error);
      }
      statusbar_plugin->alarm2Time=g_key_file_get_integer(keyFile,"alarm","alarm 2 time",&error);
      if (error)
      {
	  statusbar_plugin->alarm2Time=168;
	  g_key_file_set_integer(keyFile,"alarm","alarm 2 time",168);
	  g_error_free(error);
      }
      statusbar_plugin->alarms=g_key_file_get_boolean(keyFile,"alarm","alarms on",&error);
      if (error)
      {
      	statusbar_plugin->alarms=FALSE;
      	g_key_file_set_boolean(keyFile,"alarm","alarms on",FALSE);
      	g_error_free(error);
      }
      statusbar_plugin->updated=TRUE;

      fileData=g_key_file_to_data(keyFile,&size,NULL);
      rcFile=fopen(filename,"w");
      fputs(fileData,rcFile);
      fclose(rcFile);
      g_key_file_free(keyFile);
      g_free(fileData);

      statusbar_plugin->osso=osso_initialize("large_statusbar_clock_minutes","0.7",TRUE,NULL);
      result=osso_rpc_set_default_cb_f(statusbar_plugin->osso,dbus_req_handler,statusbar_plugin);

      statusbar_plugin->updated=TRUE;

      statusbar_plugin->eventID=g_timeout_add(1000,update_minutes,statusbar_plugin);

      g_signal_connect(statusbar_plugin->button,"clicked",G_CALLBACK(minutes_button_clicked),statusbar_plugin);
      g_free(filename);

      return statusbar_plugin;
}

void large_statusbar_clock_minutes_entry(HildonStatusBarPluginFn_st *fn)
{
    fn->initialize   = large_statusbar_clock_minutes_initialize;
    fn->update       = large_statusbar_clock_minutes_update;
    fn->destroy      = large_statusbar_clock_minutes_destroy;
    fn->get_priority = large_statusbar_clock_minutes_get_priority;
}

/*void Large_statusbar_clock_minutesApplet_entry(HildonStatusBarPluginFn_st *fn)
{
	large_statusbar_clock_minutes_entry(fn);
}*/

void large_statusbar_clock_minutes_destroy(void *data)
{
      LargeStatusbarClockMinutesPlugin *statusbar_plugin=(LargeStatusbarClockMinutesPlugin*)data;
      //osso_return_t result;
      
      g_source_remove(statusbar_plugin->eventID);
      //result=osso_statusbar_send_event(statusbar_plugin->osso,"statusbar_clock_hours",0,0,"",NULL);
      g_free(statusbar_plugin);
}

void large_statusbar_clock_minutes_update(void *data, gint value1, gint value2, const gchar *str)
{
    LargeStatusbarClockMinutesPlugin *statusbar_plugin=(LargeStatusbarClockMinutesPlugin*)data;
    if (value1 == 0)
		gtk_widget_destroy(GTK_WIDGET(statusbar_plugin->sbitem));

}

gint large_statusbar_clock_minutes_get_priority(void *data)
{
    return 42;
}

gboolean update_minutes(gpointer data)
{
  LargeStatusbarClockMinutesPlugin *statusbar_plugin=(LargeStatusbarClockMinutesPlugin*)data;
  
  const time_t curTime=time(NULL);
  struct tm *locTime;
  gchar timeString[45];
  osso_return_t ret;
  
  locTime=localtime(&curTime);
  if ((locTime->tm_min!=statusbar_plugin->previous)||(statusbar_plugin->updated))
  {
  	cookie_t *list,*iter;
  	
    statusbar_plugin->curColour=statusbar_plugin->fontColour;
    if (statusbar_plugin->alarms)
    {
		list = alarm_event_query(curTime, curTime + statusbar_plugin->alarm2Time*60*60, 0, 0);
g_debug("alarm_event_query has run");
		if (list == NULL)
		{
			g_debug("alarm_event_query failed");
		}
		else if (list[0] == (cookie_t) 0)
		{
			free(list);
		}
		else
		{
			for (iter = list; *iter != (cookie_t) 0; iter++) {
				alarm_event_t* alarm=alarm_event_get(*iter);
				if (g_strcasecmp(alarm->icon,ICON_FILE)==0)
				{
				  statusbar_plugin->curColour=statusbar_plugin->alarm2Colour;
				  break;
				}
				alarm_event_free(alarm);
			}
			free(list);
		}
		
		list = alarm_event_query(curTime, curTime + statusbar_plugin->alarm1Time*60*60, 0, 0);
		if (list == NULL)
		{
			g_debug("alarm_event_query failed");
		}
		else if (list[0] == (cookie_t) 0)
		{
			free(list);
		}
		else
		{
			for (iter = list; *iter != (cookie_t) 0; iter++) {
				alarm_event_t* alarm=alarm_event_get(*iter);
				if (g_strcasecmp(alarm->icon,ICON_FILE)==0)
				{
				  statusbar_plugin->curColour=statusbar_plugin->alarm1Colour;
				  break;
				}
				alarm_event_free(alarm);
			}
			free(list);
		}
    }

    ret = osso_rpc_async_run(statusbar_plugin->osso,
			      "com.nokia.large_statusbar_clock_hours",
			      "/com/nokia/large_statusbar_clock_hours",
			      "com.nokia.large_statusbar_clock_hours",
			      "update", NULL, NULL,
			      DBUS_TYPE_STRING,statusbar_plugin->fontDesc,
			      DBUS_TYPE_STRING,statusbar_plugin->curColour,
			      DBUS_TYPE_BOOLEAN,statusbar_plugin->ampm,
			      DBUS_TYPE_BOOLEAN,statusbar_plugin->stacked,
			      DBUS_TYPE_INVALID);
			      
	if (statusbar_plugin->ampm)
	{
	  if (statusbar_plugin->stacked)
	  {
		if (locTime->tm_hour<12)
		{
		  strftime(timeString,45,"<span size='small'>%M\nAM</span>",locTime);
		}
		else
		{
		  strftime(timeString,45,"<span size='small'>%M\nPM</span>",locTime);
		}
	  }
	  else
	  {
		if (locTime->tm_hour<12)
		{
		  strftime(timeString,45,"%M<span size='xx-small'>A</span>",locTime);
		}
		else
		{
		  strftime(timeString,45,"%M<span size='xx-small'>P</span>",locTime);
		}
	  }
	}
	else
	{
	  strftime(timeString,45,"%M",locTime);
	}

    gtk_label_set_markup(GTK_LABEL(statusbar_plugin->minutesLabel),g_strconcat("<span foreground='#",statusbar_plugin->curColour,"' font-desc='",statusbar_plugin->fontDesc,"'>",timeString,"</span>",NULL));

  }
  statusbar_plugin->previous=locTime->tm_min;
  statusbar_plugin->updated=FALSE;
  return TRUE;
}

void minutes_button_clicked(GtkButton *button, gpointer data)
{
  GtkWidget *menu_item;
  time_t curTime;
  struct tm *locTime;
  gchar timeString[45];
  LargeStatusbarClockMinutesPlugin *statusbar_plugin=(LargeStatusbarClockMinutesPlugin*)data;
  
  curTime=time(NULL);
  locTime=localtime(&curTime);
  strftime(timeString,45,"%A, %B %e, %Y",locTime);

  statusbar_plugin->minutesMenu = gtk_menu_new();
  menu_item = gtk_menu_item_new_with_label(timeString);
  /*gtk_widget_set_sensitive(menu_item,FALSE);*/
  gtk_widget_show_all(menu_item);
  g_signal_connect(menu_item, "activate", G_CALLBACK(launch_clock),statusbar_plugin);
  gtk_menu_shell_append(GTK_MENU_SHELL(statusbar_plugin->minutesMenu), menu_item);

  menu_item = gtk_separator_menu_item_new();
  gtk_widget_show_all(menu_item);
  gtk_menu_shell_append(GTK_MENU_SHELL(statusbar_plugin->minutesMenu), menu_item);

  if (statusbar_plugin->alarms)
  {
	  menu_item = gtk_menu_item_new_with_label("Alarms");
	  g_signal_connect(menu_item, "activate", G_CALLBACK(alarm_dialog),statusbar_plugin);
	  gtk_widget_show_all(menu_item);
	  gtk_menu_shell_append(GTK_MENU_SHELL(statusbar_plugin->minutesMenu), menu_item);
  }

  menu_item = gtk_menu_item_new_with_label("Configure");
  g_signal_connect(menu_item, "activate", G_CALLBACK(configure_clock),statusbar_plugin);
  gtk_widget_show_all(menu_item);
  gtk_menu_shell_append(GTK_MENU_SHELL(statusbar_plugin->minutesMenu), menu_item);

  gtk_menu_popup(GTK_MENU(statusbar_plugin->minutesMenu), NULL, NULL, popup_place, statusbar_plugin, 1,
		  gtk_get_current_event_time());

  gtk_widget_show_all(statusbar_plugin->minutesMenu);
}

void launch_clock(GtkMenuItem *item, gpointer data)
{
  osso_context_t *context;
  osso_return_t ret;

  context = osso_initialize("toptest", "0.1", 0,NULL);
  if (context == NULL)
  {
    printf("could not connect to D-Bus\n");
  }
  ret = osso_rpc_async_run(context,
    "com.nokia.osso_worldclock",
    "/com/nokia/osso_worldclock",
    "com.nokia.osso_worldclock",
    "top_application", NULL, NULL,
    DBUS_TYPE_STRING,
    "this is the top_application parameter",
    DBUS_TYPE_INVALID);

  if (ret != OSSO_OK)
  {
    printf("ERROR!\n");
  }

  osso_deinitialize(context);
}

void toggle_alarms(GtkToggleButton *button, GtkVBox *box)
{
	if (gtk_toggle_button_get_active(button))
	{
		gtk_widget_show_all(GTK_WIDGET(box));
	}
	else
	{
		gtk_widget_hide_all(GTK_WIDGET(box));
	}
	gtk_container_check_resize(GTK_CONTAINER(box));
}

void configure_clock(GtkMenuItem *item, gpointer data)
{
  LargeStatusbarClockMinutesPlugin *statusbar_plugin=(LargeStatusbarClockMinutesPlugin*)data;
  GtkWidget *dialog;
  gint response;
  GtkWidget *vbox, *alarmBox;
  GtkWidget *captionWidget;
  GtkWidget *fontButton;
  GtkWidget *alarmCheck;
  GtkWidget *colourCombo,*alarm1Colour,*alarm2Colour;
  GtkWidget *alarm1Time,*alarm2Time;
  GtkWidget *ampm;
  GtkWidget *stacked;
  gint time;
  GKeyFile *keyFile;
  gchar* filename;
  gboolean fileExists;
  gchar* colourString;
  gchar* fontString;
  GError* error=NULL;
  GdkColor colour;

  dialog=gtk_dialog_new_with_buttons("Configure Large Statusbar Clock",NULL,GTK_DIALOG_MODAL,GTK_STOCK_CANCEL,GTK_RESPONSE_REJECT,GTK_STOCK_OK,GTK_RESPONSE_ACCEPT,NULL);

  keyFile=g_key_file_new();
  filename=g_strdup(HOME_DIR);
  filename=g_strconcat(filename,"/.statusbar_clock.rc",NULL);
  fileExists=g_key_file_load_from_file(keyFile,filename,G_KEY_FILE_KEEP_COMMENTS,NULL);

  vbox=gtk_vbox_new(FALSE,0);

  fontString=g_key_file_get_string(keyFile,"initialize","font",NULL);
  if (!fontString)
  {
    fontButton=gtk_font_button_new();
  }
  else
  {
    fontButton=gtk_font_button_new_with_font(fontString);
  }
  captionWidget=hildon_caption_new(NULL,"Choose font",fontButton,NULL,HILDON_CAPTION_MANDATORY);
  gtk_box_pack_start(GTK_BOX(vbox),captionWidget,FALSE,FALSE,0);

  colourString=g_key_file_get_string(keyFile,"initialize","font colour",&error);
  if (error)
  {
    colourString="FFFFFFFFFFFF";
    g_key_file_set_string(keyFile,"initialize","font colour","FFFFFFFFFFFF");
    g_error_free(error);
  }
	colourCombo=gtk_color_button_new();
	captionWidget=hildon_caption_new(NULL,"Choose font colour",colourCombo,NULL,HILDON_CAPTION_MANDATORY);
	gtk_box_pack_start(GTK_BOX(vbox),captionWidget,FALSE,FALSE,0);
	char s[5]="";
	guint16 x;
	strncat(s,colourString,strlen(colourString)/3);
	x=strtol(s,NULL,16);
	colour.red=x;
	s[0]=NULL;
	strncat(s,colourString+strlen(colourString)/3,strlen(colourString)/3);
	x=strtol(s,NULL,16);
	colour.green=x;
	s[0]=NULL;
	strncat(s,colourString+(strlen(colourString)/3)*2,strlen(colourString)/3);
	x=strtol(s,NULL,16);
	colour.blue=x;
	gtk_color_button_set_color(colourCombo,&colour);

  ampm=gtk_check_button_new_with_label("Display time in AM/PM mode");
  if (g_key_file_get_boolean(keyFile,"initialize","ampm",NULL))
  {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ampm),TRUE);
  }
  gtk_box_pack_start(GTK_BOX(vbox),ampm,FALSE,FALSE,0);

  stacked=gtk_check_button_new_with_label("Use \"stacked\" mode (minutes display over AM/PM)");
  if (g_key_file_get_boolean(keyFile,"initialize","stacked",NULL))
  {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(stacked),TRUE);
  }
  gtk_box_pack_start(GTK_BOX(vbox),stacked,FALSE,FALSE,0);

  captionWidget=gtk_label_new("Alarm Settings");
  gtk_label_set_markup(GTK_LABEL(captionWidget),"<b>Alarm Settings</b>");
  gtk_box_pack_start(GTK_BOX(vbox),captionWidget,FALSE,FALSE,0);

  alarmBox=gtk_vbox_new(FALSE,0);

  alarmCheck=gtk_check_button_new();
  gtk_toggle_button_set_active(alarmCheck,g_key_file_get_boolean(keyFile,"alarm","alarms on",NULL));
  captionWidget=hildon_caption_new(NULL,"Enable alarms",alarmCheck,NULL,HILDON_CAPTION_MANDATORY);
  gtk_box_pack_start(GTK_BOX(vbox),captionWidget,FALSE,FALSE,0);
  g_signal_connect(alarmCheck,"toggled",G_CALLBACK(toggle_alarms),alarmBox);
  
  alarm1Time=hildon_number_editor_new(0,672);
  time=g_key_file_get_integer(keyFile,"alarm","alarm 1 time",&error);
  hildon_number_editor_set_value(HILDON_NUMBER_EDITOR(alarm1Time),time);
  captionWidget=hildon_caption_new(NULL,"Display first level alarms within (hours)",alarm1Time,NULL,HILDON_CAPTION_MANDATORY);
  gtk_box_pack_start(GTK_BOX(alarmBox),captionWidget,FALSE,FALSE,0);

  colourString=g_key_file_get_string(keyFile,"alarm","alarm 1 colour",&error);
  if (error)
  {
    colourString="FFFF00000000";
    g_key_file_set_string(keyFile,"alarm","alarm 1 colour","FFFF00000000");
    g_error_free(error);
  }
	alarm1Colour=gtk_color_button_new();
	captionWidget=hildon_caption_new(NULL,"Choose font colour for first level alarms",alarm1Colour,NULL,HILDON_CAPTION_MANDATORY);
	gtk_box_pack_start(GTK_BOX(alarmBox),captionWidget,FALSE,FALSE,0);
	s[0]=NULL;
	strncat(s,colourString,strlen(colourString)/3);
	x=strtol(s,NULL,16);
	colour.red=x;
	s[0]=NULL;
	strncat(s,colourString+strlen(colourString)/3,strlen(colourString)/3);
	x=strtol(s,NULL,16);
	colour.green=x;
	s[0]=NULL;
	strncat(s,colourString+(strlen(colourString)/3)*2,strlen(colourString)/3);
	x=strtol(s,NULL,16);
	colour.blue=x;
	gtk_color_button_set_color(alarm1Colour,&colour);

  alarm2Time=hildon_number_editor_new(0,672);
  time=g_key_file_get_integer(keyFile,"alarm","alarm 2 time",&error);
  hildon_number_editor_set_value(HILDON_NUMBER_EDITOR(alarm2Time),time);
  captionWidget=hildon_caption_new(NULL,"Display second level alarms within (hours)",alarm2Time,NULL,HILDON_CAPTION_MANDATORY);
  gtk_box_pack_start(GTK_BOX(alarmBox),captionWidget,FALSE,FALSE,0);

  colourString=g_key_file_get_string(keyFile,"alarm","alarm 2 colour",&error);
  if (error)
  {
    colourString="0000FFFF0000";
    g_key_file_set_string(keyFile,"alarm","alarm 1 colour","0000FFFF0000");
    g_error_free(error);
  }
	alarm2Colour=gtk_color_button_new();
	captionWidget=hildon_caption_new(NULL,"Choose font colour for second level alarms",alarm2Colour,NULL,HILDON_CAPTION_MANDATORY);
	gtk_box_pack_start(GTK_BOX(alarmBox),captionWidget,FALSE,FALSE,0);
	s[0]=NULL;
	strncat(s,colourString,strlen(colourString)/3);
	x=strtol(s,NULL,16);
	colour.red=x;
	s[0]=NULL;
	strncat(s,colourString+strlen(colourString)/3,strlen(colourString)/3);
	x=strtol(s,NULL,16);
	colour.green=x;
	s[0]=NULL;
	strncat(s,colourString+(strlen(colourString)/3)*2,strlen(colourString)/3);
	x=strtol(s,NULL,16);
	colour.blue=x;
	gtk_color_button_set_color(alarm2Colour,&colour);

  gtk_box_pack_start(GTK_BOX(vbox),alarmBox,FALSE,FALSE,0);

  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox),vbox);
  gtk_widget_show_all(dialog);

  if (!gtk_toggle_button_get_active(alarmCheck))
  {
  	gtk_widget_hide_all(alarmBox);
  }

  response=gtk_dialog_run(GTK_DIALOG(dialog));

  if (response==GTK_RESPONSE_ACCEPT)
  {
    gchar* fileData;
    FILE* rcFile;
    gsize size;
    GdkColor colour;
   
    statusbar_plugin->fontDesc=g_strdup(gtk_font_button_get_font_name(GTK_FONT_BUTTON(fontButton)));

    gtk_color_button_get_color(GTK_COLOR_BUTTON(colourCombo),&colour);
	statusbar_plugin->fontColour=g_strdup_printf("%04X%04X%04X",colour.red,colour.green,colour.blue);
	
    statusbar_plugin->ampm=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(ampm));
    statusbar_plugin->stacked=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(stacked));
    gtk_color_button_get_color(GTK_COLOR_BUTTON(alarm1Colour),&colour);
	statusbar_plugin->alarm1Colour=g_strdup_printf("%04X%04X%04X",colour.red,colour.green,colour.blue);
    statusbar_plugin->alarm1Time=hildon_number_editor_get_value(HILDON_NUMBER_EDITOR(alarm1Time));
    gtk_color_button_get_color(GTK_COLOR_BUTTON(alarm2Colour),&colour);
	statusbar_plugin->alarm2Colour=g_strdup_printf("%04X%04X%04X",colour.red,colour.green,colour.blue);
    statusbar_plugin->alarm2Time=hildon_number_editor_get_value(HILDON_NUMBER_EDITOR(alarm2Time));
    statusbar_plugin->alarms=gtk_toggle_button_get_active(alarmCheck);
    statusbar_plugin->updated=TRUE;
    
    
    g_key_file_set_string(keyFile,"initialize","font",statusbar_plugin->fontDesc);
    g_key_file_set_string(keyFile,"initialize","font colour",statusbar_plugin->fontColour);
    g_key_file_set_boolean(keyFile,"initialize","ampm",statusbar_plugin->ampm);
    g_key_file_set_boolean(keyFile,"initialize","stacked",statusbar_plugin->stacked);
    g_key_file_set_string(keyFile,"alarm","alarm 1 colour",statusbar_plugin->alarm1Colour);
    g_key_file_set_integer(keyFile,"alarm","alarm 1 time",statusbar_plugin->alarm1Time);
    g_key_file_set_string(keyFile,"alarm","alarm 2 colour",statusbar_plugin->alarm2Colour);
    g_key_file_set_integer(keyFile,"alarm","alarm 2 time",statusbar_plugin->alarm2Time);
    g_key_file_set_boolean(keyFile,"alarm","alarms on",statusbar_plugin->alarms);
    fileData=g_key_file_to_data(keyFile,&size,NULL);
    rcFile=fopen(filename,"w");
    fputs(fileData,rcFile);
    fclose(rcFile);
    g_free(fileData);
  }
  gtk_widget_destroy(dialog);
  g_key_file_free(keyFile);
}

void popup_place(GtkMenu *menu, gint *x, gint *y, gboolean *push_in, gpointer user_data)
{
      LargeStatusbarClockMinutesPlugin *statusbar_plugin=(LargeStatusbarClockMinutesPlugin*)user_data;
	GtkWidget *btn=statusbar_plugin->minutesLabel;

	gdk_window_get_position(btn->window, x, y);

	*y += btn->allocation.y + btn->allocation.height;
	*x += btn->allocation.x;
}

gint dbus_req_handler(const gchar* interface, const gchar* method, GArray* arguments, gpointer data, osso_rpc_t* retval)
{
  osso_rpc_t val;
  LargeStatusbarClockMinutesPlugin *statusbar_plugin=(LargeStatusbarClockMinutesPlugin*)data;
  val=g_array_index(arguments,osso_rpc_t,0);
  if (DBUS_TYPE_STRING==val.type)
  {
    if (g_strcasecmp(val.value.s,"popup menu")==0)
    {
      gtk_button_clicked(GTK_BUTTON(statusbar_plugin->button));
    }
    else
    {
      osso_return_t ret;
      ret = osso_rpc_async_run(statusbar_plugin->osso,
				"com.nokia.large_statusbar_clock_hours",
				"/com/nokia/large_statusbar_clock_hours",
				"com.nokia.large_statusbar_clock_hours",
				"update", NULL, NULL,
				DBUS_TYPE_STRING,statusbar_plugin->fontDesc,
				DBUS_TYPE_STRING,statusbar_plugin->curColour,
				DBUS_TYPE_BOOLEAN,statusbar_plugin->ampm,
				DBUS_TYPE_BOOLEAN,statusbar_plugin->stacked,
				DBUS_TYPE_INVALID);
    }
  }

  return OSSO_OK;
}
