/*
* Copyright (C) 2005-2006  Koos Vriezen <koos.vriezen@gmail.com>
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#ifndef _KMPLAYER_TYPES_H_
#define _KMPLAYER_TYPES_H_

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>

#include <glib.h>
#include <gdk/gdk.h>
#include <gconf/gconf-client.h>
#include <libgnomevfs/gnome-vfs-mime-utils.h>

#undef KDE_NO_CDTOR_EXPORT
#undef KDE_NO_EXPORT
#if __GNUC__ - 0 > 3 || (__GNUC__ - 0 == 3 && __GNUC_MINOR__ - 0 > 3)
  #define KMPLAYER_NO_CDTOR_EXPORT __attribute__ ((visibility("hidden")))
  #define KMPLAYER_NO_EXPORT __attribute__ ((visibility("hidden")))
  #define KMPLAYER_EXPORT __attribute__ ((visibility("default")))
#elif __GNUC__ - 0 > 3 || (__GNUC__ - 0 == 3 && __GNUC_MINOR__ - 0 > 2)
  #define KMPLAYER_NO_CDTOR_EXPORT
  #define KMPLAYER_NO_EXPORT __attribute__ ((visibility("hidden")))
  #define KMPLAYER_EXPORT
#else
  #define KMPLAYER_NO_CDTOR_EXPORT
  #define KMPLAYER_NO_EXPORT
  #define KMPLAYER_EXPORT
#endif
#ifndef KDE_EXPORT
  #define KDE_EXPORT KMPLAYER_EXPORT
  #define KDE_NO_EXPORT KMPLAYER_NO_EXPORT
  #define KDE_NO_CDTOR_EXPORT KMPLAYER_NO_CDTOR_EXPORT
#endif


#ifndef ASSERT
#define ASSERT g_assert
//#define ASSERT Q_ASSERT
#endif

struct GnomeVFSURI;

namespace KMPlayer {

class GnomeVFSFilePrivate;

/**
 * Type meant for screen coordinates
 */
class KMPLAYER_NO_EXPORT Single {
    int value;
    friend Single operator + (const Single s1, const Single s2);
    friend Single operator - (const Single s1, const Single s2);
    friend Single operator * (const Single s1, const Single s2);
    friend Single operator / (const Single s1, const Single s2);
    friend Single operator + (const Single s1, const int i);
    friend Single operator - (const Single s1, const int i);
    friend float operator * (const Single s, const float f);
    friend double operator * (const Single s, const double f);
    friend Single operator * (const int i, const Single s);
    friend float operator * (const float f, const Single s);
    friend double operator * (const double d, const Single s);
    friend Single operator / (const Single s, const int i);
    friend float operator / (const Single s, const float f);
    friend double operator / (const Single s, const double d);
    friend double operator / (const double d, const Single s);
    friend bool operator > (const Single s1, const Single s2);
    friend bool operator > (const Single s, const int i);
    friend bool operator > (const int i, const Single s);
    friend bool operator >= (const Single s1, const Single s2);
    friend bool operator == (const Single s1, const Single s2);
    friend bool operator != (const Single s1, const Single s2);
    friend bool operator < (const Single s1, const Single s2);
    friend bool operator < (const Single s, const int i);
    friend bool operator < (const int i, const Single s);
    friend bool operator <= (const Single s1, const Single s2);
    friend bool operator <= (const Single s, const int i);
    friend Single operator - (const Single s);
public:
    Single () : value (0) {}
    Single (const int v) : value (v << 8) {}
    Single (const float v) : value (int (256 * v)) {}
    Single (const double v) : value (int (256 * v)) {}
    Single & operator = (const Single s) { value = s.value; return *this; }
    Single & operator = (const int v) { value = v << 8; return *this; }
    Single & operator = (const float v) { value = int (256 * v); return *this; }
    Single & operator = (const double v) { value = int(256 * v); return *this; }
    Single & operator += (const Single s) { value += s.value; return *this; }
    Single & operator += (const int i) { value += (i << 8); return *this; }
    Single & operator -= (const Single s) { value -= s.value; return *this; }
    Single & operator -= (const int i) { value -= (i << 8); return *this; }
    Single & operator *= (const Single s);
    Single & operator *= (const float f) { value = int(value*f); return *this; }
    Single & operator /= (const int i) { value /= i; return *this; }
    Single & operator /= (const float f);
    operator int () const { return value >> 8; }
    operator double () const { return 1.0 * value / 256; }
};

template <class T>
class KMPLAYER_NO_EXPORT Point {
public:
    Point ();
    Point (T _x, T _y);

    bool operator == (const Point<T> &p) const;
    bool operator != (const Point<T> &p) const;

    T x;
    T y;
};

template <class T>
class KMPLAYER_NO_EXPORT Size {
public:
    Size ();
    Size (T w, T h);

    bool isEmpty () const;
    bool operator == (const Size<T> &s) const;
    bool operator != (const Size<T> &s) const;

    T width;
    T height;
};

template <class T>
class KMPLAYER_NO_EXPORT Rect {
public:
    Rect ();
    Rect (T a, T b, T w, T h);
    Rect (T a, T b, const Size<T> &s);
    Rect (const Point<T> &point, const Size<T> &s);
    T x () const;
    T y () const;
    T width () const;
    T height () const;
    Rect<T> unite (const Rect<T> &r) const;
    Rect<T> intersect (const Rect<T> &r) const;
    bool operator == (const Rect<T> &r) const;
    bool operator != (const Rect<T> &r) const;
    bool isEmpty () const;

    Point<T> point;
    Size<T> size;
};

typedef Size<Single> SSize;
typedef Rect<Single> SRect;

typedef Size<int> ISize;
typedef Point<int> IPoint;
typedef Rect<int> IRect;
template <> Rect<int> Rect<int>::intersect (const Rect<int> &r) const;


template <class T>
class KMPLAYER_EXPORT CharTmpl {
    T m_unichar;
public:
    CharTmpl ();
    CharTmpl (const T & char_impl);
    CharTmpl (const char ch);
    CharTmpl (const int unichar);
    CharTmpl (const CharTmpl <T> & other);
    ~CharTmpl ();
    bool operator == (const CharTmpl <T> & other) const;
    bool operator == (const T & char_impl) const;
    bool operator == (const char ch) const;
    bool operator != (const CharTmpl <T> & other) const;
    bool operator != (const T & char_impl) const;
    bool operator != (const char ch) const;
    CharTmpl <T> & operator = (const CharTmpl <T> & other);
    CharTmpl <T> & operator = (const T & char_impl);
    CharTmpl <T> & operator = (const char ch);
    operator T ();
    operator T () const;
    operator char () const;
    int unicode () const;
    // these isXX should be specialized
    bool isSpace () const;
    bool isLetterOrNumber () const;
};

typedef CharTmpl <gunichar> Char;


class ByteArrayImpl {
    char * m_data;
    unsigned m_size;
    unsigned m_capacity;
    bool data_owner;
public:
    ByteArrayImpl ();
    ByteArrayImpl (unsigned sz);
    ByteArrayImpl (const ByteArrayImpl & other);
    ByteArrayImpl (const char * data, unsigned size);
    ~ByteArrayImpl ();
    unsigned size () const;
    void resize (unsigned ns);
    char * data () const;
    bool isEmpty () const;
    ByteArrayImpl & operator = (const ByteArrayImpl & other);
    ByteArrayImpl & operator = (const char *str);
};

typedef ByteArrayImpl ByteArray;


class FixedSizeAllocator {
    void **pool;
    size_t size;
    int count;
public:
    FixedSizeAllocator (size_t s)
        : pool ((void**) malloc (10 * sizeof (void *))), size (s), count (0) {}

    void *alloc ();
    void dealloc (void *p);
};

template <class T>
class Shared {
public:
    ~Shared ();
    void clear ();
    bool isNull () const;
protected:
    T *data;
    Shared ();
    void attach (const Shared<T> &other);
    void detach ();
};

struct SharedChars {
    SharedChars (gchar *p, int len) : ptr (p), ref_count (1), buf_len (len) {}

    // Shared requirements
    SharedChars (const SharedChars &s)
        : ptr(g_strdup (s.ptr)), ref_count (1), buf_len (s.buf_len) {}
    ~SharedChars () { g_free (ptr); }
    void *operator new (size_t);
    void operator delete (void *);
    void ref () { ref_count++; }
    void unref () { if (--ref_count <= 0) delete this; }
    int ref_count;

    gchar *ptr;
    int buf_len; // excluding zero terminator
};

template <> void Shared<SharedChars>::attach (const Shared<SharedChars> &other);

class KMPLAYER_NO_EXPORT GLibString : public Shared <SharedChars> {
public:
    GLibString ();
    GLibString (const GLibString & str);
    GLibString (const char * str, int maxlen = -1);
    GLibString (const Char & str);
    ~GLibString ();
    void append (const char *p, int length);  // p must be valid utf8
    void takeUtf8 (char *p, int buf_len=-1);  // p must be valid utf8
    int compare (const char *str) const;
    int compare (const GLibString & str) const;
    bool operator == (const GLibString &other) const;
    bool operator == (const char * str) const;
    bool operator != (const GLibString &other) const;
    bool operator != (const char * str) const;
    GLibString & operator = (const GLibString & str);
    GLibString & operator = (const char * s);
    GLibString & operator = (const Char & ch);
    GLibString & operator << (const GLibString &other);
    GLibString & operator << (const char * str);
    GLibString operator + (const char * str) const;
    GLibString operator + (const GLibString &other) const;
    GLibString operator + (const Char & ch) const;
    GLibString & operator += (const GLibString & str);
    GLibString & operator += (const char * str);
    GLibString & operator += (const Char & ch);
    Char operator [] (int index) const;
    operator const char * () const;
    bool operator < (const GLibString &other) const;
    void take (char * s);
    GLibString lower () const;
    GLibString upper () const;
    GLibString stripWhiteSpace () const;
    static GLibString fill (Char ch, int length);
    static GLibString number (int nr);
    bool startsWith (const GLibString & prefix) const;
    bool endsWith (const GLibString & suffix) const;
    int indexOf (const GLibString & needle) const;
    int lastIndexOf (const GLibString & needle) const;
    GLibString replace (Char c1, Char c2) const;
    bool isEmpty () const;
    GLibString mid (const int index, unsigned length=(unsigned)-1) const;
    GLibString left (int index) const;
    int toInt (bool *valid=0L, int base=10) const;
    double toDouble (bool *valid=0L) const;
    int length () const;
    void truncate (int pos);
};

template < class U>
GLibString operator + (const CharTmpl <U> &c, const GLibString &s) {
    GLibString str (c);
    return str += s;
}

template <class U>
GLibString operator + (const GLibString &s, const CharTmpl <U> &c) {
    GLibString str (s);
    return str += c;
}

typedef GLibString String;


enum FileMode { IO_ReadOnly=0x01, IO_WriteOnly=0x02, IO_ReadWrite=0x03 };

class GnomeVFSFile {
    GnomeVFSFilePrivate * d;
public:
    GnomeVFSFile (const String & path);
    ~GnomeVFSFile ();
    bool open (FileMode mode);
    void close ();
    ByteArray readAll ();
    off_t readBlock (char *, off_t max);
    off_t writeBlock (const char * data, off_t length);
    bool exists ();
    off_t size ();
};

typedef GnomeVFSFile File;


class GLibTextStream {
    GString * m_gstring;
    gchar * m_utf8_str;
    gchar * m_iter;
public:
    GLibTextStream ();
    GLibTextStream (const String & init);
    GLibTextStream (ByteArray & init);
    ~GLibTextStream ();
    GLibTextStream & operator << (const String & str);
    GLibTextStream & operator << (const char * str);
    GLibTextStream & operator << (const Char & ch);
    GLibTextStream & operator << (int i);
    GLibTextStream & operator >> (Char & ch);
    GLibTextStream & readRawBytes (char * b, unsigned length);
    bool atEnd () const;
    bool isEmpty () const;
    const gchar *at () const;
    String release ();
    String readLine ();
private:
    GLibTextStream (const GLibTextStream & other);
    GLibTextStream & operator = (const GLibTextStream & other);
};

typedef GLibTextStream TextStream;


class GnomeVFSUrl {
    GnomeVFSURI * m_vfsurl;
public:
    GnomeVFSUrl ();
    GnomeVFSUrl (const String & ur);
    GnomeVFSUrl (const GnomeVFSUrl & base, const String & rel);
    GnomeVFSUrl (const GnomeVFSUrl & url);
    ~GnomeVFSUrl ();
    GnomeVFSUrl & operator = (const String & url);
    GnomeVFSUrl & operator = (const GnomeVFSUrl & url);
    bool operator == (const GnomeVFSUrl &u) const { return url() == u.url(); }
    String url () const;
    String path () const;
    String protocol () const;
    static String decode_string (const String & str);
    static String encode_string (const String & str);
    bool isLocalFile () const;
};

typedef GnomeVFSUrl URL;


class GMimeType {
public:
    static String findByContent (const char * data, int size);
    static String findByURL (const URL & url);
};

typedef GMimeType MimeType;


class IOJob;
class IOJobListener;
typedef class CurlGetJob IOJobPrivate;

IOJob *asyncGet (IOJobListener *receiver, const String &url);

class IOJob {
    friend IOJob *asyncGet (IOJobListener *receiver, const String &url);
    friend class CurlGetJob;

public:
    ~IOJob ();

    void setHeader (const String &header);
    void setHttpPostData (const ByteArray &data);
    void start ();
    void kill (bool quiet = true);
    bool error ();
    unsigned long contentLength() const;
    String contentType () const;

private:
    IOJob (IOJobPrivate *);

    IOJobPrivate *d;
};

class IOJobListener {
public:
    virtual void jobData (IOJob * job, ByteArray & data) = 0;
    virtual void jobResult (IOJob * job) = 0;
    virtual void redirected (IOJob * job, const String & uri) = 0;
};

class GConf {
    GConfClient * client;
public:
    GConf ();
    ~GConf ();
    String readEntry (const String & key, const String & def);
    int readNumEntry (const String & key, int def);
    double readNumEntry (const String & key, double def);
    void writeEntry (const String & key, const String & val);
    void writeEntry (const String & key, int val);
    void writeEntry (const String & key, double val);
    operator GConfClient* ();
};

typedef GConf Config;


class Log;

typedef Log & (*LOGFUNC) (Log &);

class Log {
    Log ();
    friend Log debugLog ();
    friend Log warningLog ();
    friend Log errorLog ();
public:
    ~Log ();
    Log & operator << (const String & s);
    Log & operator << (const char * s);
    //Log & operator << (const Char & s);
    Log & operator << (const URL & u);
    Log & operator << (int i);
    Log & operator << (unsigned int i);
    Log & operator << (double d);
    Log & operator << (const Single &s);
    Log & operator << (const SRect &s);
    Log & operator << (const IRect &s);
    Log & operator << (void *);
    Log & operator << (LOGFUNC manip);
};

Log debugLog ();
Log warningLog ();
Log errorLog ();


class GColor {
    GdkColor * m_color;
public:
    GColor ();
    GColor (const String & desc);
    GColor (int rgb);
    GColor (const GColor & other);
    ~GColor ();
    void clear ();
    GColor & operator = (const GColor & other);
    //GColor & operator = (const String & desc);
    //GColor & operator = (int rgb);
    operator GdkColor * () const;
    unsigned rgb () const;
};

typedef GColor Color;
typedef unsigned Rgb;


class GPixbuf {
    GdkPixbuf * m_pixbuf;
public:
    GPixbuf ();
    GPixbuf (int w, int h);
    GPixbuf (const ByteArray & data);
    GPixbuf (GdkPixbuf * pix, bool skip_addref=false);
    GPixbuf (const GPixbuf & other); // only a g_object_ref on other.m_pixmap
    ~GPixbuf ();
    bool isNull () const;
    int width () const;
    int height () const;
    void fill (const Color & c = Color (0));
    void reset ();
    GPixbuf scale (int w, int h);
    operator GdkPixbuf * () const;
    GPixbuf & operator = (const GPixbuf & other);
    //GPixbuf & operator = (const Pixmap & pixmap);
    GPixbuf & operator = (GdkPixbuf * pixbuf);
    GdkPixbuf * pixbuf () const;
};

typedef GPixbuf Image;

//-----------------------------------------------------------------------------

inline Single & Single::operator *= (const Single s) {
    value = (((int64_t)value) * s.value) >> 8;
    return *this;
}

inline Single & Single::operator /= (const float f) {
    value = (int) (value / f);
    return *this;
}

inline Single operator + (const Single s1, const Single s2) {
    Single s;
    s.value = s1.value + s2.value;
    return s;
}

inline Single operator - (const Single s1, const Single s2) {
    Single s;
    s.value = s1.value - s2.value;
    return s;
}

inline Single operator * (const Single s1, const Single s2) {
    Single s;
    s.value = (((int64_t)s1.value) * s2.value) >> 8;
    return s;
}

inline Single operator / (const Single s1, const Single s2) {
    Single s;
    s.value = ((int64_t)s1.value << 8) / s2.value;
    return s;
}

inline Single operator + (const Single s, const int i) {
    return s + Single (i);
}

inline Single operator - (const Single s, const int i) {
    return s - Single (i);
}

inline Single operator * (const int i, const Single s) {
    Single s1;
    s1.value = s.value * i;
    return s1;
}

inline Single operator * (const Single s, const int i) {
    return i * s;
}
inline float operator * (const Single s, const float f) {
    return s.value * f / 256;
}

inline double operator * (const Single s, const double d) {
    return s.value * d / 256;
}

inline float operator * (const float f, const Single s) {
    return s.value * f / 256;
}

inline double operator * (const double d, const Single s) {
    return s.value * d / 256;
}

inline Single operator / (const Single s, const int i) {
    Single s1;
    s1.value = s.value / i;
    return s1;
}

inline float operator / (const Single s, const float f) {
    return (s.value / f ) / 256;
}

inline double operator / (const Single s, const double d) {
    return (s.value / d ) / 256;
}

inline double operator / (const double d, const Single s) {
    return (d * 256 / s.value);
}

inline bool
operator > (const Single s1, const Single s2) { return s1.value > s2.value; }

inline bool
operator > (const Single s, const int i) { return s > Single (i); }

inline bool
operator > (const int i, const Single s) { return Single (i) > s; }

inline bool
operator >= (const Single s1, const Single s2) { return s1.value >= s2.value; }

inline bool
operator == (const Single s1, const Single s2) { return s1.value == s2.value; }

inline bool
operator != (const Single s1, const Single s2) { return s1.value != s2.value; }

inline bool
operator < (const Single s1, const Single s2) { return s1.value < s2.value; }

inline bool
operator < (const Single s, const int i) { return s < Single (i); }

inline bool
operator < (const int i, const Single s) { return Single (i) < s; }

inline bool
operator <= (const Single s1, const Single s2) { return s1.value <= s2.value; }

inline bool
operator <= (const Single s, const int i) { return s <= Single (i); }

inline Single operator - (const Single s) {
    Single s1;
    s1.value = -s.value;
    return s1;
}

//-----------------------------------------------------------------------------

template <class T> inline Point<T>::Point () : x (0), y (0) {}

template <class T> inline Point<T>::Point (T _x, T _y) : x (_x), y (_y) {}

template <class T>
inline bool Point<T>::Point::operator == (const Point<T> &p) const {
    return x == p.x && y == p.y;
}

template <class T>
inline bool Point<T>::Point::operator != (const Point<T> &p) const {
    return !(*this == p);
}

//-----------------------------------------------------------------------------

template <class T> inline Size<T>::Size () : width (0), height (0) {}

template <class T> inline Size<T>::Size (T w, T h) : width (w), height (h) {}

template <class T> inline bool Size<T>::isEmpty () const {
    return width <= 0 || height <= 0;
}

template <class T>
inline bool Size<T>::Size::operator == (const Size<T> &s) const {
    return width == s.width && height == s.height;
}

template <class T>
inline bool Size<T>::Size::operator != (const Size<T> &s) const {
    return !(*this == s);
}

//-----------------------------------------------------------------------------

template <class T> inline Rect<T>::Rect () {}

template <class T> inline  Rect<T>::Rect (T a, T b, T w, T h)
 : point (a, b), size (w, h) {}

template <class T> inline Rect<T>::Rect (T a, T b, const Size<T> &s)
 : point (a, b), size (s) {}

template <class T> inline Rect<T>::Rect (const Point<T> &pnt, const Size<T> &s)
 : point (pnt), size (s) {}

template <class T> inline T Rect<T>::x () const { return point.x; }

template <class T> inline T Rect<T>::y () const { return point.y; }

template <class T> inline T Rect<T>::width () const { return size.width; }

template <class T> inline T Rect<T>::height () const { return size.height; }

template <class T> inline bool Rect<T>::operator == (const Rect<T> &r) const {
    return point == r.point && size == r.size;
}

template <class T> inline bool Rect<T>::operator != (const Rect<T> &r) const {
    return !(*this == r);
}

template <class T> inline bool Rect<T>::isEmpty () const {
    return size.isEmpty ();
}

template <class T> inline Rect<T> Rect<T>::unite (const Rect<T> &r) const {
    if (size.isEmpty ())
        return r;
    if (r.size.isEmpty ())
        return *this;
    T a (point.x < r.point.x ? point.x : r.point.x);
    T b (point.y < r.point.y ? point.y : r.point.y);
    return Rect<T> (a, b,
            ((point.x + size.width < r.point.x + r.size.width)
             ? r.point.x + r.size.width : point.x + size.width) - a,
            ((point.y + size.height < r.point.y + r.size.height)
             ? r.point.y + r.size.height : point.y + size.height) - b);
}

template <class T> inline Rect<T> Rect<T>::intersect (const Rect<T> &r) const {
    T a (point.x < r.point.x ? r.point.x : point.x);
    T b (point.y < r.point.y ? r.point.y : point.y);
    return Rect<T> (a, b,
            ((point.x + size.width < r.point.x + r.size.width)
             ? point.x + size.width : r.point.x + r.size.width) - a,
            ((point.y + size.height < r.point.y + r.size.height)
             ? point.y + size.height : r.point.y + r.size.height) - b);
}

//-----------------------------------------------------------------------------

template <class T> inline CharTmpl<T>::CharTmpl () : m_unichar (0) {}

template <class T>
inline CharTmpl<T>::CharTmpl (const T & char_impl) : m_unichar (char_impl) {}

template <class T>
inline CharTmpl<T>::CharTmpl (const char ch) : m_unichar (ch) {}

template <class T>
inline CharTmpl<T>::CharTmpl (const int unichar) : m_unichar (unichar) {}

template <class T>
inline CharTmpl<T>::CharTmpl (const CharTmpl <T> & other) 
    :  m_unichar (other.m_unichar) {}

template <class T> inline CharTmpl<T>::~CharTmpl () {}

template <class T> inline bool CharTmpl<T>::operator == (const CharTmpl <T> & other) const {
    return m_unichar == other.m_unichar;
}

template <class T> inline bool CharTmpl<T>::operator == (const T & char_impl) const {
    return m_unichar == char_impl;
}

template <class T> inline bool CharTmpl<T>::operator == (const char ch) const {
    return m_unichar == ch;
}

template <class T> inline bool CharTmpl<T>::operator != (const CharTmpl <T> & other) const {
    return m_unichar != other.m_unichar;
}

template <class T> inline bool CharTmpl<T>::operator != (const T & char_impl) const {
    return m_unichar != char_impl;
}

template <class T> inline  bool CharTmpl<T>::operator != (const char ch) const {
    return m_unichar != ch;
}

template <class T>
inline CharTmpl <T> & CharTmpl<T>::operator = (const CharTmpl <T> & other) {
    m_unichar = other.m_unichar;
    return *this;
}

template <class T>
inline CharTmpl <T> & CharTmpl<T>::operator = (const T & char_impl) {
    m_unichar = char_impl;
    return *this;
}

template <class T>
inline CharTmpl <T> & CharTmpl<T>::operator = (const char ch) {
    m_unichar = ch;
    return *this;
}

template <class T> inline CharTmpl<T>::operator T () { return m_unichar; }

template <class T> inline CharTmpl<T>::operator T () const { return m_unichar; }

template <class T> inline CharTmpl<T>::operator char () const {
    gchar buf [8];
    int nr = g_unichar_to_utf8 (m_unichar, buf);
    return (char) buf [nr-1];
}

template <class T> inline int CharTmpl<T>::unicode () const {
    return (int) m_unichar;
}

template <class T> inline bool CharTmpl<T>::isSpace () const {
    return false;
}

template <class T> inline bool CharTmpl<T>::isLetterOrNumber () const {
    return false;
}

//-----------------------------------------------------------------------------

template <>
inline bool CharTmpl<gunichar>::isSpace () const {
    return g_unichar_isspace (m_unichar);
}

template <>
inline bool CharTmpl<gunichar>::isLetterOrNumber () const {
    return g_unichar_isalnum (m_unichar);
}

template <>
inline CharTmpl <gunichar> & CharTmpl<gunichar>::operator = (const char ch) {
    char buf [] = { ch, 0 };
    const gchar *end;
    if (!g_utf8_validate (buf, -1, &end)) {
        gsize bytes_read, bytes_written;
        gchar *utf8 =g_locale_to_utf8(buf, -1, &bytes_read, &bytes_written, 0L);
        m_unichar = g_utf8_get_char (utf8);
        g_free (utf8);
    } else
        m_unichar = g_utf8_get_char (buf);
    return *this;
}

template <>
inline CharTmpl<gunichar>::CharTmpl (const char ch) {
    operator = (ch);
}

template <>
inline  bool CharTmpl<gunichar>::operator == (const char ch) const {
    return m_unichar == Char (ch). m_unichar;
}

template <>
inline  bool CharTmpl<gunichar>::operator != (const char ch) const {
    return m_unichar != Char (ch).m_unichar;
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT inline Log::Log () {}

KMPLAYER_NO_CDTOR_EXPORT inline Log::~Log () {}

KMPLAYER_NO_EXPORT inline Log & Log::operator << (const String & s) {
    return *this << (const char *) s;
}

KMPLAYER_NO_EXPORT inline Log & Log::operator << (const URL & u) {
    return *this << u.url ();
}

KMPLAYER_NO_EXPORT inline Log & endl (Log & log) {
    log << "\n";
    return log;
}

KMPLAYER_NO_EXPORT inline Log & Log::operator << (LOGFUNC f) {
    return (*f) (*this);
}

KMPLAYER_NO_EXPORT inline Log debugLog () {
    return Log ();
}

KMPLAYER_NO_EXPORT inline Log warningLog () {
    Log log;
    return log << "[WARNING] ";
}

KMPLAYER_NO_EXPORT inline Log errorLog () {
    Log log;
    return log << "[ERROR] ";
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT inline ByteArrayImpl::ByteArrayImpl ()
 : m_data (0L), m_size (0), m_capacity (0), data_owner (true) {}

KMPLAYER_NO_CDTOR_EXPORT inline ByteArrayImpl::ByteArrayImpl (unsigned sz)
 : m_data ((char *) malloc (sz)),
   m_size (sz),
   m_capacity (sz),
   data_owner (true) {}

inline ByteArrayImpl::ByteArrayImpl (const char * d, unsigned sz)
 : m_data ((char *)d), m_size (sz), m_capacity (sz), data_owner (false) {}

KMPLAYER_NO_CDTOR_EXPORT inline ByteArrayImpl::~ByteArrayImpl () {
    if (m_data && data_owner)
        free (m_data);
}

KMPLAYER_NO_EXPORT inline unsigned ByteArrayImpl::size () const {
    return m_size;
}

KMPLAYER_NO_EXPORT inline char * ByteArrayImpl::data () const {
    return m_data;
}

KMPLAYER_NO_EXPORT inline bool ByteArrayImpl::isEmpty () const {
    return (!m_data || m_size ==0);
}

//-----------------------------------------------------------------------------

template <class T>
inline Shared<T>::Shared () : data (0) {}

template <class T>
inline Shared<T>::~Shared () {
    clear ();
}

template <class T>
inline bool Shared<T>::isNull () const {
    return !data;
}

template <class T>
inline void Shared<T>::clear () {
    if (data)
        data->unref ();
    data = 0L;
}

template <class T>
inline void Shared<T>::detach () {
    if (data && data->ref_count > 1) {
        T *tmp = new T (data);
        if (data)
            data->unref ();
        data = tmp;
    }
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT inline GLibString::GLibString () {}

KMPLAYER_NO_CDTOR_EXPORT inline GLibString::GLibString (const GLibString &s) {
    attach (s);
}

KMPLAYER_NO_CDTOR_EXPORT
inline GLibString::GLibString (const char * str, int maxlen) {
    if (str)
        take (maxlen > -1 ? g_strndup (str, maxlen) : g_strdup (str));
}

KMPLAYER_NO_EXPORT inline GLibString::~GLibString () {}

KMPLAYER_NO_EXPORT
inline GLibString & GLibString::operator = (const GLibString &str) {
    attach (str);
    return *this;
}

KMPLAYER_NO_EXPORT inline GLibString::operator const char * () const {
    return data ? data->ptr : "";
}

KMPLAYER_NO_EXPORT inline Char GLibString::operator [] (int index) const {
    // unchecked pointer, TODO add a modifying operator []
    gchar * off = g_utf8_offset_to_pointer (data->ptr, index);
    return Char (g_utf8_get_char (off));
}

KMPLAYER_NO_EXPORT inline GLibString GLibString::number (int nr) {
    GLibString str;
    str.takeUtf8 (g_strdup_printf ("%d", nr));
    return str;
}

KMPLAYER_NO_EXPORT inline bool GLibString::isEmpty () const {
    return !data || !data->ptr [0];
}

KMPLAYER_NO_EXPORT inline int GLibString::length () const {
    return data ? g_utf8_strlen (data->ptr, -1) : 0;
}

KMPLAYER_NO_EXPORT
inline bool GLibString::operator == (const GLibString &other) const {
    return !compare (other);
}

KMPLAYER_NO_EXPORT inline bool GLibString::operator == (const char *str) const {
    return !compare (str);
}

KMPLAYER_NO_EXPORT
inline bool GLibString::operator != (const GLibString &other) const {
    return compare (other);
}

KMPLAYER_NO_EXPORT inline bool GLibString::operator != (const char *str) const {
    return compare (str);
}

KMPLAYER_NO_EXPORT
inline GLibString GLibString::operator + (const GLibString &other) const {
    GLibString str (*this);
    return str += other;
}

KMPLAYER_NO_EXPORT
inline GLibString GLibString::operator + (const char *str) const {
    GLibString str1 (*this);
    return str1 += str;
}

KMPLAYER_NO_EXPORT
inline GLibString GLibString::operator + (const Char & ch) const {
    GLibString str (*this);
    return str += ch;
}


KMPLAYER_NO_EXPORT
inline GLibString & GLibString::operator << (const GLibString &other) {
    return *this += other;
}

KMPLAYER_NO_EXPORT
inline GLibString & GLibString::operator << (const char *str) {
    return *this += str;
}

KMPLAYER_NO_EXPORT
inline bool GLibString::operator < (const GLibString &other) const {
    return compare (other) < 0;
}

KMPLAYER_NO_EXPORT inline GLibString GLibString::lower () const {
    GLibString str;
    if (data)
        str.take (g_utf8_strdown (data->ptr, -1));
    return str;
}

KMPLAYER_NO_EXPORT inline GLibString GLibString::upper () const {
    GLibString str;
    if (data)
        str.take (g_utf8_strup (data->ptr, -1));
    return str;
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT inline GLibTextStream::GLibTextStream ()
    //: m_gstring (g_string_new ("")), m_utf8_str (0L), m_iter (0L) {}
    : m_gstring (g_string_sized_new (128)), m_utf8_str (0L), m_iter (0L) {}

KMPLAYER_NO_CDTOR_EXPORT
inline GLibTextStream::GLibTextStream (const String & init)
    : m_gstring (g_string_new ((const char *) init)),
      m_utf8_str (0L),
      m_iter (0L) {}

KMPLAYER_NO_CDTOR_EXPORT
inline GLibTextStream::GLibTextStream (ByteArray & init)
    : m_gstring (g_string_new_len (init.data (), init.size ())),
      m_utf8_str (0L),
      m_iter (0L) {}

KMPLAYER_NO_CDTOR_EXPORT inline GLibTextStream::~GLibTextStream () {
    if (m_gstring)
        g_string_free (m_gstring, true);
    else if (m_utf8_str)
        g_free (m_utf8_str);
}

KMPLAYER_NO_EXPORT inline String GLibTextStream::release () {
    GLibString str;
    if (m_gstring) {
        str.takeUtf8 (g_string_free (m_gstring, false));
        m_gstring = 0L;
    }
    return str;
}

KMPLAYER_NO_EXPORT inline GLibTextStream & GLibTextStream::operator << (int i) {
    return *this << String::number (i);
}

KMPLAYER_NO_EXPORT inline GLibTextStream & GLibTextStream::operator << (const char * str) {
    if (m_gstring)
        g_string_append (m_gstring, (const char *) str);
    return *this;
}

KMPLAYER_NO_EXPORT inline GLibTextStream & GLibTextStream::operator << (const String & str) {
    return *this << (const char *) str;
}

KMPLAYER_NO_EXPORT inline GLibTextStream & GLibTextStream::operator << (const Char & ch) {
    if (m_gstring)
        g_string_append_unichar (m_gstring, (gunichar) ch.unicode ());
    return *this;
}

KMPLAYER_NO_EXPORT inline
GLibTextStream & GLibTextStream::readRawBytes (char * b, unsigned len) {
    if (m_gstring)
        g_string_append_len (m_gstring, b, len);
    return *this;
}

KMPLAYER_NO_EXPORT inline bool GLibTextStream::atEnd () const {
    return (m_iter && !m_iter[0]) || (!m_gstring && !m_iter);
}

KMPLAYER_NO_EXPORT inline bool GLibTextStream::isEmpty () const {
    return (!m_gstring && (!m_utf8_str || !m_utf8_str [0]) ||
            (m_gstring && !m_gstring->len));
}

KMPLAYER_NO_EXPORT inline const gchar *GLibTextStream::at () const {
    return m_iter;
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT inline GnomeVFSUrl::GnomeVFSUrl () : m_vfsurl (0L) {}

//-----------------------------------------------------------------------------

inline String GMimeType::findByURL (const URL & url) {
    return String (gnome_vfs_get_mime_type ((const char *) url.url ()));
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_EXPORT
inline void GConf::writeEntry (const String & key, const String & val) {
    gconf_client_set_string (client, (const char *) key, (const char *)val, 0L);
}

KMPLAYER_NO_EXPORT
inline void GConf::writeEntry (const String & key, int val) {
    gconf_client_set_int (client, (const char *) key, val, 0L);
}

KMPLAYER_NO_EXPORT
inline void GConf::writeEntry (const String & key, double val) {
    gconf_client_set_float (client, (const char *) key, val, 0L);
}

KMPLAYER_NO_EXPORT inline GConf::operator GConfClient * () {
    return client;
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT inline GColor::GColor () : m_color (0L) {}

KMPLAYER_NO_CDTOR_EXPORT inline GColor::GColor (const GColor & c)
    : m_color (c.m_color ? gdk_color_copy (c.m_color) : 0L) {}

KMPLAYER_NO_EXPORT inline void GColor::clear () {
    if (m_color) {
        //gdk_colormap_free_colors (gdk_colormap_get_system (), m_color, 1);
        gdk_color_free (m_color); // ??
        m_color = 0L;
    }
}

KMPLAYER_NO_CDTOR_EXPORT inline GColor::~GColor () {
    clear ();
}

KMPLAYER_NO_EXPORT inline GColor & GColor::operator = (const GColor & c) {
    clear ();
    m_color = c.m_color ? gdk_color_copy (c.m_color) : 0L;
    return *this;
}

KMPLAYER_NO_EXPORT inline GColor::operator GdkColor * () const {
    return m_color;
}

KMPLAYER_NO_EXPORT inline unsigned GColor::rgb () const {
    return m_color ? ((m_color->red >> 8) << 16) | ((m_color->green >> 8) << 8) | (m_color->blue >> 8) : 0L;
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_EXPORT inline GPixbuf::GPixbuf () : m_pixbuf (0L) {}

KMPLAYER_NO_EXPORT inline bool GPixbuf::isNull () const { return !m_pixbuf; }

KMPLAYER_NO_EXPORT inline
GPixbuf::operator GdkPixbuf * () const { return m_pixbuf; }

KMPLAYER_NO_EXPORT
inline GdkPixbuf * GPixbuf::pixbuf () const { return m_pixbuf; }

KMPLAYER_NO_EXPORT inline int GPixbuf::width () const {
    return m_pixbuf ? gdk_pixbuf_get_width (m_pixbuf) : 0;
}

KMPLAYER_NO_EXPORT inline int GPixbuf::height () const {
    return m_pixbuf ? gdk_pixbuf_get_height (m_pixbuf) : 0;
}

} // namespace KMPlayer

#include "triestring.h"

#endif  // _KMPLAYER_TYPES_H_

