/**
  This file belong to the KMPlayer project, a movie player plugin for Konqueror
  Copyright (C) 2007  Koos Vriezen <koos.vriezen@gmail.com>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**/

#include <config.h>

#include <math.h>
#include <cairo.h>
#include <cairo-xlib.h>
#include <pango/pangocairo.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
//#include <X11/extensions/Xdbe.h>

#include "kmplayertypes.h"
#include "kmplayer_smil.h"
#include "kmplayer.h"
#include "viewarea.h"
#include "kmplayercontrol.h"
#include "kmplayer_rp.h"
#include "actor.h"

unsigned char * pixbufBits (GdkPixbuf *pixbuf, int & channels);

using namespace KMPlayer;

//-------------------------------------------------------------------------

static void clearSurface (cairo_t *cr, const IRect &rect) {
    cairo_save (cr);
    cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
    cairo_rectangle (cr, rect.x (), rect.y (), rect.width (), rect.height ());
    cairo_fill (cr);
    cairo_restore (cr);
}

void ImageData::copyImage (Surface *s, const SSize &sz, cairo_surface_t *similar,
        unsigned int *penalty, CalculatedSizer *zoom) {
    cairo_surface_t *src_sf;
    bool clear = false;
    int w = sz.width;
    int h = sz.height;

    unsigned char *bits = NULL;
    if (surface) {
        src_sf = surface;
    } else {
        int channels;
        unsigned char *bits = pixbufBits (image->pixbuf (), channels);
        has_alpha = 4 == channels;
        src_sf = cairo_image_surface_create_for_data (
                bits, has_alpha ? CAIRO_FORMAT_ARGB32 : CAIRO_FORMAT_RGB24,
                width, height, width*4);
        if (flags & ImagePixmap && !(flags & ImageAnimated)) {
            surface = cairo_surface_create_similar (similar,
                    has_alpha ? CAIRO_CONTENT_COLOR_ALPHA : CAIRO_CONTENT_COLOR,
                    width, height);
            cairo_pattern_t *pat = cairo_pattern_create_for_surface (src_sf);
            cairo_pattern_set_extend (pat, CAIRO_EXTEND_NONE);
            cairo_pattern_set_filter (pat, CAIRO_FILTER_FAST);
            cairo_t *cr = cairo_create (surface);
            cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
            cairo_set_source (cr, pat);
            cairo_paint (cr);
            cairo_destroy (cr);
            cairo_pattern_destroy (pat);
            cairo_surface_destroy (src_sf);
            cairo_surface_flush (surface);
            g_free (bits);
            src_sf = surface;
            delete image;
            image = NULL;
            *penalty += width * height / (10 * 480);
        }
    }

    cairo_pattern_t *img_pat = cairo_pattern_create_for_surface (src_sf);
    cairo_pattern_set_extend (img_pat, CAIRO_EXTEND_NONE);
    if (zoom) {
        cairo_matrix_t mat;
        Single zx, zy, zw, zh;
        zoom->calcSizes (NULL, width, height, zx, zy, zw, zh);
        cairo_matrix_init_translate (&mat, zx, zy);
        cairo_matrix_scale (&mat, 1.0 * zw/w, 1.0 * zh/h);
        cairo_pattern_set_matrix (img_pat, &mat);
        *penalty += w * h / (1 * 480);
    } else if (w != width && h != height) {
        cairo_matrix_t mat;
        cairo_matrix_init_scale (&mat, 1.0 * width/w, 1.0 * height/h);
        cairo_pattern_set_matrix (img_pat, &mat);
        *penalty += w * h / (1 * 480);
    }
    if (!s->surface)
        s->surface = cairo_surface_create_similar (similar,
                has_alpha ?
                CAIRO_CONTENT_COLOR_ALPHA : CAIRO_CONTENT_COLOR, w, h);
    else
        clear = true;
    cairo_t *cr = cairo_create (s->surface);
    if (clear)
        clearSurface (cr, IRect (0, 0, w, h));
    cairo_set_source (cr, img_pat);
    if (!has_alpha)
        cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
    cairo_paint (cr);
    cairo_destroy (cr);

    cairo_pattern_destroy (img_pat);
    if (!surface) {
        cairo_surface_destroy (src_sf);
        g_free (bits);
    }
}

//-------------------------------------------------------------------------

#define CAIRO_SET_SOURCE_RGB(cr,c)            \
    cairo_set_source_rgb ((cr),               \
            1.0 * (((c) >> 16) & 0xff) / 255, \
            1.0 * (((c) >> 8) & 0xff) / 255,  \
            1.0 * (((c)) & 0xff) / 255)

#define USE_IMG_SURFACE 0

# define CAIRO_SET_SOURCE_ARGB(cr,c)          \
    cairo_set_source_rgba ((cr),              \
            1.0 * (((c) >> 16) & 0xff) / 255, \
            1.0 * (((c) >> 8) & 0xff) / 255,  \
            1.0 * (((c)) & 0xff) / 255,       \
            1.0 * (((c) >> 24) & 0xff) / 255)

class KMPLAYER_NO_EXPORT CairoPaintVisitor : public Visitor {
    IRect clip;
    ViewArea *view_widget;
    cairo_surface_t * cairo_surface;
    cairo_surface_t * cairo_surface_widget;
    Matrix matrix;
    // stack vars need for transitions
    SMIL::MediaType *cur_media;
    cairo_pattern_t * cur_pat;
    cairo_matrix_t cur_mat;
    float opacity;
public:
    unsigned int penalty;
    bool toplevel;

    void traverseRegion (Node *reg, Surface *s);
    void updateExternal (SMIL::MediaType *av, SurfacePtr s);
    void paint(SMIL::MediaType *, Surface *, const IPoint &p, const IRect &);
    void video (Mrl *mt, Surface *s);
    cairo_t * cr;
    CairoPaintVisitor (ViewArea *v, cairo_surface_t * cs, Matrix m,
            const IRect & rect, bool toplevel=false);
    ~CairoPaintVisitor ();
    using Visitor::visit;
    void visit (Node * n);
    void visit (SMIL::Smil *);
    void visit (SMIL::Layout *);
    void visit (SMIL::RegionBase *);
    void visit (SMIL::Transition *);
    void visit (SMIL::ImageMediaType *);
    void visit (SMIL::TextMediaType *);
    void visit (SMIL::Brush *);
    void visit (SMIL::SmilText *);
    void visit (SMIL::RefMediaType *);
    void visit (RP::Imfl *);
    void visit (RP::Fill *);
    void visit (RP::Fadein *);
    void visit (RP::Fadeout *);
    void visit (RP::Crossfade *);
    void visit (RP::Wipe *);
    void visit (RP::ViewChange *);
};

KDE_NO_CDTOR_EXPORT
CairoPaintVisitor::CairoPaintVisitor (ViewArea *v, cairo_surface_t *cs,
        Matrix m, const IRect & rect, bool top)
 : view_widget (v), clip (rect),
#if USE_IMG_SURFACE
 cairo_surface_widget (cs), matrix (m), toplevel (top) {
    if (toplevel)
        cairo_surface = cairo_image_surface_create
            (CAIRO_FORMAT_RGB24, rect.width (), rect.height ());
    else
        cairo_surface = cs;
    cr = cairo_create (cairo_surface);
    if (toplevel) {
        cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
        cairo_set_tolerance (cr, 0.5 );
        cairo_set_antialias (cr, CAIRO_ANTIALIAS_NONE);
        cairo_rectangle (cr, 0, 0, rect.width (), rect.height ());
        cairo_fill (cr);
        cairo_translate (cr, -rect.x (), -rect.y ());
#else
# warning xrender
 cairo_surface (cs), matrix (m), toplevel (top) {
    cr = cairo_create (cs);
    if (toplevel) {
        penalty = rect.width () * rect.height () / (20 * 480);
        cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
        cairo_set_tolerance (cr, 0.5 );
        cairo_set_antialias (cr, CAIRO_ANTIALIAS_NONE);
        //cairo_push_group (cr);
        cairo_set_source_rgb (cr, 32.0 / 255, 32.0 / 255, 48.0 / 255);
        cairo_rectangle (cr, rect.x(), rect.y(), rect.width(), rect.height());
        cairo_fill (cr);
#endif
    } else {
        penalty = 0;
        clearSurface (cr, rect);
    }
}

KDE_NO_CDTOR_EXPORT CairoPaintVisitor::~CairoPaintVisitor () {
#if USE_IMG_SURFACE
    if (toplevel) {
        cairo_pattern_t *pat = cairo_pattern_create_for_surface (cairo_surface);
        cairo_pattern_set_extend (pat, CAIRO_EXTEND_NONE);
        cairo_matrix_t mat;
        cairo_matrix_init_translate (&mat, -clip.x (), -clip.y ());
        cairo_pattern_set_matrix (pat, &mat);
        cairo_t * wcr = cairo_create (cairo_surface_widget);
        cairo_set_source (wcr, pat);
        cairo_rectangle (wcr, clip.x(), clip.y(), clip.width(), clip.height());
        cairo_fill (wcr);
        cairo_destroy (wcr);
        cairo_pattern_destroy (pat);
        cairo_surface_destroy (cairo_surface);
    }
#endif
    cairo_destroy (cr);
}

KDE_NO_EXPORT void CairoPaintVisitor::visit (Node * n) {
    warningLog() << "Paint called on " << n->nodeName() << endl;
}

KDE_NO_EXPORT void CairoPaintVisitor::visit (SMIL::Smil *s) {
    if (s->active () && s->layout_node)
        s->layout_node->accept (this);
}

KDE_NO_EXPORT void CairoPaintVisitor::traverseRegion (Node *node, Surface *s) {
    // next visit receivers
    ConnectionList *nl = nodeMessageReceivers (node, MsgSurfaceAttach);
    if (nl) {
        for (Connection *c = nl->first(); c; c = nl->next ())
            if (c->connecter)
                c->connecter->accept (this);
    }
    /*for (SurfacePtr c = s->lastChild (); c; c = c->previousSibling ()) {
        if (c->node && c->node->id != SMIL::id_node_region &&
        c->node && c->node->id != SMIL::id_node_root_layout)
            c->node->accept (this);
        else
            break;
    }*/
    // finally visit region children
    for (SurfacePtr c = s->firstChild (); c; c = c->nextSibling ()) {
        if (c->node && c->node->id == SMIL::id_node_region)
            c->node->accept (this);
        else
            break;
    }
    s->dirty = false;
}

KDE_NO_EXPORT void CairoPaintVisitor::visit (SMIL::Layout *layout) {
    //debugLog() << "Visit " << layout->nodeName() << endl;
    if (layout->root_layout) {
        Surface *s = (Surface *)layout->root_layout->role (RoleDisplay);
        if (s) {
            //cairo_save (cr);
            Matrix m = matrix;

            IRect scr = matrix.toScreen (SRect (0, 0, s->bounds.size));

            IRect clip_save = clip;
            clip = clip.intersect (scr);

            if (s->background_color & 0xff000000) {
                CAIRO_SET_SOURCE_RGB (cr, s->background_color);
                cairo_rectangle (cr,
                        clip.x (), clip.y (), clip.width (), clip.height ());
                cairo_fill (cr);
            }

            matrix = Matrix (0, 0, s->xscale, s->yscale);
            matrix.transform (m);
            traverseRegion (layout->root_layout, s);
            //cairo_restore (cr);
            matrix = m;
            clip = clip_save;
        }
    }
}

#define REGION_SCROLLBAR_WIDTH 30

static void cairoDrawRect (cairo_t *cr, unsigned int color,
        int x, int y, int w, int h) {
    CAIRO_SET_SOURCE_ARGB (cr, color);
    cairo_rectangle (cr, x, y, w, h);
    cairo_fill (cr);
}

KDE_NO_EXPORT void CairoPaintVisitor::visit (SMIL::RegionBase *reg) {
    Surface *s = (Surface *) reg->role (RoleDisplay);
    if (s) {
        SRect rect = s->bounds;

        IRect scr = matrix.toScreen (rect);
        if (clip.intersect (scr).isEmpty ())
            return;
        Matrix m = matrix;
        matrix = Matrix (rect.x(), rect.y(), 1.0, 1.0);
        matrix.transform (m);
        IRect clip_save = clip;
        clip = clip.intersect (scr);
        cairo_save (cr);

        Surface *cs = s->firstChild ();
        if (!s->virtual_size.isEmpty ())
            matrix.translate (-s->x_scroll, -s->y_scroll);

        ImageActor *im = reg->media_info
             ? (ImageActor *) reg->media_info->media
             : NULL;
        ImageData *bg_img = im && !im->isEmpty() ? im->cached_img.ptr () : NULL;
        unsigned int bg_alpha = s->background_color & 0xff000000;
        if ((SMIL::RegionBase::ShowAlways == reg->show_background ||
                    reg->m_AttachedMediaTypes.first ()) &&
                (bg_alpha || bg_img)) {
            cairo_save (cr);
            if (bg_alpha) {
                cairo_rectangle (cr,
                        clip.x (), clip.y (), clip.width (), clip.height ());
                if (bg_alpha < 0xff000000) {
                    CAIRO_SET_SOURCE_ARGB (cr, s->background_color);
                    cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
                    cairo_fill (cr);
                    cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
                } else {
                    CAIRO_SET_SOURCE_RGB (cr, s->background_color);
                    cairo_fill (cr);
                }
            }
            if (bg_img) {
                Single w = bg_img->width;
                Single h = bg_img->height;
                matrix.getWH (w, h);
                if (!s->surface)
                    bg_img->copyImage (s, SSize(w, h), cairo_surface, &penalty);
                if (bg_img->has_alpha)
                    cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
                cairo_pattern_t *pat = cairo_pattern_create_for_surface (s->surface);
                cairo_pattern_set_extend (pat, CAIRO_EXTEND_REPEAT);
                cairo_matrix_t mat;
                cairo_matrix_init_translate (&mat, -scr.x (), -scr.y ());
                cairo_pattern_set_matrix (pat, &mat);
                cairo_set_source (cr, pat);
                cairo_rectangle (cr,
                        clip.x (), clip.y (), clip.width (), clip.height ());
                cairo_fill (cr);
                cairo_pattern_destroy (pat);
                if (bg_img->has_alpha)
                    cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
            }
            cairo_restore (cr);
        }
        traverseRegion (reg, s);
        cs = s->firstChild ();
        if (cs && (s->scroll || cs->scroll) && cs == s->lastChild ()) {
            SRect r = cs->bounds;
            if (r.width () > rect.width () || r.height () > rect.height ()) {
                if (s->virtual_size.isEmpty ())
                    s->x_scroll = s->y_scroll = 0;
                s->virtual_size = r.size;
                matrix.getWH (s->virtual_size.width, s->virtual_size.height);
                s->virtual_size.width += REGION_SCROLLBAR_WIDTH;
                s->virtual_size.height += REGION_SCROLLBAR_WIDTH;
                const int vy = s->virtual_size.height;
                const int vw = s->virtual_size.width;
                int sbw = REGION_SCROLLBAR_WIDTH;
                int sbx = scr.x () + scr.width () - sbw;
                int sby = scr.y ();
                int sbh = scr.height () - REGION_SCROLLBAR_WIDTH;
                IRect sb_clip = clip.intersect (IRect (sbx, sby, sbw, sbh));
                if (!sb_clip.isEmpty ()) {
                    int knob_h = sbh * scr.height () / vy;
                    int knob_y = scr.y () + s->y_scroll * sbh / vy;
                    IRect knob (sbx, knob_y, sbw, knob_h);
                    cairo_save (cr);
                    cairo_rectangle (cr, sb_clip.x (), sb_clip.y (),
                            sb_clip.width (), sb_clip.height ());
                    cairo_clip (cr);
                    cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
                    cairo_set_line_width (cr, 2);
                    CAIRO_SET_SOURCE_ARGB (cr, 0x80A0A0A0);
                    cairo_rectangle (cr, sbx + 1, sby + 1, sbw - 2, sbh - 2);
                    cairo_stroke (cr);
                    if (s->y_scroll)
                        cairoDrawRect (cr, 0x80000000,
                                sbx + 2, sby + 2,
                                sbw - 4, knob.y() - sby - 2);
                    cairoDrawRect (cr, 0x80808080,
                            knob.x() + 2, knob.y(),
                            knob.width() - 4, knob.height());
                    if (sby + sbh - knob.y() - knob.height() - 2 > 0)
                        cairoDrawRect (cr, 0x80000000,
                                sbx + 2, knob.y() + knob.height(),
                                sbw - 4, sby + sbh - knob.y() -knob.height()-2);
                    cairo_restore (cr);
                }
                sbh = REGION_SCROLLBAR_WIDTH;
                sbx = scr.x ();
                sby = scr.y () + scr.height () - sbh;
                sbw = scr.width () - REGION_SCROLLBAR_WIDTH;
                sb_clip = clip.intersect (IRect (sbx, sby, sbw, sbh));
                if (!sb_clip.isEmpty ()) {
                    int knob_w = sbw * scr.width () / vw;
                    int knob_x = scr.x () + s->x_scroll * sbw / vw;
                    IRect knob (knob_x, sby, knob_w, sbh);
                    cairo_save (cr);
                    cairo_rectangle (cr, sb_clip.x (), sb_clip.y (),
                            sb_clip.width (), sb_clip.height ());
                    cairo_clip (cr);
                    cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
                    cairo_set_line_width (cr, 2);
                    CAIRO_SET_SOURCE_ARGB (cr, 0x80A0A0A0);
                    cairo_rectangle (cr, sbx + 1, sby + 1, sbw - 2, sbh - 2);
                    cairo_stroke (cr);
                    if (s->x_scroll)
                        cairoDrawRect (cr, 0x80000000,
                                sbx + 2, sby + 2,
                                knob.x() - sbx - 2, sbh - 4);
                    cairoDrawRect (cr, 0x80808080,
                            knob.x(), knob.y() + 2,
                            knob.width(), knob.height() - 4);
                    if (sbx + sbw - knob.x() - knob.width() - 2 > 0)
                        cairoDrawRect (cr, 0x80000000,
                                knob.x() + knob.width(), sby + 2,
                                sbx + sbw - knob.x() - knob.width()-2, sbh - 4);
                    cairo_restore (cr);
                }
            }
        }
        cairo_restore (cr);
        matrix = m;
        clip = clip_save;
        s->dirty = false;
    }
}

#define CAIRO_SET_PATTERN_COND(cr,pat,mat)                      \
    if (pat) {                                                  \
        cairo_pattern_set_extend (cur_pat, CAIRO_EXTEND_NONE);  \
        cairo_pattern_set_matrix (pat, &mat);                   \
        cairo_pattern_set_filter (pat, CAIRO_FILTER_FAST);      \
        cairo_set_source (cr, pat);                             \
    }

KDE_NO_EXPORT void CairoPaintVisitor::visit (SMIL::Transition *trans) {
    float perc = trans->start_progress + (trans->end_progress - trans->start_progress)*cur_media->trans_gain;
    if (cur_media->trans_out_active)
        perc = 1.0 - perc;
    if (SMIL::Transition::Fade == trans->type) {
        CAIRO_SET_PATTERN_COND(cr, cur_pat, cur_mat)
        cairo_rectangle (cr, clip.x(), clip.y(), clip.width(), clip.height());
        opacity = perc;
    } else if (SMIL::Transition::BarWipe == trans->type) {
        IRect rect;
        if (SMIL::Transition::SubTopToBottom == trans->sub_type) {
            if (SMIL::Transition::dir_reverse == trans->direction) {
                int dy = (int) ((1.0 - perc) * clip.height ());
                rect = IRect (clip.x (), clip.y () + dy,
                        clip.width (), clip.height () - dy);
            } else {
                rect = IRect (clip.x (), clip.y (),
                        clip.width (), (int) (perc * clip.height ()));
            }
        } else {
            if (SMIL::Transition::dir_reverse == trans->direction) {
                int dx = (int) ((1.0 - perc) * clip.width ());
                rect = IRect (clip.x () + dx, clip.y (),
                        clip.width () - dx, clip.height ());
            } else {
                rect = IRect (clip.x (), clip.y (),
                        (int) (perc * clip.width ()), clip.height ());
            }
        }
        cairo_rectangle (cr, rect.x(), rect.y(), rect.width(), rect.height());
        CAIRO_SET_PATTERN_COND(cr, cur_pat, cur_mat)
    } else if (SMIL::Transition::PushWipe == trans->type) {
        int dx = 0, dy = 0;
        if (SMIL::Transition::SubFromTop == trans->sub_type)
            dy = -(int) ((1.0 - perc) * clip.height ());
        else if (SMIL::Transition::SubFromRight == trans->sub_type)
            dx = (int) ((1.0 - perc) * clip.width ());
        else if (SMIL::Transition::SubFromBottom == trans->sub_type)
            dy = (int) ((1.0 - perc) * clip.height ());
        else //if (SMIL::Transition::SubFromLeft == trans->sub_type)
            dx = -(int) ((1.0 - perc) * clip.width ());
        cairo_matrix_translate (&cur_mat, -dx, -dy);
        IRect rect = clip.intersect (IRect (clip.x () + dx, clip.y () + dy,
                    clip.width (), clip.height ()));
        cairo_rectangle (cr, rect.x(), rect.y(), rect.width(), rect.height());
        CAIRO_SET_PATTERN_COND(cr, cur_pat, cur_mat)
    } else if (SMIL::Transition::IrisWipe == trans->type) {
        CAIRO_SET_PATTERN_COND(cr, cur_pat, cur_mat)
        if (SMIL::Transition::SubDiamond == trans->sub_type) {
            cairo_rectangle (cr, clip.x(), clip.y(),clip.width(),clip.height());
            cairo_clip (cr);
            int dx = (int) (perc * clip.width ());
            int dy = (int) (perc * clip.height ());
            int mx = clip.x () + clip.width ()/2;
            int my = clip.y () + clip.height ()/2;
            cairo_new_path (cr);
            cairo_move_to (cr, mx, my - dy);
            cairo_line_to (cr, mx + dx, my);
            cairo_line_to (cr, mx, my + dy);
            cairo_line_to (cr, mx - dx, my);
            cairo_close_path (cr);
        } else { // SubRectangle
            int dx = (int) (0.5 * (1 - perc) * clip.width ());
            int dy = (int) (0.5 * (1 - perc) * clip.height ());
            cairo_rectangle (cr, clip.x () + dx, clip.y () + dy,
                    clip.width () - 2 * dx, clip.height () -2 * dy);
        }
    } else if (SMIL::Transition::ClockWipe == trans->type) {
        cairo_rectangle (cr, clip.x(), clip.y(), clip.width(), clip.height());
        cairo_clip (cr);
        int mx = clip.x () + clip.width ()/2;
        int my = clip.y () + clip.height ()/2;
        cairo_new_path (cr);
        cairo_move_to (cr, mx, my);
        float hw = 1.0 * clip.width ()/2;
        float hh = 1.0 * clip.height ()/2;
        float radius = sqrtf (hw * hw + hh * hh);
        float phi;
        switch (trans->sub_type) {
            case SMIL::Transition::SubClockwiseThree:
                phi = 0;
                break;
            case SMIL::Transition::SubClockwiseSix:
                phi = M_PI / 2;
                break;
            case SMIL::Transition::SubClockwiseNine:
                phi = M_PI;
                break;
            default: // Twelve
                phi = -M_PI / 2;
                break;
        }
        if (SMIL::Transition::dir_reverse == trans->direction)
            cairo_arc_negative (cr, mx, my, radius, phi, phi - 2 * M_PI * perc);
        else
            cairo_arc (cr, mx, my, radius, phi, phi + 2 * M_PI * perc);
        cairo_close_path (cr);
        CAIRO_SET_PATTERN_COND(cr, cur_pat, cur_mat)
    } else if (SMIL::Transition::BowTieWipe == trans->type) {
        cairo_rectangle (cr, clip.x(), clip.y(), clip.width(), clip.height());
        cairo_clip (cr);
        int mx = clip.x () + clip.width ()/2;
        int my = clip.y () + clip.height ()/2;
        cairo_new_path (cr);
        cairo_move_to (cr, mx, my);
        float hw = 1.0 * clip.width ()/2;
        float hh = 1.0 * clip.height ()/2;
        float radius = sqrtf (hw * hw + hh * hh);
        float phi;
        switch (trans->sub_type) {
            case SMIL::Transition::SubHorizontal:
                phi = 0;
                break;
            default: // Vertical
                phi = -M_PI / 2;
                break;
        }
        float dphi = 0.5 * M_PI * perc;
        cairo_arc (cr, mx, my, radius, phi - dphi, phi + dphi);
        cairo_close_path (cr);
        cairo_new_sub_path (cr);
        cairo_move_to (cr, mx, my);
        if (SMIL::Transition::SubHorizontal == trans->sub_type)
            cairo_arc (cr, mx, my, radius, M_PI + phi - dphi, M_PI + phi +dphi);
        else
            cairo_arc (cr, mx, my, radius, -phi - dphi, -phi + dphi);
        cairo_close_path (cr);
        CAIRO_SET_PATTERN_COND(cr, cur_pat, cur_mat)
    } else if (SMIL::Transition::EllipseWipe == trans->type) {
        cairo_rectangle (cr, clip.x(), clip.y(), clip.width(), clip.height());
        cairo_clip (cr);
        int mx = clip.x () + clip.width ()/2;
        int my = clip.y () + clip.height ()/2;
        float hw = (double) clip.width ()/2;
        float hh = (double) clip.height ()/2;
        float radius = sqrtf (hw * hw + hh * hh);
        cairo_save (cr);
        cairo_new_path (cr);
        cairo_translate (cr, (int) mx, (int) my);
        cairo_move_to (cr, - Single (radius), 0);
        if (SMIL::Transition::SubHorizontal == trans->sub_type)
            cairo_scale (cr, 1.0, 0.6);
        else if (SMIL::Transition::SubVertical == trans->sub_type)
            cairo_scale (cr, 0.6, 1.0);
        cairo_arc (cr, 0, 0, perc * radius, 0, 2 * M_PI);
        cairo_close_path (cr);
        cairo_restore (cr);
        CAIRO_SET_PATTERN_COND(cr, cur_pat, cur_mat)
    }
}

KDE_NO_EXPORT void CairoPaintVisitor::video (Mrl *m, Surface *s) {
    if (m->media_info &&
            m->media_info->media &&
            ActorAgent::AudioVideo == m->media_info->type) {
        AudioVideoActor *avm = static_cast<AudioVideoActor *> (m->media_info->media);
        if (s && avm && strcmp (m->nodeName (), "audio")) {
            s->xscale = s->yscale = 1; // either scale width/heigt or use bounds
            view_widget->setAudioVideoGeometry (s->toScreen (s->bounds.size),
                    NULL);
        } else {
            view_widget->setAudioVideoGeometry (IRect (-60, -60, 50, 50), NULL);
        }
    }
}

KDE_NO_EXPORT void CairoPaintVisitor::visit (SMIL::RefMediaType *ref) {
    Surface *s = ref->surface ();
    if (s && ref->external_tree)
        updateExternal (ref, s);
    else
        video (ref, s);
}

KDE_NO_EXPORT void CairoPaintVisitor::paint (SMIL::MediaType *mt, Surface *s,
        const IPoint &point, const IRect &rect) {
    cairo_save (cr);
    opacity = 1.0;
    cairo_matrix_init_translate (&cur_mat, -point.x, -point.y);
    cur_pat = cairo_pattern_create_for_surface (s->surface);
    if (mt->active_trans) {
        IRect clip_save = clip;
        clip = rect;
        penalty += rect.width() * rect.height() / (20 * 480);
        cur_media = mt;
        mt->active_trans->accept (this);
        clip = clip_save;
    } else {
        cairo_pattern_set_extend (cur_pat, CAIRO_EXTEND_NONE);
        cairo_pattern_set_matrix (cur_pat, &cur_mat);
        cairo_pattern_set_filter (cur_pat, CAIRO_FILTER_FAST);
        cairo_set_source (cr, cur_pat);
        cairo_rectangle (cr, rect.x(), rect.y(), rect.width(), rect.height());
    }
    opacity *= mt->opacity / 100.0;
    bool over = opacity < 0.99 ||
                CAIRO_CONTENT_COLOR != cairo_surface_get_content (s->surface);
    cairo_operator_t op;
    if (over) {
        penalty += rect.width () * rect.height () / (40 * 480);
        op = cairo_get_operator (cr);
        cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
    }
    if (opacity < 0.99) {
        cairo_clip (cr);
        cairo_paint_with_alpha (cr, opacity);
    } else {
        cairo_fill (cr);
    }
    if (over)
        cairo_set_operator (cr, op);
    cairo_pattern_destroy (cur_pat);
    cairo_restore (cr);
}

static Mrl *findActiveMrl (Node *n, bool *rp_or_smil) {
    Mrl *mrl = n->mrl ();
    if (mrl) {
        *rp_or_smil = (mrl->id >= SMIL::id_node_first &&
                mrl->id < SMIL::id_node_last) ||
            (mrl->id >= RP::id_node_first &&
             mrl->id < RP::id_node_last);
        if (*rp_or_smil ||
                (mrl->media_info &&
                 ActorAgent::AudioVideo == mrl->media_info->type))
            return mrl;
    }
    for (Node *c = n->firstChild (); c; c = c->nextSibling ())
        if (c->active ()) {
            Mrl *m = findActiveMrl (c, rp_or_smil);
            if (m)
                return m;
        }
    return NULL;
}

KDE_NO_EXPORT
void CairoPaintVisitor::updateExternal (SMIL::MediaType *av, SurfacePtr s) {
    bool rp_or_smil = false;
    Mrl *ext_mrl = findActiveMrl (av->external_tree.ptr (), &rp_or_smil);
    if (!ext_mrl)
        return;
    if (!rp_or_smil) {
        video (ext_mrl, s.ptr ());
        return;
    }
    IRect scr = matrix.toScreen (s->bounds);
    IRect clip_rect = clip.intersect (scr);
    if (clip_rect.isEmpty ())
        return;
    if (!s->surface || s->dirty) {
        Matrix m = matrix;
        m.translate (-scr.x (), -scr.y ());
        IRect r (clip_rect.x() - scr.x () - 1, clip_rect.y() - scr.y () - 1,
                clip_rect.width() + 3, clip_rect.height() + 3);
        if (!s->surface) {
            s->surface = cairo_surface_create_similar (cairo_surface,
                    CAIRO_CONTENT_COLOR_ALPHA, scr.width (), scr.height ());
            r = IRect (0, 0, scr.size);
        }
        CairoPaintVisitor visitor (view_widget, s->surface, m, r);
        ext_mrl->accept (&visitor);
        penalty += visitor.penalty;
        s->dirty = false;
    }
    paint (av, s.ptr (), scr.point, clip_rect);
}

KDE_NO_EXPORT void CairoPaintVisitor::visit (SMIL::ImageMediaType * img) {
    //debugLog() << "Visit " << img->nodeName() << " " << img->src << endl;
    if (!img->media_info)
        return;
    Surface *s = img->surface ();
    if (!s)
        return;
    if (img->external_tree) {
        updateExternal (img, s);
        return;
    }
    if (!img->media_info->media)
        return;

    IRect scr = matrix.toScreen (s->bounds);
    IRect clip_rect = clip.intersect (scr);
    if (clip_rect.isEmpty ())
        return;

    ImageActor *im = static_cast <ImageActor *> (img->media_info->media);
    ImageData *id = im ? im->cached_img.ptr () : NULL;
    if (id && id->flags == ImageData::ImageScalable)
        penalty += im->render (cairo_surface, scr.size);
    if (!id || im->isEmpty () || img->size.isEmpty ()) {
        s->remove();
        return;
    }
    if (!s->surface || s->dirty)
        id->copyImage (s, SSize (scr.width (), scr.height ()), cairo_surface, &penalty, img->pan_zoom);
    paint (img, s, scr.point, clip_rect);
    s->dirty = false;
}

static void calculateTextDimensions (PangoFontDescription *desc,
        const char *text, Single w, Single h,
        int *pxw, int *pxh, bool markup_text) {
    cairo_surface_t *img_surf = cairo_image_surface_create (
            CAIRO_FORMAT_RGB24, (int) w, h);
    cairo_t *cr_txt = cairo_create (img_surf);
    PangoLayout *layout = pango_cairo_create_layout (cr_txt);
    pango_layout_set_width (layout, 1.0 * w * PANGO_SCALE);
    pango_layout_set_wrap (layout, PANGO_WRAP_WORD_CHAR);
    if (markup_text)
        pango_layout_set_markup (layout, text, -1);
    else
        pango_layout_set_text (layout, text, -1);
    pango_layout_set_font_description (layout, desc);

    pango_cairo_show_layout (cr_txt, layout);
    pango_layout_get_pixel_size (layout, pxw, pxh);

    g_object_unref (layout);
    cairo_destroy (cr_txt);
    cairo_surface_destroy (img_surf);
}

KDE_NO_EXPORT void CairoPaintVisitor::visit (SMIL::TextMediaType * txt) {
    if (!txt->media_info || !txt->media_info->media)
        return;
    TextActor *tm = static_cast <TextActor *> (txt->media_info->media);
    Surface *s = txt->surface ();
    if (!s)
        return;
    if (!s->surface) {
        txt->size = SSize ();
        s->bounds = txt->calculateBounds ();
    }
    IRect scr = matrix.toScreen (s->bounds);
    if (!s->surface) {

        int w = scr.width ();
        int pxw, pxh;
        Single ft_size = w * txt->font_size / (double)s->bounds.width ();
        const char *text = (const char *) tm->text;

        PangoFontDescription *desc = pango_font_description_new ();
        pango_font_description_set_family (desc, "Sans");
        pango_font_description_set_absolute_size (desc, PANGO_SCALE * ft_size);

        calculateTextDimensions (desc, text, w, 2 * ft_size, &pxw, &pxh, false);
        unsigned int bg_alpha = txt->background_color & 0xff000000;
        s->surface = cairo_surface_create_similar (cairo_surface,
                bg_alpha < 0xff000000
                    ? CAIRO_CONTENT_COLOR_ALPHA
                    : CAIRO_CONTENT_COLOR,
                pxw, pxh);
        cairo_t *cr_txt = cairo_create (s->surface);

        if (bg_alpha) {
            if (bg_alpha < 0xff000000)
                CAIRO_SET_SOURCE_ARGB (cr_txt, txt->background_color);
            else
                CAIRO_SET_SOURCE_RGB (cr_txt, txt->background_color);
            cairo_paint (cr_txt);
        }

        CAIRO_SET_SOURCE_RGB (cr_txt, txt->font_color);
        PangoLayout *layout = pango_cairo_create_layout (cr_txt);
        pango_layout_set_width (layout, 1.0 * w * PANGO_SCALE);
        pango_layout_set_wrap (layout, PANGO_WRAP_WORD_CHAR);
        pango_layout_set_text (layout, text, -1);
        pango_layout_set_font_description (layout, desc);

        pango_cairo_show_layout (cr_txt, layout);

        pango_font_description_free (desc);
        g_object_unref (layout);
        cairo_destroy (cr_txt);

        // update bounds rect
        SRect rect = matrix.toUser (IRect (scr.point, ISize (pxw, pxh)));
        txt->size = rect.size;
        s->bounds = txt->calculateBounds ();

        // update coord. for painting below
        scr = matrix.toScreen (s->bounds);
    }
    IRect clip_rect = clip.intersect (scr);
    if (!clip_rect.isEmpty ())
        paint (txt, s, scr.point, clip_rect);
    s->dirty = false;
}

KDE_NO_EXPORT void CairoPaintVisitor::visit (SMIL::Brush * brush) {
    //debugLog() << "Visit " << brush->nodeName() << endl;
    Surface *s = brush->surface ();
    if (s) {
        opacity = 1.0;
        IRect clip_rect = clip.intersect (matrix.toScreen (s->bounds));
        if (clip_rect.isEmpty ())
            return;
        cairo_save (cr);
        if (brush->active_trans) {
            cur_media = brush;
            cur_pat = NULL;
            brush->active_trans->accept (this);
        } else {
            cairo_rectangle (cr, clip_rect.x (), clip_rect.y (),
                    clip_rect.width (), clip_rect.height ());
        }
        opacity *= brush->opacity / 100.0;
        if (opacity < 0.99) {
            cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
            cairo_set_source_rgba (cr,
                    1.0 * ((brush->color >> 16) & 0xff) / 255,
                    1.0 * ((brush->color >> 8) & 0xff) / 255,
                    1.0 * (brush->color & 0xff) / 255,
                    opacity);
        } else {
            CAIRO_SET_SOURCE_RGB (cr, brush->color);
        }
        cairo_fill (cr);
        if (opacity < 0.99)
            cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
        s->dirty = false;
        cairo_restore (cr);
    }
}

KDE_NO_EXPORT void CairoPaintVisitor::visit (SMIL::SmilText *txt) {
    Surface *s = txt->surface ();
    if (!s)
        return;

    SRect rect = s->bounds;
    IRect scr = matrix.toScreen (rect);

    if (!s->surface) {

        int w = scr.width ();
        Single ft_size = w * 14 / (double)s->bounds.width ();
        int pxw, pxh;
        String rtext = txt->richText ();
        const char *text = (const char *) rtext;

        PangoFontDescription *desc = pango_font_description_new ();
        pango_font_description_set_family (desc, "Sans");
        pango_font_description_set_absolute_size (desc, PANGO_SCALE * ft_size);

        calculateTextDimensions (desc, text, w, 2 * ft_size, &pxw, &pxh, true);

        s->surface = cairo_surface_create_similar (cairo_surface,
                CAIRO_CONTENT_COLOR_ALPHA, (int) w, pxh);
        cairo_t *cr_txt = cairo_create (s->surface);

        CAIRO_SET_SOURCE_RGB (cr_txt, 0);
        PangoLayout *layout = pango_cairo_create_layout (cr_txt);
        pango_layout_set_width (layout, 1.0 * w * PANGO_SCALE);
        pango_layout_set_wrap (layout, PANGO_WRAP_WORD_CHAR);
        pango_layout_set_markup (layout, text, -1);
        pango_layout_set_font_description (layout, desc);

        pango_cairo_show_layout (cr_txt, layout);

        pango_font_description_free (desc);
        g_object_unref (layout);
        cairo_destroy (cr_txt);

        // update bounds rect
        s->bounds = matrix.toUser (IRect (scr.point, ISize (w, pxh)));

        // update coord. for painting below
        scr = matrix.toScreen (s->bounds);
    }
    IRect clip_rect = clip.intersect (scr);
    if (!clip_rect.isEmpty ()) {
        cairo_save (cr);
        cairo_matrix_init_translate (&cur_mat, -scr.x (), -scr.y ());
        cairo_pattern_t *pat = cairo_pattern_create_for_surface (s->surface);
        cairo_pattern_set_extend (pat, CAIRO_EXTEND_NONE);
        cairo_pattern_set_matrix (pat, &cur_mat);
        cairo_pattern_set_filter (pat, CAIRO_FILTER_FAST);
        cairo_set_source (cr, pat);
        cairo_rectangle(cr, clip_rect.x (), clip_rect.y (),
                clip_rect.width (), clip_rect.height ());
        cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
        cairo_fill (cr);
        cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
        cairo_pattern_destroy (pat);
        cairo_restore (cr);
    }
    s->dirty = false;
}

KDE_NO_EXPORT void CairoPaintVisitor::visit (RP::Imfl * imfl) {
    if (imfl->surface ()) {
        cairo_save (cr);
        Matrix m = matrix;
        IRect scr = matrix.toScreen (SRect (0, 0, imfl->rp_surface->bounds.size));
        int w = scr.width ();
        int h = scr.height ();
        cairo_rectangle (cr, scr.x (), scr.y (), w, h);
        //cairo_clip (cr);
        cairo_translate (cr, scr.x (), scr.y ());
        cairo_scale (cr, 1.0*w/(double)imfl->size.width, 1.0*h/(double)imfl->size.height);
        if (imfl->needs_scene_img)
            cairo_push_group (cr);
        for (NodePtr n = imfl->firstChild (); n; n = n->nextSibling ())
            if (n->state >= Node::state_began &&
                    n->state < Node::state_deactivated) {
                RP::TimingsBase * tb = convertNode<RP::TimingsBase>(n);
                switch (n->id) {
                    case RP::id_node_viewchange:
                        if (!(int)tb->srcw)
                            tb->srcw = imfl->size.width;
                        if (!(int)tb->srch)
                            tb->srch = imfl->size.height;
                        // fall through
                    case RP::id_node_crossfade:
                    case RP::id_node_fadein:
                    case RP::id_node_fadeout:
                    case RP::id_node_fill:
                    case RP::id_node_wipe:
                        if (!(int)tb->w)
                            tb->w = imfl->size.width;
                        if (!(int)tb->h)
                            tb->h = imfl->size.height;
                        n->accept (this);
                        break;
                }
            }
        if (imfl->needs_scene_img) {
            cairo_pattern_t * pat = cairo_pop_group (cr);
            cairo_pattern_set_extend (pat, CAIRO_EXTEND_NONE);
            cairo_set_source (cr, pat);
            cairo_paint (cr);
            cairo_pattern_destroy (pat);
        }
        cairo_restore (cr);
        matrix = m;
    }
}

KDE_NO_EXPORT void CairoPaintVisitor::visit (RP::Fill * fi) {
    //kdDebug() << "Visit " << fi->nodeName() << endl;
    CAIRO_SET_SOURCE_RGB (cr, fi->color);
    if ((int)fi->w && (int)fi->h) {
        cairo_rectangle (cr, fi->x, fi->y, fi->w, fi->h);
        cairo_fill (cr);
    }
}

KDE_NO_EXPORT void CairoPaintVisitor::visit (RP::Fadein * fi) {
    //debugLog() << "Visit " << fi->nodeName() << endl;
    if (fi->target && fi->target->id == RP::id_node_image) {
        RP::Image *img = convertNode <RP::Image> (fi->target);
        ImageActor *im = img && img->media_info
            ? static_cast <ImageActor*> (img->media_info->media) : NULL;
        if (im && img->surface ()) {
            Single sx = fi->srcx, sy = fi->srcy, sw = fi->srcw, sh = fi->srch;
            if (!(int)sw)
                sw = img->size.width;
            if (!(int)sh)
                sh = img->size.height;
            if ((int)fi->w && (int)fi->h && (int)sw && (int)sh) {
                if (!img->img_surface->surface)
                    im->cached_img->copyImage (img->img_surface,
                            img->size, cairo_surface, &penalty);
                cairo_matrix_t matrix;
                cairo_matrix_init_identity (&matrix);
                float scalex = 1.0 * sw / fi->w;
                float scaley = 1.0 * sh / fi->h;
                cairo_matrix_scale (&matrix, scalex, scaley);
                cairo_matrix_translate (&matrix,
                        1.0*sx/scalex - (double)fi->x,
                        1.0*sy/scaley - (double)fi->y);
                cairo_save (cr);
                cairo_rectangle (cr, fi->x, fi->y, fi->w, fi->h);
                cairo_pattern_t *pat = cairo_pattern_create_for_surface (img->img_surface->surface);
                cairo_pattern_set_extend (pat, CAIRO_EXTEND_NONE);
                cairo_pattern_set_matrix (pat, &matrix);
                cairo_set_source (cr, pat);
                cairo_clip (cr);
                cairo_paint_with_alpha (cr, 1.0 * fi->progress / 100);
                cairo_restore (cr);
                cairo_pattern_destroy (pat);
            }
        }
    }
}

KDE_NO_EXPORT void CairoPaintVisitor::visit (RP::Fadeout * fo) {
    //kdDebug() << "Visit " << fo->nodeName() << endl;
    if (fo->progress > 0) {
        CAIRO_SET_SOURCE_RGB (cr, fo->to_color);
        if ((int)fo->w && (int)fo->h) {
            cairo_save (cr);
            cairo_rectangle (cr, fo->x, fo->y, fo->w, fo->h);
            cairo_clip (cr);
            cairo_paint_with_alpha (cr, 1.0 * fo->progress / 100);
            cairo_restore (cr);
        }
    }
}

KDE_NO_EXPORT void CairoPaintVisitor::visit (RP::Crossfade * cf) {
    //debugLog() << "Visit " << cf->nodeName() << endl;
    if (cf->target && cf->target->id == RP::id_node_image) {
        RP::Image *img = convertNode <RP::Image> (cf->target);
        ImageActor *im = img && img->media_info
            ? static_cast <ImageActor*> (img->media_info->media) : NULL;
        if (im && img->surface ()) {
            Single sx = cf->srcx, sy = cf->srcy, sw = cf->srcw, sh = cf->srch;
            if (!(int)sw)
                sw = img->size.width;
            if (!(int)sh)
                sh = img->size.height;
            if ((int)cf->w && (int)cf->h && (int)sw && (int)sh) {
                if (!img->img_surface->surface)
                    im->cached_img->copyImage (img->img_surface,
                            img->size, cairo_surface, &penalty);
                cairo_save (cr);
                cairo_matrix_t matrix;
                cairo_matrix_init_identity (&matrix);
                float scalex = 1.0 * sw / cf->w;
                float scaley = 1.0 * sh / cf->h;
                cairo_matrix_scale (&matrix, scalex, scaley);
                cairo_matrix_translate (&matrix,
                        1.0*sx/scalex - (double)cf->x,
                        1.0*sy/scaley - (double)cf->y);
                cairo_rectangle (cr, cf->x, cf->y, cf->w, cf->h);
                cairo_pattern_t *pat = cairo_pattern_create_for_surface (img->img_surface->surface);
                cairo_pattern_set_extend (pat, CAIRO_EXTEND_NONE);
                cairo_pattern_set_matrix (pat, &matrix);
                cairo_set_source (cr, pat);
                cairo_clip (cr);
                cairo_paint_with_alpha (cr, 1.0 * cf->progress / 100);
                cairo_restore (cr);
                cairo_pattern_destroy (pat);
            }
        }
    }
}

KDE_NO_EXPORT void CairoPaintVisitor::visit (RP::Wipe * wipe) {
    //debugLog() << "Visit " << wipe->nodeName() << endl;
    if (wipe->target && wipe->target->id == RP::id_node_image) {
        RP::Image *img = convertNode <RP::Image> (wipe->target);
        ImageActor *im = img && img->media_info
            ? static_cast <ImageActor*> (img->media_info->media) : NULL;
        if (im && img->surface ()) {
            Single x = wipe->x, y = wipe->y;
            Single tx = x, ty = y;
            Single w = wipe->w, h = wipe->h;
            Single sx = wipe->srcx, sy = wipe->srcy, sw = wipe->srcw, sh = wipe->srch;
            if (!(int)sw)
                sw = img->size.width;
            if (!(int)sh)
                sh = img->size.height;
            if (wipe->direction == RP::Wipe::dir_right) {
                Single dx = w * 1.0 * wipe->progress / 100;
                tx = x -w + dx;
                w = dx;
            } else if (wipe->direction == RP::Wipe::dir_left) {
                Single dx = w * 1.0 * wipe->progress / 100;
                tx = x + w - dx;
                x = tx;
                w = dx;
            } else if (wipe->direction == RP::Wipe::dir_down) {
                Single dy = h * 1.0 * wipe->progress / 100;
                ty = y - h + dy;
                h = dy;
            } else if (wipe->direction == RP::Wipe::dir_up) {
                Single dy = h * 1.0 * wipe->progress / 100;
                ty = y + h - dy;
                y = ty;
                h = dy;
            }

            if ((int)w && (int)h) {
                if (!img->img_surface->surface)
                    im->cached_img->copyImage (img->img_surface,
                            img->size, cairo_surface, &penalty);
                cairo_matrix_t matrix;
                cairo_matrix_init_identity (&matrix);
                float scalex = 1.0 * sw / wipe->w;
                float scaley = 1.0 * sh / wipe->h;
                cairo_matrix_scale (&matrix, scalex, scaley);
                cairo_matrix_translate (&matrix,
                        1.0*sx/scalex - (double)tx,
                        1.0*sy/scaley - (double)ty);
                cairo_pattern_t *pat = cairo_pattern_create_for_surface (img->img_surface->surface);
                cairo_pattern_set_extend (pat, CAIRO_EXTEND_NONE);
                cairo_pattern_set_matrix (pat, &matrix);
                cairo_set_source (cr, pat);
                cairo_rectangle (cr, x, y, w, h);
                cairo_fill (cr);
                cairo_pattern_destroy (pat);
            }
        }
    }
}

KDE_NO_EXPORT void CairoPaintVisitor::visit (RP::ViewChange * vc) {
    //kdDebug() << "Visit " << vc->nodeName() << endl;
    if (vc->unfinished () || vc->progress < 100) {
        cairo_pattern_t * pat = cairo_pop_group (cr); // from imfl
        cairo_pattern_set_extend (pat, CAIRO_EXTEND_NONE);
        cairo_push_group (cr);
        cairo_save (cr);
        cairo_set_source (cr, pat);
        cairo_paint (cr);
        if ((int)vc->w && (int)vc->h && (int)vc->srcw && (int)vc->srch) {
            cairo_matrix_t matrix;
            cairo_matrix_init_identity (&matrix);
            float scalex = 1.0 * vc->srcw / vc->w;
            float scaley = 1.0 * vc->srch / vc->h;
            cairo_matrix_scale (&matrix, scalex, scaley);
            cairo_matrix_translate (&matrix,
                    1.0*vc->srcx/scalex - (double)vc->x,
                    1.0*vc->srcy/scaley - (double)vc->y);
            cairo_pattern_set_matrix (pat, &matrix);
            cairo_set_source (cr, pat);
            cairo_rectangle (cr, vc->x, vc->y, vc->w, vc->h);
            cairo_fill (cr);
        }
        cairo_pattern_destroy (pat);
        cairo_restore (cr);
    }
}

//-----------------------------------------------------------------------------

namespace KMPlayer {

class KMPLAYER_NO_EXPORT MouseVisitor : public Visitor {
    ViewArea *view_area;
    Matrix matrix;
    NodePtrW source;
    MessageType event;
    int x, y;
    bool handled;
    bool bubble_up;
public:
    MouseVisitor (ViewArea *v, MessageType evt, int x, int y);
    KDE_NO_CDTOR_EXPORT ~MouseVisitor () {}
    using Visitor::visit;
    void visit (Node * n);
    void visit (Element *);
    void visit (SMIL::Smil *);
    void visit (SMIL::Layout *);
    void visit (SMIL::RegionBase *);
    void visit (SMIL::MediaType * n);
    void visit (SMIL::SmilText * n);
    void visit (SMIL::Anchor *);
    void visit (SMIL::Area *);

};

} // namespace

KDE_NO_CDTOR_EXPORT
MouseVisitor::MouseVisitor (ViewArea *v, MessageType evt, int a, int b)
  : view_area (v), event (evt), x (a), y (b),
    handled (false), bubble_up (false) {
}

KDE_NO_EXPORT void MouseVisitor::visit (Node * n) {
    debugLog () << "Mouse event ignored for " << n->nodeName () << endl;
}

KDE_NO_EXPORT void MouseVisitor::visit (SMIL::Smil *s) {
    if (s->active () && s->layout_node)
        s->layout_node->accept (this);
}

KDE_NO_EXPORT void MouseVisitor::visit (SMIL::Layout * layout) {
    Surface *s = (Surface *) layout->root_layout->role (RoleDisplay);
    if (s) {
        Matrix m = matrix;
        SRect rect = s->bounds;
        matrix = Matrix (rect.x(), rect.y(), s->xscale, s->yscale);
        matrix.transform (m);

        NodePtr node_save = source;
        source = layout;
        for (NodePtr r = layout->firstChild (); r; r = r->nextSibling ()) {
            if (r->id == SMIL::id_node_region)
                r->accept (this);
            if (!source || !source->active ())
                break;
        }
        source = node_save;

        matrix = m;
    }
}

KDE_NO_EXPORT void MouseVisitor::visit (SMIL::RegionBase *region) {
    Surface *s = (Surface *) region->role (RoleDisplay);
    if (s) {
        SRect rect = s->bounds;
        IRect scr = matrix.toScreen (rect);
        int rx = scr.x(), ry = scr.y(), rw = scr.width(), rh = scr.height();
        handled = false;
        bool inside = x > rx && x < rx+rw && y > ry && y< ry+rh;
        if (!inside && (event == MsgEventClicked || !region->has_mouse))
            return;

        if (event == MsgEventClicked && !s->virtual_size.isEmpty () &&
                x > rx + rw - REGION_SCROLLBAR_WIDTH) {
            const int sbh = rh - REGION_SCROLLBAR_WIDTH;
            const int vy = s->virtual_size.height;
            const int knob_h = sbh * rh / vy;
            int knob_y = y - ry - 0.5 * knob_h;
            if (knob_y < 0)
                knob_y = 0;
            else if (knob_y + knob_h > sbh)
                knob_y = sbh - knob_h;
            s->y_scroll = vy * knob_y / sbh;
            view_area->scheduleRepaint (scr);
            return;
        }
        if (event == MsgEventClicked && !s->virtual_size.isEmpty () &&
                y > ry + rh - REGION_SCROLLBAR_WIDTH) {
            const int sbw = rw - REGION_SCROLLBAR_WIDTH;
            const int vw = s->virtual_size.width;
            const int knob_w = sbw * rw / vw;
            int knob_x = x - rx - 0.5 * knob_w;
            if (knob_x < 0)
                knob_x = 0;
            else if (knob_x + knob_w > sbw)
                knob_x = sbw - knob_w;
            s->x_scroll = vw * knob_x / sbw;
            view_area->scheduleRepaint (scr);
            return;
        }

        Matrix m = matrix;
        matrix = Matrix (rect.x(), rect.y(), 1.0, 1.0);
        matrix.transform (m);
        if (!s->virtual_size.isEmpty ())
            matrix.translate (-s->x_scroll, -s->y_scroll);
        bubble_up = false;

        bool child_handled = false;
        if (inside)
            for (NodePtr r = region->firstChild (); r; r = r->nextSibling ()) {
                r->accept (this);
                child_handled |= handled;
                if (!source || !source->active ())
                    break;
            }
        child_handled &= !bubble_up;
        bubble_up = false;

        MessageType saved_event = event;
        if (source && source->active ()) {
            bool notify_receivers = !child_handled;
            bool pass_event = !child_handled;
            if (event == MsgEventPointerMoved) {
                pass_event = true; // always pass move events
                if (region->has_mouse && (!inside || child_handled)) {
                    notify_receivers = true;
                    region->has_mouse = false;
                    event = MsgEventPointerOutBounds;
                } else if (inside && !child_handled && !region->has_mouse) {
                    notify_receivers = true;
                    region->has_mouse = true;
                    event = MsgEventPointerInBounds;
                }
            }// else // MsgEventClicked
            if (notify_receivers) {
                Posting mouse_event (region, region, event);
                region->deliver (event, &mouse_event);
            }
            if (pass_event) {
                ConnectionList *nl = nodeMessageReceivers (region, MsgSurfaceAttach);
                if (nl) {
                    for (Connection *c = nl->first(); c; c = nl->next ()) {
                        if (c->connecter)
                            c->connecter->accept (this);
                        if (!source || !source->active ())
                            break;
                    }
                }
            }
        }
        event = saved_event;
        handled = inside;
        matrix = m;
    }
}

static void followLink (SMIL::LinkingBase * link) {
    debugLog () << "link to " << link->href << " clicked" << endl;
    if (link->href.startsWith ("#")) {
        SMIL::Smil * s = SMIL::Smil::findSmilNode (link);
        if (s)
            s->jump (link->href.mid (1));
        else
            errorLog() << "In document jumps smil not found" << endl;
    } else {
        PlayListNotify *notify = link->document ()->notify_listener;
        if (notify && !link->target.isEmpty ()) {
             notify->openUrl (link->href, link->target, String ());
        } else {
            NodePtr n = link;
            for (NodePtr p = link->parentNode (); p; p = p->parentNode ()) {
                if (n->mrl () && n->mrl ()->opener == p) {
                    p->setState (Node::state_deferred);
                    p->mrl ()->setParam (StringPool::attr_src, link->href, 0L);
                    p->activate ();
                    break;
                }
                n = p;
            }
        }
    }
}

KDE_NO_EXPORT void MouseVisitor::visit (SMIL::Anchor * anchor) {
    if (event == MsgEventPointerMoved)
        ;// set cursor to hand
    else if (event == MsgEventClicked)
        followLink (anchor);
}

KDE_NO_EXPORT void MouseVisitor::visit (SMIL::Area * area) {
    NodePtr n = area->parentNode ();
    Surface *s = (Surface *) n->role (RoleDisplay);
    if (s) {
        SRect rect = s->bounds;
        IRect scr = matrix.toScreen (rect);
        int w = scr.width (), h = scr.height ();
        if (area->nr_coords > 1) {
            Single left = area->coords[0].size (rect.width ());
            Single top = area->coords[1].size (rect.height ());
            matrix.getXY (left, top);
            if (x < left || x > left + w || y < top || y > top + h)
                return;
            if (area->nr_coords > 3) {
                Single right = area->coords[2].size (rect.width ());
                Single bottom = area->coords[3].size (rect.height ());
                matrix.getXY (right, bottom);
                if (x > right || y > bottom)
                    return;
            }
        }
        if (event == MsgEventPointerMoved)
            ;//cursor.setShape (Qt::PointingHandCursor);
        else {
            ConnectionList *nl = nodeMessageReceivers (area, event);
            if (nl)
                for (Connection *c = nl->first(); c; c = nl->next ()) {
                    if (c->connecter)
                        c->connecter->accept (this);
                    if (!source || !source->active ())
                        return;
                }
            if (event == MsgEventClicked && !area->href.isEmpty ())
                followLink (area);
        }
    }
}

KDE_NO_EXPORT void MouseVisitor::visit (Element *elm) {
    Runtime *rt = (Runtime *) elm->role (RoleTiming);
    if (rt) {
        Posting mouse_event (source, elm, event);
        rt->message (event, &mouse_event);
    }
}

KDE_NO_EXPORT void MouseVisitor::visit (SMIL::MediaType *mt) {
    if (mt->sensitivity == SMIL::MediaType::sens_transparent) {
        bubble_up = true;
        return;
    }
    Surface *s = mt->surface ();
    if (!s)
        return;
    if (s->node && s->node.ptr () != mt) {
        s->node->accept (this);
        return;
    }
    SRect rect = s->bounds;
    IRect scr = matrix.toScreen (rect);
    int rx = scr.x(), ry = scr.y(), rw = scr.width(), rh = scr.height();
    const bool inside = x > rx && x < rx+rw && y > ry && y< ry+rh;
    if (!inside && (event == MsgEventClicked || inside == mt->has_mouse))
        return;
    mt->has_mouse = inside;

    ConnectionList *nl = nodeMessageReceivers (mt,
            event == MsgEventPointerMoved ? MsgSurfaceAttach : event);
    if (nl) {
        NodePtr node_save = source;
        source = mt;
        for (Connection *c = nl->first(); c; c = nl->next ()) {
            if (c->connecter && c->connecter.ptr () != mt)
                c->connecter->accept (this);
            if (!source || !source->active ())
                break;
        }
        source = node_save;
        if (!source || !source->active ())
            return;
    }

    if (event != MsgEventPointerMoved)
        visit (static_cast <Element *> (mt));
    if (event != MsgEventPointerInBounds && event != MsgEventPointerOutBounds) {
      SMIL::RegionBase *r=convertNode<SMIL::RegionBase>(mt->region_node);
      if (r && r->role (RoleDisplay) &&
              r->id != SMIL::id_node_smil &&
              r->region_surface->node && r != r->region_surface->node.ptr ())
          return r->region_surface->node->accept (this);
    }
}

KDE_NO_EXPORT void MouseVisitor::visit (SMIL::SmilText *st) {
    if (MsgEventClicked == event) {
        ConnectionList *nl = nodeMessageReceivers (st, event);
        if (nl)
            for (Connection *c = nl->first(); c; c = nl->next ()) {
                if (c->connecter && c->connecter.ptr () != st)
                    c->connecter->accept (this);
                if (!source || !source->active ())
                    return;
            }
    }
}
//-----------------------------------------------------------------------------

namespace KMPlayer {
class KMPLAYER_NO_EXPORT ViewAreaPrivate {
public:
    ViewAreaPrivate (ViewArea *v)
        : m_view_area (v), dpy (NULL), backing_store (0) {}

    ~ViewAreaPrivate() {
        destroyBackingStore ();
    }

    Display *display () {
        if (!dpy)
            dpy = gdk_x11_display_get_xdisplay (gtk_widget_get_display (
                        m_view_area->viewWidget ()));
        return dpy;
    }
    void resizeSurface (Surface *s) {
        int w = m_view_area->width ();
        int h = m_view_area->height ();
        if (s->surface && (w != width || h != height)) {
                //cairo_xlib_surface_set_size (s->surface, w, h);
            Window wid = gdk_x11_drawable_get_xid (
                    GDK_DRAWABLE (m_view_area->viewWidget ()->window));
            destroyBackingStore ();
            backing_store = XCreatePixmap (dpy, wid, w, h,
                    gdk_drawable_get_depth (
                        GDK_DRAWABLE (m_view_area->viewWidget ()->window)));
            cairo_xlib_surface_set_drawable(s->surface, backing_store, w,h);
            width = w;
            height = h;
        }
    }
    cairo_surface_t *createSurface (int w, int h) {
        Window wid = gdk_x11_drawable_get_xid (
                GDK_DRAWABLE (m_view_area->viewWidget ()->window));
        Display *d = display ();
        destroyBackingStore ();
        width = w;
        height = h;
        backing_store = XCreatePixmap (d, wid, w, h, gdk_drawable_get_depth (
                    GDK_DRAWABLE (m_view_area->viewWidget ()->window)));
        return cairo_xlib_surface_create (d, backing_store,
                gdk_x11_visual_get_xvisual (gtk_widget_get_visual (
                        m_view_area->viewWidget ())), w, h);
        /*return cairo_xlib_surface_create_with_xrender_format (
            QX11Info::display (),
            id,
            DefaultScreenOfDisplay (QX11Info::display ()),
            XRenderFindVisualFormat (QX11Info::display (),
                DefaultVisual (QX11Info::display (),
                    DefaultScreen (QX11Info::display ()))),
            w, h);*/
    }
    void swapBuffer (int sx, int sy, int sw, int sh, int dx, int dy) {
        Window wid = gdk_x11_drawable_get_xid (
                GDK_DRAWABLE (m_view_area->viewWidget ()->window));
        GC gc = XCreateGC (display (), backing_store, 0, NULL);
        XCopyArea (dpy, backing_store, wid, gc, sx, sy, sw, sh, dx, dy);
        XFreeGC (dpy, gc);
        XFlush (dpy);
    }
    void destroyBackingStore () {
        if (backing_store)
            XFreePixmap (display (), backing_store);
        backing_store = 0;
    }
    ViewArea *m_view_area;
    Display *dpy;
    Drawable backing_store;
    int width;
    int height;
};

class KMPLAYER_NO_EXPORT RepaintUpdater {
public:
    RepaintUpdater (Node *n, RepaintUpdater *nx) : node (n), next (nx) {}

    NodePtrW node;
    RepaintUpdater *next;
};

}

//-----------------------------------------------------------------------------

KDE_NO_CDTOR_EXPORT ViewArea::ViewArea ()
 : d (new ViewAreaPrivate (this)),
   controls (NULL),
   controls_timer (0),
   m_updaters (NULL),
   aspect (0.0),
   widget (0L),
   video (0L),
   video_input (0L),
   surface (new Surface (this)),
   last_repaint_time (0),
   next_repaint_time (0),
   penalty (0),
   cur_timeout (0),
   m_repaint_timer (0),
   m_fullscreen (false),
   m_event_filter_on (false),
   m_updaters_enabled (true),
   m_controls_overlap (true),
   m_video_filter (false) {
}

KDE_NO_CDTOR_EXPORT ViewArea::~ViewArea () {
    reset ();
    delete d;
}

KDE_NO_EXPORT void ViewArea::reset () {
    if (m_repaint_timer)
        g_source_remove (m_repaint_timer);
    m_repaint_timer = 0;
    while (m_updaters) {
        RepaintUpdater *tmp = m_updaters;
        m_updaters = m_updaters->next;
        delete tmp;
    }
    m_updaters = NULL;
    aspect = 0,0;
}

KDE_NO_EXPORT void ViewArea::mousePressEvent (int x, int y) {
    if (surface->node) {
        struct timeval tv;
        surface->node->document ()->timeOfDay (tv); // update last_event_time
        MouseVisitor visitor (this, MsgEventClicked, x, y);
        surface->node->accept (&visitor);
    }
}

KDE_NO_EXPORT void ViewArea::mouseDoubleClickEvent () {
}

KDE_NO_EXPORT void ViewArea::syncVisual (const IRect & rect) {
    int ex = rect.x ();
    if (ex > 0)
        ex--;
    int ey = rect.y ();
    if (ey > 0)
        ey--;
    int ew = rect.width () + 2;
    int eh = rect.height () + 2;
    if (!surface->surface)
        surface->surface = d->createSurface (width (), height ());
    {
        CairoPaintVisitor visitor (this, surface->surface,
                Matrix (surface->bounds.x(), surface->bounds.y(), 1.0, 1.0),
                IRect (ex, ey, ew, eh), true);
        if (surface->node)
            surface->node->accept (&visitor);
        penalty = visitor.penalty;
    }
    cairo_surface_flush (surface->surface);
    d->swapBuffer (ex, ey, ew, eh, ex, ey);

    m_repaint_rect = IRect ();
}

KDE_NO_EXPORT void ViewArea::paintEvent (const IRect & rect) {
    scheduleRepaint (rect);
}

KDE_NO_EXPORT void ViewArea::updateSurfaceBounds () {
    Single x, y, w = width (), h = height ();
    Mrl *mrl = surface->node ? surface->node->mrl () : NULL;
    if (w > 0 && h > 0 && mrl && !mrl->size.isEmpty ()) {
        double wasp = (double) w / h;
        double masp = (double) mrl->size.width / mrl->size.height;
        if (wasp > masp) {
            Single tmp = w;
            w = masp * h;
            x += (tmp - w) / 2;
        } else {
            Single tmp = h;
            h = Single (w / masp);
            y += (tmp - h) / 2;
        }
        surface->xscale = 1.0 * w / mrl->size.width;
        surface->yscale = 1.0 * h / mrl->size.height;
    } else {
        surface->xscale = 1.0;
        surface->yscale = 1.0;
    }
    if (surface->node) {
        surface->bounds = SRect (x, y, w, h);
        surface->node->message (MsgSurfaceBoundsUpdate, (void *) true);
    } else {
        surface->resize (SRect (x, y, w, h), true);
    }
    scheduleRepaint (IRect (0, 0, width (), height ()));
}

KDE_NO_EXPORT void ViewArea::resizeEvent () {
    if (!video)
        return;
    Single x, y, w = width (), h = height ();
    if (w <= 0 || h <= 0)
        return;
    if (surface->node)
        d->destroyBackingStore ();
    updateSurfaceBounds ();
    d->resizeSurface (surface.ptr ());
    if (controls_timer && !m_controls_overlap)
        h -= 60;
    if (video_node && Mrl::SingleMode == video_node->mrl ()->view_mode)
        setAudioVideoGeometry (IRect (x, y, w, h), 0L);
}

KDE_NO_EXPORT void ViewArea::deleteEvent () {
    widget = NULL;
    if (m_repaint_timer)
        g_source_remove (m_repaint_timer);
    m_repaint_timer = 0;
}

KDE_NO_EXPORT
void ViewArea::setAudioVideoGeometry (const IRect &rect, unsigned int * bg_color) {
    int x = rect.x (), y = rect.y (), w = rect.width (), h = rect.height ();
    if (!surface->node && aspect > 0.01) {
        int w1 = w;
        int h1 = h;
        w = int (h * aspect);
        if (w > w1) {
            h = int (w1 / aspect);
            w = w1;
        }
        x += (w1 - w) / 2;
        y += (h1 - h) / 2;
    }
    Color color (bg_color ? (int)*bg_color : 0);
    IRect wrect = IRect (x, y, w, h);
    if (m_av_geometry != wrect) {
        if (m_av_geometry.width () > 0 && m_av_geometry.height () > 0)
            scheduleRepaint (IRect (m_av_geometry.x (), m_av_geometry.y (),
                    m_av_geometry.width (), m_av_geometry.height ()));
        m_av_geometry = IRect (x, y, w, h);
        debugLog() << "ViewArea::setAudioVideoGeometry " << x << "," << y << " "
            << w << "x" << h << endl;
        gdk_window_move_resize (video, x, y, w, h);
        enableVideoInputFilter (m_video_filter);
        gdk_window_set_background (video, (GdkColor *) color);
        gdk_window_clear (video);
    }
}

void ViewArea::enableVideoInputFilter (bool enable) {
    int x, y, w, h;
    m_video_filter = enable;
    if (enable) {
        x = m_av_geometry.x ();
        y = m_av_geometry.y ();
        w = m_av_geometry.width();
        h = m_av_geometry.height ();
    } else {
        x = y = -100;
        w = h = 50;
    }
    if (video_input)
        gdk_window_move_resize (video_input, x, y, w, h);
}

KDE_NO_EXPORT void ViewArea::setAudioVideoNode (NodePtr n) {
    video_node = n;
    m_av_geometry = IRect ();
    if (!n || Mrl::WindowMode == n->mrl ()->view_mode)
        setAudioVideoGeometry (IRect (-100, -100, 50, 50), 0L);
}

KDE_NO_EXPORT
void ViewArea::setAspect (float a) {
    aspect = a;
    /*if (aspect > 0.01) {
        int w, h, x, y;
        gdk_drawable_get_size (video, &w, &h);
        gdk_window_get_position (video, &x, &y);
        scheduleRepaint (x, y, w, h);
        int w1 = int (h * aspect);
        int h1 = h;
        if (w1 > w) {
            h1 = int (w / aspect);
            w1 = w;
        }
        int xoff = (w - w1) / 2;
        int yoff = (h - h1) / 2;
        if (xoff > 0 && yoff > 0)
            gdk_window_move_resize (video, x + xoff, y + yoff, w1, h1);
    }*/
}

KDE_NO_EXPORT Surface *ViewArea::getSurface (Mrl *mrl) {
    surface->clear ();
    surface->node = mrl;
    //m_view->viewer()->resetBackgroundColor ();
    if (mrl) {
        updateSurfaceBounds ();
        return surface.ptr ();
    } else if (surface->surface) {
        cairo_surface_destroy (surface->surface);
        surface->surface = 0L;
        d->destroyBackingStore ();
    }
    scheduleRepaint (IRect (0, 0, width (), height ()));
    return 0L;
}

KDE_NO_EXPORT void ViewArea::scheduleRepaint (const IRect &rect) {
    if (m_repaint_timer)
        m_repaint_rect = m_repaint_rect.unite (rect);
    else {
        m_repaint_rect = rect;
        scheduleRepaintTimeout (surface->node
                ? surface->node->document ()->last_event_time
                : 0);
    }
}

KDE_NO_EXPORT void ViewArea::addUpdater (Node *node) {
    m_updaters = new RepaintUpdater (node, m_updaters);
    if (m_updaters_enabled && !m_repaint_timer)
        scheduleRepaintTimeout (surface->node
                ? surface->node->document ()->last_event_time
                : 0);
}

KDE_NO_EXPORT void ViewArea::removeUpdater (Node *node) {
    RepaintUpdater *prev = NULL;
    for (RepaintUpdater *r = m_updaters; r; r = r->next) {
        if (r->node.ptr () == node) {
            if (prev)
                prev->next = r->next;
            else
                m_updaters = r->next;
            delete r;
            break;
        }
        prev = r;
    }
    if (m_repaint_timer &&
            (!m_updaters_enabled || !m_updaters) &&
            m_repaint_rect.isEmpty ()) {
        g_source_remove (m_repaint_timer);
        m_repaint_timer = 0;
    }
}

static RepaintUpdater *getFirstUpdater (RepaintUpdater *updaters) {
    for (RepaintUpdater *r = updaters; r; r = updaters) {
        if (updaters->node)
            return updaters;
        updaters = r->next;
        delete r;
    }
    return NULL;
}

static void propagateUpdatersEvent (RepaintUpdater *updaters, void *event) {
    for (RepaintUpdater *r = updaters; r; ) {
        RepaintUpdater *next = r->next;
        if (r->node)
            r->node->message (MsgSurfaceUpdate, event); // may call removeUpdater()
        r = next;
    }
}

KDE_NO_EXPORT
void ViewArea::enableUpdaters (bool enable, unsigned int skip) {
    m_updaters_enabled = enable;
    m_updaters = getFirstUpdater (m_updaters);
    if (enable && m_updaters) {
        UpdateEvent event (m_updaters->node->document (), skip);
        propagateUpdatersEvent (m_updaters, &event);
        if (!m_repaint_timer)
            scheduleRepaintTimeout (event.cur_event_time);
    } else if (!enable && m_repaint_timer && m_repaint_rect.isEmpty ()) {
        g_source_remove (m_repaint_timer);
        m_repaint_timer = 0;
    }
}

KDE_NO_EXPORT bool ViewArea::timeoutPaintEvent () {
    m_updaters = getFirstUpdater (m_updaters);
    if (m_updaters_enabled && m_updaters) {
        UpdateEvent event (m_updaters->node->document (), 0);
        next_repaint_time = event.cur_event_time;
        propagateUpdatersEvent (m_updaters, &event);
    }
    if (!m_repaint_rect.isEmpty ())
        syncVisual (m_repaint_rect);
    last_repaint_time = next_repaint_time;
    if (m_updaters_enabled && m_updaters)
        return scheduleRepaintTimeout (last_repaint_time);
    m_repaint_timer = 0;
    return false;
}

bool ViewArea::scheduleRepaintTimeout (unsigned int cur_time) {
    if (!widget)
        return false;
    int timeout;
    if (!cur_time) {
        timeout = 20;
    } else {
        Document *doc = surface->node ? surface->node->document () : NULL;
        if (doc && doc->last_event_time < last_repaint_time) {
            last_repaint_time = doc->last_event_time;
            penalty = 30;
        }
        if (last_repaint_time + penalty < cur_time)
            timeout = 20;
        else
            timeout = 20 + penalty - (cur_time - last_repaint_time);
    }
    next_repaint_time = cur_time + timeout;
    if (m_repaint_timer && abs (cur_timeout - timeout) < 5)
        return true;
    cur_timeout = timeout;
    m_repaint_timer = g_timeout_add (cur_timeout, cb_scheduledPaint, this);;
    return false;
}

KDE_NO_EXPORT int ViewArea::width () const {
    int w = 0, h;
    if (widget)
        gdk_drawable_get_size (widget->window, &w, &h);
    return w;
}

KDE_NO_EXPORT int ViewArea::height () const {
    int w, h = 0;
    if (widget)
        gdk_drawable_get_size (widget->window, &w, &h);
    return h;
}

void ViewArea::prepareFullscreenToggle () {
    // MPlayer+omapfb stays black in window mode if having the focus
    Window w = GDK_DRAWABLE_XID (GDK_DRAWABLE (widget->window));
    Display *dpy = gdk_x11_display_get_xdisplay(gtk_widget_get_display(widget));

    XSetInputFocus (dpy, w, RevertToParent, CurrentTime);
    XFlush (dpy);
}

static void setXSelectInput (Display *dpy, Window wid, long mask) {
    XWindowAttributes attr;
    XGetWindowAttributes (dpy, wid, &attr);
    long m = mask | attr.your_event_mask;
    XSelectInput (dpy, wid, m);

    Window r, p, *c;
    unsigned int nr;
    if (XQueryTree (dpy, wid, &r, &p, &c, &nr)) {
        for (int i = 0; i < nr; ++i)
            setXSelectInput (dpy, c[i], mask);
        XFree (c);
    }
}

static bool findWindowAt (Display *dpy, Window wid, long mask, int *x, int *y,
        Window *window, Window *sub_window) {
    XWindowAttributes attr;
    XGetWindowAttributes (dpy, wid, &attr);
    Window r, p, *c;
    unsigned int nr;
    *window = 0;
    if (XQueryTree (dpy, wid, &r, &p, &c, &nr)) {
        for (int i = nr-1; i >= 0; --i) {
            XWindowAttributes cattr;
            XGetWindowAttributes (dpy, c[i], &cattr);
            if (*x >= cattr.x && *x <= cattr.x + cattr.width &&
                    *y >= cattr.y && *y <= cattr.y + cattr.height) {
                int x1 = *x - cattr.x;
                int y1 = *y - cattr.y;
                *sub_window = c[i];
                if (findWindowAt(dpy, c[i], mask, &x1, &y1, window, sub_window))
                {
                    *x = x1;
                    *y = y1;
                    break;
                }
            }
        }
        XFree (c);
    }
    if (!*window && attr.all_event_masks & mask)
        *window = wid;
    return *window;

}

static int getChildWindowDepth (Display *dpy, Window p, Window c, int depth) {
    Window root = 0;
    Window parent = 0;
    Window *children;
    unsigned int nr;
    /*Gives 'BadWindow (invalid Window parameter)' when npp stops'
    while (p != c && XQueryTree (dpy, c, &root, &parent, &children, &nr)) {
        if (nr)
            XFree (children);
        if (parent == p || parent == root || c == root || !parent)
            break;
        c = parent;
    }
    return parent == p;*/
    int d = -1;
    if ( p == c) {
        d = depth;
    } else if (XQueryTree (dpy, p, &root, &parent, &children, &nr)) {
        for (int i = 0; d < 0 && i < nr; ++i) {
            if (children[i] == c )
                d = depth + 1;
            else
                d = getChildWindowDepth (dpy, children[i], c, depth + 1);
        }
        if (nr)
            XFree (children);
    }
    return d;
}

static int tree_x_down;
static int tree_y_down;
static int tree_x_root_down;
static int tree_y_root_down;
static int tree_x_root_scroll;
static int tree_y_root_scroll;
static unsigned tree_time_scroll;
static unsigned tree_window_down;
static bool tree_scrolled;
static const int tree_threshold = 14;

static bool updateScroll (GtkAdjustment *adj, int pix) {
    if (!adj || !pix)
        return false;
    double d = gtk_adjustment_get_value (adj);
    double pc = (adj->upper - adj->lower) / adj->page_size;
    d -= pc * pix;
    if (d < adj->lower)
        d = adj->lower;
    if (d > adj->upper - adj->page_size)
        d = adj->upper - adj->page_size;
    gtk_adjustment_set_value (adj, d);
    return true;
}

static
GdkFilterReturn videoWidgetFilter (GdkXEvent *xe, GdkEvent *e, Application *a) {
    XEvent *event = (XEvent *) xe;
    ViewArea *view_area = VIEWAREA(a);
    static Time down_time;
    static int down_x;
    static int down_y;

    if (view_area->controls &&
            event->xany.window == GDK_DRAWABLE_XID (view_area->controls))
        return GDK_FILTER_CONTINUE;

    switch (event->xany.type) {
    case KeyPress:
    case KeyRelease: {
        XKeyEvent keyevent = event->xkey;
        if (a->fullscreen &&
                XKeycodeToKeysym (event->xany.display, keyevent.keycode, 0) == XK_F4)
            return GDK_FILTER_REMOVE; //GDK_FILTER_CONTINUE;
        Window v = GDK_DRAWABLE_XID(GDK_DRAWABLE(view_area->videoWindow()));
        Window i = GDK_DRAWABLE_XID(GDK_DRAWABLE(view_area->videoInputWindow()));
        int depth = event->xany.window == i
            ? 0
            : getChildWindowDepth (event->xany.display, v,
                    event->xkey.window, 0);
        if (depth >= 0) {
            keyevent.window =
                GDK_DRAWABLE_XID (GDK_DRAWABLE (view_area->viewWidget ()->window));
            XSendEvent (event->xany.display, keyevent.window, true,
                    KeyPressMask, (XEvent *) &keyevent);
            return GDK_FILTER_REMOVE;
        }
        break;
    }
    case MapNotify: {
        Window v = GDK_DRAWABLE_XID(GDK_DRAWABLE(view_area->videoWindow()));
        if (!event->xmap.override_redirect) {
            int depth = getChildWindowDepth (event->xany.display,
                    v, event->xmap.window, 0);
            if (depth >= 0) {
                setXSelectInput (event->xany.display,
                        event->xmap.window,
                        KeyPressMask|ExposureMask|SubstructureNotifyMask);
            }
        }
        break;
    }
    case ConfigureNotify:
        if (view_area->videoInputWindow ()) {
            Window v = GDK_DRAWABLE_XID(GDK_DRAWABLE(view_area->videoWindow()));
            Window i = GDK_DRAWABLE_XID(GDK_DRAWABLE(view_area->videoInputWindow()));
            if (event->xconfigure.window == v &&
                    event->xconfigure.above == i)
                XRaiseWindow (event->xany.display, i);
        }
        break;
    case ButtonPress: {
        tree_scrolled = false;
        if (getChildWindowDepth (event->xany.display,
                    GDK_DRAWABLE_XID(GDK_DRAWABLE(a->playlist_view->window)),
                    event->xbutton.window, 0) >= 0) {
            if (event->xbutton.button == 1) {
                tree_x_down = event->xbutton.x;
                tree_y_down = event->xbutton.y;
                tree_x_root_down = event->xbutton.x_root;
                tree_y_root_down = event->xbutton.y_root;
                tree_window_down = event->xbutton.window;
                return GDK_FILTER_REMOVE;
            }
            return GDK_FILTER_CONTINUE;
        }
        tree_window_down = 0;
        Window i = GDK_DRAWABLE_XID(GDK_DRAWABLE(view_area->videoInputWindow()));
        Window v = GDK_DRAWABLE_XID(GDK_DRAWABLE(view_area->videoWindow()));
        Window w = GDK_DRAWABLE_XID (GDK_DRAWABLE (view_area->viewWidget ()->window));
        IRect vr = view_area->videoGeometry ();
        if (a->fullscreen &&
                (event->xany.window == w &&
                 event->xbutton.x > 720 && event->xbutton.y < 80) ||
                (event->xany.window == i &&
                 event->xbutton.x + vr.x () > 720 &&
                 event->xbutton.y + vr.y () < 80)) {
            if (!view_area->controls_timer)
                updatePlayTree (a);
            view_area->controlsVisible (a, true);
        } else if (event->xany.window == w || event->xany.window == i) {
            if (event->xbutton.time - down_time < 500 &&
                    abs (down_x - event->xbutton.x) < 80 &&
                    abs (down_y - event->xbutton.y) < 80) {
                view_area->prepareFullscreenToggle ();
                if (a->fullscreen)
                    app_normal_screen (a);
                else
                    app_full_screen (a);
                down_time = 0;
                return GDK_FILTER_REMOVE;
            } else {
                down_x = event->xbutton.x;
                down_y = event->xbutton.y;
                down_time = event->xbutton.time;
            }
        }
    }
    // fall through
    case ButtonRelease:
        if (tree_window_down) {
            GdkFilterReturn ret = GDK_FILTER_REMOVE;
            if (tree_window_down == event->xbutton.window &&
                    !tree_scrolled &&
                    abs (event->xbutton.x - tree_x_down) < tree_threshold &&
                    abs (event->xbutton.y - tree_y_down) < tree_threshold) {
                GdkEventButton evt;
                evt.type = GDK_BUTTON_PRESS;
                evt.window = gdk_window_lookup_for_display (
                        gdk_x11_lookup_xdisplay (event->xany.display),
                        event->xany.window);
                evt.send_event = false;
                evt.x = tree_x_down;
                evt.y = tree_y_down;
                evt.x_root = tree_x_root_down;
                evt.y_root = tree_y_root_down;
                evt.axes = NULL;
                evt.state = 0;
                evt.button = 1;
                evt.time = event->xbutton.time;
                evt.device = gdk_device_get_core_pointer ();
                gtk_main_do_event ((GdkEvent *) &evt);
                ret = GDK_FILTER_CONTINUE;
            }
            tree_window_down = 0;
            tree_scrolled = false;
            return ret;
        }
        if (event->xany.window == GDK_DRAWABLE_XID(GDK_DRAWABLE(view_area->videoInputWindow()))) {
            XButtonEvent evt = event->xbutton;
            int evt_mask = event->xany.type == ButtonRelease
                ? ButtonReleaseMask : ButtonPressMask;
            if (findWindowAt (event->xany.display,
                        GDK_DRAWABLE_XID(GDK_DRAWABLE(view_area->videoWindow())),
                        evt_mask,
                        &evt.x, &evt.y, &evt.window, &evt.subwindow)) {
                XSendEvent (event->xany.display, evt.window, False,
                        evt_mask, (XEvent *) &evt);
                XFlush (event->xany.display);
                if (event->xany.type == ButtonRelease) {
                    XSetInputFocus (event->xany.display, evt.window,
                            RevertToParent, CurrentTime);
                    XFlush (event->xany.display);
                }
                return GDK_FILTER_REMOVE;
            }
        }
        break;
    case MotionNotify:
        if (tree_window_down) {
            if (tree_scrolled ||
                    abs (event->xmotion.x_root - tree_x_root_down) >
                        tree_threshold ||
                    abs (event->xmotion.y_root - tree_y_root_down) >
                        tree_threshold) {
                if (!tree_scrolled) {
                    tree_scrolled = true;
                    tree_time_scroll = 0;
                    tree_x_root_scroll = tree_x_root_down;
                    tree_y_root_scroll = tree_y_root_down;
                }
                if (event->xmotion.time - tree_time_scroll > 80) {
                    bool scrolled = updateScroll (gtk_tree_view_get_vadjustment(
                                GTK_TREE_VIEW (a->playlist_view)),
                            event->xmotion.y_root - tree_y_root_scroll);
                    scrolled |= updateScroll (gtk_tree_view_get_hadjustment(
                                GTK_TREE_VIEW (a->playlist_view)),
                            event->xmotion.x_root - tree_x_root_scroll);
                    if (scrolled) {
                        tree_x_root_scroll = event->xmotion.x_root;
                        tree_y_root_scroll = event->xmotion.y_root;
                        tree_time_scroll = event->xmotion.time;
                    }
                }
            }
            return GDK_FILTER_REMOVE;
        }
    }
    return GDK_FILTER_CONTINUE;
}

static
GdkFilterReturn controlsFilter (GdkXEvent *xe, GdkEvent *e, Application *app) {
    XEvent *event = (XEvent *) xe;
    ViewArea *view_area = VIEWAREA(app);

    if (event->xany.window != GDK_DRAWABLE_XID (view_area->controls))
        return GDK_FILTER_CONTINUE;

    static int downx, downy;
    static Time down_time;

    switch (event->xany.type) {
    case ButtonPress:
        if (app->fullscreen) {
            int x, w = -1;
            if (event->xbutton.x < 100) {
                previousSelected (app);
                x = 0;
                w = 100;
            } else if (event->xbutton.x > 705) {
                nextSelected (app);
                x = 705;
                w = 95;
            } else if (event->xbutton.x > 655 && event->xbutton.x < 705) {
                cb_favorites_add (app);
                x = 655;
                w = 50;
            } else if (event->xbutton.x > 110  && event->xbutton.x < 655) {
                downx = event->xbutton.x;
                down_time = event->xbutton.time;
            }
            if (w > 0) {
                gdk_window_clear_area_e (VIEWAREA(app)->controls, x, 0, w, 60);
                view_area->controlsVisible (app, true);
            }
        }
        break;
    case ButtonRelease:
        if (event->xbutton.x > 110  && event->xbutton.x < 655) {
            int dx = event->xbutton.x - downx;
            if (-20 < dx && dx < 20 && event->xbutton.time - down_time < 1000) {
                gdk_window_clear_area_e (VIEWAREA(app)->controls, 110, 0, 545, 60);
                rowSelected (app);
            } else {
                view_area->controlsVisible (app, false);
            }
        }
        break;
    case Expose:
        if (!event->xexpose.count && event->xany.window &&
                event->xany.window == GDK_DRAWABLE_XID (view_area->controls)) {
            cairo_t *cr = gdk_cairo_create (view_area->controls);
            cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
            cairo_set_source_rgb (cr, 32.0 / 255, 32.0 / 255, 1);
            cairo_rectangle (cr, 0, 0, 655, 60);
            cairo_fill (cr);
            cairo_rectangle (cr, 705, 0, 95, 60);
            cairo_fill (cr);
            cairo_set_source_rgb (cr, 32.0 / 255, 1, 48.0 / 255);
            cairo_rectangle (cr, 655, 0, 50, 60);
            cairo_fill (cr);
            GValue icon = { 0 , };
            GValue text = { 0 , };
            if (playlist_selected (app, &icon, &text)) {
                GdkPixbuf *pix = (GdkPixbuf *) g_value_get_object (&icon);
                const gchar *val = g_value_get_string (&text);
                int pixw = gdk_pixbuf_get_width (pix);
                cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
                gdk_cairo_set_source_pixbuf (cr, pix, 127, 17);
                cairo_rectangle (cr, 127, 17, pixw, pixw);
                cairo_fill (cr);
                cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
                cairo_set_source_rgb (cr, 0.8, 0.8, 0);
                cairo_move_to (cr, 157, 10);
                PangoLayout *layout = pango_cairo_create_layout (cr);
                pango_layout_set_width (layout, 483 * PANGO_SCALE);
                pango_layout_set_text (layout, val, -1);
                PangoFontDescription *desc = pango_font_description_new ();
                pango_font_description_set_family (desc, "Sans");
                pango_font_description_set_absolute_size (desc,
                        PANGO_SCALE * 20);
                pango_layout_set_font_description (layout, desc);
                pango_font_description_free (desc);
                pango_cairo_show_layout (cr, layout);
                g_object_unref (layout);
                g_value_unset (&icon);
                g_value_unset (&text);
            }
            cairo_destroy (cr);
            return GDK_FILTER_REMOVE;
        }
    }
    return GDK_FILTER_CONTINUE;
}

KDE_NO_EXPORT void ViewArea::setEventFiltering (Application *app, bool on) {
    if (on && !m_event_filter_on)
        gdk_window_add_filter (NULL, (GdkFilterFunc) videoWidgetFilter, app);
    else if (!on && m_event_filter_on)
        gdk_window_remove_filter (NULL, (GdkFilterFunc) videoWidgetFilter, app);
    m_event_filter_on = on;
}

KDE_NO_EXPORT void ViewArea::setOutputWidget (Application *app, GtkWidget * w) {
    widget = w;
    gdk_window_clear (w->window);
    gdk_window_focus (widget->window, CurrentTime);
    if (surface->surface) {
        cairo_surface_destroy (surface->surface);
        surface->surface = 0L;
    }
    if (!video) {
        GdkWindowAttr attributes;
        attributes.window_type = GDK_WINDOW_CHILD;
        attributes.x = w->allocation.x;
        attributes.y = w->allocation.y;
        attributes.width = w->allocation.width;
        attributes.height = w->allocation.height;
        attributes.wclass = GDK_INPUT_OUTPUT;
        attributes.visual = gtk_widget_get_visual (w);
        attributes.colormap = gtk_widget_get_colormap (w);
        attributes.event_mask = gtk_widget_get_events (w) |
            GDK_SUBSTRUCTURE_MASK | GDK_EXPOSURE_MASK | GDK_KEY_PRESS_MASK;
        attributes.event_mask &= ~GDK_BUTTON_PRESS_MASK;
        //    ~(GDK_KEY_PRESS_MASK | GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK);
        gint mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
        video = gdk_window_new (w->window, &attributes, mask);
        gdk_window_show (video);

        attributes.wclass = GDK_INPUT_ONLY;
        attributes.event_mask = gtk_widget_get_events (w) |
            GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK;
        attributes.event_mask &= ~GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK;
        mask = GDK_WA_X | GDK_WA_Y | GDK_WA_WMCLASS;
        video_input = gdk_window_new (w->window, &attributes, mask);
        gdk_window_show (video_input);

        setAudioVideoGeometry (IRect (-100, -100, 50, 50), 0L);

        debugLog() << "video window "  << (int)GDK_DRAWABLE_XID (GDK_DRAWABLE (video)) << " " << endl;
        setEventFiltering (app, true);

    } else {
        gdk_window_reparent (video, widget->window, m_av_geometry.x(), m_av_geometry.y());
        gdk_window_reparent (video_input, widget->window, m_av_geometry.x(), m_av_geometry.y());
    }
    //resizeEvent ();
}

static gboolean cb_controlsTimeout (Application *app) {
    VIEWAREA(app)->controls_timer = 0;
    VIEWAREA(app)->controlsVisible (app, false);
    return false;
}

void ViewArea::controlsVisible (Application *app, bool show) {
    if (controls_timer)
        g_source_remove (controls_timer);
    if (show) {
        bool update_sizes = false;
        if (!controls_timer) {
            gdk_window_move (controls, 0, 420);
            update_sizes = true;
        }
        controls_timer = g_timeout_add (10000, (GSourceFunc) cb_controlsTimeout, app);
        if (update_sizes)
            CONTROL(app)->updateViewAreaSizes ();
    } else {
        controls_timer = 0;
        if (controls) {
            gdk_window_move (controls, 0, 490);
            CONTROL(app)->updateViewAreaSizes ();
        }
    }
}

//------------------------------------------------------------------------------

void cb_view_area_button_press (GtkWidget *w, GdkEventButton *e, Application * app) {
    VIEWAREA(app)->mousePressEvent (int (e->x), int (e->y));
}

void cb_view_area_expose (GtkWidget *w, GdkEventExpose *e, Application *app) {
    VIEWAREA(app)->paintEvent (
            IRect (e->area.x, e->area.y, e->area.width, e->area.height));
}

void cb_view_area_realized (GtkWidget *widget, Application * app) {
    g_signal_connect (G_OBJECT (widget), "configure-event",
            GTK_SIGNAL_FUNC (cb_view_area_configure), app);
    gtk_widget_set_double_buffered (widget, false);
    if (widget == app->view_area_fullscreen) {
        Color c (0x101010);
        gdk_window_set_background (widget->window, (GdkColor *) c);
    } else {
        VIEWAREA(app)->setOutputWidget (app, widget);
    }
}

static GdkBitmap *createControlsMask () {
    cairo_surface_t *surface = cairo_image_surface_create (CAIRO_FORMAT_A1, 800, 60);
    cairo_t *cr = cairo_create (surface);
    cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
    cairo_set_source_rgb (cr, 1, 1, 1);

    cairo_new_path (cr);
    cairo_move_to (cr, 30, 30);
    cairo_line_to (cr, 90, 0);
    cairo_curve_to (cr, 90 - 12, 20, 90 - 12, 40, 90, 60);
    cairo_line_to (cr, 30, 30);
    cairo_close_path (cr);
    cairo_fill (cr);

    cairo_new_path (cr);
    cairo_move_to (cr, 130, 0);
    cairo_line_to (cr, 650, 0);
    cairo_curve_to (cr, 650 - 12, 20, 650 - 12, 40, 650, 60);
    cairo_line_to (cr, 130, 60);
    cairo_curve_to (cr, 130 - 12, 40, 130 - 12, 20, 130, 0);
    cairo_close_path (cr);
    cairo_fill (cr);

    cairo_new_path (cr);
    cairo_move_to (cr, 710, 0);
    cairo_line_to (cr, 770, 30);
    cairo_line_to (cr, 710, 60);
    cairo_curve_to (cr, 710 + 12, 40, 710 + 12, 20, 710, 0);
    cairo_close_path (cr);
    cairo_fill (cr);

    cairo_translate (cr, 680, 12);
    cairo_new_path (cr);
    for (int i = 0; i < 6; ++i) {
        cairo_rotate (cr, 60 * M_PI / 180);
        cairo_line_to (cr, 0, -8);
        cairo_translate (cr, 0, -8);
        cairo_rotate (cr, 60 * M_PI / 180);
        cairo_line_to (cr, 0, -8);
        cairo_translate (cr, 0, -8);
        cairo_rotate (cr, 60 * M_PI / 180);
        cairo_line_to (cr, 0, -8);
        cairo_translate (cr, 0, -8);
        cairo_rotate (cr, 240 * M_PI / 180);
    }
    cairo_close_path (cr);
    cairo_fill (cr);

    GdkBitmap *bitmap = gdk_bitmap_create_from_data (NULL,
            (const gchar *) cairo_image_surface_get_data (surface), 800, 60);

    cairo_destroy (cr);
    cairo_surface_destroy (surface);

    return bitmap;
}

gboolean cbViewAreaFullscreenMapped (GtkWidget *w, GdkEvent *, Application *a) {
    ViewArea *view_area = VIEWAREA(a);
    gtk_window_fullscreen (GTK_WINDOW (a->window_fullscreen));
    view_area->setOutputWidget (a, a->view_area_fullscreen);
    if (!view_area->controls) {
        GdkWindowAttr attributes;
        attributes.window_type = GDK_WINDOW_CHILD;
        attributes.x = 0;
        attributes.y = 0;
        attributes.width = 800;
        attributes.height = 60;
        attributes.wclass = GDK_INPUT_OUTPUT;
        attributes.visual = gtk_widget_get_visual (w);
        attributes.colormap = gtk_widget_get_colormap (w);
        attributes.event_mask = gtk_widget_get_events (w) |
            GDK_SUBSTRUCTURE_MASK | GDK_EXPOSURE_MASK |
            GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK;
        attributes.event_mask &= ~GDK_KEY_PRESS_MASK;
        //    ~(GDK_KEY_PRESS_MASK | GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK);
        gint mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
        view_area->controls = gdk_window_new (a->view_area_fullscreen->window, &attributes, mask);
        gdk_window_add_filter (view_area->controls, (GdkFilterFunc) controlsFilter, a);
        GdkBitmap *bitmap = createControlsMask ();
        gdk_window_shape_combine_mask (view_area->controls, bitmap, 0, 0);
        gdk_window_show (view_area->controls);
    } else {
        gdk_window_raise (view_area->controls);
    }
    view_area->controlsVisible (a, false);
    return false;
}

void cbViewAreaFullscreenUnmapped (GtkWidget *w, Application *a) {
    ViewArea *view_area = VIEWAREA(a);
    view_area->setOutputWidget (a, a->view_area);
    view_area->controlsVisible (a, false);
}

void cb_fullscreen_destoyed (GtkObject *o, Application *app) {
    if (app->fullscreen) {
        ViewArea *view_area = VIEWAREA(app);
        app->fullscreen = false;
        app->window_fullscreen = NULL;
        view_area->controls = NULL;
        view_area->controlsVisible (app, false);
    }
}

gboolean cb_scheduledPaint (void * vp) {
    return static_cast <ViewArea *> (vp)->timeoutPaintEvent ();
}


//#ifdef gtk < 2.8

unsigned char * pixbufBits (GdkPixbuf *pixbuf, int & n_channels) {
    gint width = gdk_pixbuf_get_width (pixbuf);
    gint height = gdk_pixbuf_get_height (pixbuf);
    guchar *gdk_pixels = gdk_pixbuf_get_pixels (pixbuf);
    int gdk_rowstride = gdk_pixbuf_get_rowstride (pixbuf);
    n_channels = gdk_pixbuf_get_n_channels (pixbuf);
    guchar *cairo_pixels, *head;
    static cairo_user_data_key_t key;
    int j;

    cairo_pixels = (guchar*)g_malloc (4 * width * height);
    head = cairo_pixels;

    for (j = height; j; j--) {
        guchar *p = gdk_pixels;
        guchar *q = cairo_pixels;

        if (n_channels == 3) {
            guchar *end = p + 3 * width;

            while (p < end) {
#if __BYTE_ORDER == __LITTLE_ENDIAN
                q[0] = p[2];
                q[1] = p[1];
                q[2] = p[0];
                q[3] = p[1];
#warning little ending
#else
                q[1] = p[0];
                q[2] = p[1];
                q[3] = p[2];
#warning big ending
#endif
                p += 3;
                q += 4;
            }
        } else {
            guchar *end = p + 4 * width;
            guint t1,t2,t3;

#define MULT(d,c,a,t) G_STMT_START { t = c * a + 0x7f; d = ((t >> 8) + t) >> 8; } G_STMT_END

            while (p < end) {
#if G_BYTE_ORDER == G_LITTLE_ENDIAN
                MULT(q[0], p[2], p[3], t1);
                MULT(q[1], p[1], p[3], t2);
                MULT(q[2], p[0], p[3], t3);
                q[3] = p[3];
#else
                q[0] = p[3];
                MULT(q[1], p[0], p[3], t1);
                MULT(q[2], p[1], p[3], t2);
                MULT(q[3], p[2], p[3], t3);
#endif

              p += 4;
              q += 4;
            }

#undef MULT
        }

        gdk_pixels += gdk_rowstride;
        cairo_pixels += 4 * width;
    }
    return head;
}

/*
void
gdk_cairo_set_source_pixbuf (cairo_t   *cr,
                             GdkPixbuf *pixbuf,
                             double     pixbuf_x,
                             double     pixbuf_y)
{
  gint width = gdk_pixbuf_get_width (pixbuf);
  gint height = gdk_pixbuf_get_height (pixbuf);
  guchar *gdk_pixels = gdk_pixbuf_get_pixels (pixbuf);
  int gdk_rowstride = gdk_pixbuf_get_rowstride (pixbuf);
  int n_channels = gdk_pixbuf_get_n_channels (pixbuf);
  debugLog() << "gdk_cairo_set_source_pixbuf pixbuf:" << width << "x" << height << " rowstride:" << gdk_rowstride << " channels:" << n_channels << endl;
  guchar *cairo_pixels;
  cairo_format_t format;
  cairo_surface_t *surface;
  static cairo_user_data_key_t key;
  int j;

  if (n_channels == 3)
    format = CAIRO_FORMAT_RGB24;
  else
    format = CAIRO_FORMAT_ARGB32;

  cairo_pixels = (guchar*)g_malloc (4 * width * height);
  surface = cairo_image_surface_create_for_data ((unsigned char *)cairo_pixels,
                                                 format,
                                                 width, height, 4 * width);
  cairo_surface_set_user_data (surface, &key,
                               cairo_pixels, (cairo_destroy_func_t)g_free);

  for (j = height; j; j--)
    {
      guchar *p = gdk_pixels;
      guchar *q = cairo_pixels;

      if (n_channels == 3)
        {
          guchar *end = p + 3 * width;
          
          while (p < end)
            {
#if G_BYTE_ORDER == G_LITTLE_ENDIAN
              q[0] = p[2];
              q[1] = p[1];
              q[2] = p[0];
#else     
              q[1] = p[0];
              q[2] = p[1];
              q[3] = p[2];
#endif
              p += 3;
              q += 4;
            }
        }
      else
        {
          guchar *end = p + 4 * width;
          guint t1,t2,t3;
            
#define MULT(d,c,a,t) G_STMT_START { t = c * a + 0x7f; d = ((t >> 8) + t) >> 8; } G_STMT_END

          while (p < end)
            {
#if G_BYTE_ORDER == G_LITTLE_ENDIAN
              MULT(q[0], p[2], p[3], t1);
              MULT(q[1], p[1], p[3], t2);
              MULT(q[2], p[0], p[3], t3);
              q[3] = p[3];
#else     
              q[0] = p[3];
              MULT(q[1], p[0], p[3], t1);
              MULT(q[2], p[1], p[3], t2);
              MULT(q[3], p[2], p[3], t3);
#endif
              
              p += 4;
              q += 4;
            }
          
#undef MULT
        }

      gdk_pixels += gdk_rowstride;
      cairo_pixels += 4 * width;
    }

  cairo_set_source_surface (cr, surface, pixbuf_x, pixbuf_y);
  cairo_surface_destroy (surface);
}
*/
