/**
 * Copyright (C) 2005 by Koos Vriezen <koos ! vriezen ? xs4all ! nl>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License version 2 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 **/

//#include <config.h>
#include <string.h>
#include "kmplayer_atom.h"
#include "kmplayer_smil.h"

using namespace KMPlayer;

Node *ATOM::Feed::childFromTag (const String & tag) {
    if (!strcmp ((const char *) tag, "entry"))
        return new ATOM::Entry (m_doc);
    else if (!strcmp ((const char *) tag, "link"))
        return new ATOM::Link (m_doc);
    else if (!strcmp ((const char *) tag, "title"))
        return new DarkNode (m_doc, tag, id_node_title);
    return NULL;
}

void ATOM::Feed::closed () {
    for (Node *c = firstChild (); c; c = c->nextSibling ())
        if (c->id == id_node_title) {
            title = c->innerText ().stripWhiteSpace ();
            break;
        }
    Mrl::closed ();
}

Node *ATOM::Entry::childFromTag (const String &tag) {
    const char *cstr = (const char *) tag;
    if (!strcmp (cstr, "link"))
        return new ATOM::Link (m_doc);
    else if (!strcmp (cstr, "content"))
        return new ATOM::Content (m_doc);
    else if (!strcmp (cstr, "title"))
        return new DarkNode (m_doc, tag, id_node_title);
    else if (!strcmp (cstr, "summary"))
        return new DarkNode (m_doc, tag, id_node_summary);
    else if (!strcmp (cstr, "media:group"))
        return new MediaGroup (m_doc);
    else if (!strcmp (cstr, "category") ||
            !strcmp (cstr, "author:") ||
            !strcmp (cstr, "id") ||
            !strcmp (cstr, "updated") ||
            !strncmp (cstr, "yt:", 3) ||
            !strncmp (cstr, "gd:", 3))
        return new DarkNode (m_doc, tag, id_node_ignored);
    return NULL;
}

void ATOM::Entry::closed () {
    for (Node *c = firstChild (); c; c = c->nextSibling ())
        if (c->id == id_node_title) {
            title = c->innerText ().stripWhiteSpace ();
            break;
        }
    Mrl::closed ();
}

String ATOM::Entry::description () const {
    for (Node *c = firstChild (); c; c = c->nextSibling ())
        if (c->id == id_node_content) {
            Mrl *mrl = c->mrl ();
            if (mrl->mimetype.startsWith ("text/") ||
                    mrl->mimetype.indexOf ("html") > -1)
                return mrl->innerText ();
            break;
        }
    return String ();
}

Node::PlayType ATOM::Link::playType () {
    return src.isEmpty () ? play_type_none : play_type_unknown;
}

void ATOM::Link::closed () {
    String href;
    String rel;
    for (Attribute *a = attributes ()->first (); a; a = a->nextSibling ()) {
        if (a->name () == StringPool::attr_href)
            href = a->value ();
        else if (a->name () == StringPool::attr_title)
            title = a->value ();
        else if (a->name () == "rel")
            rel = a->value ();
    }
    if (!href.isEmpty () && rel == "enclosure")
        src = href;
    else if (title.isEmpty ())
        title = href;
    Mrl::closed ();
}

void ATOM::Content::closed () {
    for (Attribute *a = attributes ()->first (); a; a = a->nextSibling ()) {
        if (a->name () == StringPool::attr_src)
            src = a->value ();
        else if (a->name () == StringPool::attr_type) {
            String v = a->value ().lower ();
            if (v == "text")
                mimetype = "text/plain";
            else if (v == "html")
                mimetype = "text/html";
            else if (v == "xhtml")
                mimetype = "application/xhtml+xml";
            else
                mimetype = v;
        }
    }
    Mrl::closed ();
}

Node::PlayType ATOM::Content::playType () {
    if (!hasChildNodes () && !src.isEmpty ())
        return play_type_unknown;
    return play_type_none;
}

Node *ATOM::MediaGroup::childFromTag (const String &tag) {
    const char *cstr = (const char *) tag;
    if (!strcmp (cstr, "media:content"))
        return new ATOM::MediaContent (m_doc);
    else if (!strcmp (cstr, "media:title"))
        return new DarkNode (m_doc, tag, id_node_media_title);
    else if (!strcmp (cstr, "media:description"))
        return new DarkNode (m_doc, tag, id_node_media_description);
    else if (!strcmp (cstr, "media:thumbnail"))
        return new DarkNode (m_doc, tag, id_node_media_thumbnail);
    else if (!strcmp (cstr, "media:category") ||
            !strcmp (cstr, "media:keywords"))
        return new DarkNode (m_doc, tag, id_node_ignored);
    else if (!strcmp (cstr, "smil"))
        return new SMIL::Smil (m_doc);
    return NULL;
}

void *ATOM::MediaGroup::message (MessageType msg, void *content) {
    if (MsgChildFinished == msg &&
            ((Posting *) content)->source->isPlayable ())
        finish (); // only play one
    return Element::message (msg, content);
}

//http://code.google.com/apis/youtube/2.0/developers_guide_protocol.html
void ATOM::MediaGroup::closed () {
    String images;
    String desc;
    String title;
    int img_count = 0;
    for (Node *c = firstChild (); c; c = c->nextSibling ()) {
        switch (c->id) {
        case id_node_media_title:
            title = c->innerText ();
            break;
        case id_node_media_description:
            desc = c->innerText ();
            break;
        case id_node_media_thumbnail:
        {
            Element *e = static_cast <Element *> (c);
            String url = e->getAttribute (StringPool::attr_url);
            if (!url.isEmpty ()) {
                images += String ("<img region=\"image\" src=\"") + url + Char ('"');
                String w = e->getAttribute (StringPool::attr_width);
                if (!w.isEmpty ())
                    images += String (" width=\"") + w + Char ('"');
                String h = e->getAttribute (StringPool::attr_height);
                if (!h.isEmpty ())
                    images += String (" height=\"") + h + Char ('"');
                String t = e->getAttribute (TrieString ("time"));
                if (!t.isEmpty ())
                    images += String (" dur=\"") +
                        String::number (Mrl::parseTimeString (t) / 10) +
                        Char ('"');
                images += String (" transIn=\"fade\" transOut=\"ellipsewipe\" fit=\"meet\"/>");
                img_count++;
            }
            break;
        }
        }
    }
    if (img_count) {
        TextStream out;
        out << "<smil><head>";
        if (!title.isEmpty ())
            out << "<title>" << title << "</title>";
        out << "<layout><root-layout width=\"400\" height=\"300\" background-color=\"#FFFFF0\"/>"
            "<region id=\"image\" left=\"5\" top=\"20\" width=\"130\" bottom=\"20\"/>"
            "<region id=\"text\" left=\"140\" top=\"10\" bottom=\"10\" right=\"10\" fit=\"scroll\"/>"
            "</layout>"
            "<transition id=\"fade\" dur=\"0.3\" type=\"fade\"/>"
            "<transition id=\"ellipsewipe\" dur=\"0.5\" type=\"ellipseWipe\"/>"
            "</head><body>"
            "<par><seq repeatCount=\"indefinite\">";
        out << images;
        out << String ("</seq><smilText region=\"text\">");
        out << XMLStringlet (desc);
        out << String ("</smilText></par></body></smil>");
        TextStream inxml (out.release ());
        KMPlayer::readXML (this, inxml, String (), false);
        NodePtr n = lastChild();
        n->normalize ();
        n->auxiliary_node = true;
        removeChild (n);
        insertBefore (n.ptr (), firstChild ());

    }
    Element::closed ();
}

void ATOM::MediaContent::closed () {
    for (Attribute *a = attributes ()->first (); a; a = a->nextSibling ()) {
        if (a->name () == StringPool::attr_url)
            src = a->value();
        else if (a->name () == StringPool::attr_type)
            mimetype = a->value ();
    }
    Mrl::closed ();
}

Node::PlayType ATOM::MediaContent::playType () {
    return src.isEmpty () ? play_type_none : play_type_unknown;
}
