/**
  This file belong to the KMPlayer project, a movie player plugin for Konqueror
  Copyright (C) 2007  Koos Vriezen <koos.vriezen@gmail.com>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**/

#ifndef _KMPLAYER__MEDIA_OBJECT_H_
#define _KMPLAYER__MEDIA_OBJECT_H_

#include <config.h>

#include "kmplayerplaylist.h"

typedef struct _GdkPixbufAnimation GdkPixbufAnimation;
typedef struct _GdkPixbufAnimationIter GdkPixbufAnimationIter;
typedef struct _GdkPixbufLoader GdkPixbufLoader;

namespace KMPlayer {

class Control;
class ImageData;
class CalculatedSizer;
class Surface;

/*
 * Class that creates Actor and keeps track objects
 */
class KMPLAYER_EXPORT ActorAgent {
public:
    enum ActorType { Any, Audio, AudioVideo, Image, Text };
    //typedef QValueList <Actor *> MediaList;

    ActorAgent (Control *ctrl);
    ~ActorAgent ();

    //MediaList &medias () { return m_media_objects; }
    Control *control () const { return m_control; }

private:
    //MediaList m_media_objects;
    Control *m_control;
};


//------------------------%<----------------------------------------------------

/*
 * Abstract base of Actor types, handles downloading
 */

class KMPLAYER_EXPORT Actor {
    friend class ActorAgent;
public:
    virtual ActorAgent::ActorType type () const = 0;

    virtual bool play () = 0;
    virtual void pause () {}
    virtual void unpause () {}
    virtual void stop () {}
    /*virtual*/ void dismiss ();

    Mrl *mrl ();

protected:
    Actor (ActorAgent *manager, Node *node);
    virtual ~Actor ();

    ActorAgent *m_agent;
    NodePtrW m_node;
};

//------------------------%<----------------------------------------------------

class KMPLAYER_EXPORT MediaInfo : public AsyncJobInterface {
public:
    MediaInfo (Node *node, ActorAgent::ActorType type);
    ~MediaInfo ();

    bool wget (const String &url);
    void killWGet ();
    void clearData ();
    String mimetype ();
    bool downloading () const;
    void create ();

    ByteArray &rawData () { return data; }
    Actor *media;
    String url;
    ByteArray data;
    String mime;
    ActorAgent::ActorType type;

private:
    virtual void jobData (Job *job, ByteArray &data);
    virtual void jobResult (Job *job);

    void ready ();
    bool readChildDoc ();

    Node *node;
    Job *job;
};


//------------------------%<----------------------------------------------------

/*
 * Actor for audio/video, groups Mrl, Process and Viewer
 */

class KMPLAYER_NO_EXPORT AudioVideoActor : public Actor {
    friend class ActorAgent;
public:
    //enum Request { ask_nothing, ask_play, ask_stop, ask_delete };

    AudioVideoActor (ActorAgent *manager, Node *node);

    ActorAgent::ActorType type () const { return ActorAgent::AudioVideo; }

    virtual bool play ();
    virtual void stop ();
    virtual void pause ();
    virtual void unpause ();

    //Request request;
    bool ignore_pause;

protected:
    ~AudioVideoActor ();
};


//------------------------%<----------------------------------------------------

/*
 * Actor for (animated)images
 */

struct KMPLAYER_NO_EXPORT ImageData {
    enum ImageFlags {
        ImageAny=0, ImagePixmap=0x1, ImageAnimated=0x2, ImageScalable=0x4
    };
    ImageData( const String & img);
    ~ImageData();
    void setImage (Image *img);
    void copyImage (Surface *s, int w, int h, cairo_surface_t *similar,
            unsigned int *penalty, CalculatedSizer *zoom=NULL);
    bool isEmpty () const {
         return !image && !surface; }

    unsigned short width;
    unsigned short height;
    short flags;
    bool has_alpha;
private:
    Image *image;
    cairo_surface_t *surface;
    String url;
};

typedef SharedPtr <ImageData> ImageDataPtr;
typedef WeakPtr <ImageData> ImageDataPtrW;

class KMPLAYER_NO_EXPORT ImageActor : public Actor {
public:
    ImageActor (ActorAgent *manager, Node *node,
            const String &url, const ByteArray &data);
    ImageActor (Node *node, ImageDataPtr id);

    ActorAgent::ActorType type () const { return ActorAgent::Image; }

    bool play ();
    void stop ();
    void pause ();
    void unpause ();

    bool isEmpty () const;
    bool movieTimer ();

    ImageDataPtr cached_img;

protected:
    ~ImageActor ();

private:
    void setupImage (const String &url, const ByteArray &ba);

    GdkPixbufAnimation *img_movie;
    GdkPixbufAnimationIter *img_movie_iter;
    GdkPixbufLoader *pix_loader;
    int img_movie_timer;
    int timeout;
};

//------------------------%<----------------------------------------------------

/*
 * Actor for text
 */
class KMPLAYER_NO_EXPORT TextActor : public Actor {
public:
    TextActor (ActorAgent *manager, Node *node, const ByteArray &ba);

    ActorAgent::ActorType type () const { return ActorAgent::Text; }

    bool play ();

    String text;
    static int defaultFontSize ();

protected:
    ~TextActor ();
};

} // namespace

#endif
