/**
  This file belong to the KMPlayer project, a movie player plugin for Konqueror
  Copyright (C) 2007  Koos Vriezen <koos.vriezen@gmail.com>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**/

#include <string.h>
#include <map>

#include "actor.h"
#include "kmplayercontrol.h"
#include "kmplayerprocess.h"
#include "viewarea.h"

using namespace KMPlayer;

namespace {
    typedef std::map <String, ImageDataPtrW> ImageDataMap;
    static ImageDataMap *image_data_map;
}

//------------------------%<----------------------------------------------------

ActorAgent::ActorAgent (Control *ctrl) : m_control (ctrl) {
    if (!image_data_map)
        image_data_map = new ImageDataMap;
}

ActorAgent::~ActorAgent () {
    if (image_data_map->size () == 0) {
        delete image_data_map;
        image_data_map = 0;
    }
}

//------------------------%<----------------------------------------------------

Actor::Actor (ActorAgent *manager, Node *node)
 : m_agent (manager), m_node (node) {
   //manager->medias ().push_back (this);
}

Actor::~Actor () {
    //m_agent->medias ().remove (this);
}

Mrl *Actor::mrl () {
    return m_node ? m_node->mrl () : NULL;
}

KDE_NO_EXPORT void Actor::dismiss () {
    delete this;
}

//------------------------%<----------------------------------------------------

static bool isPlayListMime (const String & mime) {
    const char * mimestr = (const char *) mime;
    //debugLog () << "isPlayListMime  mime:" << mimestr << endl;
    return mimestr && (!strcmp (mimestr, "audio/mpegurl") ||
            !strcmp (mimestr, "audio/x-mpegurl") ||
            !strncmp (mimestr, "video/x-ms", 10) ||
            !strncmp (mimestr, "audio/x-ms", 10) ||
            //!strcmp (mimestr, "video/x-ms-wmp") ||
            //!strcmp (mimestr, "video/x-ms-asf") ||
            //!strcmp (mimestr, "video/x-ms-wmv") ||
            //!strcmp (mimestr, "video/x-ms-wvx") ||
            //!strcmp (mimestr, "video/x-msvideo") ||
            !strcmp (mimestr, "video/x-real") ||
            !strcmp (mimestr, "audio/x-scpls") ||
            !strcmp (mimestr, "audio/x-pn-realaudio") ||
            !strcmp (mimestr, "audio/vnd.rn-realaudio") ||
            !strcmp (mimestr, "audio/m3u") ||
            !strcmp (mimestr, "audio/x-m3u") ||
            !strncmp (mimestr, "text/", 5) ||
            (!strncmp (mimestr, "application/", 12) &&
             strstr (mimestr + 12,"+xml")) ||
            !strncasecmp (mimestr, "application/smil", 16) ||
            !strncasecmp (mimestr, "application/xml", 15) ||
            !strcmp (mimestr, "application/news_reader") ||
            !strcmp (mimestr, "application/rss+xml") ||
            !strcmp (mimestr, "application/atom+xml") ||
            !strcmp (mimestr, "image/vnd.rn-realpix") ||
            !strcmp (mimestr, "application/x-mplayer2"));
}

static bool isAudioOnlyMime (const String & mime) {
    const char * mimestr = (const char *) mime;
    return mimestr && (!strncmp ((const char *) mime, "audio/", 6));
}

MediaInfo::MediaInfo (Node *n, ActorAgent::ActorType t)
 : node (n), media (NULL), type (t), job (NULL) {
}

MediaInfo::~MediaInfo () {
    clearData ();
    if (media)
        media->dismiss ();
}

KDE_NO_EXPORT void MediaInfo::killWGet () {
    if (job) {
        job->kill (); // quiet, no result signal
        job = 0L;
        clearData (); // assume data is invalid
    }
}

/**
 * Gets contents from url and puts it in data
 */
KDE_NO_EXPORT bool MediaInfo::wget (const String &str) {
    clearData ();
    url = str;

    if (ActorAgent::Any == type || ActorAgent::Image == type) {
        ImageDataMap::iterator i = image_data_map->find (str);
        if (i != image_data_map->end ()) {
            media = new ImageActor (node, i->second);
            ready ();
            return true;
        }
    }

    Mrl *mrl = node->mrl ();
    bool only_playlist = false;
    bool maybe_playlist = false;

    if (mrl && (ActorAgent::Any == type || ActorAgent::AudioVideo == type))
    {
        only_playlist = true;
        mime = mrl->mimetype;
        if (mime == "application/x-shockwave-flash" ||
                mime == "application/futuresplash") {
            ready ();
            return true; // FIXME
        }
    }

    URL uri (str);
    if (mime.isEmpty () && uri.isLocalFile ()) {
        mime = MimeType::findByURL (uri);
        if (mrl)
            mrl->mimetype = mime;
    }
    if (!mime.isEmpty ())
        maybe_playlist = isPlayListMime (mime);

    if (mrl) {
        if (mime.isEmpty ()) {
            for (NodePtr p = mrl->parentNode (); p; p = p->parentNode ()) {
                Mrl *m = p->mrl ();
                if (m && m->linkNode ()->audio_only) {
                    mrl->linkNode ()->audio_only = true;
                    break;
                }
            }
        } else {
            mrl->linkNode ()->audio_only |= isAudioOnlyMime (mime);
        }
    }
    debugLog () << "Actor::wget " << str << endl;
    if (uri.isLocalFile ()) {
        File file (uri.path ());
        if (file.exists () && file.open (IO_ReadOnly)) {
            if (only_playlist) {
                maybe_playlist &= file.size () < 2000000;
                if (maybe_playlist) {
                    char buf[256];
                    off_t nr = file.readBlock (buf, sizeof (buf) - 1);
                    if (nr > 0) {
                        buf[nr] = 0;
                        String mimestr = MimeType::findByContent (buf, nr);
                        if (mimestr.startsWith (String ("text/")) &&
                                strncmp (buf, "RIFF", 4))
                            maybe_playlist = false;
                    }
                }
                if (!maybe_playlist) {
                    ready ();
                    return true;
                }
            }
            data = file.readAll ();
            file.close ();
        }
        ready ();
        return true;
    }

    String protocol = uri.protocol ();
    if (protocol == "mms" || protocol == "rtsp" || protocol == "rtp" ||
            (only_playlist && !maybe_playlist && !mime.isEmpty ())) {
        ready ();
        return true;
    }

    job = asyncGet (this, url);
    return false;
}

KDE_NO_EXPORT bool MediaInfo::readChildDoc () {
    TextStream textstream (data);
    String line;
    do {
        line = textstream.readLine ();
    } while (!line.isNull () && line.stripWhiteSpace ().isEmpty ());
    if (!line.isNull ()) {
        NodePtr cur_elm = node;
        if (cur_elm->isPlayable ())
            cur_elm = cur_elm->mrl ()->linkNode ();
        if (cur_elm->mrl ()->mimetype == String ("audio/x-scpls")) {
            bool groupfound = false;
            int nr = -1;
            struct Entry {
                String url, title;
            } * entries = 0L;
            do {
                line = line.stripWhiteSpace ();
                if (!line.isEmpty ()) {
                    if (line.startsWith (String ("[")) && line.endsWith (String ("]"))) {
                        if (!groupfound && line.mid (1, line.length () - 2).stripWhiteSpace () == String ("playlist"))
                            groupfound = true;
                        else
                            break;
                    } else if (groupfound) {
                        int eq_pos = line.indexOf (Char ('='));
                        if (eq_pos > 0) {
                            if (line.lower ().startsWith (String ("numberofentries"))) {
                                nr = line.mid (eq_pos + 1).stripWhiteSpace ().toInt ();
                                if (nr > 0 && nr < 1024)
                                    entries = new Entry[nr];
                                else
                                    nr = 0;
                            } else if (nr > 0) {
                                String ll = line.lower ();
                                if (ll.startsWith (String ("file"))) {
                                    int i = line.mid (4, eq_pos-4).toInt ();
                                    if (i > 0 && i <= nr)
                                        entries[i-1].url = line.mid (eq_pos + 1).stripWhiteSpace ();
                                } else if (ll.startsWith (String ("title"))) {
                                    int i = line.mid (5, eq_pos-5).toInt ();
                                    if (i > 0 && i <= nr)
                                        entries[i-1].title = line.mid (eq_pos + 1).stripWhiteSpace ();
                                }
                            }
                        }
                    }
                }
                line = textstream.readLine ();
            } while (!line.isNull ());
            NodePtr doc = node->document ();
            for (int i = 0; i < nr; i++)
                if (!entries[i].url.isEmpty ())
                    cur_elm->appendChild (new GenericURL (doc, URL::decode_string (entries[i].url), entries[i].title));
            delete [] entries;
        } else if (line.stripWhiteSpace ().startsWith (Char ('<'))) {
            readXML (cur_elm, textstream, line);
            //cur_elm->normalize ();
        } else if (line.lower () != String ("[reference]")) {
            bool extm3u = line.startsWith ("#EXTM3U");
            String title;
            NodePtr doc = node->document ();
            if (extm3u)
                line = textstream.readLine ();
            while (!line.isNull ()) {
             /* TODO && m_document.size () < 1024 / * support 1k entries * /);*/
                String mrl = line.stripWhiteSpace ();
                if (line == String ("--stop--"))
                    break;
                if (mrl.lower ().startsWith (String ("asf ")))
                    mrl = mrl.mid (4).stripWhiteSpace ();
                if (!mrl.isEmpty ()) {
                    if (extm3u && mrl.startsWith (Char ('#'))) {
                        if (line.startsWith ("#EXTINF:"))
                            title = line.mid (9);
                        else
                            title = mrl.mid (1);
                    } else if (!line.startsWith (Char ('#'))) {
                        cur_elm->appendChild (new GenericURL (doc, mrl, title));
                        title.truncate (0);
                    }
                }
                line = textstream.readLine ();
            }
        }
    }
}

KDE_NO_EXPORT String MediaInfo::mimetype () {
    if (data.size () > 0 && mime.isEmpty ())
        mime = MimeType::findByContent (data.data (), data.size ());
    return mime;
}

KDE_NO_EXPORT void MediaInfo::clearData () {
    killWGet ();
    url.truncate (0);
    mime.truncate (0);
    data.resize (0);
}

KDE_NO_EXPORT bool MediaInfo::downloading () const {
    return !!job;
}

KDE_NO_EXPORT void MediaInfo::create () {
    ActorAgent *mgr = (ActorAgent *)node->document()->message(
            MsgQueryActorAgent);
    if (!media) {
        switch (type) {
            case ActorAgent::Audio:
            case ActorAgent::AudioVideo:
                debugLog() << "create AudioVideo " << data.size () << endl;
                if (!data.size () || !readChildDoc ())
                    media = new AudioVideoActor (mgr, node);
                break;
            case ActorAgent::Image:
                if (data.size () &&
                        (!(mimetype ().startsWith ("text/") ||
                           mime == "image/vnd.rn-realpix") ||
                         !readChildDoc ()))
                    media = new ImageActor (mgr, node, url, data);
                break;
            case ActorAgent::Text:
                media = new TextActor (mgr, node, data);
                break;
            default: // Any
                break;
        }
    }
    data = ByteArray ();
}

KDE_NO_EXPORT void MediaInfo::ready () {
    create ();
    node->document()->post (node, new Posting (node, MsgMediaReady));
}

KDE_NO_EXPORT void MediaInfo::jobResult (Job *jb) {
    if (jb->error ())
        data.resize (0);
    job = 0L; // signal KIO::Job::result deletes itself
    if (mime.isEmpty ()) {
        mime = jb->contentType ();
        if (node->mrl ()) {
            node->mrl ()->mimetype = mime;
            node->mrl ()->audio_only = isAudioOnlyMime (mime);
        }
    }
    ready ();
}

KDE_NO_EXPORT void MediaInfo::jobData (Job *jb, ByteArray &ba) {
    //debugLog () << "ActorTypeRuntime::jobData " << data.size () << endl;
    if (ba.size ()) {
        int old_size = data.size ();
        int newsize = old_size + ba.size ();
        switch (type) {
            case ActorAgent::Any:
                //TODO
            case ActorAgent::Audio:
            case ActorAgent::AudioVideo:
                if (newsize > 2000000 ||
                        (!old_size &&
                         (!MimeType::findByContent (ba.data (), ba.size ()).startsWith ("text/") ||
                          (newsize > 4 && !strncmp (ba.data (), "RIFF", 4))))) {
                    data.resize (0);
                    job->kill (false); // not quiet, wants jobResult
                    return;
                }
                break;
            default:
                //TODO
                break;
        }
        data.resize (newsize);
        memcpy (data.data () + old_size, ba.data (), ba.size ());
    }
}

//------------------------%<----------------------------------------------------

AudioVideoActor::AudioVideoActor (ActorAgent *manager, Node *node)
 : Actor (manager, node),
   ignore_pause (false) {
    debugLog() << "AudioVideoActor::AudioVideoActor" << endl;
}

AudioVideoActor::~AudioVideoActor () {
    stop ();
    debugLog() << "AudioVideoActor::~AudioVideoActor" << endl;
}

bool AudioVideoActor::play () {
    Process *process = m_agent->control ()->process ();
    if (process && process->playing () && process->mrl () == mrl ()) {
        errorLog () << "play: already playing " << mrl ()->src << endl;
    } else if (m_agent->control ()->requestPlayURL (m_node)) {
        if (!mrl ()->audio_only)
            m_agent->control ()->viewArea ()->setAudioVideoNode (m_node);
        ignore_pause = true;
        m_node->defer ();
        ignore_pause = false;
        return true;
    }
    return false;
}

void AudioVideoActor::stop () {
    Process *process = m_agent->control ()->process ();
    if (process)
        process->stop ();
    m_agent->control ()->viewArea ()->setAudioVideoNode (NULL);
}

void AudioVideoActor::pause () {
    Process *process = m_agent->control ()->process ();
    if (!ignore_pause && process)
        process->pause ();
}

void AudioVideoActor::unpause () {
    Process *process = m_agent->control ()->process ();
    if (!ignore_pause && process)
        process->pause ();
}

//------------------------%<----------------------------------------------------

#include <cairo.h>

ImageData::ImageData( const String & img)
 : width (0),
   height (0),
   flags (0),
   has_alpha (false),
   image (0L),
   surface (NULL),
   url (img) {
    //if (img.isEmpty ())
    //    //debugLog() << "New ImageData for " << this << endl;
    //else
    //    //debugLog() << "New ImageData for " << img << endl;
}

ImageData::~ImageData() {
    if (!url.isEmpty ())
        image_data_map->erase (url);
    if (surface)
        cairo_surface_destroy (surface);
    delete image;
}

void ImageData::setImage (Image *img) {
    if (image != img) {
        delete image;
        if (surface)
            cairo_surface_destroy (surface);
        image = img;
        width = img->width ();
        height = img->height ();
        has_alpha = false;
    }
}

ImageActor::ImageActor (ActorAgent *manager, Node *node,
        const String &url, const ByteArray &ba)
 : Actor (manager, node),
   img_movie (NULL),
   img_movie_iter (NULL),
   pix_loader (NULL),
   img_movie_timer (0),
   timeout (0) {
    setupImage (url, ba);
}

ImageActor::ImageActor (Node *node, ImageDataPtr id)
 : Actor ((ActorAgent *)node->document()->message(MsgQueryActorAgent),
         node),
   img_movie (NULL),
   img_movie_iter (NULL),
   pix_loader (NULL),
   img_movie_timer (0),
   timeout (0) {
    cached_img = id;
}

ImageActor::~ImageActor () {
    stop ();
    if (img_movie_iter)
        g_object_unref (img_movie_iter);
    if (pix_loader)
        //gdk_pixbuf_loader_close (pix_loader, 0L);
        g_object_unref (pix_loader);
    else if (img_movie)
        g_object_unref (img_movie);
}

KDE_NO_EXPORT bool ImageActor::play () {
    if (img_movie)
        unpause ();
    return img_movie;
}

KDE_NO_EXPORT void ImageActor::stop () {
    pause ();
}

void ImageActor::pause () {
    if (img_movie_timer)
        g_source_remove (img_movie_timer);
    img_movie_timer = 0;
}

gboolean movieUpdateTimeout (void * p) {
   return ((ImageActor *)p)->movieTimer ();
}

bool ImageActor::movieTimer () {
    bool ret = false;
    if (gdk_pixbuf_animation_iter_advance (img_movie_iter, 0L)) {
        timeout = 0;
        cached_img->setImage (new Image (GPixbuf (
                    gdk_pixbuf_animation_iter_get_pixbuf (img_movie_iter))));
        cached_img->flags = (int) (ImageData::ImagePixmap | ImageData::ImageAnimated);
        int to = gdk_pixbuf_animation_iter_get_delay_time (img_movie_iter);
        if (to == timeout) {
            ret = true; // re-use timer
        } else {
            timeout = to;
            if (to > 0)
                img_movie_timer = g_timeout_add (to, movieUpdateTimeout, this);
        }
        if (m_node)
            m_node->document()->post (m_node, new Posting (m_node, MsgMediaUpdated));
    } else if (m_node) {
        m_node->document()->post (m_node, new Posting(m_node, MsgMediaFinished));
    }
    return ret;
}

void ImageActor::unpause () {
    if (img_movie_iter && !img_movie_timer) {
        timeout = gdk_pixbuf_animation_iter_get_delay_time (img_movie_iter);
        if (timeout > 0)
            img_movie_timer = g_timeout_add (timeout, movieUpdateTimeout, this);
    }
}

KDE_NO_EXPORT
void ImageActor::setupImage (const String &url, const ByteArray &data) {
    GError *gerror = 0L;

    pix_loader = gdk_pixbuf_loader_new ();
    if (gdk_pixbuf_loader_write (pix_loader,
                (const guchar *) data.data (),
                data.size (), &gerror) &&
            gdk_pixbuf_loader_close (pix_loader, 0L)) {
        cached_img = ImageDataPtr (new ImageData (url));
        img_movie = gdk_pixbuf_loader_get_animation (pix_loader);
    }

    if (img_movie && !gdk_pixbuf_animation_is_static_image (img_movie)) {
        img_movie_iter = gdk_pixbuf_animation_get_iter (img_movie, NULL);
        cached_img->setImage (new Image (GPixbuf (
                    gdk_pixbuf_animation_iter_get_pixbuf (img_movie_iter))));
        cached_img->flags = (int) (ImageData::ImagePixmap | ImageData::ImageAnimated);
    } else if (img_movie) {
        cached_img->setImage (new Image (GPixbuf (
                    gdk_pixbuf_animation_get_static_image (img_movie))));
        cached_img->flags = (int) ImageData::ImagePixmap;
        (*image_data_map)[url] = ImageDataPtrW (cached_img);
    }
    g_free (gerror);
}

bool ImageActor::isEmpty () const {
    return !cached_img || cached_img->isEmpty ();
}

//------------------------%<----------------------------------------------------

static int default_font_size = 14;

TextActor::TextActor (ActorAgent *manager, Node *node, const ByteArray &ba)
 : Actor (manager, node) {
    text = String (ba.data (), ba.size ());
    debugLog() << "TextActor::ready " << ba.size () << " " << text << endl;
}

TextActor::~TextActor () {
}

bool TextActor::play () {
    return !text.isEmpty ();
}

int TextActor::defaultFontSize () {
    return default_font_size;
}
