/*
  Kandis - A tea steepming timer.
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Selector.h"

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/Value.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Label.h>
#include <Lum/Panel.h>
#include <Lum/Slider.h>
#include <Lum/Table.h>
#include <Lum/TextValue.h>

Selector::Selector(Lum::Model::Int* minutes, Lum::Model::Int *seconds)
 : value(new Lum::Model::String()),
   minutes(minutes),
   seconds(seconds),
   list(new TeasModel(teas,new TeasDataProvider())),
   selection(new Lum::Model::SingleLineSelection()),
   addAction(new Lum::Model::Action()),
   updateAction(new Lum::Model::Action()),
   removeAction(new Lum::Model::Action()),
   okAction(new Lum::Model::Action())
{
  updateAction->Disable();
  removeAction->Disable();

  Observe(minutes);
  Observe(seconds);

  Observe(selection);

  Observe(addAction);
  Observe(updateAction);
  Observe(removeAction);
  Observe(okAction);

  Observe(GetClosedAction());
}

Lum::Object* Selector::GetContent()
{
  Lum::Label            *label;
  Lum::Panel            *panel,*horiz;
  Lum::Slider           *slider;
  Lum::Table            *table;
  Lum::TextValue        *value;
  Lum::Model::HeaderRef headerModel;

  panel=Lum::VPanel::Create(true,true);

  label=Lum::Label::Create(true,true);

  value=Lum::TextValue::Create(this->value,Lum::TextValue::centered,true,false);
  value->SetFont(Lum::OS::display->GetFont(Lum::OS::Display::fontScaleCaption2));
  value->SetStyle(Lum::OS::Font::bold);
  panel->Add(value);

  panel->AddSpace();

  slider=new Lum::HSlider();
  slider->SetWidth(Lum::Base::Size::stdCharWidth,30);
  slider->SetFlex(true,false);
  slider->SetModel(minutes);
  slider->SetScale(true);
  label->AddLabel(_(L"LBL_MINUTES",L"Minutes"),slider);

  slider=new Lum::HSlider();
  slider->SetWidth(Lum::Base::Size::stdCharWidth,30);
  slider->SetFlex(true,false);
  slider->SetScale(true);
  slider->SetModel(seconds);

  label->AddLabel(_(L"LBL_SECONDS",L"Seconds"),slider);
  panel->Add(label);

  panel->AddSpace();

  horiz=Lum::HPanel::Create(true,true);
  horiz->Add(Lum::VPanel::Create(false,true)
             ->Add(Lum::Button::Create(_(L"BTN_ADD",L"_Add"),addAction,true,false))
             ->Add(Lum::Button::Create(_(L"BTN_UPDATE",L"_Update"),updateAction,true,false))
             ->Add(Lum::Button::Create(_(L"BTN_REMOVE",L"_Remove"),removeAction,true,false)));

  horiz->AddSpace();

  headerModel=new Lum::Model::HeaderImpl();
  headerModel->AddColumn(_(L"TBL_HEADER_TEA",L"Tea"),Lum::Base::Size::stdCharWidth,20,true);
  headerModel->AddColumn(_(L"TBL_HEADER_TIME",L"Time"),Lum::Base::Size::stdCharWidth,5);

  table=new Lum::Table();
  table->SetFlex(true,true);
  table->SetShowHeader(true);
  //table->GetTableView()->SetAutoHFit(true);
  table->SetModel(list);
  table->SetHeaderModel(headerModel);
  table->SetSelection(selection);
  table->SetDoubleClickAction(okAction);
  horiz->Add(table);

  panel->Add(horiz);

  return panel;
}

void Selector::GetActions(std::vector<Lum::Dlg::ActionInfo>& actions)
{
  Lum::Dlg::ActionDialog::CreateActionInfosOkCancel(actions,okAction,GetClosedAction());
}

void Selector::Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
{
  if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
    Exit();
  }
  else if (model==okAction && okAction->IsFinished()) {
    minutes->Save();
    seconds->Save();
    Exit();
  }
  else if (model==minutes || model==seconds) {
    std::wstring text;

    text=Lum::Base::NumberToWString(minutes->Get());
    text.append(L":");
    if (seconds->Get()<10) {
      text.append(L"0");
    }
    text.append(Lum::Base::NumberToWString(seconds->Get()));

    value->Set(text);
  }
  else if (model==selection && dynamic_cast<const Lum::Model::Selection::Selected*>(&msg)!=NULL) {
    if (selection->HasSelection()) {
      updateAction->Enable();
      removeAction->Enable();

      minutes->Set(teas[selection->GetLine()-1].minutes);
      seconds->Set(teas[selection->GetLine()-1].seconds);
    }
    else {
      updateAction->Disable();
      removeAction->Disable();
    }
  }
  else if (model==addAction && addAction->IsFinished()) {
    Lum::Model::StringRef name=new Lum::Model::String(_(L"TEA_NAME_DEFAULT",L"Some tea"));

    if (Lum::Dlg::TextInput::GetText(this->GetWindow(),
                                     L"",
                                     _(L"DLG_TEA_NAME_TEXT",
                                       L"Please enter the name of the tea:"),
                                       name)) {
      Tea tea;

      // The tea list
      tea.name=name->Get();
      tea.minutes=minutes->Get();
      tea.seconds=seconds->Get();

      // The selection list
      list->Append(tea);

      configurationChanged=true;
    }
  }
  else if (model==updateAction && updateAction->IsFinished()) {
    assert(selection->HasSelection());

    teas[selection->GetLine()-1].minutes=minutes->Get();
    teas[selection->GetLine()-1].seconds=seconds->Get();

    list->RedrawRow(selection->GetLine());

    configurationChanged=true;
  }
  else if (model==removeAction && removeAction->IsFinished()) {
    assert(selection->HasSelection());

    list->Delete(selection->GetLine());

    configurationChanged=true;
  }

  Dialog::Resync(model,msg);
}

