/*
  Kandis - A tea steepming timer.
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "StopWatch.h"

#include <Lum/Base/String.h>

StopWatch::StopWatch()
 : clickAction(new Lum::Model::Action()),
   blinkTimer(new Lum::Model::Action()),
   alarmColor(1.0,0,0,Lum::OS::Display::textColor),
   showInvert(false)
{
  if (Lum::OS::display->GetType()==Lum::OS::Display::typeTextual) {
    minFont=Lum::OS::display->GetFont(Lum::OS::Display::fontTypeProportional,
                                      Lum::OS::Display::fontScaleNormal);
    font=minFont;
    }
  else {
    minFont=Lum::OS::display->GetFont(Lum::OS::Display::fontTypeProportional,
                                      1000);
    font=minFont;
  }

  AttachModel(blinkTimer);
}

StopWatch::~StopWatch()
{
  UnattachModel(blinkTimer);
}

Lum::Model::Action* StopWatch::GetClickAction() const
{
  return clickAction.Get();
}

void StopWatch::SetText(const std::wstring& text)
{
  this->text=text;
  Redraw();
}

void StopWatch::SetAlarm(bool alarm)
{
  if (this->alarm!=alarm) {
    this->alarm=alarm;
    showInvert=false;
    Redraw();
  }
}

void StopWatch::CalcSize()
{
  size_t w;

  Lum::OS::FontExtent extent;

  w=0;
  for (size_t i=1; i<=9; i++) {
    minFont->StringExtent(Lum::Base::NumberToWString(i),extent,Lum::OS::Font::bold);

    w=std::max(w,extent.width);
  }

  minFont->StringExtent(L":",extent,Lum::OS::Font::bold);

  minWidth=4*w+extent.width;
  minHeight=minFont->height;

  w=0;
  for (size_t i=1; i<=9; i++) {
    font->StringExtent(Lum::Base::NumberToWString(i),extent,Lum::OS::Font::bold);

    w=std::max(w,extent.width);
  }

  font->StringExtent(L":",extent,Lum::OS::Font::bold);

  width=4*w+extent.width;
  height=font->height;

  Lum::Control::CalcSize();
}

Lum::OS::Font* StopWatch::FindFont()
{
  Lum::OS::FontRef font;
  bool             fits;
  size_t           fontSize;

  font=minFont;
  fontSize=font->pixelHeight+1;

  do {
    Lum::OS::FontRef tmpFont;
    size_t           w=0,h=0;

    tmpFont=Lum::OS::display->GetFontByPixel(Lum::OS::Display::fontTypeProportional,fontSize);

    if (tmpFont.Valid()) {
      Lum::OS::FontExtent extent;

      for (size_t i=1; i<=9; i++) {
        tmpFont->StringExtent(Lum::Base::NumberToWString(i),extent,Lum::OS::Font::bold);

        w=std::max(w,extent.width);
        h=std::max(h,extent.height);
      }

      tmpFont->StringExtent(L":",extent,Lum::OS::Font::bold);
      h=std::max(h,extent.height);

      fits=(4*w+extent.width<=width && h<=height);

      if (fits) {
        font=tmpFont;
        fontSize++;
      }
    }
    else {
      fits=(fontSize<=height);
    }
  } while (fits);

  return font.Get();
}

void StopWatch::Layout()
{
  font=FindFont();

  Control::Layout();
}

void StopWatch::Draw(int x, int y, size_t w, size_t h)
{
  Control::Draw(x,y,w,h);

  if (!Intersect(x,y,w,h)) {
    return;
  }

  /* --- */

  Lum::OS::DrawInfo   *draw=GetDrawInfo();
  Lum::OS::FontExtent extent;

  DrawBackground(x,y,w,h);

  draw->PushFont(font,Lum::OS::Font::bold);

  if (alarm && showInvert) {
    draw->PushForeground(alarmColor);
  }
  else if (alarm) {
    draw->PushForeground(alarmColor);
    draw->FillRectangle(this->x,this->y,this->width,this->height);
    draw->PopForeground();

    draw->PushForeground(Lum::OS::Display::textColor);
  }
  else {
    draw->PushForeground(Lum::OS::Display::textColor);
  }

  font->StringExtent(text,extent,Lum::OS::Font::bold);

  draw->DrawString(this->x+(width-extent.width)/2,
                   this->y+(height-extent.height)/2+extent.ascent,
                   text);

  draw->PopForeground();
  draw->PopFont();

  if (alarm) {
    Lum::OS::display->RemoveTimer(blinkTimer);
    Lum::OS::display->AddTimer(1,0,blinkTimer);
  }
}

bool StopWatch::HandleMouseEvent(const Lum::OS::MouseEvent& event)
{
  if (event.type==Lum::OS::MouseEvent::down && PointIsIn(event) && event.button==Lum::OS::MouseEvent::button1) {
    return true;
  }
  else if (event.type==Lum::OS::MouseEvent::up && PointIsIn(event) && event.IsGrabed() && event.button==Lum::OS::MouseEvent::button1) {
    clickAction->Trigger();
  }

  return false;
}

void StopWatch::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==blinkTimer) {
    showInvert=!showInvert;
    Redraw();
  }

  Control::Resync(model,msg);
}

