/*
  This source is part of the Illumination library
  Copyright (C) 2008  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Wizard.h>

#include <Lum/Base/L10N.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Multi.h>
#include <Lum/Panel.h>
#include <Lum/Space.h>
#include <Lum/Text.h>

#include <iostream>
namespace Lum {

  Wizard::Wizard()
  : backAction(new Lum::Model::Action()),
    page(new Lum::Model::SizeT(0))
  {
    Observe(backAction);
  }

  Wizard::~Wizard()
  {
    // no code
  }

  void Wizard::SetCancelAction(Lum::Model::Action* cancelAction)
  {
    this->cancelAction=cancelAction;
  }

  void Wizard::AddPage(const std::wstring& title,
                       Lum::Object* content,
                       Lum::Model::Action* continueAction,
                       Lum::Model::Action* helpAction)
  {
    assert(!inited);
    assert(content!=NULL);

    Page page;

    page.title=title;
    page.content=content;
    page.continueAction=continueAction!=NULL ? continueAction : new Lum::Model::Action();
    page.helpAction=helpAction;

    Observe(page.continueAction);

    pages.push_back(page);
  }

  void Wizard::CalcSize()
  {
    assert(pages.size()>0);

    Multi *multi;

    multi=new Multi();
    multi->SetFlex(true,true);
    multi->SetModel(page);

    for (size_t i=0; i<pages.size(); i++) {
      Button    *button;
      ButtonRow *row;
      Panel     *panel;
      Text      *title;

      panel=new Lum::VPanel();
      panel->SetFlex(true,true);

      title=new Lum::Text(pages[i].title,
                          OS::Font::bold,
                          Text::left,
                          OS::display->GetFont(OS::Display::fontTypeProportional,
                                               OS::Display::fontScaleCaption2));
      title->SetFlex(true,false);
      panel->Add(title);

      panel->Add(new VSpace());

      panel->Add(pages[i].content);
      panel->Add(new VSpace());

      row=new Lum::ButtonRow();
      row->SetFlex(true,false);

      if (pages[i].helpAction.Valid()) {
        button=Button::CreateHelp(pages[i].helpAction);
        button->SetFlex(true,true);

        row->AddOptional(button);
      }

      if (i>0) {
        button=new Lum::Button();
        button->SetFlex(true,true);
        button->SetModel(backAction);
        button->SetText(L"_Previous");
        row->Add(button);
      }

      if (i==pages.size()-1) {
        button=new Lum::Button();
        button->SetFlex(true,true);
        button->SetModel(pages[i].continueAction);
        button->SetText(L"_Finish");
        row->Add(button);
      }
      else {
        button=new Lum::Button();
        button->SetFlex(true,true);
        button->SetModel(pages[i].continueAction);
        button->SetText(L"_Next");
        row->Add(button);
      }

      if (cancelAction.Valid()) {
        button=Button::CreateCancel(cancelAction);
        button->SetFlex(true,true);
        row->Add(button);
      }

      panel->Add(row);

      multi->Append(panel);
    }

    container=multi;

    Component::CalcSize();
  }

  void Wizard::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    if (model==backAction && backAction->IsFinished()) {
      page->Dec();
    }
    else if (pages.size()>0) {
      for (size_t i=0; i<pages.size()-1; i++) {
        if (model==pages[i].continueAction && pages[i].continueAction->IsFinished()) {
          page->Inc();
        }
      }
    }

    Component::Resync(model,msg);
  }
}

