/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/TableView.h>

#include <cmath>
#include <cstdlib>
#include <memory>

#include <Lum/Base/Size.h>

#include <Lum/Table.h>

namespace Lum {

#define STEP 10

  static ControllerRef controller(new TableView::Controller);

  TableView::FormatProvider::~FormatProvider()
  {
    // no code
  }

  TableView::Alignment TableView::FormatProvider::GetAlignment(size_t /*column*/, size_t /*row*/) const
  {
    return left;
  }

  OS::Color TableView::FormatProvider::GetTextColor(size_t column, size_t row) const
  {
    return OS::display->GetColor(OS::Display::tableTextColor);
  }

  OS::Fill* TableView::FormatProvider::GetBackground(size_t column, size_t row) const
  {
    if (row%2!=0) {
      return OS::display->GetFill(OS::Display::tableBackgroundFillIndex);
    }
    else {
      return OS::display->GetFill(OS::Display::tableBackground2FillIndex);
    }
  }

  TableView::TableView()
  : selectionAction(new Model::Action),
    doubleClickAction(new Model::Action),
    mouseSelectionAction(new Model::Action),
    updateScrollerAction(new Model::Action),
    autoFitColumns(false),
    autoHSize(false),
    autoVSize(false),
    currentPos(1),
    scrolling(false),
    scrollRefresh(false),
    scrollTimerAction(new Model::Action()),
    px(0),
    py(0),
    ppos(0),
    my(0),
    sy(0),
    mouseTimerAction(new Model::Action()),
    tableWidth(0),
    rowHeight(0),
    formatProvider(new FormatProvider()),
    font(OS::display->GetFont())
  {
    SetBackground(OS::display->GetFill(OS::Display::tableBackgroundFillIndex));

    SetCanFocus(true);
    RequestFocus();

    SetController(::Lum::controller);

    Observe(updateScrollerAction);
    Observe(scrollTimerAction);
    Observe(mouseTimerAction);
  }

  TableView::~TableView()
  {
    delete formatProvider;
  }

  /**
    Return the cell under the given coordinates. Returns false if there is no
    cell at the given position.
  */
  bool TableView::GetCell(size_t mx, size_t my, size_t& x, size_t& y) const
  {
    assert(model.Valid() && PointIsIn(mx,my));

    if (model->GetRows()==0) {
      return false;
    }

    assert(rowHeight!=0);

    size_t start;

    y=((my-this->y)+currentPos-1+rowHeight-1)/rowHeight;

    if (y==0 || y>model->GetRows()) {
      return false;
    }

    mx-=this->x-hAdjustment->GetTop();

    start=0;
    for (x=0; x<header->GetColumns(); x++) {
      if ((mx>=start) && (mx<start+header->GetColumnWidth(x))) {
        return true;
      }
      start+=header->GetColumnWidth(x);
    }

    return false;
  }

  /**
    Redisplay text in that way, that the given point can be seen
  */
  void TableView::MakeVisible(size_t x, size_t y)
  {
    size_t pos;

    vAdjustment->MakeVisible(y*rowHeight-(rowHeight-1));
    vAdjustment->MakeVisible(y*rowHeight);

    pos=0;
    for (size_t i=0; i<x-1; i++) {
      pos+=header->GetColumnWidth(i);
    }

    hAdjustment->MakeVisible(pos);
  }

  /**
    If \p autoFitColumns is true and the table is wider than to sum of the size of all columns
    the table searches for flexible columns and will resize them to fit
    all remaining space.
  */
  void TableView::SetAutoFitColumns(bool autoFit)
  {
    autoFitColumns=autoFit;
  }

  /**
    If \p autoSize is 'true', the initial width of the
    table is be set that way, that all columns are as small as their content
    and all columns are visible if the maximum width of the table allows.

    Use Object.SetMaxWidth to restrict the maximum
    width of the table. Be aware that the table might exceed
    screen width, if you do not.
  */
  void TableView::SetAutoHSize(bool autoSize)
  {
    autoHSize=autoSize;
  }

  /**
    If \p autoSize is 'true', the initial height of the
    table is be set that way, that all entries are visible.

    Use Object.SetMaxHeight the restrict the maximum
    height of the table. Be aware that the table might exceed
    screen height, if do not.
  */
  void TableView::SetAutoVSize(bool autoSize)
  {
    autoVSize=autoSize;
  }

  /**
    Set the height of a single row in the table to a non-standard value.
    By default, the row height is choosen that way, that text using the
    standard font is completly visible. A non-standard height might be
    necessary if you choose a non-standard font are graphical elements.
  */
  void TableView::SetRowHeight(int height)
  {
    rowHeight=height;
  }

  void TableView::RecalcTableWidth()
  {
    tableWidth=0;
    for (size_t column=0; column<header->GetColumns(); column++) {
      tableWidth+=header->GetColumnWidth(column);
    }
  }

  bool TableView::SetModel(Base::Model* model)
  {
    this->model=dynamic_cast<Model::Table*>(model);

    if (this->model.Valid())  {
      currentPos=1;
    }

    Control::SetModel(this->model);

    return this->model.Valid();
  }

  void TableView::SetHeader(Model::Header *header)
  {
    if (this->header.Valid()) {
      Forget(this->header);
    }

    this->header=header;

    if (this->header.Valid()) {
      Observe(this->header);
    }
  }

  /**
    Set the Model::Selection object responsible for holding all information
    about the current selection. By default Model::NoneSelection ist set.

    You must assign a valid instance of a class derived from Model::Selection.
  */
  void TableView::SetSelection(Model::Selection* selection)
  {
    assert(selection!=NULL);

    if (this->selection.Valid()) {
      this->selection->Clear();
      Forget(this->selection);
    }

    this->selection=selection;

    if (this->selection.Valid()) {
      Observe(this->selection);
    }
  }

  void TableView::SetSelectionAction(Model::Action* action)
  {
    selectionAction=action;
  }

  void TableView::SetMouseSelectionAction(Model::Action* action)
  {
    mouseSelectionAction=action;
  }

  void TableView::SetDoubleClickAction(Model::Action* action)
  {
    doubleClickAction=action;
  }

  /**
    Assign a new FormatProvider.

    Previously assigned provider will be deleted.
    The resource ownership of the assigned format provider will be hold by the table.
    You must not free an assigned format provider anytime after the assignment.
  */
  void TableView::SetFormatProvider(FormatProvider* formatProvider)
  {
    assert(formatProvider!=NULL);

    delete this->formatProvider;
    this->formatProvider=formatProvider;

    Redraw();
  }

  Model::Table* TableView::GetModel() const
  {
    return model.Get();
  }

  Model::Selection* TableView::GetSelection() const
  {
    return selection.Get();
  }

  Model::Action* TableView::GetSelectionAction() const
  {
    return selectionAction.Get();
  }

  Model::Action* TableView::GetDoubleClickAction() const
  {
    return doubleClickAction.Get();
  }

  Model::Action* TableView::GetMouseSelectionAction() const
  {
    return mouseSelectionAction.Get();
  }

  void TableView::FitColumn(size_t column)
  {
    size_t width;

    if (!model.Valid() || !header.Valid()) {
      return;
    }

    width=2*OS::display->GetSpaceHorizontal(OS::Display::spaceObjectBorder);
    for (size_t y=1; y<=model->GetRows(); y++) {
      std::wstring string=model->GetString(column,y);
      if (!string.empty()) {
        OS::FontExtent extent;

        font->StringExtent(string,extent);
        width=std::max(width,extent.width);
      }
      else {
        Object *object=model->GetObject(column,y);
        if (object!=NULL)  {
          if (!inited || (object->GetWindow()!=GetWindow()))  {
            object->SetWindow(GetWindow());
            object->SetParent(this);
            object->CalcSize();
          }
          width=std::max(width,(size_t)object->GetOWidth());
        }
      }
    }

    header->SetColumnWidth(column-1,width);
  }

  void TableView::CalculateRowHeight()
  {
    if (rowHeight!=0 || !model.Valid() || model->GetRows()==0) {
      return;
    }

    for (size_t x=1; x<=this->header->GetColumns(); x++) {
      Object *object=this->model->GetObject(x,1);

      if (object!=NULL) {
        if (!object->IsInited()) {
          object->SetParent(this);
          object->SetWindow(GetWindow());
          object->CalcSize();
        }

        rowHeight=std::max(rowHeight,object->GetOHeight());

      }
      else {
        rowHeight=std::max(rowHeight,font->height);
      }
    }
  }

  void TableView::FitColumns()
  {
    size_t targetWidth=this->width;

    // This is a hack!
    // TableView does know, that GetParent()->GetParent() is normally of type
    // Table and requests width of vertical scroller to resize labels that way
    // that content does not scroll out of view...
    if (header->GetColumns()>1 &&
        firstDraw &&
        GetParent()!=NULL && dynamic_cast<Table*>(GetParent()->GetParent())!=NULL/* &&
        vAdjustment->GetVisible()<hAdjustment->GetTotal()*/) {
      targetWidth-=dynamic_cast<Table*>(GetParent()->GetParent())->GetVerticalScrollerWidth();
    }

    if (header->GetColumns()==1) {
      header->SetColumnWidth(0,targetWidth);
    }
    else {
      size_t w=0;
      size_t flexs=0;

      for (size_t column=0; column<header->GetColumns(); column++) {
        w+=header->GetColumnWidth(column);
        if (header->CanColumnChangeSize(column)) {
          ++flexs;
        }
      }

      if (w!=targetWidth && flexs>0) {
        for (size_t column=0; column<header->GetColumns(); column++) {
          if (header->CanColumnChangeSize(column)) {
            if (w<targetWidth) {
              header->SetColumnWidth(column,header->GetColumnWidth(column)+(targetWidth-w)/flexs);
              w+=(targetWidth-w)/flexs;
            }
            else {
              header->SetColumnWidth(column,header->GetColumnWidth(column)-(w-targetWidth)/flexs);
              w+=(w-targetWidth)/flexs;
            }
          }
        }

        RecalcTableWidth();
      }
    }
  }

  /**
    Returns the vertical on-screen bounds for the given line. Returns false if the either
    the table or the line is not visible.
  */
  bool TableView::GetLineBounds(size_t line, int& ly, size_t& lheight) const
  {
    if (!visible) {
      return false;
    }

    size_t start,end;

    start=(line-1)*rowHeight;
    end=line*rowHeight-1;

    // Is line visible at all?

    if (end+1<currentPos) {
      return false;
    }

    if (start+1>currentPos+height-1) {
      return false;
    }

    // From now on at least something must be visible

    if (start<currentPos-1) {
      start=currentPos-1;
    }

    if (end>currentPos-1+height-1) {
      end=currentPos-1+height-1;
    }

    ly=start+1-currentPos+y;
    lheight=end-start+1;

    return true;
  }

  void TableView::UpdateScroller()
  {
    if (model.Valid() && !model->IsNull()) {
      hAdjustment->SetStepSize(font->StringWidth(L"m"));
      hAdjustment->SetDimension(width,tableWidth);
      if (rowHeight!=0) {
        vAdjustment->SetStepSize(rowHeight);
        vAdjustment->SetDimension(height,model->GetRows()*rowHeight);
      }
    }
    else {
      hAdjustment->SetInvalid();
      vAdjustment->SetInvalid();
    }
  }

  /* ---------- Printing --------------- */

  OS::Color TableView::GetTextColor(OS::DrawInfo *draw) const
  {
    if (draw->selected) {
      return OS::display->GetColor(OS::Display::fillTextColor);
    }
    else {
      return OS::display->GetColor(OS::Display::tableTextColor);
    }
  }

  void TableView::DrawCell(OS::DrawInfo* draw,
                           int xPos,
                           int yPos,
                           size_t width,
                           size_t height,
                           int x,
                           int y,
                           bool drawBackground)
  {
    Alignment    alignment;
    bool         selected,cliped;
    std::wstring string;
    Object       *object;
    OS::Fill*    back;

    selected=selection.Valid() && (selection->IsLineSelected(y) || selection->IsCellSelected(x,y));
    alignment=formatProvider->GetAlignment(x,y);

    if (selected) {
      back=OS::display->GetFill(OS::Display::tableBackgroundFillIndex);
      draw->selected=true;
    }
    else {
      back=formatProvider->GetBackground(x,y);
    }

    cliped=false;
    string=model->GetString(x,y);
    object=model->GetObject(x,y);

    if (!string.empty()) {
      OS::FontExtent extent;
      int            textStart=xPos; // To make compiler happy

      if (drawBackground) {
        back->Draw(draw,xPos,yPos,width,height,xPos,yPos,width,height);
      }

      if (selected) {
        draw->PushForeground(OS::Display::fillTextColor);
      }
      else {
        draw->PushForeground(formatProvider->GetTextColor(x,y));
      }

      font->StringExtent(string,extent);
      cliped=extent.width>width;

      if (cliped) {
        draw->PushClip(xPos,yPos,width,height);
      }

      switch (alignment) {
      case center:
        textStart=xPos+(width-extent.width) / 2;
        break;
      case right:
        textStart=xPos+width-extent.width;
        break;
      case left:
        textStart=xPos;
        break;
      }

      draw->DrawString(textStart,yPos+font->ascent+((int)height-(int)font->height) / 2,string);

      draw->PopForeground();
    }
    else if (object!=NULL) {
      bool focused=draw->focused;

      draw->focused=false;

      if (!inited || (object->GetWindow()!=GetWindow())) {
        object->SetWindow(GetWindow());
        object->SetParent(this);
        object->CalcSize();
      }

      object->Resize(width,height);
      object->Move(xPos,yPos+((int)height-(int)object->GetOHeight()) / 2);

      if (drawBackground) {
        back->Draw(draw,xPos,yPos,width,height,xPos,yPos,width,height);
      }

      cliped=object->GetOWidth()>width;

      if (cliped) {
        draw->PushClip(xPos,yPos,width,height);
      }

      if (object->GetBackground()==NULL) {
        object->SetBackground(new Lum::OS::EmptyFill());
      }
      object->Draw(xPos,yPos,object->GetOWidth(),object->GetOHeight());

      draw->focused=focused;
    }
    else if (drawBackground) {
      back->Draw(draw,xPos,yPos,width,height,xPos,yPos,width,height);
    }

    if (selected && selection.Valid() && !selection->IsLineSelected(y) && HasFocus()) {
      OS::FrameRef focus=OS::display->GetFrame(OS::Display::focusFrameIndex);

      /* Draw a frame arond one cell */
      focus->Draw(draw,xPos,yPos,width,height);
    }

    if (cliped) {
      draw->PopClip();
    }

    draw->selected=false;
  }

  /**
    Print the given line at the given display position.
  */
  void TableView::DrawLine(OS::DrawInfo* draw, int y, size_t line)
  {
    if (line>model->GetRows()) {
      DrawBackground(this->x,y,width,rowHeight);
      return;
    }

    int  x=this->x-hAdjustment->GetTop()+1;
    bool drawBackground=true;

    if (selection.Valid() && selection->IsLineSelected(line)) {
      OS::Fill* back;

      back=OS::display->GetFill(OS::Display::tableBackgroundFillIndex);

      drawBackground=false;

      draw->selected=true;
      back->Draw(draw,x,y,tableWidth,rowHeight,x,y,tableWidth,rowHeight);
      draw->selected=false;
    }

    for (size_t column=0; column<header->GetColumns(); column++) {
      size_t columnWidth;

      columnWidth=header->GetColumnWidth(column);

      if (!(x>=this->x+(int)this->width) && !(x+(int)columnWidth-1<this->x)) { /* Draw only if visible */
        DrawCell(draw,x,y,columnWidth,rowHeight,column+1,line,drawBackground);
      }
      x+=(int)columnWidth;
    }

    if (x<(int)(this->x+width)) { /* Fill space behind last cell in row */
      OS::Fill* back;

      if (line%2!=0) {
        back=OS::display->GetFill(OS::Display::tableBackgroundFillIndex);
      }
      else {
        back=OS::display->GetFill(OS::Display::tableBackground2FillIndex);
      }

      back->Draw(draw,
                 x,y,this->x+width-x+1,rowHeight,
                 x,y,this->x+width-x+1,rowHeight);

    }

    if (selection.Valid() && selection->IsLineSelected(line) && HasFocus()) {
      OS::FrameRef focus=OS::display->GetFrame(OS::Display::focusFrameIndex);

      /* Draw a frame arond whole line */
      if (header->GetColumns()==1) {
        focus->Draw(draw,this->x,y,this->width,rowHeight);
      }
      else {
        focus->Draw(draw,this->x-hAdjustment->GetTop()+1,y,tableWidth,rowHeight);
      }
    }
  }

  void TableView::DrawRect(OS::DrawInfo *draw, size_t start, size_t height)
  {
    size_t startLine;
    size_t endLine;
    int    startPos;
    int    endPos;

    draw->PushClip(x,y+start,width,height);
    draw->PushFont(font);
    draw->focused=HasFocus();

    startLine=(currentPos-1+start)/rowHeight;
    endLine=(currentPos-1+start+height-1)/rowHeight;

    startPos=startLine*rowHeight+1-currentPos;
    endPos=(endLine+1)*rowHeight+1-currentPos-1;

    startPos+=y;
    endPos+=y;

    while (startPos<=endPos) {
      DrawLine(draw,startPos,startLine+1);
      startPos+=rowHeight;
      startLine++;
    }

    draw->focused=false;
    draw->PopFont();
    draw->PopClip();
  }

  void TableView::DrawFull(OS::DrawInfo *draw)
  {
    DrawRect(draw,0,height);
  }

  void TableView::RedrawLine(size_t line)
  {
    int    ly=0;
    size_t lh=0;

    if (GetLineBounds(line,ly,lh)) {
      scrollRefresh=false;
      Redraw(this->x,ly,width,lh);
    }
  }

  void TableView::CalcSize()
  {
    minWidth=10;
    minHeight=3*font->height;

    width=minWidth;

    if (autoVSize && model.Valid()) {
      height=model->GetRows()*font->height;
      if (header->GetColumns()==1 && !autoHSize) {
        // This is a hack!
        // TableView does know, that GetParent()->GetParent() is normally of type
        // Table and requests width of vertical scroller to resize width that all labels
        // fit.
        if (GetParent()!=NULL && dynamic_cast<Table*>(GetParent()->GetParent())!=NULL) {
          height+=dynamic_cast<Table*>(GetParent()->GetParent())->GetHorizontalScrollerHeight();
        }
      }
    }
    else {
      height=minHeight;
    }

    if (autoHSize && model.Valid() && header.Valid()) {
      size_t w=0;

      for (size_t column=0; column<header->GetColumns(); column++) {
        if (header->GetMinColumnWidth(column)==0) {
          FitColumn(column+1);
        }
        w+=header->GetColumnWidth(column);
      }

      // This is a hack!
      // TableView does know, that GetParent()->GetParent() is normally of type
      // Table and requests width of vertical scroller to resize width that all labels
      // fit.
      if (GetParent()!=NULL && dynamic_cast<Table*>(GetParent()->GetParent())!=NULL) {
        w+=dynamic_cast<Table*>(GetParent()->GetParent())->GetVerticalScrollerWidth();
      }

      width=w;
      minWidth=w;
      tableWidth=w;
    }
    else {
      RecalcTableWidth();
    }

    Scrollable::CalcSize();
  }

  void TableView::Layout()
  {
    // If we are still scorlling during a resize event, stop scrolling
    scrollRefresh=false;

    CalculateRowHeight();

    if (model.Valid() && header.Valid() && autoFitColumns) {
      FitColumns();
    }

    UpdateScroller();

    Scrollable::Layout();
  }

  void TableView::Draw(int x, int y, size_t w, size_t h)
  {
    Scrollable::Draw(x,y,w,h);

    if (!OIntersect(x,y,w,h)) {
      return;
    }

    /* --- */

    CalculateRowHeight();

    if (model.Valid() && model->GetRows()>0) {
      RecalcTableWidth();

      OS::display->QueueActionForEventLoop(updateScrollerAction);

      Lum::OS::DrawInfo *draw=GetDrawInfo();

      scrolling=(currentPos!=vAdjustment->GetTop());

      if (OS::display->GetType()==OS::Display::typeTextual) {
        currentPos=vAdjustment->GetTop();
        DrawFull(draw);
        return;
      }

      if (scrolling && scrollRefresh) {
        size_t recentPos=currentPos;
        size_t diff;

        if (currentPos<vAdjustment->GetTop()) {
          size_t step;

          step=(vAdjustment->GetTop()-currentPos)/STEP;
          //step=(int)(sqrt(vAdjustment->GetTop()-currentPos)+0.5);

          if (step==0) {
            step=1;
          }

          currentPos+=step;
        }
        else {
          size_t step;

          step=(currentPos-vAdjustment->GetTop())/STEP;
          //step=(int)(sqrt(currentPos-vAdjustment->GetTop())+0.5);

          if (step==0) {
            step=1;
          }

          currentPos-=step;
        }

        diff=std::max(recentPos,currentPos)-std::min(recentPos,currentPos);

        if (diff>=height) {
          DrawFull(draw);
        }
        else if (currentPos>recentPos) {
          draw->CopyArea(this->x,this->y+diff,width,height-diff,this->x,this->y);
          DrawRect(draw,height-diff,diff);
        }
        else if (recentPos>currentPos) {
          draw->CopyArea(this->x,this->y,width,height-diff,this->x,this->y+diff);
          DrawRect(draw,0,diff);
        }

        scrolling=(currentPos!=vAdjustment->GetTop());

        if (scrolling) {
          Lum::OS::display->AddTimer(0,1000000/25,scrollTimerAction);
        }
        else {
          scrollRefresh=false;
        }
      }
      else {
        scrollRefresh=false;
        DrawRect(draw,y-this->y,h);
      }
    }
    else {
      DrawBackground(x,y,w,h);
    }
  }

  void TableView::Hide()
  {
    scrollRefresh=false;
    scrolling=false;

    Scrollable::Hide();
  }

  void TableView::DrawFocus()
  {
    if (!model.Valid()) {
      return;
    }

    Model::SingleCellSelection *singleCell;
    Model::SingleLineSelection *singleLine;

    if ((singleCell=dynamic_cast<Model::SingleCellSelection*>(selection.Get()))!=NULL) {
      if (singleCell->HasSelection()) {
        RedrawLine(singleCell->GetRow());
      }
    }
    else if ((singleLine=dynamic_cast<Model::SingleLineSelection*>(selection.Get()))!=NULL) {
      if (singleLine->HasSelection()) {
        RedrawLine(singleLine->GetLine());
      }
    }
  }

  void TableView::HideFocus()
  {
    if (!model.Valid()) {
      return;
    }

    Model::SingleCellSelection *singleCell;
    Model::SingleLineSelection *singleLine;

    if ((singleCell=dynamic_cast<Model::SingleCellSelection*>(selection.Get()))!=NULL) {
      if (singleCell->HasSelection()) {
        RedrawLine(singleCell->GetRow());
      }
    }
    else if ((singleLine=dynamic_cast<Model::SingleLineSelection*>(selection.Get()))!=NULL) {
      if (singleLine->HasSelection()) {
        RedrawLine(singleLine->GetLine());
      }
    }
  }

  void TableView::OnSelection()
  {
    if (selectionAction.Valid()) {
      selectionAction->Trigger();
    }
  }

  void TableView::OnMouseSelection()
  {
    if (mouseSelectionAction.Valid()) {
      mouseSelectionAction->Trigger();
    }
  }

  void TableView::ActionLeft()
  {
    Model::SingleCellSelection *singleCell;

    if ((singleCell=dynamic_cast<Model::SingleCellSelection*>(selection.Get()))!=NULL) {
      if (singleCell->HasSelection() && singleCell->GetColumn()>1) {
        singleCell->SelectCell(singleCell->GetColumn()-1,singleCell->GetRow());
        MakeVisible(singleCell->GetColumn(),singleCell->GetRow());
      }
    }
    else {
      hAdjustment->DecTop();
    }
  }

  void TableView::ActionRight()
  {
    Model::SingleCellSelection *singleCell;

    if ((singleCell=dynamic_cast<Model::SingleCellSelection*>(selection.Get()))!=NULL) {
      if (singleCell->HasSelection() && singleCell->GetColumn()<header->GetColumns()) {
        singleCell->SelectCell(singleCell->GetColumn()+1,singleCell->GetRow());
        MakeVisible(singleCell->GetColumn(),singleCell->GetRow());
      }
    }
    else {
      hAdjustment->IncTop();
    }
  }

  void TableView::ActionUp()
  {
    Model::SingleCellSelection *singleCell;
    Model::SingleLineSelection *singleLine;

    if ((singleCell=dynamic_cast<Model::SingleCellSelection*>(selection.Get()))!=NULL) {
      if (singleCell->GetRow()>1) {
        singleCell->SelectCell(singleCell->GetColumn(),singleCell->GetRow()-1);
        MakeVisible(singleCell->GetColumn(),singleCell->GetRow());
      }
    }
    else if ((singleLine=dynamic_cast<Model::SingleLineSelection*>(selection.Get()))!=NULL) {
      if (singleLine->HasSelection() && singleLine->GetLine()>1) {
        singleLine->SelectLine(singleLine->GetLine()-1);
        MakeVisible(1,singleLine->GetLine());
      }
    }
    else {
      vAdjustment->DecTop();
    }
  }

  void TableView::ActionDown()
  {
    Model::SingleCellSelection *singleCell;
    Model::SingleLineSelection *singleLine;

    if ((singleCell=dynamic_cast<Model::SingleCellSelection*>(selection.Get()))!=NULL) {
      if (singleCell->HasSelection()) {
        if (singleCell->GetRow()<model->GetRows()) {
          singleCell->SelectCell(singleCell->GetColumn(),singleCell->GetRow()+1);
        }
      }
      else {
        singleCell->SelectCell(1,1);
      }
      MakeVisible(singleCell->GetColumn(),singleCell->GetRow());
    }
    else if ((singleLine=dynamic_cast<Model::SingleLineSelection*>(selection.Get()))!=NULL) {
      if (singleLine->HasSelection()) {
        if (singleLine->GetLine()<model->GetRows()) {
          singleLine->SelectLine(singleLine->GetLine()+1);
          MakeVisible(1,singleLine->GetLine());
        }
      }
      else {
        singleLine->SelectLine(1);
        MakeVisible(1,singleLine->GetLine());
      }
    }
    else {
      vAdjustment->IncTop();
    }
  }

  void TableView::ActionStart()
  {
    Model::SingleCellSelection *singleCell;
    Model::SingleLineSelection *singleLine;

    if ((singleCell=dynamic_cast<Model::SingleCellSelection*>(selection.Get()))!=NULL) {
      singleCell->SelectCell(1,singleCell->GetRow());
      MakeVisible(singleCell->GetColumn(),singleCell->GetRow());
    }
    else if ((singleLine=dynamic_cast<Model::SingleLineSelection*>(selection.Get()))!=NULL) {
      singleLine->SelectLine(1);
      MakeVisible(1,singleLine->GetLine());
    }
    else {
      vAdjustment->SetTop(1);
    }
  }

  void TableView::ActionEnd()
  {
    Model::SingleCellSelection *singleCell;
    Model::SingleLineSelection *singleLine;

    if ((singleCell=dynamic_cast<Model::SingleCellSelection*>(selection.Get()))!=NULL) {
      singleCell->SelectCell(header->GetColumns(),singleCell->GetRow());
      MakeVisible(singleCell->GetColumn(),singleCell->GetRow());
    }
    else if ((singleLine=dynamic_cast<Model::SingleLineSelection*>(selection.Get()))!=NULL) {
      singleLine->SelectLine(model->GetRows());
      MakeVisible(1,singleLine->GetLine());
    }
    else {
      vAdjustment->SetTop(vAdjustment->GetTotal()-vAdjustment->GetVisible());
    }
  }

  void TableView::ActionPageUp()
  {
    Model::SingleCellSelection *singleCell;
    Model::SingleLineSelection *singleLine;

    if ((singleCell=dynamic_cast<Model::SingleCellSelection*>(selection.Get()))!=NULL) {
      if (singleCell->HasSelection()) {
        if (singleCell->GetRow()*rowHeight>vAdjustment->GetVisible()) {
          singleCell->SelectCell(singleCell->GetColumn(),
                                 std::max(1,(int)singleCell->GetRow()-(int)(height/rowHeight)-1));
        }
        else {
          singleCell->SelectCell(singleCell->GetColumn(),1);
        }
        MakeVisible(singleCell->GetColumn(),singleCell->GetRow());
      }
      else {
        vAdjustment->PageBack();
      }
    }
    else if ((singleLine=dynamic_cast<Model::SingleLineSelection*>(selection.Get()))!=NULL) {
      if (singleLine->HasSelection()) {
        if (singleLine->GetLine()*rowHeight>vAdjustment->GetVisible()) {
          singleLine->SelectLine(std::max(1,(int)singleLine->GetLine()-(int)(height/rowHeight)-1));
        }
        else {
          singleLine->SelectLine(1);
        }
        MakeVisible(1,singleLine->GetLine());
      }
      else {
        vAdjustment->PageBack();
      }
    }
    else {
      vAdjustment->PageBack();
    }
  }


  void TableView::ActionPageDown()
  {
    Model::SingleCellSelection *singleCell;
    Model::SingleLineSelection *singleLine;

    if ((singleCell=dynamic_cast<Model::SingleCellSelection*>(selection.Get()))!=NULL) {
      if (singleCell->HasSelection()) {
        singleCell->SelectCell(singleCell->GetColumn(),
                               std::min(model->GetRows(),
                                        singleCell->GetRow()+height/rowHeight+1));
        MakeVisible(singleCell->GetColumn(),singleCell->GetRow());
      }
      else {
        vAdjustment->PageForward();
      }
    }
    else if ((singleLine=dynamic_cast<Model::SingleLineSelection*>(selection.Get()))!=NULL) {
      if (singleLine->HasSelection()) {
        singleLine->SelectLine(std::min(model->GetRows(),
                                        singleLine->GetLine()+height/rowHeight+1));
        MakeVisible(1,singleLine->GetLine());
      }
      else {
        vAdjustment->PageForward();
      }
    }
    else {
      vAdjustment->PageForward();
    }
  }

  void TableView::ActionDoubleClick()
  {
    if (doubleClickAction.Valid()) {
      doubleClickAction->Trigger();
    }
  }

  bool TableView::HandleMouseEvent(const OS::MouseEvent& event)
  {
    size_t xc,yc;

    if (!visible || !model.Valid() || !model->IsEnabled()) {
      return false;
    }

    if (event.type==OS::MouseEvent::down && PointIsIn(event)) {
      if (event.button==OS::MouseEvent::button1) {
        if (GetCell(event.x,event.y,xc,yc) && selection.Valid() && yc<=model->GetRows()) {
          px=event.x;
          py=event.y;
          ppos=currentPos;

          selection->SelectCell(xc,yc);
          OnMouseSelection();
          if (GetWindow()->IsDoubleClicked()) {
            ActionDoubleClick();
          }

          my=event.y;
          sy=event.y;
          mouseTimer.SetToNow();
          Lum::OS::display->AddTimer(0,1000000/20,mouseTimerAction);
          return true;
        }
      }
      else if (event.button==OS::MouseEvent::button4) {
        scrollRefresh=false;
        vAdjustment->DecTop();
      }
      else if (event.button==OS::MouseEvent::button5) {
        scrollRefresh=false;
        vAdjustment->IncTop();
      }
    }
    else if (event.type==OS::MouseEvent::move &&
             (event.qualifier & OS::MouseEvent::button1) &&
             event.IsGrabed()) {
      my=event.y;

      if (event.y>=py && (size_t)(event.y-py)>=ppos) {
        vAdjustment->SetTop(1);
      }
      else {
        vAdjustment->SetTop(ppos+(py-event.y));
      }

      return true;
    }
    else if (event.IsGrabEnd()) {
      Lum::OS::display->RemoveTimer(mouseTimerAction);

      Lum::Base::SystemTime now;

      now.Sub(mouseTimer);

      // Happens for unknown reason sometimes under windows!
      if (now.GetTime()==0 && now.GetMicroseconds()==0) {
        return true;
      }

      int step=((sy-event.y)*1000000)/(now.GetMicroseconds()+1000000*now.GetTime());

      if (abs(step)<5) {
        return true;
      }

      if (step>0) {
        scrollRefresh=true;
        vAdjustment->SetTop(vAdjustment->GetTop()+step);
      }
      else if (step<0) {
        scrollRefresh=true;
        if ((size_t)abs(step)<vAdjustment->GetTop()-1) {
          vAdjustment->SetTop(vAdjustment->GetTop()+step);
        }
        else {
          vAdjustment->SetTop(1);
        }
      }
    }

    return false;
  }

  /* -------- model feetback ---------- */

  void TableView::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    const Model::Table::RefreshCell     *refreshCell;
    const Model::Table::RefreshRow      *refreshRow;
    const Model::Table::InsertRow       *insertRow;
    const Model::Table::DeleteRow       *deleteRow;
    const Model::Header::ResizedColumn  *resizedColumn;
    const Model::Header::FitColumn      *fitColumn;
    const Model::Selection::RefreshCell *selRefreshCell;
    const Model::Selection::RefreshRow  *selRefreshRow;
    const Model::Selection::Selected    *selected;

    if ((refreshCell=dynamic_cast<const Model::Table::RefreshCell*>(&msg))!=NULL) {
      RedrawLine(refreshCell->row);
    }
    else if ((refreshRow=dynamic_cast<const Model::Table::RefreshRow*>(&msg))!=NULL) {
      RedrawLine(refreshRow->row);
    }
    else if ((insertRow=dynamic_cast<const Model::Table::InsertRow*>(&msg))!=NULL) {
      scrollRefresh=false;
      Redraw(); // TODO: Better calculate to be redrawn region...
    }
    else if ((deleteRow=dynamic_cast<const Model::Table::DeleteRow*>(&msg))!=NULL) {
      scrollRefresh=false;
      Redraw(); // TODO: Better calculate to be redrawn region...
    }
    else if ((resizedColumn=dynamic_cast<const Model::Header::ResizedColumn*>(&msg))!=NULL) {
      scrollRefresh=false;
      Redraw(); // TODO: Calculate rectangle to get redrawn...
    }
    else if ((fitColumn=dynamic_cast<const Model::Header::FitColumn*>(&msg))!=NULL) {
      FitColumn(fitColumn->column+1);
    }
    else if ((selRefreshCell=dynamic_cast<const Model::Selection::RefreshCell*>(&msg))!=NULL) {
      RedrawLine(selRefreshCell->row);
    }
    else if ((selRefreshRow=dynamic_cast<const Model::Selection::RefreshRow*>(&msg))!=NULL) {
      RedrawLine(selRefreshRow->row);
    }
    else if ((selected=dynamic_cast<const Model::Selection::Selected*>(&msg))!=NULL) {
      OnSelection();
    }
    else { // unknown/empty message
      if (model==this->model) {
        if (selection.Valid() && selection->HasSelection()) {
          selection->OnClear();
        }

        scrollRefresh=false;
        currentPos=1; // We don't want to scroll on a complete refresh...
        vAdjustment->SetTop(1);
        scrollRefresh=false;
        Redraw();
      }
      else if (model==vAdjustment->GetTopModel()) {
        if (!scrollRefresh) {
          currentPos=vAdjustment->GetTop();
        }
        Redraw();
      }
      else if (model==hAdjustment->GetTopModel() || model==selection) {
        Redraw();
      }
      else if (model==scrollTimerAction && scrollTimerAction->IsFinished()) {
        Redraw();
      }
      else if (model==mouseTimerAction && mouseTimerAction->IsFinished()) {
        sy=my;
        mouseTimer.SetToNow();
        Lum::OS::display->AddTimer(0,1000000/20,mouseTimerAction);
      }
      else if (model==updateScrollerAction && updateScrollerAction->IsFinished()) {
        UpdateScroller();
      }
    }
  }

  TableView::Controller::Controller()
  {
    RegisterKeyAction(L"Left",actionLeft);
    RegisterKeyAction(L"Right",actionRight);
    RegisterKeyAction(L"Up",actionUp);
    RegisterKeyAction(L"Down",actionDown);
    RegisterKeyAction(L"Home",actionStart);
    RegisterKeyAction(L"End",actionEnd);
    RegisterKeyAction(L"Prior",actionPageUp);
    RegisterKeyAction(L"Next",actionPageDown);
    RegisterKeyAction(L"Space",actionDoubleClick);
    RegisterKeyAction(L"space",actionDoubleClick);
  }

  bool TableView::Controller::DoAction(Lum::Object* object, Action action)
  {
    TableView *tableView;

    tableView=dynamic_cast<TableView*>(object);

    switch (action) {
    case actionLeft:
      tableView->ActionLeft();
      break;
    case actionRight:
      tableView->ActionRight();
      break;
    case actionUp:
      tableView->ActionUp();
      break;
    case actionDown:
      tableView->ActionDown();
      break;
    case actionStart:
      tableView->ActionStart();
      break;
    case actionEnd:
      tableView->ActionEnd();
      break;
    case actionPageUp:
      tableView->ActionPageUp();
      break;
    case actionPageDown:
      tableView->ActionPageDown();
      break;
    case actionDoubleClick:
      tableView->ActionDoubleClick();
      break;
    default:
      return false;
    }

    return true;
  }
}

